/**
 * $Id: HLServerTrackerTable.java,v 1.2 2001/10/07 04:43:39 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.hotline;

import java.util.Hashtable;
import java.util.Enumeration;
import java.net.InetAddress;
import java.io.IOException;

import redlight.crypto.UnixCrypt;
import redlight.utils.DebuggerOutput;

/**
 * A trackertable maintains a table of tracker hostnames / IP addresses 
 * and (optionally) passwords.
 */
public class HLServerTrackerTable extends HLServerComponent {
    
    protected Hashtable table;

    /**
     * Creates an empty trackertable.
     */
    public HLServerTrackerTable(HLServer h) {

        super(h);        
        table = new Hashtable();
        
    }
    
    /**
     * Returns an enumeration of the elements in the bantable.
     * Each element has type {@link HLServerTrackerTable.TrackerRecord}.
     */
    public Enumeration elements() {

        checkModified();
        return table.elements();
        
    }
    
    /**
     * Checks whether the bantable contains a given IP address.
     * @param ip the IP address.
     * @return true if the address is in the bantable.
     */
    public boolean contains(InetAddress ip) {

        checkModified();
        return table.containsKey(ip.getHostAddress());
        
    }

    /**
     * Inserts / replaces an address to the trackertable.
     * @param ip the address to insert / replace.
     * @param password the password (may be null).
     * @throws IllegalArgumentException if ip is null.  
     */
    public void put(InetAddress ip, String password) {
        
        if(ip == null)
            throw new IllegalArgumentException("ip == null");
        
        TrackerRecord t = new TrackerRecord(ip, password);
        table.put(ip.getHostAddress(), t);
        hasChanged();
        
    }
    
    /**
     * Removes an address from the tracker table.
     * @param ip the address to remove.
     * @throws IllegalArgumentException if ip is null.
     */
    public void remove(InetAddress ip) {
    
        if(ip == null)
            throw new IllegalArgumentException("ip == null");

        if(table.containsKey(ip.getHostAddress())) {

            table.remove(ip.getHostAddress());
            hasChanged();
            
        }
        
    }

    /**
     * Broadcasts the server information to all trackers in the
     * table.
     */
    public synchronized void broadcast() {

        checkModified();

        for(Enumeration en = table.elements(); en.hasMoreElements(); ) {
            
            TrackerRecord t = (TrackerRecord) en.nextElement();

            try {

                new HLTracker(t.address, HLProtocol.HTRK_UDPPORT).sendServerInfo(hls.getPort(), hls.getServerName(), hls.getServerDescription(), (short) hls.getClients().length, t.password);
                DebuggerOutput.debug("HLServerTrackerTable.broadcast: sent server info to " + t.address.getHostName());

            } catch(IOException e) {

                hls.log("Error connecting to tracker " + t.address.toString() + " (" + e.toString() + ")");

            }
            
        }
        
    }        

    public String toString() {

        Enumeration en = table.elements();

        String s = "HLServerTrackerTable[";

        while(en.hasMoreElements())
            s += en.nextElement().toString() + ", ";
        
        return s + "]";

    }

    /**
     * Called whenever the bantable changes. Subclasses can override
     * this method to implement e.g. persistent storage.  
     */
    public void hasChanged() {}

    /** 
     * Called whenever the latest version of the agreement is required.
     * Can be overridden by subclasses to implement persistent storage.
     */    
    public void checkModified() {}

    /**
     * Encapsulates IP address and time.
     */
    public class TrackerRecord {
       
        /* The IP address (read-only). */

        public InetAddress address;

        /* Time in seconds since the epoch (read-only). */

        public String password;
        
        public TrackerRecord(InetAddress a, String p) {
            
            address = a;
            password = p;
            
        }
        
        public String toString() {

            return "TrackerRecord[address = " + address.getHostName() + ", password = " + password + "]";

        }
        
    }

}
