package redlight.hotline.iconlib;
import java.util.Vector;
import redlight.hotline.iconlib.*;
import redlight.hotline.iconlib.HLExtractor;
import redlight.hotline.iconlib.HLUtils;

/*
                Read the license.txt file for license information!
                If you don't have it, get it at www.MarkusStengel.de.
*/

/**
 *          data structure where all the important icon information data is stored
 *          @author     Markus Stengel
 *          @see          <a href="http://www.MarkusStengel.de">developer site</a>
 */
public class HLIconData {

     int          width;
     int          height;
     int          widthInBytes;
     int          palLengthBytes;
     int          palLengthLW;
     int          transparency;
     byte[][]     palette;
     byte[]       rawData;
     int[]        data;
     int          datCounter;
     int          datSize;
     
     public HLIconData() {
             datCounter = 0;
             datSize = 0;
     }
     
     /**
      *     allocate the memory necessary for the HLRGB objects
      *        @param size     number of HLRGB units for which the memory should be allocated
      */                
     public void malloc(int size) {
             datSize = size;
             data = new int[datSize];
     }
     
     /** 
      *        add a new RGB value of a HLRGB object to the data-field
      *        @param h      HLRGB object to be added
      *        @see            HLRGB
      *        @return      true when everything went alright, false otherwise 
      */
     public boolean add (HLRGB h) {
             if ((data == null) || (datCounter>=datSize)) return false;
             data[datCounter] = h.get();
             datCounter++;
             return true;
     }
     /** 
      *        add a new RGB entry to the data-field
      *        @param RGB     int with red in bits 23-16, green 15-8, blue 7-0 
      *        @return      true when everything went alright, false otherwise 
      */
     public boolean add (int RGB) {
             if ((data == null) || (datCounter>=datSize)) return false;
             data[datCounter] = RGB;
             datCounter++;
             return true;
     }

     /** 
      *        add a new RGB entry to the dat-field
      *        @param RGB     int with red in bits 23-16, green 15-8, blue 7-0 
      *        @return      true when everything went alright, false otherwise 
      */
     public boolean add (byte red, byte green, byte blue) {
             if ((data == null) || (datCounter>=datSize)) return false;
             int val = 0;
             val     = (HLUtils.signedToUnsigned(red)      << 16) |
                         (HLUtils.signedToUnsigned(green)     << 8) |
                      (HLUtils.signedToUnsigned(blue));
             data[datCounter] = val;
             datCounter++;
             return true;
     }
     
     /**
      *         get the desired HLRGB object
      *        @param index        index of the HLRGB object in the dat-field, starting at index 0 
      *        @return              HLRGB object, null if failure 
      */
     public HLRGB get (int index) {
             if ((index>=datCounter) || (index>=datSize) || (data == null)) return null;
             return new HLRGB(data[index]);
     }

     /**
      *         get the desired RGB as an integer
      *        @param index        index of the RGB value in the data-field, starting at index 0 
      *        @return              RGB value, <b>-1</b> if an error occured 
      */
     public int getRGB (int index) {
             if ((index>=datCounter) || (index>=datSize) || (data == null)) return -1;
             return data[index];
     }

     /**
      *         get the desired RGB as an integer, including the alpha channel
      *        @param index        index of the RGB value in the data-field, starting at index 0 
      *        @return              AlphaRGB value 
      */
     public int getARGB (int index) {     
             if ((index>=datCounter) || (index>=datSize) || (data == null)) return 0; // this should be changed to exception
             int item = data[index];
             if (getTransparency() != item) item = item | 255 << 24;
             return  item;
     }
     
     /**
      *         get RGB data including the alpha channel, whereas the color-byte-order is: Alpha|Red|Green|Blue
      *        @return           ARGB data in an int[] field, null if there was an error
      */
     public int[] getARGBField() {
             if ((datCounter==0) || (datSize==0) || (data == null)) return null;
             int[] argbField = new int[datCounter];
             for (int i=0; i<datCounter; i++) argbField[i] = getARGB(i); 
             return argbField;             
     }
     /**
      *         get RGB data , whereas the color-byte-order is: *|Red|Green|Blue
      *        @return           RGB data in an int[] field, null if there was an error
      */
     public int[] getRGBField() {
             if ((datCounter==0) || (datSize==0) || (data == null)) return null;
             return data;             
     }
     /**
      *          returns the icon width in pixel
      *          @return               width in pixel
      */
     public int               getWidth()                              { return width; }
     /**
      *          returns the icon height in pixel
      *          @return               height in pixel
      */
     public int               getHeight()                              { return height; }
     /**
      *          returns the icon width in bytes
      *          @return               width in bytes
      */
     public int               getWidthInBytes()                    { return widthInBytes; }
     /**
      *          returns the palette length in bytes, note: In order to receive the number of palette
      *          entries you need to divide this value by 4 (four) since they are aligned to 4-byte-boundaries,
      *          or you can simply use the <a href="#getPalLengthLW()">getPalLengthLW()</a> method
      *          @return               palette length in bytes
      */
     public int               getPalLengthBytes()               { return palLengthBytes; }
     /**
      *          returns the palette length in longwords, identical to the number of palette entries
      *          @return               palette length in longwords
      */
     public int               getPalLengthLW()                    { return palLengthLW; }
     /**
      *          Returns the color used for transparency as RGB in big endian format.
      *          You should use this function and not access transparency directly since values are stored 
      *          shifted 8 bits by the hotline app
      *          @return               transparency color as RGB in big endian format
      */
     public int               getTransparency()                    { return transparency>>8; }
     /**
      *          get access to the palette as a two dimensional field, where the first dimension is the entry number
      *          and the second dimension has red, green and blue at indices 0, 1, 2<br>
      *          this example extracts the red value from the 16th palette color:
      *          <pre>
                     ...
                     HLIconData hlid = new HLIconData();
                    ...
                    byte     red = hlid.getPalette()[15][0];
                    ...
      *          </pre>
      *          @return               RGB palette
      */
     public byte[][]     getPalette()                         { return palette; }

     /**        returns the red component of the RGB entry specified by index
      *        @param index        index in the data-field, starting at 0       
      *        @return red component, <b>-1</b> on error
      */
     public byte getRed(int index) {     
             if ((index>=datCounter) || (index>=datSize) || (data == null)) return -1;
             return (byte) HLUtils.intToBytes(data[index])[1];
     }

     /**        returns the green component of the RGB entry specified by index
      *        @param index        index in the data-field, starting at 0       
      *        @return green component, <b>-1</b> on error
      */
     public byte getGreen(int index) {     
             if ((index>=datCounter) || (index>=datSize) || (data == null)) return -1;
             return (byte) HLUtils.intToBytes(data[index])[2];
     }
     /**        returns the blue component of the RGB entry specified by index
      *        @param index        index in the data-field, starting at 0       
      *        @return blue component, <b>-1</b> on error
      */
     public byte getBlue(int index) {     
             if ((index>=datCounter) || (index>=datSize) || (data == null)) return -1;
             return (byte) HLUtils.intToBytes(data[index])[3];
     }


     /**
      *          get access to raw data: all pixels are stored as 1 byte entries, their values indicate their palette color entry
      *          @return               raw data
      */
     public byte[]          getRawData()                         { return rawData; }

     /**
      *          get access to the int[] data field
      *          @return               data-field
      */
     public int[] getData() { return data;     }
}