/**
 * $Id: ScriptBroker.java,v 1.1.1.1 2001/07/22 02:44:48 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.script;

import java.util.Vector;
import java.util.BitSet;
import redlight.utils.DebuggerOutput;

/**
 * A ScriptBroker mediates between Scriptable objects.
 * First, a Scriptable objects must register itself
 * with a ScriptBroker. Then it uses the ScriptBroker to
 * send messages to another registered object. The 
 * ScriptBroker can also broadcast a message to all 
 * registered objects that can understand a particular kind of 
 * message.
 */
public class ScriptBroker {
    Vector targets;

    class Target {
	public Scriptable target;
	public long knowsMessages;
	public Target(Scriptable s, long m) {
	    target = s;
	    knowsMessages = m;
	}
    }

    /**
     * Constructs a ScriptBroker.
     */
    public ScriptBroker() {
	targets = new Vector();
    }

    /**
     * Adds a scripting target.
     * @param s the scripting target
     */
    public void addTarget(Scriptable s) {
	targets.addElement(new Target(s, s.getKnownMessages()));
	s.gotTarget(this);
    }

    /**
     * Removes a scripting target.
     * @param s the scripting target
     */
    public synchronized void removeTarget(Scriptable s) {
	Target t = getTarget(s);
	if(t != null) {

	    targets.removeElement(t);
	    s.lostTarget(this);

	} else {

	    DebuggerOutput.tell("ScriptBroker: illegal target remove for "+s.toString());
	    
	}
    }

    private Target getTarget(Scriptable s) {
	for(int i=0; i<targets.size(); i++) 
	    if(((Target) targets.elementAt(i)).target == s)
		return (Target) targets.elementAt(i);

	return null;
    }
    
    /**
     * Send a message to a specified target.
     * @param t the target object
     * @param s the message
     * @return the result
     */
    public ScriptResult executeScript(Scriptable t, ScriptObject s) 
	throws UnknownMessageException {
	ScriptResult r = t.executeScript(s);
	return r;
    }

    /**
     * Broadcast a message to all targets that can
     * understand the message.
     * @param s the message
     * @return an array of results
     * @throws UnknownMessageException when the message was not understood
     * by any targets.
     */
    public ScriptResult[] executeScript(ScriptObject s) 
	throws UnknownMessageException {
	ScriptResult[] sTemp = new ScriptResult[targets.size()];
	int j=0;
	for(int i=0; i<targets.size(); i++) {
	    try {
		ScriptResult result = 
		    executeScript((Target) targets.elementAt(i), s);
		sTemp[j++] = result;
	    } catch (UnknownMessageException ex) {
	    }
	}
	if(j == 0) 
	    throw new UnknownMessageException(s.toString());

	ScriptResult[] sr = new ScriptResult[j];
	for(int i=0; i<j; i++) 
	    sr[i] = sTemp[i];
	return sr;
    }

    private ScriptResult executeScript(Target t, ScriptObject s) 
	throws UnknownMessageException {
	if((t.knowsMessages & s.type) == s.type) 
	    return executeScript(t.target, s);	   
    
	throw new UnknownMessageException(s.toString());
    }    
}










