/**
 * $Id: Console.java,v 1.16 2001/10/10 03:03:21 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.server;

import java.net.ServerSocket;
import java.net.Socket;
import java.net.InetAddress;
import java.net.URL;
import java.net.InetAddress;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.DataOutputStream;
import java.io.OutputStream;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.LineNumberReader;
import java.io.InputStreamReader;
import java.io.File;
import java.util.StringTokenizer;
import java.util.Date;
import java.util.Hashtable;
import java.util.Enumeration;

import redlight.utils.InterruptableInputStream;
import redlight.utils.DebuggerOutput;
import redlight.utils.BytesFormat;
import redlight.utils.TimeFormat;
import redlight.utils.TextUtils;
import redlight.utils.HTMLStripperInputStream;
import redlight.hotline.HLProtocol;
import redlight.hotline.HLServer;
import redlight.hotline.HLServerDispatcher;
import redlight.hotline.HLServerAccountsTable;
import redlight.crypto.UnixCrypt;

/**
 * The  administrator console.
 */
public class Console extends Thread {

    boolean interrupted = false;
    String consolePassword;
    ServerSocket serverSocket;
    Socket client;
    HLServer hls;
    DataOutputStream outputStream;
    LineNumberReader lineReader;
    File initFile;

    Command[] commands = {

        new Command("help [CMD]", "show this message") {
                
                public void perform(String[] args) throws IOException {
                    
                    if(args.length > 1) {

                        Command c = resolveCommand(args[1], false);

                        if(c != null) {

                            writeln(c.command);

                            if(c.command.indexOf(' ') != -1)
                                args[1] = c.command.substring(0, c.command.indexOf(' '));
                            
                            else 
                                args[1] = c.command;

                        }

                        URL u = ClassLoader.getSystemResource("redlight/Documentation/rld/" + args[1] + ".ihtml");

                        if(u == null) {

                            writeln("No help found for '" + args[1] + "'.");
                            return;

                        }

                        try {

                            LineNumberReader lr = new LineNumberReader(new InputStreamReader(new HTMLStripperInputStream(u.openStream())));

                            for(String line = lr.readLine(); line != null; line = lr.readLine())
                                writeln(line);

                            if(args[1].equals("set")) {

                                writeln("List of valid NAMEs:");

                                Enumeration en = Main.symbolTable.getSymbols();

                                String symbol;

                                while(en.hasMoreElements()) {

                                    symbol = (String) en.nextElement();

                                    writeln(TextUtils.pad(symbol, 25) + ": " + Main.symbolTable.getDescription(symbol) + ".");

                                }

                            }

                        } catch(IOException e) {}
                            
                    } else {

                        write(Main.copyright);
                        writeln("Commands:");
                        writeln();
                        
                        for(int i = 0; i < commands.length; i++)
                            writeln("   " + TextUtils.pad(commands[i].command, 20) + " " + commands[i].description + ".");
                        
                        writeln();
                        writeln("Commands can be abbreviated to their shortest unique prefix.");
                        writeln();
                        
                    }

                }

            },
        
        new Command("adduser LOGIN [PASS [PRIV [NICK]]]", "create new account") {
                
                public void perform(String[] args) throws IOException {
                    
                    if(args.length < 2) {
                        
                        writeln("Must specify LOGIN.");
                        return;

                    }

                    String l = args[1];
                    String p = "guest";
                    long pr = 0;
                    String n = l;

                    if(args.length > 2) {

                        p = args[2];
                        
                        if(args.length > 3) {
                            
                            try {
                                
                                pr = Long.parseLong(args[3]);
                                
                            } catch(NumberFormatException e) {
                                
                                writeln("'" + args[3] + "' is not a number.");
                                return;
                                
                            }
                            
                            if(args.length > 4)
                                n = args[4];
                            
                        }                            
                        
                    }

                    try {

                        HLServerAccountsTable accountsTable = hls.getAccountsTable();
                        
                        accountsTable.lock();
                        
                        if(!accountsTable.exists(l)) {

                            HLProtocol.AccountInfo newAccount = Main.hlp.new AccountInfo(l, n, UnixCrypt.crypt("aa", p), pr, null);
                
                            String accountCreateAllowed = hls.getHLServerPolicy().approveAccountCreate(newAccount);
                
                            if(accountCreateAllowed == null) {

                                accountsTable.put(l, newAccount);
                                writeln("Created account '" + l + "'.");

                            } else {
                    
                                writeln("Could not create account '" + l + "': " + accountCreateAllowed);

                            }
                
                        } else {
                
                            writeln("Could not create account '" + l + "' because it already exists.");
                
                        }

                        accountsTable.unlock();

                    } catch(InterruptedException e) {

                        writeln("Could not lock accounts table.");

                    }

                }

            },
        
        new AccountCommand("deluser LOGIN...", "delete account", true) {
                
                public void perform(HLProtocol.AccountInfo account) throws IOException {
                    
                    HLServerAccountsTable accountsTable = hls.getAccountsTable();
                    String l = account.login;
                    
                    if(accountsTable.exists(l)) {

                        String accountDeleteAllowed = hls.getHLServerPolicy().approveAccountDelete(accountsTable.get(l));
                    
                        if(accountDeleteAllowed == null) {
                            
                            accountsTable.remove(l);
                            writeln("Deleted account '" + l + "'.");
                            
                        } else {
                            
                            writeln("Could not delete account '" + l + "': " + accountDeleteAllowed);
                        
                        }

                    } else {

                        writeln("Could not delete account '" + l + "' because it does not exist.");

                    }

                }
                        
            },
        
        new AccountCommand("listusers [LOGIN...]", "list accounts", false) {

                public void perform(String args[]) throws IOException {

                    writeln(" login             | nick              | privileges    | home directory");
                    writeln("-------------------+-------------------+---------------+------------------");
                    super.perform(args);

                }

                public void perform(HLProtocol.AccountInfo account) throws IOException {
                    
                    String row = " " +
                        TextUtils.pad(account.login, 16);

                    row += " | ";

                    row += TextUtils.pad(account.nick, 16);

                    row += " | ";

                    row += TextUtils.pad(new Long(account.privileges).toString(), 12);

                    row += " | ";

                    if(account.homeDirectory == null) 
                        row += "<default>";
                    else
                        row += account.homeDirectory;

                    writeln(row);
                    
                }
                
            },
        
        new Command("status", "show server status") {
                
                public void perform(String[] args) throws IOException {
                    
                    writeln("Settings (see also 'help set'):");
                    writeln("");
                    String symbol = null;

                    Enumeration en = Main.symbolTable.getSymbols();

                    while(en.hasMoreElements()) {

                        symbol = (String) en.nextElement();
                        writeln(TextUtils.pad(Main.symbolTable.getDescription(symbol), 25) + ": " + Main.symbolTable.get(symbol));
                        
                    }
                    
                    writeln("");
                    writeln("Running totals:");
                    writeln("");
                    writeln("Since                     : " + new Date(hls.uptime) + " (" + TimeFormat.format((System.currentTimeMillis() - hls.uptime) / 1000) + ")");
                    writeln("Currently connected       : " + hls.getClients().length);
                    writeln("Downloads in progress     : " + hls.downloadsInProgress);
                    writeln("Uploads in progress       : " + hls.uploadsInProgress);
                    writeln("Downloads in queue        : " + hls.downloadsInQueue);
                    writeln("Uploads in queue          : " + hls.uploadsInQueue);
                    writeln("Connection peak           : " + hls.connectionPeak);
                    writeln("Connection counter        : " + hls.connectionCounter);
                    writeln("Download counter          : " + hls.downloadCounter);
                    writeln("Upload counter            : " + hls.uploadCounter);

                }

            },
        
        new Command("set NAME [VALUE]", "set or unset parameter") {
                
                public void perform(String[] args) throws IOException {
                    
                    if(args.length < 2) {
                        
                        writeln("NAME omitted.");
                        return;
                        
                    }
                    
                    String newValue = args.length == 2 ? 
                        null : Console.glueArgs(2, args).trim();

                    writeln(Main.symbolTable.put(args[1].trim(), 
                                                 newValue));
                    
                }
                
            },

        new Command("xfkill XFREF", "kill transfer") {
                
                public void perform(String[] args) throws IOException {
                    
                    if(args.length < 2) {

                        writeln("Must specify XFREF.");
                        return;

                    }

                    int xfref;

                    try {

                        xfref = (int) Long.decode("0x" + args[1]).intValue();

                    } catch(NumberFormatException e) {

                        writeln("Not a number: " + e.getMessage());
                        return;

                    }

                    HLServer.TransferRequest tr;

                    Enumeration en = 
                        hls.getTransferRequests();

                    while(en.hasMoreElements()) {
                        
                        tr = (HLServer.TransferRequest) en.nextElement();

                        if(tr.ref == xfref) {
                        
                            hls.getTransferQueue().destroy(tr.ref);

                            writeln("Killed " + tr.local.getFile().getName() + ".");
                        
                            return;

                        }
                        
                    }
                    
                }
                
            },

        new Command("xflist", "list transfers") {
                
                public void perform(String[] args) throws IOException {

                    writeln("    xfref | type |  sock | state | progress        | filename");
                    writeln("----------+------+-------+-------+-----------------+-------------");


                    HLServer.TransferRequest tr;

                    Enumeration en = 
                        hls.getTransferRequests();

                    while(en.hasMoreElements()) {
                        
                        tr = (HLServer.TransferRequest) en.nextElement();

                        String row = " " + TextUtils.prepad(Integer.toHexString(tr.ref), 7) + " | ";
                        
                        if(tr.type == tr.FILE_UPLOAD)
                            row += "up  ";
                        else
                            row += "down";

                        row += " | ";

                        if(tr.client != null) {

                            HLProtocol.UserListComponent u = tr.client.getUser();
                            row += TextUtils.prepad(new Integer(u.sock).toString(), 4);

                        } else {

                            row += "  ???";

                        }
                        
                        row += " | ";

                        if(tr.queuePosition == 0 && !tr.lock) 
                            row += "wait ";
                        else if(tr.queuePosition == 0 && tr.lock)
                            row += "xfer ";
                        else
                            row += "queue";

                        row += " | ";
                        
                        if(tr.queuePosition == 0 && !tr.lock) {

                            row += "               ";

                        } else if(tr.queuePosition == 0 && tr.lock) {
                            
                            if(tr.progressDone > 0)
                                row += TextUtils.prepad("" + (int) (100 * ((double) tr.progressDone / tr.totalSize)), 2);
                            else
                                row += "  0";
                            
                            row += "% / " + TextUtils.prepad(BytesFormat.format(tr.totalSize), 7);
                            
                        } else
                            row += TextUtils.prepad("#" + TextUtils.prepadCharacter(new Integer(tr.queuePosition).toString(), 2, '0'), 14);
                        
                        row += " | ";
                        row += tr.local.getFile().getName();

                        writeln(row);
                        
                        if(DebuggerOutput.on)
                            writeln(tr.toString());

                    }
                    
                }
                
            },

        new Command("shutdown [MSG]", "takes thee server down") {
                
                public void perform(String[] args) throws IOException {
                    
                    String reason = Console.glueArgs(1, args);
                    throw new ShutdownException(reason);
                    
                }

            },
        
        new Command("quit [MSG]", "quit console (shutdown if console on stdin)") {
                
                public void perform(String[] args) throws IOException {
                    
                    String reason = Console.glueArgs(1, args);
                    throw new TerminateSessionException(reason);
                    
                }

            },

        new Command("version", "show program version") {
                
                public void perform(String[] args) throws IOException {
                    
                    writeln();
                    writeln(Main.getVersionString());
                    
                }
                
            },
        
        new Command("message MSG", "broadcast administrator message") {
                
                public void perform(String[] args) throws IOException {
                    
                    String msg = Console.glueArgs(1, args);
                    
                    if(msg == null) {
                        
                        writeln("Won't send empty message.");

                    } else {

                        writeln("Broadcasting: '" + msg + "'");
                        hls.broadcastAdministratorMessage(msg);

                    }
                    
                }
                
            },
        
        new SocketCommand("clients [SOCK...]", "show connections", false) {
                
                public void perform(HLServerDispatcher client) throws IOException {
                    
                    HLProtocol.UserListComponent u = client.getUser();

                    String row =
                        TextUtils.prepad(new Integer(u.sock).toString(), 5) + " | " +
                        TextUtils.prepad(new Integer(u.icon).toString(), 5) + " | " +
                        TextUtils.pad(client.client.getInetAddress().getHostAddress(), 14) + " | ";

                    char[] flags = new char[5];
                    flags[0] = ' ';
                    flags[1] = ' ';
                    flags[2] = ' ';
                    flags[3] = ' ';
                    flags[4] = ' ';

                    if((u.clr & HLProtocol.UserListComponent.HAS_BEEN_IDLE) ==
                       HLProtocol.UserListComponent.HAS_BEEN_IDLE)
                        flags[0] = 'I';

                    if((u.clr & HLProtocol.UserListComponent.CAN_DISCONNECT_USERS) ==
                       HLProtocol.UserListComponent.CAN_DISCONNECT_USERS)
                        flags[1] = 'A';

                    row += new String(flags) + " | " + u.nick;

                    writeln(row);
                    
                }
                
                public void perform(String args[]) throws IOException {
                    
                    writeln("  sock |   icon | address         | flags | nick");
                    writeln("-------+--------+-----------------+-------+------------");
                    super.perform(args);
                    
                }
                
            },
        
        new SocketCommand("info SOCK...", "get info on clients", true) {
                
                public void perform(HLServerDispatcher client) throws IOException {
                    
                    writeln(hls.getUserInfo(client.sock));
                    
                }
                
            },
        
        new SocketCommand("kick SOCK...", "kick user", true) {
                
                public void perform(HLServerDispatcher client) throws IOException {
                    
                    if(hls.kickUser(client.sock, false))
                        writeln("Kicked " + client.sock);
                    else
                        writeln("Can't kick " + client.sock);
                    
                }
                
            },
        
        new SocketCommand("ban SOCK...", "ban user", true) {
                
                public void perform(HLServerDispatcher client) throws IOException {
                    
                    if(hls.kickUser(client.sock, true))
                        writeln("Banned " + client.sock);
                    else
                        writeln("Can't ban " + client.sock);
                    
                }
                
            },

        new Command("debug", "enable / disable debug spew") {
                
                public void perform(String[] args) throws IOException {
                    
                    String msg = Console.glueArgs(1, args);

                    Main.DebugOutputEnabled = !Main.DebugOutputEnabled;
                    DebuggerOutput.setEnabled(Main.DebugOutputEnabled);
                    write("Debugging output ");
                    writeln(Main.DebugOutputEnabled ? "enabled." : "disabled.");

                }
                
            },

        
    };

    /**
     * Creates the console and runs the init file.
     */
    Console(HLServer hls, File initFile) {

        consolePassword = null;
        this.hls = hls;
        this.setName("Console " + serverSocket);
        this.initFile = initFile;
        start();

    }

    public synchronized void run() {

        DebuggerOutput.debug("Console starting ...");

        try {

            if(initFile.isFile() && initFile.exists() && initFile.canRead()) {
                
                DebuggerOutput.debug("Console parsing init file ...");
                FileInputStream fis = new FileInputStream(initFile);
                setIO(fis, null);
                prompt();
                DebuggerOutput.debug("Console parsing init file: done.");

            }

            while(!interrupted && 
                  Main.symbolTable.get("admin.console") != null) {
                    
                try {
                    
                    setupIO();
                
                    if(auth())
                        prompt();
                    
                } catch(IOException e) {

                    try {

                        if(e.getMessage() != null)
                            writeln(e.getMessage());

                    } catch(IOException _e) {}

                    DebuggerOutput.debug("Exception in auth prompt loop:");
                    DebuggerOutput.stackTrace(e);
                    
                    Main.logLine("Closing administrator console connection.");
                    
                    if(e instanceof ShutdownException)
                        throw e;
                    
                    /* Terminate server on errors if console on stdin. */

                    if(Main.symbolTable.get("admin.console") != null) {

                        if(serverSocket == null && Main.symbolTable.get("admin.console").equals("stdin")) {
                            
                            throw new ShutdownException(e.getMessage());
                            
                        }

                        terminateSession();
                            
                    }

                }

                DebuggerOutput.debug("Console: interrupted = " + interrupted);

            }
            
        } catch(IOException e) {

            DebuggerOutput.debug("Exception in large loop:");
            DebuggerOutput.stackTrace(e);
            terminateServer(e.getMessage());

        }

        if(Main.symbolTable.get("admin.console") == null && !interrupted) {

            try {

                writeln("Disabling console ...");

            } catch(IOException e) {}

            terminateSession();

            Main.logLine("Administrator console disabled.");

            try {

                wait();

            } catch(InterruptedException e) {}

        }

        DebuggerOutput.debug("Console exiting");

    }
    
    void setupIO() throws IOException {

        if(Main.symbolTable.get("admin.console") == null)
            Main.symbolTable.put("admin.console", "stdin");
        
        if(Main.symbolTable.get("admin.console").equals("stdin")) {
            
            terminateSession();

            try {

                setIO(System.in, System.out);
            
            } catch(IOException e) {
            
                throw new TerminateSessionException(e.getMessage());

            }

        } else {
            
            String hostPort = Main.symbolTable.get("admin.console");

            String host = hostPort;
            int port = hls.getPort() + 2;

            if(hostPort.lastIndexOf(':') != -1) {
                
                /* Get port. */
                
                try {
                    
                    port = Integer.parseInt(hostPort.substring(hostPort.lastIndexOf(':') + 1).trim());
                    
                } catch(NumberFormatException e) {
                    
                    throw new TerminateSessionException("admin.console property has illegal value " + e.getMessage());

                }

                /* Get host. */

                host = hostPort.substring(0, hostPort.lastIndexOf(':'));
                
            }
            
            InetAddress address = InetAddress.getByName(host);
            
            Main.logLine("Administrator console on " + address + ", port " + port);

            if(serverSocket == null)
                serverSocket = new ServerSocket(port, 0, address);
            
            tcpAccept();
            setIO(client.getInputStream(), client.getOutputStream());
            
        }

    }

    void tcpAccept() throws IOException {

        while(!interrupted) {

            try {

                serverSocket.setSoTimeout(5000);
                client = serverSocket.accept();
                client.setSoTimeout(10000);
                Main.logLine("Accepted administrator console connection.");
                return;

            } catch(InterruptedIOException e) {}

        }

        throw new TerminateSessionException("interrupted");

    }

    void setIO(InputStream input, OutputStream output) throws IOException {
        
        lineReader = new LineNumberReader(new InputStreamReader(input));

        if(output == null)
            outputStream = null;
        else
            outputStream = new DataOutputStream(output);

    }

    void terminateSession() {

        try {

            if(serverSocket != null) {

                serverSocket.close();
                serverSocket = null;

            }

        } catch(IOException e) {}

        try {

            if(client != null) {

                client.close();
                client = null;

            }

        } catch(IOException e) {}

    }

    void terminateServer(final String reason) {

        try {

            writeln("Shutting down ...");

        } catch(IOException e) {}

        hls.shutdown(reason);
        hls = null;
        
    }

    public void writeln() throws IOException {

        if(outputStream != null)
            outputStream.write(new String("\n").getBytes());

    }

    public void writeln(String s) throws IOException {

        if(outputStream != null)
            outputStream.write(new String(s + "\n").getBytes());

    }

    public void write(String s) throws IOException {

        if(outputStream != null)
            outputStream.write(s.getBytes());

    }

    boolean auth() throws IOException {
        
        String cpw = Main.symbolTable.get("admin.password");

        Main.logLine("Authenticating login from " + (client != null ? client.getInetAddress().toString() : "stdin") + " ...");

        if(cpw == null) {

            Main.logLine("No authentication required.");
            return true;

        }

        write("Password: ");
        String pw = lineReader.readLine();        
        
        if(pw == null)
            throw new TerminateSessionException("didn't get password");

        if(pw.equals(cpw)) {
            
            Main.logLine("Successfully authenticated.");
            return true;
            
        } else {
            
            Main.logLine("Authentication failed.");
            
        }

        try {

            Thread.currentThread().sleep(4000);

        } catch(InterruptedException e) {}

        throw new TerminateSessionException("Authentication failed.");
        
    }

    void prompt() throws IOException {

        if(client != null)
            client.setSoTimeout(5 * 60000);

        writeln("Welcome to rld.");
        writeln("Type 'help' for help.");
        writeln("");
        write("> ");
        
        for(String line = lineReader.readLine(); 
            line != null; 
            line = lineReader.readLine()) {
            
            DebuggerOutput.debug("Console.prompt: read line: " + line);
            doCommand(line);
            write("> ");
            
        }

    }

    void doCommand(String line) throws IOException {

        line.trim();

        /* Skip comments and empty lines. */

        if(line.startsWith("#") || line.equals(""))
            return;

        StringTokenizer st = new StringTokenizer(line);
        String args[] = new String[st.countTokens()];

        if(args.length == 0)
            return;

        for(int i = 0; st.hasMoreTokens(); i++)
            args[i] = st.nextToken();
            
        Command c = resolveCommand(args[0], true);

        if(c != null) {

            DebuggerOutput.debug("Console.doCommand: executing: " + line);
            c.perform(args);
            
        }

    }

    Command resolveCommand(String c, boolean complain) throws IOException {

        String command = c;

        int matches = 0;
        int matchPos = -1;

        for(int i = 0; i < commands.length; i++) {
            
            if(commands[i].command.startsWith(command)) {

                matches++;
                matchPos = i;

            }

        }
                
        if(matches == 0) {

            if(complain)
                writeln("Unknown command. Try help.");

        } else if(matches == 1) {

            return commands[matchPos];

        } else {

            if(complain)
                writeln("Ambiguous command. Try help.");

        }

        return null;

    }

    static String glueArgs(int from, String[] args) {
        
        String tmp = "";
        
        for(int i = from; i < args.length; i++)
            tmp += args[i] + " ";
        
        return tmp.equals("") ? null : tmp.trim();
        
    }

    abstract class AccountCommand extends Command {

        private boolean accountsRequired;

        AccountCommand(String command, 
                       String description,
                       boolean accountsRequired) {
            
            super(command, description);
            this.accountsRequired = accountsRequired;

        }

        public abstract void perform(HLProtocol.AccountInfo client) throws IOException;

        public void perform(String[] args) throws IOException {
                        
            if(args.length == 1 && accountsRequired) {
                
                writeln("'" + command + "' requires account arguments. Try help.");
                return;
                
            }
            
            try {
                
                HLServerAccountsTable accountsTable = 
                    hls.getAccountsTable();
                
                accountsTable.lock();
                
                HLProtocol.AccountInfo[] accounts = 
                    accountsTable.list();
                
                for(int i = 0; i < accounts.length; i++) {
                    
                    if(args.length == 1) {
                        
                        /* Walk all accounts. */
                        
                        perform(accounts[i]);
                        
                    } else {
                        
                        /* Walk specific accounts. */
                        
                        for(int j = 1; j < args.length; j++)
                            if(args[j].equals(accounts[i].login))
                                perform(accounts[i]);
                        
                    }
                    
                }
                
                accountsTable.unlock();
                
            } catch(InterruptedException e) {
                
                writeln("Could not lock accounts table.");
                
            }
            
        }

    }

    abstract class SocketCommand extends Command {

        private boolean socketsRequired;

        SocketCommand(String command, 
                      String description,
                      boolean socketsRequired) {
        
            super(command, description);
            this.socketsRequired = socketsRequired;

        }
        
        public abstract void perform(HLServerDispatcher client) throws IOException;

        public void perform(String[] args) throws IOException {
            
            if(args.length == 1 && socketsRequired) {
                
                writeln("'" + command + "' requires socket arguments. Try help.");
                return;
                
            }
            
            HLServerDispatcher clients[] = hls.getClients();
            
            try {
                
                for(int i = 0; i < clients.length; i++) {
                    
                    if(clients[i].sock == -1)
                        continue;

                    if(args.length == 1) {
                        
                        /* Walk over all sockets. */
                        
                        perform(clients[i]);
                        
                    } else {
                        
                        /* Walk over specific sockets. */
                        
                        for(int j = 1; j < args.length; j++)
                            if(Integer.parseInt(args[j]) == 
                               clients[i].sock)
                                perform(clients[i]);
                        
                    }
                    
                }
                
            } catch(NumberFormatException e) {
                
                writeln("Invalid socket: " + e.getMessage());
                
            }
            
        }
                
    }

    abstract class Command {

        String command;
        String description = "command";

        Command(String command, String description) {

            this.command = command;
            this.description = description;

        }

        abstract public void perform(String[] args) throws IOException;

    }

    public void interrupt() {

        interrupted = true;

        try {

            if(serverSocket != null)
                serverSocket.close();

        } catch(IOException e) {
            
            DebuggerOutput.stackTrace(e);

        }

        try {

            if(client != null)
                client.close();

        } catch(IOException e) {
            
            DebuggerOutput.stackTrace(e);

        }

        try {

            if(lineReader != null)
                lineReader.close();

        } catch(IOException e) {
            
            DebuggerOutput.stackTrace(e);

        }
            
        super.interrupt();

    }

}

class TerminateSessionException extends IOException {

    TerminateSessionException(String s) {

        super(s); 

    }

}

class ShutdownException extends IOException {

    ShutdownException(String s) {

        super(s); 

    }

}
