/**
 * The graphical user interface of {@link retypar}.
 * @author Jens Heuser
 */
package retypar.gui;


import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGImageDecoder;
import heuser.multilang.Ml;
import heuser.simpleLogger.SimpleLoggerListener;
import heuser.updater.Update;
import retypar.converter.Converter;
import thirdParty.GifEncoder.GifDecoder;
import thirdParty.ImagePreviewPanel;

import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.border.BevelBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.util.Hashtable;

/**
 * This is the graphical user interface of retypar.
 * @see retypar
 * @author Jens Heuser
 */
public class MainFrame extends JFrame implements ActionListener, ChangeListener, MouseListener, SimpleLoggerListener{

    private String filename = "";
    private float wI, hI;

    private File imgF;
    private BufferedImage convertedImage;
    private Image imConvertedImage;

    private File[] fileArray;

    private File currentDir;

    private Container c = getContentPane();



    //JPanel userPanel= new JPanel(new GridLayout(0,1));
    //private JTextArea userInfo = new JTextArea();
    /**
     * The <code>Box</code> containing the (eventually scaled) images.
     * @see MainFrame#original
     * @see MainFrame#converted
     */
    private Box pictureBox  = Box.createVerticalBox();

    /**
     * The left side of the GUI containing buttons, checkboxes, sliders, ...
     */
    private Box userBox     = Box.createVerticalBox();

    /**
     * Contains the checkboxes "A-Z", "a-z", "0-9"
     */
    private Box optionsBox1 = Box.createHorizontalBox();

    /**
     * Contains the checkboxes "Blank" and "colored"
     */
    private Box optionsBox2 = Box.createHorizontalBox();

    /**
     * Used to arrange <code>optionsBox1</code> and <code>optionsBox2</code>.
     */
    private Box optionsBoxC = Box.createVerticalBox();

    /**
     * Contains all the buttons.
     */
    private Box buttonBox   = Box.createHorizontalBox();

    private Box langBox1    = Box.createVerticalBox();

    private Box langBox2    = Box.createVerticalBox();

    private Box langBox3    = Box.createVerticalBox();

    private Box langBoxC    = Box.createHorizontalBox();

    private final String uCase          = "A-Z";
    private final String lCase          = "a-z";
    private final String numbers        = "0-9";

    private final int strut_small   = 10;
    private final int strut_big     = 20;

    private JButton loadButton          = new JButton();
    private JButton storeButton         = new JButton();
    private JButton convertButton       = new JButton();
    private JButton viewConvertedButton = new JButton();
    private JButton optionsButton       = new JButton();
    private JButton aboutButton         = new JButton();
    private JButton cancelButton        = new JButton();

    private JCheckBox uppercase = new JCheckBox(uCase, true);
    private JCheckBox lowercase = new JCheckBox(lCase, true);
    private JCheckBox digits    = new JCheckBox(numbers, true);
    private JCheckBox specials  = new JCheckBox();

    private JCheckBox colored   = new JCheckBox();

    //private JLabel renderLaber;
    protected JComboBox renderCombo;
    private JSpinner thresholdSpinner;
    private JButton algoInfoButton;
    private Box algoSelectionBox;

    private JComboBox fontCombo;
    private JLabel fontSelectionLabel;
    private Box fontSelectionBox;

    private JScrollPane fontExamplePane;

    private JLabel  fontExample1, fontExample2;

    private JSpinner fontSizeSpinner;
    private JLabel fontSizer;

    private JRadioButton german     = new JRadioButton("Deutsch", true);
    private JRadioButton english    = new JRadioButton("English", false);
    private JRadioButton french     = new JRadioButton("Franaise", false);
    private JRadioButton italian    = new JRadioButton("Italiano", false);
    private JRadioButton spanish    = new JRadioButton("Espaol", false);
    private JRadioButton blank      = new JRadioButton("", false);

    ButtonGroup bg = new ButtonGroup();

    /**
     * Contains the original (but eventually scaled) image.
     */
    private JScrollPane original;

    /**
     * Contains the converted (and eventually scaled) image.
     */
    private JScrollPane converted;

    private JLabel qualityLabel;

    private Hashtable labelTable;

    /**
     * The quality of jpeg compression when saving to disk.
     */
    private JSlider qualitySlider;

    private JLabel scaleLabel;
    /**
     * The scale factor when converting.
     */
    private JSpinner scaleSpinner;

    private JPanel scalePanel;
    private JPanel showColor;

    private JLabel exportFormatLabel;
    private JCheckBox opacityBox;

    private long origFileSizeKB;
    private long convFileSizeKB;


    /**
     * Shows the progress of converting.
     * @see retypar.gui.Walker
     */
    public Walker walker;

    private Converter converter;

    private JLabel duration;

    protected Ml l;

    private JLabel previewLabel = new JLabel();

    protected String plaf;
    
    private JSlider slider1,
                    slider2,
                    slider3;



    // PopupMenu and its MenuItems
    private PopupMenu popUp;

    private MenuItem popReload;
    private MenuItem popLoadNew;
    private MenuItem popSave;
    private MenuItem popConvert;
    private MenuItem popView;
    private MenuItem popAbout;

    private JComboBox exportFormatCombo;

    private Timer memoryCheckTimer;


    public MainFrame(){
        retypar.retypar.addLoggerListenerError((SimpleLoggerListener) this);
        
        initMultiLang();

        createInterface();

        doPossibleUpdate();
    }


    private void initMultiLang(){
        l = new Ml();
        l.setOriginalLang("ENG");
        if (!retypar.retypar.lang.equals("")){
            l.setLang(retypar.retypar.lang);
            return;
        }

        try{
            String lang = System.getProperty("user.language");
            if (lang.equals("de")) l.setLang("GER");
            if (lang.equals("fr")) l.setLang("FRA");
            if (lang.equals("it")) l.setLang("ITA");
        }catch(Exception e){
            retypar.retypar.logError("Could not initialize multilanguage module!");
        }
    }


    private void createInterface(){
        //setPlaf();
        checkMaxRes();

        BorderLayout borderLayout = new BorderLayout();

        c.setLayout(borderLayout);
        //new BoxLayout(c, BoxLayout.X_AXIS);
        setResizable(true);
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        setTitle("retypar " + retypar.retypar.version + " - a tool to convert images into ASCII (Jens Heuser, 2005), http://www.retypar.de.vu");
        //setIconImage(null);

        Box verticalBox = Box.createVerticalBox();

        buttonBox.add(Box.createVerticalStrut(strut_small));

        loadButton.setEnabled(true);
        loadButton.setText(l.pr("Open image..."));
        loadButton.addActionListener(this);
        loadButton.setActionCommand("load");
        loadButton.setAlignmentX(Component.RIGHT_ALIGNMENT);

        storeButton.setEnabled(false);
        storeButton.setText(l.pr("Store image..."));
        storeButton.addActionListener(this);
        storeButton.setActionCommand("store");
        storeButton.setAlignmentX(Component.RIGHT_ALIGNMENT);

        verticalBox.add(loadButton);
        verticalBox.add(Box.createVerticalStrut(10));
        verticalBox.add(storeButton);
        //verticalBox.setMaximumSize(verticalBox.getPreferredSize());

        buttonBox.add(verticalBox);
        buttonBox.add(Box.createHorizontalStrut(strut_small));

        convertButton.setEnabled(false);
        convertButton.setText(l.pr("Convert image!"));
        convertButton.addActionListener(this);
        convertButton.setActionCommand("convert");
        convertButton.setAlignmentX(Component.LEFT_ALIGNMENT);

        viewConvertedButton.setEnabled(false);
        viewConvertedButton.setText(l.pr("View converted image."));
        viewConvertedButton.addActionListener(this);
        viewConvertedButton.setActionCommand("view");
        viewConvertedButton.setAlignmentX(Component.LEFT_ALIGNMENT);

        verticalBox = Box.createVerticalBox();

        verticalBox.add(convertButton);
        verticalBox.add(Box.createVerticalStrut(10));
        verticalBox.add(viewConvertedButton);
        //verticalBox.setMaximumSize(verticalBox.getPreferredSize());

        buttonBox.add(verticalBox);

        userBox.add(buttonBox);
        userBox.add(Box.createVerticalStrut(strut_small));

        optionsButton.setEnabled(true);
        optionsButton.setText(l.pr("Options") + "...");
        optionsButton.addActionListener(this);
        optionsButton.setActionCommand("options");
        optionsButton.setAlignmentX(Component.CENTER_ALIGNMENT);
        userBox.add(optionsButton);

        userBox.add(Box.createVerticalStrut(strut_small));

        aboutButton.setEnabled(true);
        aboutButton.setText(l.pr("About"));
        aboutButton.addActionListener(this);
        aboutButton.setActionCommand("about");
        aboutButton.setAlignmentX(Component.CENTER_ALIGNMENT);
        userBox.add(aboutButton);

        userBox.add(Box.createVerticalStrut(strut_small));

        cancelButton.setEnabled(false);
        cancelButton.setText(l.pr("Cancel!"));
        cancelButton.addActionListener(this);
        cancelButton.setActionCommand("cancel");
        cancelButton.setAlignmentX(Component.CENTER_ALIGNMENT);
        userBox.add(cancelButton);

        userBox.add(Box.createVerticalStrut(strut_big));
        userBox.add(Box.createVerticalGlue());

        walker = new Walker(userBox);
        duration = new JLabel("00:00:00");
        duration.setAlignmentX(JLabel.CENTER_ALIGNMENT);

        userBox.add(duration);

        userBox.add(Box.createVerticalGlue());
        userBox.add(Box.createVerticalStrut(strut_big));

        uppercase.setEnabled(true);
        uppercase.addActionListener(this);
        uppercase.setActionCommand("uppercase");
        optionsBox1.add(uppercase);

        lowercase.setEnabled(true);
        lowercase.addActionListener(this);
        lowercase.setActionCommand("lowercase");
        optionsBox1.add(lowercase);

        digits.setEnabled(true);
        digits.addActionListener(this);
        digits.setActionCommand("digits");
        optionsBox1.add(digits);

        specials.setSelected(false);
        specials.setText(l.pr("Blank"));
        colored.setSelected(true);
        colored.setText(l.pr("colored"));

        optionsBox2.add(specials);
        optionsBox2.add(colored);

        optionsBoxC.add(optionsBox1);
        optionsBoxC.add(optionsBox2);

        optionsBoxC.setBorder(BorderFactory.createTitledBorder(l.pr("Characters")));

        showColor = new JPanel(new BorderLayout());
        showColor.setBorder(BorderFactory.createTitledBorder(l.pr("Background color: ")));

        final JPanel showColor2 = new JPanel();
        showColor2.setBackground(Color.WHITE);
        showColor2.setForeground(Color.WHITE);
        showColor.setMaximumSize(new Dimension( (int)optionsBoxC.getPreferredSize().getWidth() * 75 / 100,
                                                (int)optionsBoxC.getPreferredSize().getHeight()));

        showColor.add(showColor2, BorderLayout.CENTER);

        showColor2.addMouseListener(new MouseAdapter(){
            public void mouseClicked(MouseEvent e){
                Color colOld = showColor2.getBackground();
                Color colNew;
                try{
                    colNew = JColorChooser.showDialog(userBox, l.pr("Choose a color"), Color.WHITE);
                    if (colNew != null){
                        showColor2.setBackground(colNew);
                        showColor2.setForeground(colNew);
                    }
                }catch(Exception ex){
                    //System.out.println("caught");
                    showColor2.setBackground(colOld);
                    showColor2.setForeground(colOld);
                }
            }
        });


        Box horizontalBox = Box.createHorizontalBox();
        horizontalBox.add(optionsBoxC);
        horizontalBox.add(showColor);

        userBox.add(horizontalBox);
        //userBox.add(Box.createVerticalGlue());

        userBox.add(Box.createVerticalStrut(strut_big));

        //userBox.add(Box.createVerticalStrut(35));



        algoSelectionBox = Box.createVerticalBox();
        algoSelectionBox.setBorder(BorderFactory.createTitledBorder(l.pr("Algorithm and threshold:")));

        Box renderBox = Box.createHorizontalBox();
        thresholdSpinner = new JSpinner(new SpinnerNumberModel());
        thresholdSpinner.setEditor(new JSpinner.NumberEditor(thresholdSpinner));
        //thresholdSpinner.setMaximumSize(thresholdSpinner.getPreferredSize());

        renderCombo = new JComboBox();
        renderCombo.addActionListener(this);
        renderCombo.setActionCommand("render");

        fillRenderCombo();
        //renderCombo.setMaximumSize(renderCombo.getPreferredSize());

        algoInfoButton = new JButton("?");
        algoInfoButton.addActionListener(this);
        algoInfoButton.setActionCommand("algoInfo");

        renderBox.add(renderCombo);
        renderBox.add(thresholdSpinner);
        renderBox.add(algoInfoButton);
        //algoSelectionBox.add(Box.createHorizontalGlue());
        algoSelectionBox.add(renderBox);

        algoSelectionBox.setMaximumSize(new Dimension(290, (int)algoSelectionBox.getPreferredSize().getHeight()));

        userBox.add(algoSelectionBox);

        userBox.add(Box.createVerticalStrut(strut_big));

        fontCombo = new JComboBox();
        for (int i = 0; i < retypar.retypar.fonts.length; i++) fontCombo.addItem(retypar.retypar.fonts[i].getFontName());
        fontCombo.addActionListener(this);
        fontCombo.setActionCommand("newFont");

        fontCombo.addMouseWheelListener(new MouseWheelListener(){
            public void mouseWheelMoved(MouseWheelEvent e){
                if (fontCombo.getSelectedIndex() + e.getWheelRotation() >= 0)
                    fontCombo.setSelectedIndex(fontCombo.getSelectedIndex() + e.getWheelRotation());
            }
        });


        fontSizeSpinner = new JSpinner(new SpinnerNumberModel(10, 6, 72, 1));
        fontSizeSpinner.addChangeListener(this);
        fontSizeSpinner.setEditor(new javax.swing.JSpinner.NumberEditor(fontSizeSpinner));

        fontSizeSpinner.addMouseWheelListener(new MouseWheelListener(){
            public void mouseWheelMoved(MouseWheelEvent e){
                if (e.getWheelRotation() < 0)
                    if (fontSizeSpinner.getModel().getNextValue() != null)
                        fontSizeSpinner.getModel().setValue(fontSizeSpinner.getModel().getNextValue());

                if (e.getWheelRotation() > 0)
                    if (fontSizeSpinner.getModel().getPreviousValue() != null)
                        fontSizeSpinner.getModel().setValue(fontSizeSpinner.getModel().getPreviousValue());
            }
        });

        fontExample1 = new JLabel();
        fontExample1.setFont(new Font(  fontCombo.getSelectedItem().toString(),
                                        retypar.retypar.fonts[fontCombo.getSelectedIndex()].getStyle(),
                                        22));
        fontExample1.setAlignmentX(JLabel.CENTER_ALIGNMENT);
        fontExample1.setText("Font example:");

        fontExample2 = new JLabel();
        fontExample2.setFont(new Font(  fontCombo.getSelectedItem().toString(),
                                        retypar.retypar.fonts[fontCombo.getSelectedIndex()].getStyle(),
                                        22));
        fontExample2.setAlignmentX(JLabel.CENTER_ALIGNMENT);
        fontExample2.setText("AaBbCcDdEe 0123456789");

        Box fontExampleBox = Box.createVerticalBox();
        fontExampleBox.add(fontExample1);
        fontExampleBox.add(fontExample2);

        fontExamplePane = new JScrollPane(fontExampleBox);
        fontExamplePane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        fontExamplePane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER);

        fontSelectionBox = Box.createVerticalBox();
        Box fontSelectionBoxInner = Box.createHorizontalBox();
        fontSelectionBox.setBorder(BorderFactory.createTitledBorder(l.pr("Font and font size: ")));
        fontSelectionBoxInner.add(fontCombo);
        fontSelectionBoxInner.add(fontSizeSpinner);
        fontSelectionBox.add(fontSelectionBoxInner);
        fontSelectionBox.add(fontExamplePane);

        fontSelectionBox.setMaximumSize(fontSelectionBox.getPreferredSize());

        userBox.add(fontSelectionBox);

        userBox.add(Box.createVerticalStrut(strut_small));
        userBox.add(Box.createVerticalGlue());


        scaleSpinner = new JSpinner(new SpinnerNumberModel(100, 1, 10000, 1));
        scaleSpinner.setEditor(new javax.swing.JSpinner.NumberEditor(scaleSpinner));

        scaleSpinner.addMouseWheelListener(new MouseWheelListener(){
            public void mouseWheelMoved(MouseWheelEvent e){
                if (e.getWheelRotation() < 0)
                    if (scaleSpinner.getModel().getNextValue() != null)
                        scaleSpinner.getModel().setValue(scaleSpinner.getModel().getNextValue());

                if (e.getWheelRotation() > 0)
                    if (scaleSpinner.getModel().getPreviousValue() != null)
                        scaleSpinner.getModel().setValue(scaleSpinner.getModel().getPreviousValue());
            }
        });


        exportFormatCombo = new JComboBox(new DefaultComboBoxModel(retypar.retypar.availableGraphicalExportFormats));
        exportFormatCombo.addActionListener(this);
        exportFormatCombo.setActionCommand("newExportFormat");
        exportFormatCombo.setEnabled(false);

        exportFormatLabel = new JLabel(l.pr("Preview as:"));

        Box exportFormatBox = Box.createHorizontalBox();
        exportFormatBox.add(exportFormatLabel);
        exportFormatBox.add(exportFormatCombo);
        exportFormatBox.add(Box.createHorizontalGlue());
        exportFormatBox.setMaximumSize(new Dimension(290, (int) exportFormatBox.getPreferredSize().getHeight()));


        scaleLabel = new JLabel(l.pr("Scalefactor") + " (in %): ");

        scalePanel = new JPanel();

        scalePanel.add(scaleLabel);
        scalePanel.add(scaleSpinner);
        //userBox.add(scalePanel);

        //userBox.add(Box.createVerticalStrut(strut_small));


        //opacityLabel = new JLabel(l.pr("Background color is transparent"));
        opacityBox = new JCheckBox(l.pr("Background color is transparent"), false);
        opacityBox.setActionCommand("opacity");
        opacityBox.addActionListener(this);
        opacityBox.setEnabled(false);



        qualityLabel = new JLabel(l.pr("Compression:"));

        qualitySlider = new JSlider(0, 0, 100, 0);
        labelTable = new Hashtable();
        labelTable.put(new Integer(0),  new JLabel(l.pr("low")));
        labelTable.put(new Integer(50), new JLabel(l.pr("medium")));
        labelTable.put(new Integer(100),new JLabel(l.pr("high")));
        qualitySlider.setLabelTable(labelTable);
        qualitySlider.setMinorTickSpacing(10);
        qualitySlider.setMajorTickSpacing(25);
        qualitySlider.setPaintTicks(true);
        qualitySlider.setPaintLabels(true);
        qualitySlider.setEnabled(false);

        qualitySlider.addMouseListener(new MouseAdapter(){
            public void mouseReleased(MouseEvent e){
                if (!qualitySlider.isEnabled())
                    return;

                updateConverted();
                qualitySlider.setToolTipText("" + qualitySlider.getValue());
            }
        });

        Component hGlue = Box.createHorizontalGlue();

        JPanel qualityPanel = new JPanel(new BorderLayout());
        //qualityPanel.setBorder(BorderFactory.createEtchedBorder());
        qualityPanel.add(qualityLabel, BorderLayout.SOUTH);
        qualityPanel.add(hGlue, BorderLayout.CENTER);
        qualityPanel.setMaximumSize(new Dimension(290, (int)qualityPanel.getPreferredSize().getHeight()));
        //userBox.add(qualityPanel);

        //userBox.add(qualitySlider);

        Box outputOptionsBox = Box.createVerticalBox();
        outputOptionsBox.setBorder(BorderFactory.createTitledBorder("Output"));

        outputOptionsBox.add(exportFormatBox);
        outputOptionsBox.add(scalePanel);
        outputOptionsBox.add(opacityBox);
        outputOptionsBox.add(qualityPanel);
        outputOptionsBox.add(qualitySlider);

        userBox.add(outputOptionsBox);

        userBox.add(Box.createVerticalGlue());

        german.addActionListener(this);
        german.setActionCommand("german");
        english.addActionListener(this);
        english.setActionCommand("english");
        french.addActionListener(this);
        french.setActionCommand("french");
        italian.addActionListener(this);
        italian.setActionCommand("italian");
        spanish.addActionListener(this);
        spanish.setActionCommand("spanish");
        blank.addActionListener(this);
        blank.setActionCommand("italian");
        blank.setEnabled(false);
        bg.add(german);
        bg.add(english);
        bg.add(french);
        bg.add(italian);
        bg.add(spanish);
        bg.add(blank);

        langBox1.add(german);
        langBox1.add(english);
        langBox2.add(french);
        langBox2.add(italian);
        langBox3.add(spanish);
        langBox3.add(blank);

        langBoxC.add(langBox1);
        langBoxC.add(langBox2);
        langBoxC.add(langBox3);

        langBoxC.setBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED));
        //langBoxC.setAlignmentY(Box.CENTER_ALIGNMENT);
        //langBoxC.setMaximumSize(new Dimension(300, langBoxC.getHeight()));

        userBox.add(langBoxC);

        userBox.setBorder(BorderFactory.createTitledBorder(l.pr("Operations")));

        //userInfo.setBorder(BorderFactory.createTitledBorder(l.pr("User information")));
        //userInfo.setEditable(false);

        pictureBox.setBorder(BorderFactory.createTitledBorder(l.pr("Images")));
        pictureBox.addMouseListener(this);

        c.add(userBox, BorderLayout.WEST);
        c.add(pictureBox, BorderLayout.CENTER);

        //pack();
        //setBounds(1280, 0, 1280, 1024);
        setBounds(0, 0, 1024, 768);
        setExtendedState(MAXIMIZED_BOTH);

        if (retypar.retypar.lang.equals("GER")) german.doClick();
        if (retypar.retypar.lang.equals("ENG")) english.doClick();
        if (retypar.retypar.lang.equals("FRA")) french.doClick();
        if (retypar.retypar.lang.equals("ITA")) italian.doClick();
        if (retypar.retypar.lang.equals("SPA")) spanish.doClick();

        setPlaf();
        setVisible(true);

        initPopUp();
        pictureBox.add(popUp);
        //userBox.setPreferredSize(userBox.getSize());
        fontCombo.setSelectedIndex(fontCombo.getSelectedIndex());
        //showColor2.setSize(showColor.getSize());
        //System.out.println("userBox: " + userBox.getWidth());
        //System.out.println("langBoxC: " + langBoxC.getWidth());
        addSplashBackground();

        //startMemoryCheckTimer();
    }


    /**
     * Triggers every second <code>doMemoryCheck()</code>.
     */
    private void startMemoryCheckTimer(){
        memoryCheckTimer = new Timer(1000, new ActionListener(){
            public void actionPerformed(ActionEvent e){
                memoryCheckTimer.stop();
                doMemoryCheck();
                memoryCheckTimer.setInitialDelay(10000);
                memoryCheckTimer.restart();
            }
        });

        memoryCheckTimer.setRepeats(true);
        memoryCheckTimer.setCoalesce(true);
        memoryCheckTimer.start();
    }


    /**
     * Displays the retypar logo in the <code>pictureBox</code> until an image is loaded.
     */
    private void addSplashBackground(){
        File    splashImageFile;
        JLabel  splashImageLabel;

        if (retypar.retypar.runningInJar){
            try{
                InputStream in = getClass().getResourceAsStream("/retypar_splash.gif");

                GifDecoder decoder = new GifDecoder();
                decoder.read(in);

                in.close();

                splashImageLabel = new JLabel(new ImageIcon(decoder.getImage()));
            }catch(Exception e){
                retypar.retypar.logError("error accessing splash-image in jar: " + e.toString());
                splashImageLabel = new JLabel();
                }
        }
        else {
            splashImageFile = new File(retypar.retypar.basedir + "retypar_splash.gif");
            splashImageLabel=new JLabel(new ImageIcon(splashImageFile.getPath()));
            }


        splashImageLabel.setAlignmentX(JLabel.CENTER_ALIGNMENT);

        pictureBox.add(Box.createVerticalGlue());
        pictureBox.add(splashImageLabel);
        pictureBox.add(Box.createVerticalGlue());
        pictureBox.doLayout();
    }


    private void initPopUp(){
        popUp =  new PopupMenu();

        popReload    = new MenuItem(l.pr("Reload"));
        popLoadNew   = new MenuItem(l.pr("Load new..."));
        popSave      = new MenuItem(l.pr("Save..."));
        popConvert   = new MenuItem(l.pr("Convert"));
        popView      = new MenuItem(l.pr("View"));
        popAbout     = new MenuItem(l.pr("About"));

        popReload.setActionCommand( "reload");
        popLoadNew.setActionCommand("load");
        popSave.setActionCommand(   "store");
        popConvert.setActionCommand("convert");
        popView.setActionCommand(   "view");
        popAbout.setActionCommand(  "about");

        popUp.add(popReload);
        popUp.addSeparator();
        popUp.add(popLoadNew);
        popUp.add(popSave);
        popUp.addSeparator();
        popUp.add(popConvert);
        popUp.add(popView);
        popUp.addSeparator();
        popUp.add(popAbout);

        popUp.addActionListener(this);
    }


    private void labelComponents(){

        pictureBox.remove(popUp);
        initPopUp();
        pictureBox.add(popUp);


        loadButton.setText(         l.pr("Open image..."));
        convertButton.setText(      l.pr("Convert image!"));
        viewConvertedButton.setText(l.pr("View converted image."));
        storeButton.setText(        l.pr("Store image..."));
        optionsButton.setText(      l.pr("Options") + "...");
        aboutButton.setText(        l.pr("About"));
        cancelButton.setText(       l.pr("Cancel!"));
        specials.setText(           l.pr("Blank"));
        colored.setText(            l.pr("colored"));
        qualityLabel.setText(       l.pr("Compression:"));
        scaleLabel.setText(         l.pr("Scalefactor") + " (in %): ");

        labelTable.put(new Integer(0),  new JLabel(l.pr("low")));
        labelTable.put(new Integer(50), new JLabel(l.pr("medium")));
        labelTable.put(new Integer(100),new JLabel(l.pr("high")));

        qualitySlider.setLabelTable(labelTable);

        exportFormatLabel.setText(l.pr(                                     "Preview as:"));
        opacityBox.setText(l.pr(                                            "Background color is transparent"));

        showColor.setBorder(BorderFactory.createTitledBorder(l.pr(          "Background color: ")));
        optionsBoxC.setBorder(BorderFactory.createTitledBorder(l.pr(        "Characters")));
        algoSelectionBox.setBorder(BorderFactory.createTitledBorder(l.pr(   "Algorithm and threshold:")));
        fontSelectionBox.setBorder(BorderFactory.createTitledBorder(l.pr(   "Font and font size: ")));
        userBox.setBorder(BorderFactory.createTitledBorder(l.pr(            "Operations")));
        userBox.revalidate();

        try{
            slider1.setToolTipText(l.pr("adjust brightness: ") + slider1.getValue());
            slider2.setToolTipText(l.pr("adjust contrast: ") + slider2.getValue());
            slider3.setToolTipText(l.pr("adjust saturation: ") + slider3.getValue());
        }catch (Exception e){}

        if (!convertButton.isEnabled())
            pictureBox.setBorder(BorderFactory.createTitledBorder(l.pr("Images")));
    }


    protected void fillRenderCombo(){
        renderCombo.removeAllItems();
        for (int i = 0; i < retypar.retypar.converters.size(); i++)
            renderCombo.addItem(retypar.retypar.converters.get(i));
        renderCombo.invalidate();
    }


    protected void setPlaf(){
        if (retypar.retypar.plaf != null) plaf = retypar.retypar.plaf;
            else plaf = UIManager.getSystemLookAndFeelClassName();
        try{
            UIManager.setLookAndFeel(plaf);
            SwingUtilities.updateComponentTreeUI(this);
        }catch (Exception e){
            retypar.retypar.logError("Error setting look-and-feel.");
        }
    }


    public void stateChanged(ChangeEvent e){                 
        // trigger actionPerformed on fontCombo
        fontCombo.setSelectedIndex(fontCombo.getSelectedIndex());
    }


    public void actionPerformed(ActionEvent e){

        //System.out.println(e.getActionCommand());

        if(e.getActionCommand().equals("load"))
            {   File tmpFile = chooseImage(null);
                if(tmpFile!=null){
                    imgF = tmpFile;
                    convertButton.setEnabled(true);
                }
            }

        if(e.getActionCommand().equals("reload")){
            chooseImage(imgF);
        }

        if(e.getActionCommand().equals("store"))
            if (!storeImage())
                JOptionPane.showMessageDialog(this, l.pr("Could not write to file!"), l.pr("I/O Error"), JOptionPane.ERROR_MESSAGE);

        if(e.getActionCommand().equals("convert"))
            triggerConversion();

        if(e.getActionCommand().equals("options"))
            displayOptions();

        if(e.getActionCommand().equals("uppercase") |
                e.getActionCommand().equals("lowercase") |
                e.getActionCommand().equals("digits"))
            checkOptions();

        if(e.getActionCommand().equals("render")) {
            checkThresholdSpinner();
            //checkOutputFormats();
        }

        if(e.getActionCommand().equals("algoInfo")) popUpAlgoInfo((Converter)renderCombo.getSelectedItem());

        if(e.getActionCommand().equals("view"))
            viewConverted();

        if(e.getActionCommand().equals("cancel"))
            cancelConversion();

        if(e.getActionCommand().equals("newFont")) {
            fontExample1.setFont(new Font(  fontCombo.getSelectedItem().toString(),
                                            retypar.retypar.fonts[fontCombo.getSelectedIndex()].getStyle(),
                                            ((Integer)fontSizeSpinner.getValue()).intValue()));

            fontExample2.setFont(new Font(  fontCombo.getSelectedItem().toString(),
                                            retypar.retypar.fonts[fontCombo.getSelectedIndex()].getStyle(),
                                            ((Integer)fontSizeSpinner.getValue()).intValue()));
            //fontExample1.setText(""+fontCombo.getSelectedIndex());//().toString());
        }

        if (e.getActionCommand().equals("newExportFormat")){
            if (exportFormatCombo.getSelectedItem().toString().equals("JPEG")){
                qualitySlider.setEnabled(true);
                //qualitySlider.getMouseListeners()[0].mouseReleased(null);
            }
            else{
                //qualitySlider.getMouseListeners()[0].mouseReleased(null);
                qualitySlider.setEnabled(false);
            }
            //reEncodeConverted(convertedImage, qualitySlider.getValue());

            if (!exportFormatCombo.getSelectedItem().toString().equals("GIF")){
                opacityBox.setSelected(false);
                opacityBox.setEnabled(false);
            }
            else
                opacityBox.setEnabled(true);

            updateConverted();
        }


        if (e.getActionCommand().equals("opacity")){
            //doOpacity here!
        }

        if(e.getActionCommand().equals("german"))
            switchLang("GER");

        if(e.getActionCommand().equals("english"))
            switchLang("ENG");

        if(e.getActionCommand().equals("french"))
            switchLang("FRA");

        if(e.getActionCommand().equals("italian"))
            switchLang("ITA");

        if(e.getActionCommand().equals("spanish"))
            switchLang("SPA");

        if (e.getActionCommand().equals("about"))
            displayAbout();
    }


    // MouseListener

    public void mouseEntered(MouseEvent e){}

    public void mouseExited(MouseEvent e){}

    public void mouseReleased(MouseEvent e){

        //System.out.println("click!");

        if (!e.isPopupTrigger()){
            return;
        }

        if (convertButton.isEnabled()) popReload.setEnabled(true);
            else popReload.setEnabled(false);

        if (convertButton.isEnabled()) popConvert.setEnabled(true);
            else popConvert.setEnabled(false);

        if (storeButton.isEnabled()) popSave.setEnabled(true);
            else popSave.setEnabled(false);

        if (viewConvertedButton.isEnabled()) popView.setEnabled(true);
            else popView.setEnabled(false);

        //System.out.println("popping up... " + e.getComponent());

        popUp.show(e.getComponent(), e.getX(), e.getY());
    }

    public void mouseClicked(MouseEvent e){}

    public void mousePressed(MouseEvent e){}


    private void popUpAlgoInfo(Converter c){
        String message, title, website;

        if (c.getWebsite().length() == 0) website = "";
            else website = ", " + c.getWebsite();

        message =   "Algorithm " + c.toString() +
                    "\nAuthor: " + c.getAuthor() + website +
                    "\n\n" + c.getDescription() +
                    "\n";

        title = "Algorithm information";


        JOptionPane.showMessageDialog(this, message, title, JOptionPane.INFORMATION_MESSAGE);
    }


    private void popUpAlgoError(String errors){
        String message, title;

        title = "Conversion Error";
        message = errors;


        JOptionPane.showMessageDialog(this, message, title, JOptionPane.INFORMATION_MESSAGE);
    }


    public void errorLogged(String err){
        String title = "Error!";

        JOptionPane.showMessageDialog(this, err, title, JOptionPane.ERROR_MESSAGE);
    }


    public void messageLogged(String msg){

    }


    /**
     * The file dialog to open an image.
     * @return The image to load as a <code>File</code>.
     */
    private File chooseImage(File imageFile){
        final JFileChooser chooser;

        try{
            chooser = new JFileChooser();
        }catch (Exception ex){return null;}


        chooser.setDialogTitle(l.pr("Open image..."));

        FileFilter images = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {	"jpg",
                                                "jpeg",
                                                "gif",
                                                "bmp",
                                                "png",
                                                "tga" };

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < 6; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("Images");
                    }
                };

        chooser.setFileFilter(images);
        chooser.setMultiSelectionEnabled(true);

        try{
            currentDir = new File(retypar.retypar.lastPath);
        }catch(Exception e){
            currentDir = new File(System.getProperty("user.home"));
        }


        chooser.setCurrentDirectory(currentDir);

        ImagePreviewPanel preview = new ImagePreviewPanel();
        chooser.setAccessory(preview);
        chooser.addPropertyChangeListener(preview);

        /**
        chooser.setAccessory(previewLabel);
        //previewLabel.setDebugGraphicsOptions(DebugGraphics.LOG_OPTION);


        chooser.addPropertyChangeListener(new PropertyChangeListener(){
            protected ImageIcon imageIcon   = new ImageIcon();
            protected ImageIcon imageIconTmp= new ImageIcon();
            protected Image     image;
            protected String    path        = "";

            public void propertyChange(PropertyChangeEvent e){
                if(e.getPropertyName().equals(JFileChooser.SELECTED_FILE_CHANGED_PROPERTY))
                    try{
                        if (((File)e.getNewValue()).isDirectory())
                            return;
                    }catch(Exception ex){}
                else return;

                try{
                    path = ((File)e.getNewValue()).getPath();
                }catch(Exception ex){
                    return;
                }

                int cx = chooser.getWidth()/4;
                int cy = chooser.getHeight()/4;

                imageIconTmp = new ImageIcon(path);
                image = imageIconTmp.getImage().getScaledInstance(cx, -1, Image.SCALE_FAST);
                imageIcon = new ImageIcon(image);

                previewLabel.removeAll();
                previewLabel.setIcon(imageIcon);
                previewLabel.invalidate();

                imageIconTmp= null;
                image       = null;
                imageIcon   = null;

                System.gc();
            }
        });
         **/

        if (imageFile != null){
            filename = imageFile.getName();
        }
        else if(chooser.showOpenDialog(this)==JFileChooser.APPROVE_OPTION){
            fileArray = chooser.getSelectedFiles();
            //((DefaultComboBoxModel)exportFormatCombo.getModel()).removeElement("TVID");
            //if (fileArray.length > 1) ((DefaultComboBoxModel)exportFormatCombo.getModel()).addElement("TVID");

            imageFile = fileArray[0];
            filename = imageFile.getName();

            // store last path in INI
            currentDir = imageFile.getParentFile();
            retypar.retypar.lastPath = currentDir.getPath();
            retypar.retypar.writeINI();
        }
            else {
                paint(this.getGraphics());
                return null;
            }

        try{
            //pictureBox.remove(original);
            //pictureBox.remove(converted);
            pictureBox.removeAll();
            converted = null;
            qualitySlider.setEnabled(false);
            qualitySlider.setValue(0);
        }catch(Exception npe){}

        //if (chooser.getSelectedFile() == null) return null;

        previewLabel.removeAll();

        //Image orig = getScaledFromDisk(chooser.getSelectedFile());
        //original = new JScrollPane(new JLabel(new ImageIcon(orig)));
        //original.setBorder(BorderFactory.createTitledBorder(""));
        //orig.flush();

        addSlidersAndOriginal(imageFile);
        //pictureBox.add(original);
        //setVisible(true);

        walker.setValue(0);

        currentDir = imageFile.getParentFile();
        pictureBox.setBorder(BorderFactory.createTitledBorder(filename + " ("+ (int)wI + "x" + (int)hI + ", " + origFileSizeKB + "KB)"));

        storeButton.setEnabled(false);
        viewConvertedButton.setEnabled(false);
        chooser.removeAll();
        System.gc();

        doMemoryCheck();

        return imageFile;
    }


    /**
     * Calls <code>retypar.retypar.hasEnoughMemoryToConvert(int picSize)</code> with
     * <code>picSize = width * height</code>.
     */
    private void doMemoryCheck(){
        if (!retypar.retypar.hasEnoughMemoryToConvert((int)(wI * hI))){
                JOptionPane.showMessageDialog(this, l.pr("Most probably not enough memory for conversion!"),
                                                    l.pr("Out-of-memory warning"),
                                                    JOptionPane.WARNING_MESSAGE);
            }
    }

    /**
     * The two sliders on the left and right and the one beneath the original image get added to the <code>pictureBox</code>.
     * So gets the original image.
     * @param origFile The original image to be added to the <code>pictureBox</code>.
     */
    private void addSlidersAndOriginal(final File origFile){
        slider1 = new JSlider(JSlider.VERTICAL,     -100, 100, 0);
        slider2 = new JSlider(JSlider.HORIZONTAL,   -100, 100, 0);
        slider3 = new JSlider(JSlider.VERTICAL,     -100, 100, 0);

        slider1.setToolTipText(l.pr("adjust brightness: ") + slider1.getValue());
        slider2.setToolTipText(l.pr("adjust contrast: ") + slider2.getValue());
        slider3.setToolTipText(l.pr("adjust saturation: ") + slider3.getValue());

        slider1.addMouseListener(new MouseAdapter(){
            public void mouseReleased(MouseEvent e){
                adjustBCS(origFile);
            }
        });

        slider2.addMouseListener(new MouseAdapter(){
            public void mouseReleased(MouseEvent e){
                adjustBCS(origFile);
            }
        });

        slider3.addMouseListener(new MouseAdapter(){
            public void mouseReleased(MouseEvent e){
                adjustBCS(origFile);
            }
        });

        original = new JScrollPane(new JLabel(new ImageIcon(getScaledFromDisk(origFile))));
        updateOriginal();
    }


    private void updateOriginal(){
        //Image orig = getScaledFromDisk(origFile);

        try{
            pictureBox.remove(0);
        }catch(Exception npe){}

        Box sliderBoxH = Box.createHorizontalBox();
        Box sliderBoxV = Box.createVerticalBox();

        //original = new JScrollPane(new JLabel(new ImageIcon(orig)));
        original.setBorder(BorderFactory.createTitledBorder(""));

        sliderBoxV.add(original);
        sliderBoxV.add(slider2);

        sliderBoxH.add(slider1);
        sliderBoxH.add(sliderBoxV);
        sliderBoxH.add(slider3);

        pictureBox.add(sliderBoxH, 0);

        pictureBox.invalidate(); //necessary when initially triggered by popUp
        pictureBox.validate();

        original.addMouseListener(this);
    }


    private void adjustBCS(File origFile){
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        Image img = getScaledFromDisk(origFile);

        if (slider1.getValue() != 0)
            img = retypar.retypar.adjustBrightness(img, slider1.getValue());

        if (slider2.getValue() != 0)
            img = retypar.retypar.adjustContrast(img, slider2.getValue());

        if (slider3.getValue() != 0)
            img = retypar.retypar.adjustSaturation(img, slider3.getValue());

        original = new JScrollPane(new JLabel(new ImageIcon(img)));

        slider1.setToolTipText(l.pr("adjust brightness: ")  + slider1.getValue());
        slider2.setToolTipText(l.pr("adjust contrast: ")    + slider2.getValue());
        slider3.setToolTipText(l.pr("adjust color: ")       + slider3.getValue());

        updateOriginal();
        setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    }



    /**
     * File dialog to save a converted image. Checks if the file already exists and asks to overwrite it.
     * @see MainFrame#permForOverwrite(java.io.File)
     */
    private boolean storeImage(){
        boolean success = false;
        JFileChooser chooser = new JFileChooser();
        chooser.setCurrentDirectory(currentDir);
        chooser.setMultiSelectionEnabled(false);
        chooser.setAcceptAllFileFilterUsed(false);
        chooser.setApproveButtonText(l.pr("Save"));

        chooser.setDialogTitle(l.pr("Save image..."));

        FileFilter jpeg = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {   "jpg",
                                                "jpeg"};

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < fileType.length; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("JPEG Images");
                    }
                };


        FileFilter gif = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {"gif"};

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < fileType.length; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("GIF Images");
                    }
                };


        FileFilter txt = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {"txt"};

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < fileType.length; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("Textfiles");
                    }
                };

        FileFilter html = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {	"html",
                                                "htm"};

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < fileType.length; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("HTML documents");
                    }
                };

        FileFilter ansi = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {"ans"};

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < fileType.length; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("ANSI files");
                    }
                };

        FileFilter txv = new FileFilter(){
                    public boolean accept(File file){
                        String fileType[] = {"txv"};

                        if (file.isDirectory()) return true;

                        for(int i = 0; i < fileType.length; i++)
                            if (file.getName().toLowerCase().endsWith("." + fileType[i])) return true;

                        return false;}

                    public String getDescription(){
                        return l.pr("Textvideo");
                    }
                };

        if (converter.returnsASCII()){
            chooser.setFileFilter(txt);
            chooser.setFileFilter(html);
            chooser.setFileFilter(ansi);
        }

        if (fileArray.length > 1)
            chooser.setFileFilter(txv);

        chooser.setFileFilter(jpeg);
        chooser.setFileFilter(gif);


        if(chooser.showSaveDialog(this) != JFileChooser.APPROVE_OPTION) return true;


        if (!permForOverwrite(chooser.getSelectedFile())) return true;

        if (chooser.getFileFilter() == jpeg)
            success = retypar.retypar.storeImage(chooser.getSelectedFile(), convertedImage, qualitySlider.getValue(), retypar.retypar.TYPE_JPEG, colored.isSelected(), showColor.getComponent(0).getBackground(), converter.getASCII(), converter.getColors(), false);

        if (chooser.getFileFilter() == gif)
            success = retypar.retypar.storeImage(chooser.getSelectedFile(), convertedImage, qualitySlider.getValue(), retypar.retypar.TYPE_GIF, colored.isSelected(), showColor.getComponent(0).getBackground(), converter.getASCII(), converter.getColors(), opacityBox.isSelected());

        if (chooser.getFileFilter() == txt)
            success = retypar.retypar.storeImage(chooser.getSelectedFile(), convertedImage, qualitySlider.getValue(), retypar.retypar.TYPE_TEXT, colored.isSelected(), showColor.getComponent(0).getBackground(), converter.getASCII(), converter.getColors(), false);

        if (chooser.getFileFilter() == html)
            success = retypar.retypar.storeImage(chooser.getSelectedFile(), convertedImage, qualitySlider.getValue(), retypar.retypar.TYPE_HTML, colored.isSelected(), showColor.getComponent(0).getBackground(), converter.getASCII(), converter.getColors(), false);

        if (chooser.getFileFilter() == ansi)
            success = retypar.retypar.storeImage(chooser.getSelectedFile(), convertedImage, qualitySlider.getValue(), retypar.retypar.TYPE_ANSI, colored.isSelected(), showColor.getComponent(0).getBackground(), converter.getASCII(), converter.getColors(), false);

        if (chooser.getFileFilter() == txv){
            setButtonState(false);
            setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

            cancelButton.setEnabled(true);
            paint(this.getGraphics());

            int i = 0;
            while (i < fileArray.length){
                if (!convertImage(fileArray[i]))
                    return false;

                success = retypar.retypar.storeImage(chooser.getSelectedFile(), convertedImage, qualitySlider.getValue(), retypar.retypar.TYPE_TVID, colored.isSelected(), showColor.getBackground(), converter.getASCII(), converter.getColors(), false);
                i++;
                walker.setValue(100/fileArray.length*i);
                walker.updateUI();
                }
            duration.setText("--:--:---");
            walker.setValue(100);
            setButtonState(true);
            setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            cancelButton.setEnabled(false);
            validate();
            }

        convertedImage.flush();
        return success;
    }


    /**
     * Converts the image using {@link retypar#triggerConversion()}.
     */
    private void triggerConversion(){
        setButtonState(false);
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        cancelButton.setEnabled(true);
        paint(this.getGraphics());

        converter = (Converter)renderCombo.getSelectedItem();

        walker.setEnabled(true);
        if (converter.monitorsProgess()) walker.walk();
            else walker.setEnabled(false);

        try{
                pictureBox.remove(converted);
            }catch(Exception npe){}


        if (!convertImage(imgF))
                return;

        imConvertedImage = getScaled(reEncodeConverted(convertedImage, qualitySlider.getValue()));
        converted = new JScrollPane(new JLabel(new ImageIcon(imConvertedImage)));
        converted.setBorder(BorderFactory.createTitledBorder(convertedImage.getWidth() + "x" + convertedImage.getHeight() + ", " + convFileSizeKB + "kB"));
        converted.addMouseListener(this);
        pictureBox.add(converted);
        setButtonState(true);
        setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        cancelButton.setEnabled(false);
        validate();
        duration.setText(retypar.retypar.getDuration());
    }



    private boolean convertImage(File imageFile){
        int threshold = ((Integer)thresholdSpinner.getValue()).intValue();
        //int renderMode = retypar.retypar.RENDER_MRULES;
        converter = (Converter)renderCombo.getSelectedItem();
        //if (renderCombo.getSelectedItem().toString().equals("Noise")) renderMode = retypar.retypar.RENDER_NOISE;

        retypar.retypar.setFont((String)fontCombo.getSelectedItem());
        retypar.retypar.setFontSize(((Integer)fontSizeSpinner.getValue()).intValue());
        convertedImage = retypar.retypar.convertImage(imageFile,
                                                    uppercase.isSelected(),
                                                    lowercase.isSelected(),
                                                    digits.isSelected(),
                                                    specials.isSelected(),
                                                    colored.isSelected(),
                                                    showColor.getComponent(0).getBackground(), //bgColor of showColor2
                                                    ((Integer)scaleSpinner.getValue()).intValue(),
                                                    slider1.getValue(),
                                                    slider2.getValue(),
                                                    slider3.getValue(),
                                                    converter,
                                                    threshold);
        retypar.retypar.stopTimer();

        if (converter.getErrors() != null) {
            popUpAlgoError(converter.getErrors());
            return false;
        }
        return true;
    }


    /**
     * Checks the state of the checkboxes ("A-Z", "a-z", ...) and disables conversion button if none is selected.
     */
    private void checkOptions(){
        if (uppercase.isSelected() | lowercase.isSelected() | digits.isSelected() && imgF!=null)
            convertButton.setEnabled(true);
        else convertButton.setEnabled(false);
    }


    private void displayOptions(){
        OptionsFrame opt = new OptionsFrame(this);
    }

    private void displayAbout(){
        String message, title, thirdPartyCredits;
        String algos = "";

        // credit authors of third party software
        thirdPartyCredits = "AnimatedGifEncoder 1.03 by Kevin Weiner, FM Software";
        thirdPartyCredits +="<br>ImagePreviewPanel by Michael Urban, javalobby.org";


        // compile list of installed converters
        for (int i = 0; i < retypar.retypar.converters.size(); i++){
            algos += "<br><b>" +
                    (((Converter)retypar.retypar.converters.get(i)).toString() +
                    "</b> (" +
                    ((Converter)retypar.retypar.converters.get(i)).getShortName() +
                    ") by " +
                    ((Converter)retypar.retypar.converters.get(i)).getAuthor() +
                    " (<a href=\"" + ((Converter)retypar.retypar.converters.get(i)).getWebsite() + "\">" +
                    ((Converter)retypar.retypar.converters.get(i)).getWebsite() +
                    "</a>)");
        }

        message =   "<HTML><table><tr><td><i>Author:</i></td><td>Jens Heuser</td></tr>" +
                    "<tr><td><i>Homepage:</i></td><td><a href=\""+ retypar.retypar.homepage + "\">" + retypar.retypar.homepage + "</a></td></tr>" +
                    "<tr><td><i>Email:</i></td><td><a href=\"mailto:" + retypar.retypar.email + "\">"   + retypar.retypar.email    + "</a></td></tr>" +
                    "<tr><td><i>Version:</i></td><td>" + retypar.retypar.version  + "</td></tr></table>" +
                    "<br><br><i>Installed algorithms:</i>" +
                    algos +
                    "<br><br><i>Third party software:</i><br>" +
                    thirdPartyCredits
                    +"</HTML>";

        title = l.pr("About") + " retypar";

        JOptionPane.showMessageDialog(this, message, title, JOptionPane.INFORMATION_MESSAGE);
    }

    private void checkThresholdSpinner(){
        if (renderCombo.getItemCount() == 0) return;
        Converter cv = (Converter)renderCombo.getSelectedItem();

        if(cv.usesThreshold()){
            int init= cv.getInitThreshold();
            int min = cv.getMinThreshold();
            int max = cv.getMaxThreshold();
            int step= cv.getThresholdStep();

            thresholdSpinner.setModel(new SpinnerNumberModel(init, min, max, step));
            thresholdSpinner.setEditor(new JSpinner.NumberEditor(thresholdSpinner));
            thresholdSpinner.setEnabled(true);
        }
        else if (!cv.usesThreshold()) thresholdSpinner.setEnabled(false);
    }


    private void checkOutputFormats(){
        if (renderCombo.getItemCount() == 0) return;
        Converter cv = (Converter)renderCombo.getSelectedItem();

        //if algorithm does not support getASCII(), switch selected item to JPEG
        if(!cv.returnsASCII()){
            if (exportFormatCombo.getSelectedIndex() > 1)
                exportFormatCombo.setSelectedIndex(0);

            removeTextExport();
        }
        else{
            addAllExport();
        }
    }


    private void removeTextExport(){// method no used
        try{
            if (exportFormatCombo.getItemCount() < 3) return;

            ((DefaultComboBoxModel)exportFormatCombo.getModel()).removeElement("Text");
            ((DefaultComboBoxModel)exportFormatCombo.getModel()).removeElement("HTML");
            ((DefaultComboBoxModel)exportFormatCombo.getModel()).removeElement("ANSI");
            ((DefaultComboBoxModel)exportFormatCombo.getModel()).removeElement("TVID");
        }catch(Exception e){}
    }

    private void addAllExport(){// method not used
        try{
            if (exportFormatCombo.getItemCount() > 2) return;

            exportFormatCombo.setModel(new DefaultComboBoxModel(retypar.retypar.availableGraphicalExportFormats));
        }catch(Exception e){}
    }


    /**
     * Scales the given image to fit the GUI.
     * @param img The image to scale.
     * @return The scaled image.
     */
    private Image getScaled(BufferedImage img){
        float wB, hB;
        float scaleFactor;

        wB = pictureBox.getWidth();
        hB = (pictureBox.getHeight() / 2) * .9f;

        wI = img.getWidth();
        hI = img.getHeight();


        if ((wI < wB) && (hI < hB)) return img;

        scaleFactor = wB / wI;

        if(hI * scaleFactor <= hB) return img.getScaledInstance((int)(scaleFactor*wI), -1, BufferedImage.SCALE_SMOOTH);
                else scaleFactor = hB / hI;
        return img.getScaledInstance(-1, (int)(scaleFactor*hI), BufferedImage.SCALE_SMOOTH);
    }


    /**
     * Loads an image and scales it to fit the GUI. Uses {@link MainFrame#getScaled(java.awt.image.BufferedImage)}.
     * @param file The image to load.
     * @return The scaled image.
     */
    private Image getScaledFromDisk(File file){
        try{
            origFileSizeKB = file.length() / 1024;
            BufferedImage img = ImageIO.read(file);
            return getScaled(img);
        }catch (Exception e){
            return new BufferedImage(100,100,BufferedImage.TYPE_BYTE_BINARY);
        }
    }


    /**
     * A modal JFrame showing the converted image in its original size.
     */
    private void viewConverted(){
        JFrame cv = new JFrame(l.pr("ASCII version, " + convFileSizeKB + "KB"));
        JScrollPane sc = new JScrollPane(new JLabel(new ImageIcon(reEncodeConverted(convertedImage, qualitySlider.getValue()))));
        cv.getContentPane().add(sc);
        cv.setBounds(getWidth()/2, getHeight()/2,getWidth()/2, getHeight()/2);
        cv.setVisible(true);
        cv.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        cv.setExtendedState(JFrame.MAXIMIZED_BOTH);
    }


    /**
     * Enables or disables the buttons. Useful while converting.
     * @param state <code>true</code> enables buttons, <code>false</code> disables them.
     */
    private void setButtonState(boolean state){
        loadButton.setEnabled(state);
        storeButton.setEnabled(state);
        convertButton.setEnabled(state);
        viewConvertedButton.setEnabled(state);
        optionsBox1.setEnabled(state);
        qualitySlider.setEnabled(state);
        exportFormatCombo.setEnabled(state);
        paint(this.getGraphics());
    }


    /**
     * Checks wether the given File exists and asks for permission to overwrite it. Used by {@link MainFrame#storeImage()}.
     * @param file The file to check.
     * @return <code>true</code> if file may be overwritten.
     */
    public boolean permForOverwrite(File file){
        if (!file.exists()) return true;

        if (JOptionPane.showConfirmDialog(this, l.pr("File already exists. Overwrite?"), l.pr("Warning!"), JOptionPane.YES_NO_OPTION)
                == JOptionPane.YES_OPTION) return true;

        return false;
    }


    /**
     * Immediately stop conversion. Useful, but does not work...
     */
    private void cancelConversion(){
        System.out.println("cancelling at " + walker.getValue() + "%");
        converter.stop();
   }

    private void doPossibleUpdate(){//todo: into core
        if (!retypar.retypar.checkUpd)
            return;

        if (!Update.newVersionAvailable(retypar.retypar.appName, retypar.retypar.version, retypar.retypar.updateURL))
            return;

        if (JOptionPane.showConfirmDialog(this, l.pr("There is a newer version available. Install?"), l.pr("Internet update"), JOptionPane.YES_NO_OPTION)
                == JOptionPane.YES_OPTION) Update.installNew();

        if (Update.error) JOptionPane.showMessageDialog(this, Update.getErrors(), l.pr("Error while updating!"), JOptionPane.ERROR_MESSAGE);
    }

    private void switchLang(String language){
        //if (l.getTargetLang().equals(language)) return;
        l.setLang(language);
        retypar.retypar.lang = language;
        labelComponents();
        retypar.retypar.plaf = UIManager.getSystemLookAndFeelClassName();

        if (!l.error)
            retypar.retypar.writeINI();
        else
            JOptionPane.showMessageDialog(this, l.pr(l.getError()),
                                                    l.pr("Error in multi language module!"),
                                                    JOptionPane.WARNING_MESSAGE);
        l.clearErrors();
    }


    private void checkMaxRes(){
        if(Toolkit.getDefaultToolkit().getScreenSize().height < 768){
            JOptionPane.showMessageDialog(this, "Your screen resolution is too low. Please use the command line interface."
                                                +" \nSee 'java -jar " + retypar.retypar.appName + "_"  + retypar.retypar.version + " -?'",
                                            "Fatal error:", JOptionPane.ERROR_MESSAGE);
            System.exit(1);
        }
    }



    private void updateConverted(){
        pictureBox.remove(converted);
        imConvertedImage = reEncodeConverted(convertedImage, qualitySlider.getValue());
        converted = new JScrollPane(new JLabel(new ImageIcon(getScaled((BufferedImage)imConvertedImage))));
        converted.setBorder(BorderFactory.createTitledBorder(convertedImage.getWidth() + "x" + convertedImage.getHeight() + ", " + convFileSizeKB + "kB"));
        pictureBox.add(converted);
        pictureBox.validate();
    }



    private BufferedImage reEncodeConverted(BufferedImage img, int quality){
        if (exportFormatCombo.getSelectedItem().toString().equals("JPEG"))
            return reEncodeConvertedAsJPEG(img, quality);

        if (exportFormatCombo.getSelectedItem().toString().equals("GIF"))
            return reEncodeConvertedAsGIF(img, quality);

        /**
        if (exportFormatCombo.getSelectedItem().toString().equals("Text"))
            return reEncodeConvertedAsASCII(img, retypar.retypar.TYPE_TEXT);

        if (exportFormatCombo.getSelectedItem().toString().equals("HTML"))
            return reEncodeConvertedAsASCII(img, retypar.retypar.TYPE_HTML);

        if (exportFormatCombo.getSelectedItem().toString().equals("ANSI"))
            return reEncodeConvertedAsASCII(img, retypar.retypar.TYPE_ANSI);**/

        // txt, html and ansi displayed as uncompressed jpeg
        //return reEncodeConvertedAsJPEG(img, 0);
        return null;
    }


    private BufferedImage reEncodeConvertedAsASCII(BufferedImage img, int type){
        //use uncompressed JPEG to resemble ASCII formats
        img = reEncodeConvertedAsJPEG(img, 0);

        switch (type){
            case retypar.retypar.TYPE_TEXT:
                convFileSizeKB = converter.getASCII().length * converter.getASCII()[0].length() / 1024;
                converted.setBorder(BorderFactory.createTitledBorder(converter.getASCII()[0].length() + "x" + converter.getASCII().length + ", " + converter.getASCII().length * converter.getASCII()[0].length() + "B"));
                break;
            case retypar.retypar.TYPE_HTML:
                convFileSizeKB = 0;
                converted.setBorder(BorderFactory.createTitledBorder(converter.getASCII()[0].length() + "x" + converter.getASCII().length + ", ??? kB"));
                break;
            case retypar.retypar.TYPE_ANSI:
                convFileSizeKB = 0;
                converted.setBorder(BorderFactory.createTitledBorder(converter.getASCII()[0].length() + "x" + converter.getASCII().length + ", ??? kB"));
                break;
        }

        return img;
    }


    private BufferedImage reEncodeConvertedAsJPEG(BufferedImage img, int quality){
        ByteArrayOutputStream out;
        BufferedImage bi = new BufferedImage((int)wI, (int)hI, BufferedImage.TYPE_INT_RGB);

        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        try{
            out = new ByteArrayOutputStream();
            retypar.retypar.streamJPEG(out, img, quality);

            InputStream in = new ByteArrayInputStream(out.toByteArray());
            JPEGImageDecoder decoder = JPEGCodec.createJPEGDecoder(in);
            bi = decoder.decodeAsBufferedImage();

            convFileSizeKB = out.size() / 1024;

        }catch(Exception e){
            retypar.retypar.logError("JPEG error: " + e.toString());
        }
        img.flush();
        bi.flush();

        setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        return bi;
    }


    private BufferedImage reEncodeConvertedAsGIF(BufferedImage img, int quality){
        Color transparentColor;
        ByteArrayOutputStream out;
        BufferedImage bi = new BufferedImage((int)wI, (int)hI, BufferedImage.TYPE_INT_RGB);

        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        if (opacityBox.isSelected())
            transparentColor = showColor.getBackground();
        else
            transparentColor = null;

        try{
            out = new ByteArrayOutputStream();
            retypar.retypar.streamGIF(out, img, quality, transparentColor);

            GifDecoder decoder = new GifDecoder();
            InputStream in = new ByteArrayInputStream(out.toByteArray());
            decoder.read(in);
            bi = decoder.getImage();

            convFileSizeKB = out.size() / 1024;

        }catch(Exception e){
            retypar.retypar.logError("GIF error: " + e.toString());
        }
        img.flush();
        bi.flush();

        setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        return bi;
    }
}
