/**
 * retypar - a tool to convert images into ASCII<br>
 * Creates images out of letters and numbers (colored or b/w) from a given picture.
 *
 * @author Jens Heuser, 2005; heuserjens@users.sourceforge.net
 * @version 0.9.4
 */
package retypar;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageEncoder;
import heuser.simpleLogger.SimpleLogger;
import retypar.converter.Converter;
import retypar.converter.helper.ImageProcessor;
import retypar.gui.MainFrame;
import thirdParty.GifEncoder.AnimatedGifEncoder;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.LinkedList;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;



/**
 * retypar - a tool to convert images into ASCII (Main Class)<br>
 * Creates images out of letters and numbers (colored or b&w) from a given picture.
 *
 * @author Jens Heuser, 2005; heuserjens@users.sourceforge.net
 * @version 0.9.4
 * @see retypar.gui
 */
public class retypar extends SimpleLogger{

    /**
     * Current Version. Needed by package {@link heuser.updater}
     */
    public static final String version      = "0.9.4";
    /**
     * The URL with a possibly newer version. Needed by package {@link heuser.updater}
     */
    public static final String updateURL    = "http://retypar.sourceforge.net/download/";
    /**
     * The name of this application. Needed by package {@link heuser.updater}
     */
    public static final String appName      = "retypar";


    public static final String homepage     = "http://www.retypar.de.vu";

    public static final String email        = "retypar.image2ascii@gmail.com";

    private static final String textVideoVersion = "1";


    /**
     * Download new algorithms from this URL
     */
    private static final String algosOnline = "http://retypar.sourceforge.net/download/algorithms/";

    /**
     * The directory external algorithms get installed to.
     */
    private static String algoSubDir        = "retypar/converter/";


    /**
     * Path of the last opened image.
      */
    public static String lastPath = "";

    /**
     * The GUI.
     * @see retypar.gui.MainFrame
     */
    private static MainFrame mF;

    /**
     * The converter itself. <code>static</code>, so that the letters don't have to be calculated over and over again.
     */
    //public static ImageProcessor imP;

    /**
     * The current progress of conversion. Accessed by {@link MainFrame#walker}.
     */
    //public static double progress = 0;

    public static int fontSize = 10;
    public static String font = "DialogInput";
    public static Font fonts[] = GraphicsEnvironment.getLocalGraphicsEnvironment().getAllFonts();

    public static String[] availableGraphicalExportFormats = {"JPEG", "GIF"};

    public static final int TYPE_TEXT   = 0;
    public static final int TYPE_HTML   = 1;
    public static final int TYPE_JPEG   = 2;
    public static final int TYPE_TVID   = 3;
    public static final int TYPE_ANSI   = 4;
    public static final int TYPE_GIF    = 5;

    public static final String[] EXTENSIONS_TYPE_JPEG   = new String[]{"jpg", "jpeg"};
    public static final String[] EXTENSIONS_TYPE_GIF    = new String[]{"gif"};
    public static final String[] EXTENSIONS_TYPE_HTML   = new String[]{"htm", "html"};
    public static final String[] EXTENSIONS_TYPE_TEXT   = new String[]{"txt"};
    public static final String[] EXTENSIONS_TYPE_TVID   = new String[]{"txv"};
    public static final String[] EXTENSIONS_TYPE_ANSI   = new String[]{"ans"};


    /**
     * @see AnimatedGifEncoder
     */
    public static final int GIF_MAX_QUALITY_IN_BITS = 24;

    private static File input;
    private static File output;

    //public final static int RENDER_MRULES = 0;
    //public final static int RENDER_NOISE = 1;

    public static LinkedList converters = new LinkedList();

    private static Calendar now;
    private static String duration = "--:--:---";
    public static Converter activeConverter;

    public static boolean checkUpd = true;
    public static String plaf;
    public static String lang;
    public static LinkedList markedForDeletion = new LinkedList();
    public static boolean runningInJar = false;
    private static boolean importingExternal = false;
    private static String fileSeparator = System.getProperty("file.separator");
    public static String basedir = "";


    /**
     * Creates instances of {@link ImageProcessor} and {@link MainFrame}.<br>
     * Control is passed to MainFrame if no command-line arguments are given.
     * @param args See <I>java -jar retypar_{version}.jar -h</I>.
     */
    public static void main(String[] args){
        setErrorPrefix("*** ");
        setConsoleOutput(true);
        setBasedir();

        readINI();
        converters = loadConverters();
        converters.addAll(importConverters());

        if (args.length > 0) runInBatchMode(args);
            else{
                logMsg("\n--- " + appName + " " + version + " (Jens Heuser, 2005) ---\n");
                logMsg("No arguments given. Starting GUI...\n");
                //setConsoleOutput(false);
                mF = new MainFrame();
        }
    }


    private static void setBasedir(){
        basedir = retypar.class.getResource("retypar.class").toString();
        if(basedir.startsWith("jar:")){
            basedir = basedir.substring(10, basedir.length());
            basedir = basedir.substring(0, basedir.lastIndexOf("retypar_" + version + ".jar!"));
        }
        if(basedir.startsWith("file:")){
            basedir = basedir.substring(6, basedir.length());
            basedir = basedir.substring(0, basedir.lastIndexOf("retypar/retypar.class"));
        }
        //System.out.println("basedir " + basedir);
    }


    protected static LinkedList loadConverters(){
        return loadConverters(null);
    }


    protected static LinkedList loadConverters(File fileX){
        LinkedList converters = new LinkedList();
        String path;
        String files[];
        Converter conv;
        File file;

        if (fileX == null){
            URL url = retypar.class.getResource("/retypar/converter/");

            path = url.toString();
            if (path.startsWith("jar:"))
                path = url.toString().substring(4);

            file = new File(path.substring(6));
        }
        else {
            file = fileX;
            path = fileX.getAbsolutePath();
        }

        if ((!file.isDirectory()) && (!file.isFile()))
            files = scanJar(file);
        else files = file.list();

        if (files == null) return converters;

        //System.out.println("loading from path " + path);
        //for(int x = 0; x < files.length; x++) System.out.println("\t" + files[x]);



        for (int i = 0; i < files.length; i++){

            if (files[i].endsWith(".class")){

                ClassLoader cl = new AlgoClassLoader();

                try{
                    if ((!importingExternal) || (!runningInJar)) conv = (Converter)retypar.class.getClassLoader().loadClass("retypar.converter." + files[i].substring(0, files[i].length() - 6)).newInstance();
                    else {
                        conv = (Converter)cl.loadClass(path + System.getProperty("file.separator") + files[i]).newInstance();
                        String packageString = "";
                        if ((packageString = conv.broughtPackage()) != null){
                            File f = new File(path + System.getProperty("file.separator") + packageString);
                            String[] fl = f.list();
                            for (int z = 0; z < fl.length; z++){
                                //System.out.println("caching " + fl[z]);
                                if(!files[z].endsWith(".class")) continue;
                                cl.loadClass(path + System.getProperty("file.separator") + packageString + System.getProperty("file.separator") + fl[z]).newInstance();
                            }
                        }
                    }


                    if (Converter.class.isInstance(conv)){
                        converters.add(conv);
                        //System.out.println("added " + conv.toString());
                    }
                }catch(Exception e) {
                    //System.out.println("not added " + files[i] + "\n" + e.toString());
                    if (!   (files[i].equals("Converter.class")
                            || files[i].equals("FontRenderer.class")
                            || files[i].equals("ImageProcessor.class")
                            || files[i].equals("Parameter.class")
                            || files[i].equals("Letter.class")))
                    logError("Could not load algorithm file '" + files[i] + "': " + e.toString());
                }
            }
        }
        //if (deletedAlgo) writeINI();

        return converters;
    }


    private static LinkedList importConverters(){
        importingExternal = true;
        //System.out.println("IMPORTING");
        LinkedList imported = new LinkedList();
        if (!runningInJar) return imported;
        String basedir = "";
        File fileX;

        basedir = retypar.class.getResource("retypar.class").toString();
        basedir = basedir.substring(0, basedir.lastIndexOf(appName + "_" + version + ".jar"));
        basedir += algoSubDir;

        if (basedir.startsWith("jar:"))
            basedir = basedir.substring(10);
        else basedir = basedir.substring(6);

        fileX = new File(basedir);
        fileX.mkdir();

        imported = loadConverters(fileX);

        importingExternal = false;
        return imported;
    }


    private static String[] scanJar(File file){
        if (file.getAbsolutePath().indexOf(".jar!") == -1) return null;
        //System.out.println("scanning jar " + file.getAbsolutePath());
        runningInJar = true;
        String files[];
        LinkedList fileList = new LinkedList();
        JarFile jFile;
        String rootPrefix = "";

        try{
            if (System.getProperty("os.name").toLowerCase().indexOf("linux") > -1)
                rootPrefix = "/";

            jFile = new JarFile(rootPrefix + file.getParentFile().getParent().replaceAll("!", ""));
            Enumeration enumeration = jFile.entries();
            while(enumeration.hasMoreElements()){
                String obj = enumeration.nextElement().toString();
                if (obj.indexOf("retypar/converter/") > -1){
                    if (obj.substring(obj.lastIndexOf('/')).length() < 2) continue;
                    //System.out.println("appending " + obj.substring(obj.lastIndexOf('/') + 1));
                    fileList.add(obj.substring(obj.lastIndexOf('/') + 1));
                }
            }
        }catch(Exception e){
            //logError("Error accessing jar "+ file.getParentFile().getParent().replaceAll("!", "") + ": " + e.toString());
            logError("Error accessing jar "+ file.getPath() + "! Stacktrace:");
            //e.printStackTrace();
        }

        files = new String[fileList.size()];
        int i = 0;
        while(!fileList.isEmpty()){
            files[i] = (String)fileList.removeLast();
            i++;
        }
        return files;
    }


    private static void deleteDirectory(File dir){
        //System.out.println("removing directory " + dir.getAbsolutePath() + ", " + dir.getName());
        File files[] = dir.listFiles();
        //System.out.println("dir: " + dir.getAbsolutePath());

        /*for (int v = 0; v < files.length; v++)
            System.out.println("entry " + files[v].getAbsolutePath());*/

        for (int i = 0; i < files.length; i++){
            if (files[i].isDirectory()) {
                deleteDirectory(files[i]);  //endlosschleife!
                files[i].delete();
            }
            files[i].delete();
            //System.out.println("removing directory file " + files[i].getName() + " ..." + files[i].delete());
            //System.out.println("\tdeleted " + files[i].toString());
        }

        boolean dirDeleted = dir.delete();
        //System.out.println("removing dir " + dir.getName() + " ..." + dirDeleted);
        //System.out.println("\tdeleted " + dir.toString());
    }


    /**
     * Removes the given algorithm.
     * @param c
     */
    public static void deleteAlgo(Converter c){
        String broughtPackage = "";

        try{
            broughtPackage = c.broughtPackage();
        }catch(Exception e){
            broughtPackage = "";
        }

        String path = c.getClass().getName();
        path = path.replace('.', fileSeparator.charAt(0));


        File f = new File(basedir + fileSeparator + path + ".class");
        if (f.exists()) f.delete();
            //System.out.println("deleting " + path + ".class ..." + (deletedAlgo = f.delete()));

        if (broughtPackage.length() > 0){
            String newFile = basedir + "retypar" + fileSeparator + "converter" + fileSeparator + broughtPackage;
            //System.out.println(newFile);
            deleteDirectory(new File(newFile));
        }

        converters.remove(c);
        //writeINI();
        converters = loadConverters();
        converters.addAll(importConverters());
    }



    /**
     * Checks if there is enough memory to convert an image of the given size (in Bytes).
     * Currently simply multiplies the number of pixels by 4.5. <code>Runtime.getRuntime().freeMemory()</code> as comparison.
     * @param picSize The size of an image in Bytes.
     * @return <code>true</code> if <code>Runtime.getRuntime().freeMemory() >= picSize</code>.
     */
    public static boolean hasEnoughMemoryToConvert(int picSize){
        //System.out.println(Runtime.getRuntime().freeMemory() + "\n" + picSize);
        return Runtime.getRuntime().freeMemory() >= picSize * 3 * 1.5;
    }


    /**
     * Initiates conversion with the given parameters and returns the converted image.
     * @param img The image to convert.
     * @param uppercase
     * @param lowercase
     * @param numbers
     * @param colored
     * @param special
     * @param scale The factor to scale the image before converting.
     * @return The converted image.
     */
    public static BufferedImage convertImage(File img,
                                             boolean uppercase,
                                             boolean lowercase,
                                             boolean numbers,
                                             boolean special,
                                             boolean colored,
                                             Color bgColor,
                                             int scale,
                                             int brightness,
                                             int contrast,
                                             int saturation,
                                             Converter converter,
                                             int threshold){

        Image image;

        try{
            image = ImageIO.read(img);
        }catch(Exception e){
            logError("Could not read image '" + img.getPath() + "'!");

            return null;
        }

        if (brightness != 0)
            image = adjustBrightness(image, brightness);

        if (contrast != 0)
            image = adjustContrast(image, contrast);

        if (saturation != 0)
            image = adjustSaturation(image, saturation);

        converter.prepare(image, uppercase, lowercase, numbers, special, colored, bgColor, scale, threshold);

        activeConverter = converter;

        startTimer();

        if (activeConverter.returnsImage()) return activeConverter.getImage();
            else return drawImageFromASCII(activeConverter.getASCII(), activeConverter.getColors(), img, colored, bgColor, scale);
    }


    public static int getProgress(){
        try{
            if (!activeConverter.monitorsProgess()) return 101;
        }catch(Exception e){
            return 101;
        }
        return activeConverter.getProgress();
    }


    /**
     * Called before every conversion to measure its duration.
     * @see retypar#getDuration()
     * @see retypar#stopTimer()
     */
    private static void startTimer(){
        now = GregorianCalendar.getInstance();
    }


    /**
     * Called after every conversion to measure its duration.
     * @see #getDuration()
     * @see #startTimer()
     */
    public static void stopTimer(){
        long time1, time2, dur;
        String tmp;

        Calendar fin = GregorianCalendar.getInstance();
        time1 = now.getTimeInMillis();
        time2 = fin.getTimeInMillis();

        dur = time2 - time1;

        tmp = "" + dur/60000;
        if (tmp.length() == 1) tmp = "0" + tmp;
        duration = tmp;

        tmp = "" + dur/1000;
        if (tmp.length() == 1) tmp = "0" + tmp;
        duration += ":" + tmp;

        tmp = "" + dur%1000;
        if (tmp.length() == 1) tmp = "0" + tmp;
        duration += ":" + tmp;
    }


    /**
     * Returns the duration of the last conversion.
     * @return The duration of the last conversion as <code>String</code> (m:s:ms)
     * @see #startTimer()
     * @see #stopTimer()
     */
    public static String getDuration(){
        return duration;
    }


    private static BufferedImage drawImageFromASCII(String[] ascii, Color[][] colors, File img, boolean colored, Color bgColor, int scale){
        BufferedImage bImg;
        int w, h, t;
        Graphics2D  g;

        try{
            bImg = ImageIO.read(img);
            w = bImg.getWidth();
            h = bImg.getHeight();
        }catch(Exception e){
            logError("Error drawing BufferedImage from ASCII.");
            return null;
        }

        if (colored) t = BufferedImage.TYPE_INT_RGB;
            else t = BufferedImage.TYPE_BYTE_GRAY;

        if (scale != 100){
            w = (w * scale / 100);
            h = (h * scale / 100);
        }

        w = (w / fontSize) * fontSize;
        h = (h / fontSize) * fontSize;

        bImg = new BufferedImage(w, h, t);
        g = bImg.createGraphics();
        g.setBackground(bgColor);
        g.clearRect(0, 0, w, h);
        g.setFont(new Font(font, Font.PLAIN, fontSize+1));
        g.setColor(Color.BLACK);

        for (int i = 0; i < ascii.length; i++){
            for (int j = 0; j < ascii[i].length(); j++){
                if (colored)
                    g.setColor(colors[i][j]);
                g.drawString("" + ascii[i].charAt(j), j * fontSize, i * fontSize + fontSize);
            }
        }
        return bImg;
    }


    public static Image adjustBrightness(Image img, int level){
        BufferedImage bImg;

        bImg = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
        bImg.getGraphics().drawImage(img, 0, 0, img.getWidth(null), img.getHeight(null), null);

        level = (int)((float)255 * ((float)level / 100));

        int r = 0;
        int g = 0;
        int b = 0;

        int rgb = 0;

        for (int i = 0; i < bImg.getWidth(); i++)
            for (int j = 0; j < bImg.getHeight(); j++){
                rgb = bImg.getRGB(i, j);

                r = (int)(rgb & 0xFF0000) >> 16;
                g = (int)(rgb & 0x00FF00) >> 8;
                b = (int)(rgb & 0x0000FF);

                r = r + level;
                g = g + level;
                b = b + level;

                if (r > 255) r = 255;
                if (g > 255) g = 255;
                if (b > 255) b = 255;

                if (r < 0) r = 0;
                if (g < 0) g = 0;
                if (b < 0) b = 0;

                rgb = ((r << 16) & 0xFF0000) | ((g << 8) & 0x00FF00) | (b & 0x0000FF);

                bImg.setRGB(i, j, rgb);
            }

        return bImg;
    }


    public static Image adjustContrast(Image img, int level){
        BufferedImage bImg;

        bImg = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
        bImg.getGraphics().drawImage(img, 0, 0, img.getWidth(null), img.getHeight(null), null);

        int contrast[] = new int[256];

        level = (int)((float)255 * ((float)level / 100));

        int r = 0;
        int g = 0;
        int b = 0;

        int rgb = 0;

        for (int i = 0; i < 128; i++)
            contrast[i] = i - ((Math.abs(128 - i) * level) / 255);

        for (int i = 128; i < 256; i++)
            contrast[i] = i + ((Math.abs(128 - i) * level) / 255);


        for (int i = 0; i < bImg.getWidth(); i++)
            for (int j = 0; j < bImg.getHeight(); j++){
                rgb = bImg.getRGB(i, j);

                r = (int)(rgb & 0xFF0000) >> 16;
                g = (int)(rgb & 0x00FF00) >> 8;
                b = (int)(rgb & 0x0000FF);

                r = contrast[r];
                g = contrast[g];
                b = contrast[b];


                if (r > 255) r = 255;
                if (g > 255) g = 255;
                if (b > 255) b = 255;

                if (r < 0) r = 0;
                if (g < 0) g = 0;
                if (b < 0) b = 0;


                rgb = ((r << 16) & 0xFF0000) | ((g << 8) & 0x00FF00) | (b & 0x0000FF);

                bImg.setRGB(i, j, rgb);
            }

        return bImg;
    }


    public static Image adjustSaturation(Image img, int level){
        BufferedImage bImg;

        bImg = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
        bImg.getGraphics().drawImage(img, 0, 0, img.getWidth(null), img.getHeight(null), null);

        level = (int)((float)255 * ((float)level / 100));

        int r = 0;
        int g = 0;
        int b = 0;

        int rgb = 0;

        for (int i = 0; i < bImg.getWidth(); i++)
            for (int j = 0; j < bImg.getHeight(); j++){
                rgb = bImg.getRGB(i, j);

                r = (int)(rgb & 0xFF0000) >> 16;
                g = (int)(rgb & 0x00FF00) >> 8;
                b = (int)(rgb & 0x0000FF);

                r = (int)(r + (float)(r * ((float)level / 100)));
                g = (int)(g + (float)(g * ((float)level / 100)));
                b = (int)(b + (float)(b * ((float)level / 100)));

                if (r > 255) r = 255;
                if (g > 255) g = 255;
                if (b > 255) b = 255;

                if (r < 0) r = 0;
                if (g < 0) g = 0;
                if (b < 0) b = 0;

                rgb = ((r << 16) & 0xFF0000) | ((g << 8) & 0x00FF00) | (b & 0x0000FF);

                bImg.setRGB(i, j, rgb);
            }

        return bImg;
    }



    public static boolean storeImage(File file, BufferedImage convertedImage, int quality, int type, boolean colored, Color bgColor, String[] txtVerMono, Color[][] colors, boolean isTransparent){
        //if(file == null) file = new File("ASCII of " + input.getName());
        switch (type){
            case TYPE_JPEG:
                return storeJPEG(checkExtension(file, EXTENSIONS_TYPE_JPEG), convertedImage, quality);
            case TYPE_GIF:
                return storeGIF(checkExtension(file, EXTENSIONS_TYPE_GIF), convertedImage, quality, isTransparent? bgColor : null);
            case TYPE_HTML:
                return storeHTML(checkExtension(file, EXTENSIONS_TYPE_HTML), colored, bgColor, txtVerMono, colors);
            case TYPE_TEXT:
                return storeTEXT(checkExtension(file, EXTENSIONS_TYPE_TEXT), txtVerMono);
            case TYPE_TVID:
                return storeTextVideo(checkExtension(file, EXTENSIONS_TYPE_TVID), txtVerMono);
            case TYPE_ANSI:
                return storeANSI(checkExtension(file, EXTENSIONS_TYPE_ANSI), bgColor, txtVerMono, colors);
        }

        return false;
    }


    private static boolean storeJPEG(File file, BufferedImage convertedImage, int quality){
        if (file == null)
                    return true;

        BufferedOutputStream out;
        try{
            out = new BufferedOutputStream(new FileOutputStream(file));

            if (!streamJPEG(out, convertedImage, quality))
                throw(new Exception());

            out.close();
        }catch (Exception e) {
            logError("Output file error: " + e.toString());
            return false;
        }

        return true;
    }


    public static boolean streamJPEG(OutputStream out, BufferedImage convertedImage, int quality){

        try{
            JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
            JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(convertedImage);

            param.setQuality((float)(100-quality) / 100.0f, false);
            encoder.setJPEGEncodeParam(param);
            encoder.encode(convertedImage);

        }catch(Exception e){
            logError("Error encoding JPEG: " + e.toString());
            return false;
        }

        return true;
    }


    private static boolean storeGIF(File file, BufferedImage convertedImage, int quality, Color transparentColor){
        if (file == null)
                    return true;

        BufferedOutputStream out;

        try{
            out = new BufferedOutputStream(new FileOutputStream(file));

            if (!streamGIF(out, convertedImage, quality, transparentColor))
                            throw(new Exception());

            out.close();
        }catch (Exception e) {
            logError("Output file error: " + e.toString());
            return false;
        }

        return true;
    }


    public static boolean streamGIF(OutputStream out, BufferedImage convertedImage, int quality, Color transparentColor){
        AnimatedGifEncoder gifEncoder = new AnimatedGifEncoder();

        float longQuality = (float)GIF_MAX_QUALITY_IN_BITS / (float)100 * (float)quality;
        System.out.println("quality = " + longQuality + ", int = " + (int)longQuality);

        gifEncoder.setTransparent(transparentColor);
        gifEncoder.setCompression((int)longQuality);
        gifEncoder.start(out);
        gifEncoder.addFrame(convertedImage);
        //success = gifEncoder.finish();

        return true;
    }


    private static boolean storeANSI(File file, Color bgColor, String[] txtVerMono, Color[][] colors){
        if (file == null)
            return true;

        FileWriter out;
        int i = 0;
        int color = 0;
        int lastColor = 0;

        final String ESC = (char)27 + "[";
        final String SAUCE = getSAUCEstring(file.getName());

        try{
            out = new FileWriter(file);
            out.write(ESC + getANSIcolor(bgColor, false) + "m");
            while (i < txtVerMono.length){
                for (int j = 0; j < txtVerMono[i].length(); j++){
                    color = getANSIcolor(colors[i][j], true);

                    if (color != lastColor)
                        out.write(ESC + color + "m" + txtVerMono[i].charAt(j));
                    else out.write(txtVerMono[i].charAt(j));

                    lastColor = color;
                }
                i++;
                out.write("\r\n");
            }
            out.write(26);
            out.write(SAUCE);
            out.flush();
            out.close();
        }catch(Exception e){
                logError("output file error: " + e.toString());
                return false;
        }

        return true;
    }


    private static String getSAUCEstring(String title){
        Calendar cal = Calendar.getInstance();
        String record = "";
        String SAUCE = "SAUCE";
        String Sversion = "00";
        String author = "retypar " + version;
        String group = "www.retypar.de.vu";
        String date = cal.get(Calendar.YEAR) + ""
                    + (((cal.get(Calendar.MONTH) + 1) < 10) ? "0" + (cal.get(Calendar.MONTH) + 1)
                                                            : "" + (cal.get(Calendar.MONTH) + 1))
                    + (((cal.get(Calendar.DAY_OF_MONTH) + 1) < 10) ? "0" + cal.get(Calendar.DAY_OF_MONTH)
                                                                    : "" + cal.get(Calendar.DAY_OF_MONTH));
        String size = "0000";
        String dateType = "1";
        String fileType = "1";
        String tInfo1 = "00"; //ANSI width  2 byte
        String tInfo2 = "00"; //ANSI height 2 byte
        String tInfo3 = "00";
        String tInfo4 = "00";
        String comment = "0";
        String flags = "0";


        if (title.length() > 35)
            title = title.substring(0, 34);

        for (int i = 0; i < 35; i++){
            if (title.length() == 35)
                break;

            title += " ";
        }


        if (author.length() > 20)
            author = author.substring(0, 19);

        for (int i = 0; i < 20; i++){
            if (author.length() == 20)
                break;

            author += " ";
        }


        if (group.length() > 20)
            group = group.substring(0, 19);

        for (int i = 0; i < 20; i++){
            if (group.length() == 20)
                break;

            group += " ";
        }

        record = SAUCE
                + Sversion
                + title
                + author
                + group
                + date
                + size
                + dateType
                + fileType
                + tInfo1
                + tInfo2
                + tInfo3
                + tInfo4
                + comment
                + flags;

        while (record.length() < 128)
            record += " ";

        return record;
    }


    private static int getANSIcolor(Color color, boolean foreground){
        long min;
        int val;

        Color colBlack    = Color.BLACK;
        Color colRed      = Color.RED;
        Color colGreen    = Color.GREEN;
        Color colYellow   = Color.YELLOW;
        Color colBlue     = Color.BLUE;
        Color colMagenta  = Color.MAGENTA;
        Color colCyan     = Color.CYAN;
        Color colWhite    = Color.WHITE;

        int valBlack    = 30;
        int valRed      = 31;
        int valGreen    = 32;
        int valYellow   = 33;
        int valBlue     = 34;
        int valMagenta  = 35;
        int valCyan     = 36;
        int valWhite    = 37;

        Color[] cols = new Color[]{colBlack, colRed, colGreen, colYellow, colBlue, colMagenta, colCyan, colWhite};
        int[] vals = new int[]{valBlack, valRed, valGreen, valYellow, valBlue, valMagenta, valCyan, valWhite};

        min = 0;
        min += Math.abs(colBlack.getRed()   - color.getRed());
        min += Math.abs(colBlack.getGreen() - color.getGreen());
        min += Math.abs(colBlack.getBlue()  - color.getBlue());
        val = valBlack;

        int minTmp = 0;
        for (int i = 0; i < cols.length; i++){
            minTmp =  Math.abs(cols[i].getRed()   - color.getRed());
            minTmp += Math.abs(cols[i].getGreen() - color.getGreen());
            minTmp += Math.abs(cols[i].getBlue()  - color.getBlue());

            if (minTmp < min){
                min = minTmp;
                val = vals[i];
            }            
        }

        if (!foreground)
            val += 10;

        return val;
    }


    private static boolean storeHTML(File file, boolean colored, Color bgColor, String[] txtVerMono, Color[][] colors){
        if (file == null)
                    return true;

        FileWriter out;
        int i = 0;

        try{
            out = new FileWriter(file);
            out.write("<HTML><BODY bgcolor=#" + getHexColor(bgColor) + ">\r\n");
            out.write("<DIV STYLE=\"line-height: 75%;\">\r\n");
            out.write("<tt>\r\n");
            while (i < txtVerMono.length){
                for (int j = 0; j < txtVerMono[i].length(); j++){
                    if (colored) out.write("<font color=#" + getHexColor(colors[i][j]) + ">");
                    out.write(txtVerMono[i].charAt(j));
                    if (colored) out.write("</font>");
                }
                i++;
                out.write("<BR>\r\n");
            }
            out.write("</tt>\r\n</DIV>\r\n</BODY>\r\n</HTML>");
            out.flush();
            out.close();
        }catch(Exception e){
                logError("Output file error: " + e.toString());
                return false;
        }

        return true;
    }


    private static String getHexColor(Color color){
        String r, g, b;

        r= Integer.toHexString(color.getRed());
        g= Integer.toHexString(color.getGreen());
        b= Integer.toHexString(color.getBlue());

        return fillLeadZeros(r) + fillLeadZeros(g) + fillLeadZeros(b);
    }

    private static String fillLeadZeros(String hex){
        if (hex.length() == 0) hex = "00";
        if (hex.length() == 1) hex = "0" + hex;

        return hex;
    }


    private static boolean storeTEXT(File file, String[] txtVerMono){
        if (file == null)
                    return true;

        FileWriter out;
        int i = 0;

        try{
            out = new FileWriter(file);
            while (i < txtVerMono.length){
                out.write(txtVerMono[i]);
                out.write("\r\n");
                i++;
            }
            out.close();
        }catch(Exception e){
                logError("Output file error: " + e.toString());
                return false;
        }

        return true;
    }


    private static boolean storeTextVideo(File file, String[] txtVerMono){
        if (file == null){
            logError("Output file is null! Skipping.");
            return true;
        }

        FileWriter out;
        int i = 0;
        boolean firstFrame = !file.exists();

        if (output == null){
            output = file;
            firstFrame = true;
        }
            else file = output;

        try{
            out = new FileWriter(file, !firstFrame);
            if (firstFrame) out.write("retypar text video version " + textVideoVersion + "\r\n" + txtVerMono.length + "\r\n");
            while (i < txtVerMono.length){
                out.write(txtVerMono[i]);
                out.write("\r\n");
                i++;
            }
            out.close();
        }catch(Exception e){
                logError("Output file error: " + e.toString());
                return false;
        }

        return true;
    }

    public static File checkExtension(File file, String ext[]){
        String path = "";
        String pathSep = System.getProperty("file.separator");

        boolean textVideo = false;

        try{
            path = file.getAbsolutePath().substring(0, file.getAbsolutePath().length() - file.getName().length());
        }catch(Exception e){
            logError("Error referencing file " + file.getAbsolutePath());
            return null;
        }
        boolean ok = false;

        for(int i = 0; i < ext.length; i++){
            if (file.getName().toLowerCase().endsWith("." + ext[i])) ok = true;
            if (ext[i].toLowerCase().equals("txv")) textVideo = true;
        }
        if (!ok) {
            if (file.getName().lastIndexOf(".") == -1)
                file = new File(path + pathSep + file.getName() + "." + ext[0]);
            else
                file = new File(path + pathSep + file.getName().substring(0, file.getName().lastIndexOf(".")) + "." + ext[0]);
        }

        if ((file.exists() && (!textVideo))){
            //if (mF == null){
                logError("File already exists! Do not supply one filename when converting multiple files.");
                //return null;
            //}

            /**
            if (mF.permForOverwrite(file))
                return file;**/

            return null;
        }

        return file;
    }


    public static  void readINI(){
        BufferedReader in;
        String checkUpdStr = "";
        String mfd;

        try{
            in = new BufferedReader(new FileReader((new File(appName + ".ini"))));
            lang = in.readLine();
            plaf = in.readLine();
            checkUpdStr = in.readLine();
            lastPath = in.readLine();

            while ((mfd = in.readLine()) != null)
                markedForDeletion.add(mfd);

            in.close();
            if (checkUpdStr.equals("n")) checkUpd = false;
            else checkUpd = true;
        }catch(Exception e){
            //logError("readINI:\n" + e.toString());
            }

        if (lang == null) lang = "";
        if (lastPath == null) lastPath = "";
    }


    public static void writeINI(){
        BufferedWriter out;
        String checkUpdStr = "y";

        if (!checkUpd) checkUpdStr = "n";

        try{
            out = new BufferedWriter(new FileWriter(new File(appName + ".ini")));
            out.write(lang);
            out.newLine();
            out.write(plaf);
            out.newLine();
            out.write(checkUpdStr);
            out.newLine();
            out.write(lastPath);

            /*if (markedForDeletion.size() > 0){
                while(!markedForDeletion.isEmpty()){
                    out.newLine();
                    out.write((String)markedForDeletion.removeFirst());
                }
            }*/

            out.close();
        }catch(Exception e){
            logError("writeINI:\n"); e.printStackTrace(); //toString());
        }
    }


    private static LinkedList getAllOnlineAlgos(){
        LinkedList algs = new LinkedList();
        BufferedReader in;
        URL url;
        String algo;

        try{
            url = new URL(algosOnline + "algos.txt");
            in = new BufferedReader(new InputStreamReader(url.openStream()));

            while((algo = in.readLine()) != null){
                algs.add(algo);
                algs.add(in.readLine());
            }

            in.close();
        } catch (Exception e){
            //logError("Error retrieving list of algorithms from server.\n");
        }

        return algs;
    }



    public static LinkedList getOnlineAlgos(){
        LinkedList ls = getAllOnlineAlgos();

        for(int y = 0; y < ls.size(); y++){
            String oa = (String)ls.get(y);
            for (int i = 0; i < retypar.converters.size(); i++){
                if (((Converter)retypar.converters.get(i)).toString().equals(oa)){
                    ls.removeFirst();
                    ls.removeFirst();
                    //System.out.println("removed " + oa);
                }
            }
            y++;
        }
        return ls;
    }


    /**
     * Triggers the installation of an online available algorithm.
     * @param algo The zip filename of the algorithm
     * @return <code>true<code> if the installation succeeded
     */
    public static boolean downloadAndInstallAlgo(String algo) {
        File file = downloadAlgo(algo);
        if (file == null) return false;

        if(!installAlgo(file)) return false;

        converters = loadConverters();
        converters.addAll(importConverters());
        return true;
    }


    private static File downloadAlgo(String algo){
        //System.out.println("getting " + algosOnline + algo);
        URL url;
        //BufferedOutputStream bOut;
        FileOutputStream bOut;
        File file;

        try{
            url = new URL(algosOnline + algo);
            URLConnection connection = url.openConnection();
            BufferedInputStream in = new BufferedInputStream(connection.getInputStream());
            //ByteArrayOutputStream baOut = new ByteArrayOutputStream();
            //bOut = new BufferedOutputStream(baOut);
            file = new File(basedir + algo);
            bOut = new FileOutputStream(file);

            int i;
            while ((i = in.read()) != -1)
                bOut.write(i);

            bOut.close();
        }catch(Exception e){
            logError("Error downloading new algorithm.");
            return null;
        }
        return file;
    }


    private static boolean installAlgo(File file){
        ZipFile zip;
        Enumeration enumeration;
        InputStream stream;
        File nf;
        ZipEntry ze;
        FileOutputStream fOut;
        LinkedList createdFiles = new LinkedList();

        createdFiles.add(file);

        try{
            zip = new ZipFile(file);
        }catch (Exception e){
            logError("Cannot access zip file " + file.toString());
            deleteFiles(createdFiles);
            return false;
        }

        if (basedir.length() == 0) {
            logError("You should never see this...");
            return false;     //duerfte nie der fall sein...
        }

        String algoSubDirAbs = basedir + algoSubDir;


        if (! new File(algoSubDirAbs).exists()){
            File dir = new File(algoSubDirAbs);
            if (!dir.mkdirs()) {
                logError("Cannot create directory '" + algoSubDirAbs + "'.");
                return false;
            }
        }


        enumeration = zip.entries();
        while (enumeration.hasMoreElements()){
            try{
                ze = (ZipEntry)enumeration.nextElement();
                stream = zip.getInputStream(ze);
                BufferedInputStream bIn = new BufferedInputStream(stream);

                nf = new File(algoSubDirAbs + ze.getName());

                if (ze.isDirectory()) {
                    if (!nf.mkdir()) {
                        logError("Cannot create directory " + nf.getName());
                        zip.close();
                        deleteFiles(createdFiles);
                        return false;
                    }
                    createdFiles.add(nf);
                    continue;
                }

                if (!nf.createNewFile()) {
                    logError("Cannot create file " + nf.getName());
                    zip.close();
                    deleteFiles(createdFiles);
                    return false;
                }

                fOut = new FileOutputStream(nf);

                int i;
                while((i = bIn.read()) != -1)
                    fOut.write(i);

                fOut.close();
                createdFiles.add(nf);
            }catch (Exception e){
                logError("Zip file maybe corrupt: " + e.toString());
                deleteFiles(createdFiles);
                return false;
            }
        }
        try{
            zip.close();
        }catch(Exception e){
            logError("Cannot close zip file!");
            deleteFiles(createdFiles);
            return false;
        }
        file.delete();
        return true;
    }


    private static void deleteFiles(LinkedList fileList){
        File[] files = new File[fileList.size()];

        for (int j = 0; j < fileList.size(); j++)
            files[j] = (File)fileList.get(j);

        for (int i = 0; i < files.length; i++){
            if (files[i].isFile()) files[i].delete();
                else if (files[i].isDirectory())
                        if (!files[i].getName().equals("converter"))
                            deleteDirectory(files[i]);
        }
    }


    public static void setFont(String fontname){
        font = fontname;
    }

    public static void setFontSize(int size){
        fontSize = size;
    }

    private static void runInBatchMode(String args[]){
        new batchMode(args);
    }
}