/* JExportOptionsDialog - jfig.gui.JExportOptionsDialog.java
 *
 * specify the options for jfig fig2dev/GIF export and handle export
 *
 * 09.09.06 - correct output filename for pstex_t command
 * 08.09.06 - pstex/pstex_t stuff via second command line
 * 06.09.06 - integrate pstex/pstex_t support (Rob Managan)
 * 08.05.05 - add HIDE_ON_CLOSE
 * 18.04.05 - save file to temporary directory
 * 23.09.04 - some support for anti-aliasing for PPM/PNG/GIF export
 * 10.07.03 - use reflection to check for doAutoSave()
 * 22.10.02 - Swing version
 * 21.01.02 - epic, eepic support by Jose Tomas Palma Mendez, jpalma@dif.um.es
 * 28.03.01 - use SetupManager to init magnification, PS center, orientation
 * 19.11.00 - some more redesign code
 * 17.11.00 - complete redesign started
 * 12.10.00 - use double values in get/setMagnification
 * 11.12.97 - first version
 *
 * (C) 1997-2006, F.N.Hendrich, hendrich@informatik.uni-hamburg.de
 */

package jfig.gui;

import jfig.canvas.FigCanvas;
import jfig.commands.FigBasicEditor;
import jfig.objects.FigAttribs;
import jfig.objects.FigConstants;
import jfig.objects.FigBbox;
import jfig.utils.*;


import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.lang.reflect.*;
import javax.swing.*;
import javax.swing.border.EmptyBorder;



/**
 * dialog window to specify fig2dev export options and handle export.
 * <p>
 */
public class  JExportOptionsDialog
       extends  JDialog  
       implements  ActionListener, ItemListener
{


   public final int FULL_COLORS    = 2;
   public final int IO_COLORS      = 3;
   public final int BLACK_WHITE    = 4;

   public final String FULL_COLORS_STRING = "full color";
   public final String IO_COLORS_STRING   = "black and white, I/O colored";
   public final String BLACK_WHITE_STRING = "black and white only";

   public final int A4_WIDTH       = 26173;  // at 2400 dpi, with 1 cm border
   public final int A4_HEIGHT      = 17952;

   FigBasicEditor editor = null;
   FigBbox        bbox = null;   // bounding box of all FIG objects, for fit
   File           tmpdir = null;

   String       figFilename;
   String       outFilename;
   String       tmpFilename;
   String       psFilename;


   Frame        parent;
   JTextField   magnificationTF, xoffsetTF, yoffsetTF;
   JComboBox    languageChoice;
   JComboBox    orientationChoice;
   JComboBox    centerChoice;
   JComboBox    colorChoice;
   JCheckBox    quoteNamesCB;

   JTimeoutButton  exportNowButton;
   JButton      closeButton;
   JButton      filenameButton, fitToPageButton;
   JTextField   filenameTF, directoryTF, fig2devPathTF, commandLineTF;
   JTextField   commandLine2TF;
   JTextField   tmpFilenameTF;
   JLabel       sizeLabel;
   JFileChooser fileChooser;
   JConsole     console;



   boolean     debug = true;
   boolean     autosave = true;
   boolean     waitExec = true; // wait for subprocess?
   boolean     showColorOptions = false;
   boolean     requestAntiAliasing = false; // for native PPM/PNG/GIF export

   String[][] data = {
// language,                  extension,    methodname,     internal
{  "Postscript",              "ps",         "doExportPS",   "no" },
{  "Encapsulated Postscript", "eps",        "doExportEPS",  "no" },
{  "IBM GL",                  "gl",         "doExportIBMGL","no" },
{  "Latex",                   "latex",      "doExportLATEX","no" },
{  "Pictex",                  "pictex",     "doExportPICTEX","no" },
{  "PSTEX + PSTEX_T",         "ps",         "doExportPSTEX","no" },
{  "PPM",                     "ppm",        "doExportPPM",  "no" },
{  "JPEG",                    "jpg",        "doExportJPG",  "no" },
{  "GIF",                     "gif",        "doExportGIF",  "no" },
{  "PNG",                     "png",        "doExportPNG",  "no" },
{  "EPIC",                    "epic",       "doExportEPIC", "no" },
{  "EEPIC",                   "eepic",      "doExportEEPIC","no" },
//{"PSTEX_T",                 "tex",        "doExportPSTEX_T","no" },
};



   String  printCommand,
           fig2devPath,
           fig2devPS,
           fig2devEPS,
           fig2devIBMGL,
           fig2devLATEX,
           fig2devPSTEX,
           fig2devPICTEX,
           fig2devJPG,
           fig2devGIF,
           fig2devTIFF,
           fig2devPNG,
           fig2devPPM,
           fig2devEPIC,
           fig2devEEPIC,
           fig2devPSTEX_T;

   boolean fig2devPSCentered = true;
   boolean fig2devPSOrientation = true; // means Portrait
   double  fig2devMagnification = 1.0;





   /**
    * construct a new JExportOptionsDialog,
    * ask SetupManager for the fig2dev parameters
    */
   public JExportOptionsDialog( Frame parent ) {
      super( parent, "Select Export Options" );
      this.parent = parent;
      getAndSetDefaultFont();
      loadExportSetup();
      buildGUI();
      buildCallbacks();
      updateCommandLine();
      setDefaultCloseOperation( JDialog.HIDE_ON_CLOSE );
   }


   public void setEditor( FigBasicEditor editor ) {
     this.editor = editor;
     updateCommandLine();
   }

   public void setConsole( JConsole console ) {
     this.console = console;
   }



   public void buildGUI() {
     try {
       Color c = SetupManager.getColor(
          "jfig.gui.EditDialog.BackgroundColor", null );
       if (c != null) this.setBackground( c );
     }
     catch( Throwable t )  { } // ignore

     // the "bottom" panel with the "OK" and "Export now" buttons
     //
     //jfig.gui.InsetsPanel okPanel = new jfig.gui.InsetsPanel(10,10);
     //okPanel.setLayout( new FlowLayout( FlowLayout.CENTER ) );
     //okPanel.setLowered();
     closeButton    = new JButton( "Close" );
     exportNowButton = new JTimeoutButton( "Export now" );
     if (KeyManager.getKeyManager().isValid()) {
      ; // ok
     }
     else {
       exportNowButton.setTimeout( 30 );
     }

     closeButton.setToolTipText( "Close dialog window" );
     exportNowButton.setToolTipText( 
       "Export figure using the current settings and command line" );

     // the "label" panel on the left of the window
     //
     JPanel labelPanel = new JPanel( new GridLayout( 0, 1 ));
     labelPanel.setBorder( new EmptyBorder(10,10,10,5) ); // tlbr
     labelPanel.add( new JLabel( "output format:", SwingConstants.RIGHT));
     if (showColorOptions) {
       labelPanel.add( new JLabel( "color option:", SwingConstants.RIGHT));
       labelPanel.add( new JLabel( " " ));
     }
     labelPanel.add( new JLabel( "magnification:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( "orientation:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( "offsets x,y:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( " " ));
     labelPanel.add( new JLabel( "output filename:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( "directory:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( "tmp filename:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( "fig2dev path:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( " " ));
     // command line and second command line for pstex_t option
     labelPanel.add( new JLabel( "command line:", SwingConstants.RIGHT ));
     labelPanel.add( new JLabel( "", SwingConstants.RIGHT ));
     JPanel closePanel = new JPanel( new FlowLayout( FlowLayout.RIGHT ));
     closePanel.add( closeButton );
     labelPanel.add( closePanel );

     // the "options" panel on the right of the window
     //
     JPanel optionsPanel = new JPanel( new GridLayout( 0, 1 ));
     optionsPanel.setBorder( new EmptyBorder(10,5,10,10) ); // tlbr
       JPanel languagePanel = new JPanel();
       languagePanel.setLayout( new FlowLayout( FlowLayout.LEFT ));
       languageChoice = new JComboBox();
       for( int i=0; i < data.length; i++ ) {
         languageChoice.addItem( data[i][0] );
       }
       languageChoice.setSelectedIndex( 0 );
       languagePanel.add( languageChoice );
     optionsPanel.add( languagePanel );

       JPanel colorPanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       colorChoice = new JComboBox();
       colorChoice.addItem( FULL_COLORS_STRING );
       //colorChoice.addItem( IO_COLORS_STRING ); // only useful in HADES
       colorChoice.addItem( BLACK_WHITE_STRING );
       colorChoice.setSelectedItem( FULL_COLORS_STRING );
       colorPanel.add( colorChoice );
     if (showColorOptions) {
       optionsPanel.add( colorPanel );
       optionsPanel.add( new JLabel( "" ));
     }

       JPanel magnificationPanel = new JPanel();
       magnificationPanel.setLayout( new FlowLayout( FlowLayout.LEFT ) );
       magnificationTF = new JTextField( "100", 6 );
       sizeLabel       = new JLabel( "  [10.0 x 10.0 inches]  " );
       fitToPageButton = new JButton( "Fit to A4" );
       setMagnification( fig2devMagnification );
       magnificationPanel.add( magnificationTF );
       magnificationPanel.add( new JLabel( "% " ));
       magnificationPanel.add( sizeLabel );
       magnificationPanel.add( fitToPageButton );
     optionsPanel.add( magnificationPanel );

       JPanel orientationPanel = new JPanel();
       orientationPanel.setLayout( new FlowLayout( FlowLayout.LEFT ) );
       orientationChoice = new JComboBox();
       orientationChoice.addItem( "landscape" );
       orientationChoice.addItem( "portrait" );
       setFig2devPSOrientation( fig2devPSOrientation );
       centerChoice = new JComboBox();
       centerChoice.addItem( "center" );
       centerChoice.addItem( "flush left" );
       setFig2devPSCentered( fig2devPSCentered );
       orientationPanel.add( orientationChoice );
       orientationPanel.add( centerChoice );
     optionsPanel.add( orientationPanel );

       JPanel offsetPanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       xoffsetTF = new JTextField( "0", 4 );
       yoffsetTF = new JTextField( "0", 4 );
       offsetPanel.add( xoffsetTF );
       offsetPanel.add( yoffsetTF );
       offsetPanel.add( new JLabel( "inches" ));
     optionsPanel.add( offsetPanel );
     optionsPanel.add( new JLabel( " " ));

       JPanel filenamePanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       filenameTF     = new JTextField( "", 40 );
       //filenameTF.setEditable( false );
       filenameButton = new JButton( "Browse" );
       filenamePanel.add( filenameTF );
       filenamePanel.add( filenameButton );
     optionsPanel.add( filenamePanel );

       JPanel directoryPanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       directoryTF     = new JTextField( "", 40 );
       //directoryTF.setEditable( false );
       directoryPanel.add( directoryTF );
     optionsPanel.add( directoryPanel );

       JPanel tmpfnamePanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       if (tmpFilename == null) tmpFilename = createTmpFigFilename();
       tmpFilenameTF = new JTextField( "", 40 );
       tmpFilenameTF.setText( tmpFilename );
       tmpfnamePanel.add( tmpFilenameTF );
     optionsPanel.add( tmpfnamePanel );

       JPanel fig2devPathPanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       fig2devPathTF = new JTextField( "", 40 );
       fig2devPathTF.setText( fig2devPath );
       fig2devPathPanel.add( fig2devPathTF );
     optionsPanel.add( fig2devPathPanel );

       JPanel quotePanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       quoteNamesCB = new JCheckBox( "quote \"filenames containing spaces\"" );
       quoteNamesCB.setToolTipText( "add quotes around filenames which contain space chars?");
       quoteNamesCB.setSelected( SetupManager.getBoolean(
         "jfig.gui.ExportOptionsDialog.QuoteNames", true ));
       quotePanel.add( quoteNamesCB );
     optionsPanel.add( quotePanel );

       JPanel commandLinePanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       commandLineTF = new JTextField( "", 50 );
       commandLinePanel.add( commandLineTF );
     optionsPanel.add( commandLinePanel );

       // second command line only enabled and used for pstex_t mode
       JPanel commandLine2Panel = new JPanel( new FlowLayout( FlowLayout.LEFT));
       commandLine2TF = new JTextField( "", 50 );
       commandLine2TF.setVisible( true );
       commandLine2Panel.add( commandLine2TF );
     optionsPanel.add( commandLine2Panel );

       JPanel exportNowPanel = new JPanel( new FlowLayout( FlowLayout.LEFT ));
       exportNowPanel.add( exportNowButton );
       exportNowPanel.add( new JLabel( "using above command" ));
     optionsPanel.add( exportNowPanel );


     this.getContentPane().setLayout( new BorderLayout() );
     this.getContentPane().add( "West",  labelPanel );
     this.getContentPane().add( "East", optionsPanel );
     //this.getContentPane().add( "South", okPanel );
     this.pack();
   } // buildGUI



  public void buildCallbacks() {
     KeyMultiplexer mux = new KeyMultiplexer();
     filenameTF     .addKeyListener( mux );
     directoryTF    .addKeyListener( mux );
     tmpFilenameTF  .addKeyListener( mux );
     xoffsetTF      .addKeyListener( mux );
     yoffsetTF      .addKeyListener( mux );
     magnificationTF.addKeyListener( mux );

     //filenameTF.addActionListener( this );
     //directoryTF.addActionListener( this );
     //magnificationTF.addActionListener( this );
     //xoffsetTF.addActionListener( this );
     //yoffsetTF.addActionListener( this );

     languageChoice.addItemListener( this );
     orientationChoice.addItemListener( this );
     centerChoice.addItemListener( this );
     colorChoice.addItemListener( this );

     fitToPageButton.addActionListener( this );
     closeButton.addActionListener( this );
     exportNowButton.addActionListener( this );
     filenameButton.addActionListener( this );

     // setDefaultCloseOperation( WindowConstants.HIDE_ON_CLOSE );
     this.addWindowListener( new JWindowCloser(this,false) );
  } // buildCallbacks

  
  public void actionPerformed( ActionEvent e ) {
    Object source = e.getSource();
    if (source == fitToPageButton) {
         dbg( "-#- fitToPageButton: " + e );
         fitToLandscapeA4();
         updateCommandLine();
    }
    else if (source == closeButton) {
         dbg( "-#- closeButton: " + e );
         JExportOptionsDialog.this.setVisible( false );
    }
    else if (source == exportNowButton) {
         dbg( "-#- exportNowButton: " + e );
         doExportNow();
         if (KeyManager.getKeyManager().isValid()) {
          ; // ok
         }
         else {
           exportNowButton.setTimeout( 30 );
          }
    }
    else if (source == filenameButton) {
         dbg( "-#- filenameButton: " + e );
         doBrowseFilename();
    }
  }



  public void itemStateChanged( ItemEvent e ) {
      dbg( "-#- optionChangeHandler: " + e );

      int index = languageChoice.getSelectedIndex();
      updateOutFilenameExtension( index );
      updateCommandLine();
      showSize();
  }


  class KeyMultiplexer implements KeyListener {

    public void keyPressed ( KeyEvent e ) { dbg( "-*- " + e ); mux(); }
    public void keyReleased( KeyEvent e ) { dbg( "-*- " + e ); mux(); }
    public void keyTyped   ( KeyEvent e ) { dbg( "-*- " + e ); mux(); }

    public void mux() {
        updateCommandLine();
        showSize();
    }
  }



  /**
   * load the export and fig2dev related SetupManager parameters.
   * We use the following parameter keys:
     <pre>
       jfig.gui.JExportOptionsDialog.Debug
       jfig.gui.JExportOptionsDialog.Autosave
       jfig.gui.JExportOptionsDialog.Print
       jfig.gui.JExportOptionsDialog.fig2devPath
       jfig.gui.JExportOptionsDialog.fig2devPS
       jfig.gui.JExportOptionsDialog.fig2devEPS
       jfig.gui.JExportOptionsDialog.fig2devIBMGL
       jfig.gui.JExportOptionsDialog.fig2devLATEX
       jfig.gui.JExportOptionsDialog.fig2devPSTEX
       jfig.gui.JExportOptionsDialog.fig2devPICTEX
       jfig.gui.JExportOptionsDialog.fig2devPPM
       jfig.gui.JExportOptionsDialog.fig2devJPG
       jfig.gui.JExportOptionsDialog.fig2devGIF
       jfig.gui.JExportOptionsDialog.fig2devPNG
       jfig.gui.JExportOptionsDialog.fig2devWaitFor
     </pre>
   */
  public void loadExportSetup() {
    String base = "jfig.gui.ExportOptionsDialog.";
    debug         = SetupManager.getBoolean(  base+"Debug", true );
    autosave      = SetupManager.getBoolean(  base+"Autosave", true );
    waitExec      = SetupManager.getBoolean(  base+"Autosave", true );
    showColorOptions 
                  = SetupManager.getBoolean( base+"ShowColorOptions",false);
    printCommand  = SetupManager.getProperty( base+"Print",  "lpr" );

    // try to guess a useful default, even if not specified in the .jfigrc
    fig2devPath   = SetupManager.getProperty( base+"fig2devPath",  null );
    if (fig2devPath == null) {
      String tmpPath = "fig2dev";
      try {
        String os = System.getProperty( "os.name" ).toLowerCase();
        if (os.indexOf( "windows" ) >= 0) tmpPath = "c:\\windows\\fig2dev.exe";
        else                              tmpPath = "/usr/X11/bin/fig2dev";
      }
      catch( Throwable t ) {
        ; // applet or other security manager...
      }
      SetupManager.setProperty( base+"fig2devPath", tmpPath ); // allow saving
      fig2devPath   = SetupManager.getProperty( base+"fig2devPath",  null );
// System.err.println( fig2devPath + " # " + tmpPath );
    }

    // get temporary directory
    String _tmp   = SetupManager.getProperty( base+"tmpDirectory", null );
    if (_tmp != null) {
      try {
         tmpdir = new File( _tmp );
      }
      catch( Exception e ) {
         tmpdir = new File( "." );
      }
    }

    fig2devPS     = SetupManager.getProperty( base+"fig2devPS",    "-L ps -P" );
    fig2devEPS    = SetupManager.getProperty( base+"fig2devEPS",   "-L eps" );
    fig2devIBMGL  = SetupManager.getProperty( base+"fig2devIBMGL", "-L ibmgl" );
    fig2devLATEX  = SetupManager.getProperty( base+"fig2devLATEX", "-L latex" );
    fig2devPSTEX  = SetupManager.getProperty( base+"fig2devPSTEX", "-L pstex" );
    fig2devPICTEX = SetupManager.getProperty( base+"fig2devPICTEX","-L pictex");

    fig2devGIF    = SetupManager.getProperty( base+"fig2devGIF",   "-L gif" );
    fig2devJPG    = SetupManager.getProperty( base+"fig2devJPG",   "-L jpeg" );
    fig2devPPM    = SetupManager.getProperty( base+"fig2devPPM",   "-L ppm" );
    fig2devPNG    = SetupManager.getProperty( base+"fig2devPNG",   "-L png" );
    fig2devEEPIC  = SetupManager.getProperty( base+"fig2devEEPIC", "-L eepic" );
    fig2devEPIC  = SetupManager.getProperty(  base+"fig2devEPIC",  "-L epic" );
    fig2devPSTEX_T = SetupManager.getProperty( base+"fig2devPSTEX_T", 
                                                                 "-L pstex_t" );

    fig2devMagnification=SetupManager.getDouble(
                                              base+"fig2devMagnification",1.0);
    fig2devPSCentered=SetupManager.getBoolean(base+"fig2devPSCentered", true);
    fig2devPSOrientation=SetupManager.getBoolean(
                                             base+"fig2devPSOrientation", true);


    if (debug) {
      System.err.println( "-#- JExportOptionsDialog.loadExportSetup: \n"
        + "debug " + debug + " Autosave " + autosave + "\n"
        + "fig2dev path " + fig2devPath + "\n"
        + "fig2devPS " + fig2devPS + "\n"
        + "fig2devEPS " + fig2devEPS + "\n"
        + "fig2devIBMGL " + fig2devIBMGL + "\n"
        + "fig2devLATEX " + fig2devLATEX + "\n"
        + "fig2devPSTEX " + fig2devPSTEX + "\n"
        + "fig2devPICTEX " + fig2devPICTEX + "\n"
        + "fig2devPSTEX_T " + fig2devPSTEX_T + "\n"
      );
    }

  }


  public void getAndSetDefaultFont() {
    String name; int style, size;
    String base = "jfig.gui.JExportOptionsDialog";
    name=SetupManager.getProperty( base+"FontName","SansSerif");
    size=SetupManager.getInteger(  base+"FontSize",12 );
    style=Font.PLAIN;
    Font font = new Font( name, style, size );
    if (font != null) this.setFont( font );
  }






   /**
    * get the currently selected color option.
    * Possible values are FULL_COLORS, IO_COLORS, and BLACK_WHITE.
    */
   public int getColorOption() {
     String s = (String) colorChoice.getSelectedItem();

     if      (s.equals( FULL_COLORS_STRING)) return FULL_COLORS;
     else if (s.equals( IO_COLORS_STRING))   return IO_COLORS;
     else if (s.equals( BLACK_WHITE_STRING)) return BLACK_WHITE;
     else {
       msg( "-E- unkwown color selection: " + s );
       return IO_COLORS;
     }
   }





   /**
    * display the size of the current drawing at selected magnification,
    * in inches, using the sizeLabel GUI component.
    * Note: This method assumes that the bounding box of the FIG objects
    * has already been calculated via updateFigObjectsBoundingBox().
    */
   public void showSize() {
     if (bbox == null) updateFigObjectsBoundingBox();
     Dimension d   = new Dimension( (int) (bbox.getXr()-bbox.getXl()),
                                    (int) (bbox.getYb()-bbox.getYt()) );
     double mag    = getMagnification();
     double size_x = (mag*d.width)  / 2400.0;
     double size_y = (mag*d.height) / 2400.0;

     sizeLabel.setText( "  ["
       + new jfig.utils.Format("%3.1f").form( size_x ) + " x "
       + new jfig.utils.Format("%3.1f").form( size_y ) + " "
       + "inches]  " );
   }


   /**
    * set a new value for the magnification factor.
    * For example, setMagnification( 1.0 );
    */
   public void setMagnification( double d ) {
     fig2devMagnification = d;
     int percent = (int) (100 * d + 0.5);
     magnificationTF.setText( "" + percent );
   }


   /**
    * return the magnification factor to use for exporting or printing.
    */
   public double getMagnification() {
     double mag;
     try {
       mag = 0.01 * Double.valueOf(
                              magnificationTF.getText().trim() ).doubleValue();
     }
     catch( Exception e ) {
       // msg( "-W- Illegal NumberFormat in getMagnification: " +
       //      magnificationTF.getText().trim() );
       // msg( "-W- using mag=100% instead" );
       mag = 1.0;
     }
     fig2devMagnification = mag;
     return mag;
   }


   public int getOrientation() {
     return
       ("landscape".equals( orientationChoice.getSelectedItem()))
       ? FigConstants.LANDSCAPE : FigConstants.PORTRAIT;
   }


   public int getJustification() {
     return
       ("center".equals( centerChoice.getSelectedItem()))
       ? FigConstants.CENTER_JUSTIFICATION
       : FigConstants.FLUSH_LEFT_JUSTIFICATION;
   }


  /**
   * set and show the filename selected by the user
   */
  public void setFilename( String filename ) {
    filenameTF.setText( filename );
  }



  /**
   * get filename extensions corresponding to current export
   * language selection. <p>
   */
  public String getExtension( int index ) {
    // return data[languageChoice.getSelectedIndex() ][1];
    return data[index][1];
  }


  /**
   * update the current filename's extension based on the currently
   * selected export language. 
   * For example, updates "foo.gl" to "foo.ppm" if PPM format is selected.
   * Note: This method assumes that filename is non-null.
   */
  public void updateOutFilenameExtension( int language ) {
    dbg( "-#- updateOutFilenameExtension..." );
    String    raw = filenameTF.getText();
    String cooked = replaceExtension( raw, getExtension(language) );
    filenameTF.setText( cooked );
  }


  public String replaceExtension( String filename, String extension ) {
    try {
      int ix = filename.lastIndexOf( '.' );
      if (ix >= 0) filename = filename.substring( 0, ix ) + '.' + extension;
      else         filename = filename + '.' + extension;

    }
    catch( Exception e ) { dbg( "-#- " + e ); }
    return filename;
  }


  public void updateCommandLine() {
    dbg( "-#- updateCommandLine..." );

    int  lang    = languageChoice.getSelectedIndex();
    figFilename  = getFigFilename(); 
    tmpFilename  = getTmpFigFilename(); 
    psFilename   = getPsFigFilename(); 
    outFilename  = getOutFilename();

    String  fig2devPath = getFig2devPath();
    String  commandLine  = "";
    String  commandLine2 = "";
    boolean enableLine2  = false;

    if (quoteNamesCB.isSelected()) {
      if (fig2devPath.indexOf( ' ' ) >= 0) { // contains space(s)
        fig2devPath = "\"" + fig2devPath + "\"";
      }
      if (tmpFilename.indexOf( ' ' ) >= 0) { // contains space(s)
        tmpFilename = "\"" + tmpFilename + "\"";
      }
      if (psFilename.indexOf( ' ' ) >= 0) { // contains space(s)
        psFilename = "\"" + psFilename + "\"";
      }
      if (outFilename.indexOf( ' ' ) >= 0) { // contains space(s)
        outFilename = "\"" + outFilename + "\"";
      }
    }

    switch( lang ) {
      case  0: // Postscript
               commandLine = fig2devPath + " " + fig2devPS
                             + getFig2devMagnification()
                             + getFig2devPSCentered()
                             + getFig2devPSOrientation()
                             + getFig2devPSOffset()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  1: // EPS
               commandLine = fig2devPath + " " + fig2devEPS
                             + getFig2devMagnification()
                             + getFig2devPSCentered()
                             + getFig2devPSOrientation()
                             + getFig2devPSOffset()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  2: // IBM GL
               commandLine = fig2devPath + " " + fig2devIBMGL
                             + getFig2devMagnification() + " "
                             + tmpFilename + " "
                             + outFilename;
               break;
      case  3: // LATEX
               commandLine = fig2devPath + " " + fig2devLATEX
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  4: // PICTEX
               commandLine = fig2devPath + " " + fig2devPICTEX
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  5: // PSTEX plus PSTEX_T
               commandLine = fig2devPath + " " + fig2devPSTEX
                             + " " + tmpFilename
                             + " " + outFilename;

               commandLine2= fig2devPath + " " + fig2devPSTEX_T
                             + " -p " + outFilename
                             + " " + tmpFilename
                             + " " + replaceExtension( outFilename, "tex" );
// FIXME XXXZZZ

               enableLine2 = true;
               break;

      case  6: // PPM
               commandLine = fig2devPath + " " + fig2devPPM
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;

               break;
      case  7: // JPG
               commandLine = fig2devPath + " " + fig2devJPG
                             + getFig2devMagnification()
                             + " -q 70"        // quality
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  8: // GIF
               commandLine = fig2devPath + " " + fig2devGIF
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  9: // PNG
               commandLine = fig2devPath + " " + fig2devPNG
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  10: // EPIC
               commandLine = fig2devPath + " " + fig2devEPIC
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  11: // EEPIC
               commandLine = fig2devPath + " " + fig2devEEPIC
                             + getFig2devMagnification()
                             + " " + tmpFilename
                             + " " + outFilename;
               break;
      case  12: // PSTEX_T
               commandLine = fig2devPath + " " + fig2devPSTEX_T
			                 + " -p " + psFilename
                             + " " + tmpFilename
                             + " " + outFilename;
               break;


      default:
               msg( "-E- internal: unknown export language: " + lang );
               return;
    }

    commandLineTF.setText( commandLine );
    commandLine2TF.setText( commandLine2 );
    commandLine2TF.setVisible( enableLine2 );
    commandLine2TF.getParent().validate(); // stupid FlowLayout requires this;
    commandLine2TF.getParent().repaint();  // and this, just to make sure...
dbg( "-X- commandLine2TF enable: " + enableLine2 );
  }


  public String getFigFilename() {
    String s = null;
    try { 
      s = editor.getFilename();
    }
    catch( Exception e ) {
      s = "unnamed.fig";
    }

    if (s == null) return "unnamed.fig";
    else           return s;
  }


  /** return the current contents of the tmpFilename textfield.
   *  We also check that the contents is a valid filename and 
   *  call createTmpFigFilename() if necessary...
   */
  public String getTmpFigFilename() {
    String tmp = tmpFilenameTF.getText();
    if (tmp == null) {
      tmp = createTmpFigFilename();
      tmpFilenameTF.setText( tmp );
      updateCommandLine();
    }
    return tmp;
  }


  /** get the name of a temporary file via File.createTempFile() */
  public static String createTmpFigFilename() {
    try {
      String prefix = "jfig-tmp";
      String suffix = ".fig";

      File tmpfile = File.createTempFile( prefix, suffix );
      tmpfile.deleteOnExit();

      return tmpfile.getPath();
    }
    catch( Throwable e ) {
e.printStackTrace();
      return getTmpFigFilename11();
    }
  }


  /** get the name of a temporary file with JDK 1.1 compatibility */
  public static String getTmpFigFilename11() {
    String prefix = "jfig-tmp";
    String suffix = ".fig";
    String  fname = null;

    // try a few times
    for( int i=0; i < 100; i++ ) {
      fname = prefix + i + suffix;
      File f = new File( fname );
      if (f.exists()) fname = null;
      else            break;
    }

    // give up. The following name is long but should be unique...
    if (fname == null) {
      fname = prefix + System.currentTimeMillis() + suffix;
    }
System.out.println( "-#- getTmpFigFilename11: '" + fname + "'" );

    return fname;
  }


  public String getOutFilename() {
    String dir  = directoryTF.getText();
    String name = filenameTF.getText();

    if (dir != null && dir.length() > 0) {
      return dir + File.separator + name;
    }
    else if (name == null || name.length() == 0) {
      return "unnamed.ps";
    }
    else {
      return name;
    }
  }


  public String getPsFigFilename() {
    String dir  = directoryTF.getText();
    String name = getFigFilename();
	
    int  i = name.lastIndexOf( File.separatorChar );
    if (i < 0) {  // no separator char
       name = name.substring( i+1 );
    }
    else {
       name = name.substring( i+1 );
    }
// strip extension, if any
    i = name.lastIndexOf( "." );
    if (i > 0) {  
       name = name.substring( 0, i );
    }
	name = name + ".ps" ;

    if (dir != null && dir.length() > 0) {
      return dir + File.separator + name;
    }
    else if (name == null || name.length() == 0) {
      return "unnamed.ps";
    }
    else {
      return name;
    }
  }


  /**
   * this method should be called whenever a new FIG file is loaded
   * into the editor, to guarantee that the output filename is
   * changed to match the new input file.
   * It should not be called for the same file, in order not to overwrite
   * any name changes by the user (via the browse button and file dialog).
   */
  public void updateOutFilenameFromFigFilename() {
    dbg( "-#- updateOutFilenameFromFigFilename..." );
    String source = null, path = null, name = null;
    try {
      source = getFigFilename();
      int  i = source.lastIndexOf( File.separatorChar );

      if (i < 0) {  // no separator char: use working directory
         path = "";
         name = source.substring( i+1 );
      }
      else {
         path = source.substring( 0, i );
         name = source.substring( i+1 );
      }
      filenameTF.setText( name );
      directoryTF.setText( path );

      updateOutFilenameExtension( languageChoice.getSelectedIndex() );
    }
    catch( Exception e ) {
      msg( "-E- internal: " + e );
      msg( "-E- filenames: " + source + " " + path + " " + name );
      e.printStackTrace();
    }
  }


  public String getFig2devPath() {
    return fig2devPathTF.getText().trim();
  }

  public String getFig2devMagnification() {
    return " -m " + getRoundedValue( getMagnification());
  }

  public String getRoundedValue( double d ) {
    return new jfig.utils.Format( "%3.2f" ).form( d );
  }


  public String getFig2devPSOrientation() {
    if ("landscape".equals( orientationChoice.getSelectedItem()))
         return " -l dummy";
    else return "";
  }

  public void setFig2devPSOrientation( boolean portrait ) {
    if (portrait) orientationChoice.setSelectedItem( "portrait" );
    else          orientationChoice.setSelectedItem( "landscape" );
  }


  public String getFig2devPSCentered() {
    if ("center".equals( centerChoice.getSelectedItem())) return " -c";
    else                                                  return " -e"; // edge
  }

  public void setFig2devPSCentered( boolean centered ) {
    if (centered) centerChoice.setSelectedItem( "center" );
    else          centerChoice.setSelectedItem( "flush left" );
  }

  public String getFig2devPSOffset() {
    double xoffset = 0.0;
    double yoffset = 0.0;
    String       s = "";

    try {
      xoffset = Double.valueOf( xoffsetTF.getText().trim() ).doubleValue();
    }
    catch( Exception e ) { }

    try {
      yoffset = Double.valueOf( yoffsetTF.getText().trim() ).doubleValue();
    }
    catch( Exception e ) { }

    if (xoffset != 0) s = s + " -x " + getRoundedValue(xoffset/72.0);
    if (yoffset != 0) s = s + " -y " + getRoundedValue(yoffset/72.0);

    return s;
  }


/* not used anymore
  private void call( String methodname ) {
    try {
      Class[]  formals = new Class[1];
      formals[0]       = Object.class;
      Method   method  = getClass().getMethod( methodname, formals );

      Object[] args = new Object[0];
      method.invoke( this, args );
    }
    catch( InvocationTargetException ite  ) {
      Throwable t = ite.getTargetException();
      msg( "-E- call: " + methodname + " " + t );
      t.printStackTrace();
    }
    catch( NoSuchMethodException e ) {
      msg( "-E- call: " + e + ":" + methodname );
    }
    catch( Exception e ) {
      msg( "-E- call: " + e + " for command: " + methodname );
      e.printStackTrace();
    }
  }
*/


  /**
   * set the magnification factor to fit all objects into a page of
   * format A4, landscape orientation.
   * Show the corresponding magnification factor and size in the
   * exportOptionsDialog.
   * Note: This method re-calculates the bounding box of all Fig objects
   * via updateFigObjectsBoundingBox() before doing the fitting.
   */
  public void fitToLandscapeA4() {
    updateFigObjectsBoundingBox();
    Dimension  d = new Dimension( (int)( bbox.getXr()-bbox.getXl()),
                                  (int)( bbox.getYb()-bbox.getYt()));
    double factor, fx, fy, dx, dy;

    fx = 1.0 * A4_WIDTH / d.width;
    fy = 1.0 * A4_HEIGHT / d.height;

    // the smaller of both, but at least 0.01
    factor = Math.min( fx, fy );
    factor = Math.max( 0.01, factor );
    dx     = factor * d.width;
    dy     = factor * d.height;

    setMagnification( factor );
    showSize();
  }


  /**
   * (re-) calculate the bounding box of all FIG objects on objectCanvas
   */
  public void updateFigObjectsBoundingBox() {
     try {
       bbox = BoundingBoxCalculator.getBoundingBox(
               editor.getObjects() );
     }
     catch( Throwable t ) {
       bbox = new FigBbox( 0, 0, A4_WIDTH, A4_HEIGHT );
     }
     if (debug) msg( "-#- updateFigObjectBoundingBox: " + bbox );

     showSize();
  }


  /**
   * this method just calls Runtime.exec() with the current value
   * of the 'command line'.
   * The motivation for this is to allow the user to edit the command line
   * as desired.
   */
  public void doExportNow() {
    dbg( "-#- doExportNow: " + commandLineTF.getText() );
    doAutoSave();
    try {
      Process pr = Runtime.getRuntime().exec( commandLineTF.getText() );
      if (waitExec) {
        pr.waitFor();
        printSubprocessOutput( pr );
      }

      // hack for automagic pstex/pstex_t export
      if (commandLine2TF.isEnabled()) {
        String cmd2 = commandLine2TF.getText();
        if ((cmd2 != null) && (cmd2.length() > 0)) {
          Process pr2 = Runtime.getRuntime().exec( cmd2 );
          if (waitExec) {
            pr2.waitFor();
            printSubprocessOutput( pr2 );
          }
        }
      }

    }
    catch( Exception e ) {
      msg( "-E- exception during export: " + e );
      e.printStackTrace();
    }
  };



  /**
   * export the current figure in Postscript format via fig2dev.
   * This method is a convenience method only called by JModularEditor.
   */
  public void doExportPS() {
    languageChoice.setSelectedIndex( 0 ); // Postscript
    updateOutFilenameExtension( 0 );
    updateCommandLine();
    doExportNow();
  }


  /**
   * export the current figure in encapsulated Postscript format via fig2dev.
   * This method is a convenience method only called by JModularEditor.
   */
  public void doExportEPS() {
    languageChoice.setSelectedIndex( 1 ); // EPS
    updateOutFilenameExtension( 1 );
    updateCommandLine();
    doExportNow();
  }


  public void requestAntiAliasing( boolean b ) {
    this.requestAntiAliasing = b;
  }


  public void checkRequestAntiAliasing( Graphics g ) {
    try {
      Graphics2D g2d = (Graphics2D) g;
      RenderingHints hints = new RenderingHints( null );
      if (requestAntiAliasing) {
        hints.put( RenderingHints.KEY_ANTIALIASING,
                   RenderingHints.VALUE_ANTIALIAS_ON );
        hints.put(RenderingHints.KEY_RENDERING,
                   RenderingHints.VALUE_RENDER_QUALITY);
        hints.put(RenderingHints.KEY_INTERPOLATION,
                   RenderingHints.VALUE_INTERPOLATION_BILINEAR);
      }
      else {
         hints.put( RenderingHints.KEY_ANTIALIASING,
                    RenderingHints.VALUE_ANTIALIAS_OFF);
      }
      g2d.setRenderingHints( hints );
    }
    catch( Exception e ) {
      msg( "-#- Internal in checkRequestAntiAliasing: "  + e );
    }
  }


  public void doExportNativeGIF() {
    languageChoice.setSelectedIndex( 8 ); // GIF
    updateOutFilenameExtension( 8 );
    updateCommandLine();
    outFilename = getOutFilename();
    msg( "-I- exporting GIF to file '" + outFilename + "'..." );

    //setWaitCursor();
    FigCanvas objectCanvas = editor.getObjectCanvas();
    try {
      int xsize = objectCanvas.getSize().width;
      int ysize = objectCanvas.getSize().height;
      Image im = this.createImage( xsize, ysize );
      if (im == null) {
         msg( "-E- Failed to create buffer image, cannot write GIF..." );
         return;
      }

      msg( "...drawing objects..." );
      Graphics g = im.getGraphics();
      checkRequestAntiAliasing( g );
      g.setColor( Color.white );
      g.fillRect( 0, 0, xsize, ysize );
      objectCanvas.drawAllObjects( g );
      // g.dispose();
try {
  Class.forName( "gifencoder.GIFEncoder" );
}
catch( Exception e ) {
  throw new RuntimeException( "GIFEncoder no longer included!" );
} 
      // msg( "...encoding GIF..." );
      // gifencoder.GIFEncoder encoder = new gifencoder.GIFEncoder( im );
      // msg( "...writing GIF to file..." );
      // FileOutputStream gifFile = new FileOutputStream( outFilename );
      // OutputStream of = new BufferedOutputStream( gifFile );
      // encoder.Write( of );
      // gifFile.close();
    }
    catch( Exception e ) {
      msg( "-E- Exception during GIF export: " + e.toString() );
      ExceptionTracer.trace( e );
    }
    //setDefaultCursor();
    msg( "...ready" );
  }

  public void doExportNativePNG() {
    languageChoice.setSelectedIndex( 9 ); // PNG
    updateOutFilenameExtension( 9 );
    updateCommandLine();
    outFilename = getOutFilename();
    msg( "-I- exporting PNG to file '" + outFilename + "'..." );

    //setWaitCursor();
    FigCanvas objectCanvas = editor.getObjectCanvas();
    try {
      int xsize = objectCanvas.getSize().width;
      int ysize = objectCanvas.getSize().height;
      BufferedImage im = new BufferedImage( xsize, ysize, BufferedImage.TYPE_INT_ARGB );
      if (im == null) {
         msg( "-E- Failed to create buffer image, cannot write GIF..." );
         return;
      }

      msg( "...drawing objects..." );
      Graphics g = im.getGraphics();
      checkRequestAntiAliasing( g );
      g.setColor( Color.white );
      g.fillRect( 0, 0, xsize, ysize );
      objectCanvas.drawAllObjects( g );
      // g.dispose();

      msg( "...encoding PNG..." );
      FileOutputStream pngFile = new FileOutputStream( outFilename );
      OutputStream of = new BufferedOutputStream( pngFile );
      javax.imageio.ImageIO.write( im, "PNG", pngFile );
      pngFile.close();
    }
    catch( Exception e ) {
      msg( "-E- Exception during PNG export: " + e.toString() );
      ExceptionTracer.trace( e );
    }
    //setDefaultCursor();
    msg( "...ready" );
  }



  public void doExportNativePPM() {
    languageChoice.setSelectedIndex( 6 ); // PPM
    updateOutFilenameExtension( 6 );
    updateCommandLine();
    outFilename = getOutFilename();
    msg( "-I- exporting PPM to file '" + outFilename + "'..." );

    //setWaitCursor();
    FigCanvas objectCanvas = editor.getObjectCanvas();

    try {
      int xsize = objectCanvas.getSize().width;
      int ysize = objectCanvas.getSize().height;
      Image im = this.createImage( xsize, ysize );
      if (im == null) {
         msg( "-E- Failed to create buffer image, cannot write PPM..." );
         return;
      }

      msg( "...drawing objects..." );
      Graphics g = im.getGraphics();
      checkRequestAntiAliasing( g );
      g.setColor( Color.white );
      g.fillRect( 0, 0, xsize, ysize );
      objectCanvas.drawAllObjects( g );
      // g.dispose();

      msg( "...encoding and writing PPM..." );
      DataOutputStream ppmstream = new DataOutputStream(
                                   new BufferedOutputStream(
                                     new FileOutputStream( outFilename )));
      new PPMWriter().writePPM( im, ppmstream );
      ppmstream.close();
      msg( "...ok. " );
    }
    catch( Exception e ) {
      msg( "-E- Exception during PPM export: " + e.toString() );
      ExceptionTracer.trace( e );
    }
    //setDefaultCursor();
    msg( "...ready" );
  }


  //public void doExportEPIC() {
  //  languageChoice.setSelectedIndex( 10 ); // EPIC
  //  updateOutFilenameExtension( 10 );
  //  updateCommandLine();
  //  doExportNow();
  //}

  //public void doExportEEPIC() {
  //  languageChoice.setSelectedIndex( 11 ); // EEPIC
  //  updateOutFilenameExtension( 11 );
  //  updateCommandLine();
  //  doExportNow();
  //}


  /**
   * combined pstex/pstex_t (Postscript non-texts plus LaTeX texts) export.
   * 
   * This method uses a hack proposed and first implemented by Rob Managan:
   * We first call fig2dev with -pstex language and .ps filename, but we
   * are not finished yet. Next, we select pstex_t as the language,
   * update the languageChoice and the command line, and call fig2dev again.
   * Finally, we reset the language choice to the pstex setting.
   */
/*
  public void doExportPSTEX() {
    if (KeyManager.getKeyManager().isValid()) {
      ; // ok
    }
    else {
      JTextViewerDialog viewer = new JTextViewerDialog( parent, 25, 80 );
      viewer.setText( 
        "Please register jfig to skip this dialog..." 
      );
      // viewer.parseResource( url );
      viewer.setCaretPosition( 0 );
      viewer.setModal( true );
      viewer.setTimeoutToEnableClosing( 30000 ); // 30 secs...
      viewer.show();
    }

    languageChoice.setSelectedIndex( 5 ); // PSTEX
    updateOutFilenameExtension( 5 );
    updateCommandLine();
    doExportNow();

    languageChoice.setSelectedIndex( 12 ); // PSTEX_T
    updateOutFilenameExtension( 12 );
    updateCommandLine();
    doExportNow();

    // reset gui to pstex
    languageChoice.setSelectedIndex( 5 ); // PSTEX
    updateOutFilenameExtension( 5 );
    updateCommandLine();
  }
*/


  public void doExportPrintPS() {
    languageChoice.setSelectedIndex( 0 ); // Postscript
    updateOutFilenameExtension( 0 );
    updateCommandLine();

    figFilename = getFigFilename();
    tmpFilename = getTmpFigFilename();
    outFilename = getOutFilename();

    doAutoSave();

    String cmd1 = fig2devPath + " " + fig2devPS
                  + getFig2devMagnification()
                  + getFig2devPSCentered()
                  + getFig2devPSOrientation()
                  + getFig2devPSOffset()
                  + " " + tmpFilename
                  + " " + outFilename;

    String cmd2 = printCommand + " " + outFilename;

    try {
       Process PR;
       PR = Runtime.getRuntime().exec( cmd1 );
       msg( "-I- " + PR.toString() + ": " + cmd1 );
       PR.waitFor();
       printSubprocessOutput( PR );

       PR = Runtime.getRuntime().exec( cmd2 );
       msg( "-I- " + PR.toString() + ": " + cmd2 );
       PR.waitFor();
       printSubprocessOutput( PR );
    }
    catch( Exception e ) {
       msg( "-E- Exception during export: " + e.toString() );
       ExceptionTracer.trace( e );
    }
  }




  public void doAutoSave() {
System.out.println( "-#- doAutoSave: saving to tmpFile '" + tmpFilename + "'" );
    try {
      JModularEditor   jme = (JModularEditor) editor;
      FileOutputStream fos = new FileOutputStream( tmpFilename );

      jme.writeFigObjects( fos );
      fos.flush();
      fos.close();
    }
    catch( Exception e ) {
      msg( "-E- JExportOptionsDialog.doAutoSave: Internal: " + e );
      e.printStackTrace();
    }
  }




  public void printSubprocessOutput( Process PR ) throws IOException {
    BufferedReader out, err;
    out = new BufferedReader( new InputStreamReader( PR.getInputStream() ));
    err = new BufferedReader( new InputStreamReader( PR.getErrorStream() ));

    String  s = null;
    if (err == null) return;
    while( (s = err.readLine()) != null) {
      msg( "-E- subprocess: " + s );
    }
    if (out == null) return;
    while( (s = out.readLine()) != null) {
      msg( "-I- subprocess: " + s );
    }
  }



  public void doBrowseFilename() {
    if (fileChooser == null) {
      fileChooser = new JFileChooser();
      fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
      fileChooser.setDialogTitle( "Select output filename:" );
    }

    try {
      String defaultDirectory = SetupManager.getProperty(
                                  "jfig.gui.CurrentFigFileDirectory", null );
      if (defaultDirectory != null) 
        fileChooser.setCurrentDirectory( new File(defaultDirectory)  );
    }
    catch( Exception e ) { } // no problem at all if this does not work
    
    int status = fileChooser.showSaveDialog( parent );
    if (status == JFileChooser.APPROVE_OPTION) {
       File selectedFile = fileChooser.getSelectedFile();
       filenameTF.setText( selectedFile.getName() );
       directoryTF.setText( selectedFile.getParent() );
    }

    updateCommandLine();
  }



  public void msg( String s ) {
    if (console != null) console.println( s );
    else System.out.println( s );
  }


  public void dbg( String s ) {
    if (debug) System.out.println( s );
  }


  /**
   * selftest for debugging
   */
  public static void main( String argv[] ) {
    SetupManager.loadLocalProperties( "jfig.cnf" );
    JFrame frame = new JFrame( "JExportOptionsDialog test." );
    JExportOptionsDialog f = new JExportOptionsDialog( frame );
    f.updateFigObjectsBoundingBox();
    f.setEditor( null );
    f.show();

    f.updateOutFilenameFromFigFilename();
  }
}


/* end JExportOptionsDialog.java */
