/**
	  Album Cover Grabber - Grabs album covers automatically off the internet
	  Copyright (C) 2002  Kevin M Schmidt

	  This program is free software; you can redistribute it and/or modify
	  it under the terms of the GNU General Public License as published by
	  the Free Software Foundation; either version 2 of the License, or
	  (at your option) any later version.

	  This program is distributed in the hope that it will be useful,
	  but WITHOUT ANY WARRANTY; without even the implied warranty of
	  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	  GNU General Public License for more details.

	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.bluedragontavern.albumgrabber.util;

import java.util.*;

import com.bluedragontavern.albumgrabber.*;

/**
 * This class compares a given string against a list of other strings and finds
 * the one in the list that matches the given string the closest.  It uses a
 * simple heuristic of comparing the numbers of letters that match.
 */
public class CompareStrings
{
	/**
	 * Find the order of the strings in the vector based on how they match
	 * against another given string.
	 * @param list Vector of strings to match
	 * @param s String to match against
	 * @return A vector of the indexs in the origianl vector
	 */
	public static Vector findBestMatchingOrder(Vector list, String s)
	{
		Vector indexScores = new Vector();

		//Loop through the list of strings and compare
		for (int i = 0; i < list.size(); ++i)
		{
			indexScores.add(new IndexScore(i,
													 compareStrings(
														 (String) list.elementAt(i), s)));
		}

		//Sort the index/scores
		Collections.sort(indexScores);
		Collections.reverse(indexScores);

		//Dont bother with more than the top 10 matches
		if (indexScores.size() > 10)
		{
			indexScores.setSize(10);
		}

		return indexScores;
	}

	/**
	 * Find the best matching string Returns the index of the best matching
	 * string in the list
	 * @param list DOCUMENT ME!
	 * @param s DOCUMENT ME!
	 * @return DOCUMENT ME!
	 */
	public static int findBestMatch(Vector list, String s)
	{
		String bestMatch = "";
		int bestMatchIndex = 0;
		int bestScore = 0;

		//Loop through the list of strings and compare
		for (int i = 0; i < list.size(); ++i)
		{
			String match = (String) list.elementAt(i);
			int score = compareStrings(match, s);

			//If a better fit, update best fit
			//Or, if they are equal, use the one with the closest size
			if (score > bestScore ||
				 (score == bestScore &&
				  (Math.abs(match.length() - s.length()) < Math.abs(
					  bestMatch.length() - s.length()))))
			{
				bestScore = score;
				bestMatch = match;
				bestMatchIndex = i;
			}
		}

		return bestMatchIndex;
	}

	/**
	 * Compare two strings and return an int representing the difference
	 * @param s1 String to compare
	 * @param s2 Other string to compare
	 * @return score A number representing how alike the two strings are
	 */
	private static int compareStrings(String s1, String s2)
	{
		int score = 0;

		byte[] chars1 = s1.getBytes();
		byte[] chars2 = s2.getBytes();
		byte[] letters1 = new byte[256];
		byte[] letters2 = new byte[256];

		for (int i = 0; i < chars1.length; ++i)
		{
			if (0 <= chars1[i] && chars1[i] < 256)
			{
				++letters1[chars1[i]];
			}
		}

		for (int i = 0; i < chars2.length; ++i)
		{
			if (0 <= chars2[i] && chars2[i] < 256)
			{
				++letters2[chars2[i]];
			}
		}

		for (int i = 0; i < 255; ++i)
		{
			if (letters1[i] > letters2[i])
			{
				score += letters2[i];
			}
			else
			{
				score += letters1[i];
			}
		}

		return score;
	}

	/**
	 * Compare the two album search results and return the better match
	 */
	public static Album compareAlbumSearchResults(Album album1, Album album2)
	{
		//Determine score for album1
		int score1 = 0;
		score1 += compareStrings(album1.getSearchArtistName(),
										 album1.getFoundArtistName());
		score1 += compareStrings(album1.getSearchAlbumName(),
										 album1.getFoundAlbumName());

		//Determine score for album2
		int score2 = 0;
		score2 += compareStrings(album2.getSearchArtistName(),
										 album2.getFoundArtistName());
		score2 += compareStrings(album2.getSearchAlbumName(),
										 album2.getFoundAlbumName());

		if (score1 > score2)
		{
			return album1;
		}
		else
		{
			return album2;
		}
	}
}