/*
 * Copyright (c) 2005 Jens Schou, Staffan Gustafsson, Bjrn Lanneskog, 
 * Einar Pehrson and Sebastian Kekkonen
 *
 * This file is part of
 * CleanSheets Extension for Test Cases
 *
 * CleanSheets Extension for Test Cases is free software; you can
 * redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * CleanSheets Extension for Test Cases is distributed in the hope that
 * it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CleanSheets Extension for Test Cases; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */
package csheets.ext.test;

import java.io.Serializable;

import csheets.core.Value;
import csheets.core.formula.Literal;

/**
 * A class holding information about a Test Case Parameter.
 * It contains the value of the parameter.
 * @author Staffan Gustafsson
 * @author Jens Schou
 */
public class TestCaseParam extends Literal implements Serializable {

	/** The unique version identifier used for serialization */
	private static final long serialVersionUID = -6821036933997460416L;

	private TestableCell cell;
	private int userEntered = 0;
	private int systemGenerated = 0;
	private int derived = 0;

	/** The test case parameters origin can be either user entered, 
	 * system generated within the cells assertion or derived from
	 * the values of its test cases
	 */
	public enum Type {

		/** Entered manually by the user. */
		USER_ENTERED,

		/** Auto-generated by the system based on the cell's assertion. */
		SYSTEM_GENERATED,

		/** Derived from the cell's test case value. */
		DERIVED
	};

	/**
	 * Creates the Test Case Parameter.
	 * @param cell the cell to which the parameter belongs
	 * @param value the value of the parameter.
	 * @param type the origin of the test case parameter
	 */
	public TestCaseParam(TestableCell cell, Value value, Type type) {
		super(value);
		this.cell = cell;

		switch(type){
		case USER_ENTERED:
			 userEntered++;
			 break;
		case SYSTEM_GENERATED:
			systemGenerated++;
			break;
		case DERIVED:
			derived++;
			break;
		}
	}

	/**
	 * Returns the Test Case Parameters cells address.
	 * @return The Test Case Parameters cells address.
	 */
	public TestableCell getCell(){
		return cell;
	}

	public boolean hasType(Type type) {
		switch(type){
		case USER_ENTERED:
			 return userEntered > 0;
		case SYSTEM_GENERATED:
			return systemGenerated > 0;
		case DERIVED:
			return derived > 0;
		default:
			return false;
		}
	}
	
	public void setType(Type type, boolean toggle) {
		// TO DO fulhack fr att hinna klart i tid
		switch(type){
		case USER_ENTERED:
			if(toggle)
				userEntered++;
			else
				userEntered--;
			break;
		case SYSTEM_GENERATED:
			if(toggle)
				systemGenerated++;
			else
				systemGenerated--;
			break;
		case DERIVED:
			if(toggle)
				derived++;
			else
				derived--;
			break;
		}
	}
	
	public boolean hasNoType(){
		return !(userEntered > 0 || systemGenerated > 0 || derived > 0);
	}

	public boolean isUserEntered() {
		return userEntered > 0;
	}
	public boolean isSystemGenerated() {
		return systemGenerated > 0;
	}
	public boolean isDerived() {
		return derived > 0;
	}

	public String toString2() {
		String temp = toString();

		if(userEntered > 0) temp += " user entered";
		if(systemGenerated > 0) temp += " system generated";
		if(derived > 0)  temp += " derived";

		return temp;
	}
		
	/**
	 * Returns whether the other object has the same value
	 * and type as this has. Could also check against address
	 * but that seems unecessary since TCPs are stored on
	 * the cell they belong to, and this way you can compare
	 * TCP's from different cells if you want.
	 * @param other the object to check for equality
	 * @return true if the objects are equal
	 */
	public boolean equals(Object other) {
		if (!(other instanceof TestCaseParam) || other == null)
			return false;
		TestCaseParam otherTCP = (TestCaseParam)other;
		return getValue().equals(otherTCP.getValue());
	}

	/**
	 * Returns the hash code of the test case parameter.
	 * @return the hash code of the test case parameter.
	 */
	public int hashCode() {
		return cell.hashCode() + getValue().hashCode();
	}
}