MODULE N2CPS;
(* ========================================================================= *)
(*                                                                           *)
(*  Main module for the .NET to Gardens Point Component Pascal Symbols tool. *)
(*      Copyright (c) Siu-Yuen Chan 2001.                                    *)
(*                                                                           *)
(*  This module handles the command line parameter parsing and makes calls   *)
(*  to various functions in other modules to accomplish the job.             *)
(* ========================================================================= *)

IMPORT
    MP := MetaParser,
    MS := MetaStore,
    SW := SymWriter,
    SR := SymReader,
    ST := AscString,
    GF := GPBinFiles,
    GPFiles,
    CS := Console,
    Error,
    ProgArgs,
    CPmain;

CONST
    DummyAssemblyName = "GPCP.dll";

TYPE
    CharOpen = POINTER TO ARRAY OF CHAR;
    AssemblyFiles = ARRAY MP.MaxAssemblies OF CharOpen;
    AssemblyMain = ARRAY MP.MaxAssemblies OF MS.Assembly;

VAR
    arg: ARRAY 128 OF CHAR;
    args: AssemblyFiles;
    acount: INTEGER;
    verbose: BOOLEAN;
    warn: BOOLEAN;

    ao: MS.Assembly;           (* assembly reference *)
    main: AssemblyMain;        (* main assemblies *)
    other: MS.Assembly;        (* all other assemblies referenced by main assembly *)

    atv: MS.OTraverser;        (* assemblies traverser *)
    stv: MS.OTraverser;        (* namespaces traverser *)
    pMain: MP.Parser;          (* meta-parser for main assembly *)
    pOther: MP.Parser;         (* meta-parser for other assemblies *)

    name: CharOpen;            (* assembly name *)
    dllfname: CharOpen;        (* dll assembly file name *)
    mclfname: CharOpen;        (* mcl assembly file name *)
    missingSym: BOOLEAN;       (* whether symbol file exist *)
    symfile: GF.FILE;          (* symbol file reference *)

    fns: MS.Namespace;         (* foreign namespaces in other assemblies *)
    asbname: CharOpen;         (* assembly name *)
    asbfile: CharOpen;         (* assembly filename *)
    nsname: CharOpen;          (* namespace name *)
    modname: CharOpen;         (* module name *)
    symfname: CharOpen;        (* symbol file name *)
    fullname: GPFiles.FileNameArray;    (* full path name for the assembly file *)
    idx: INTEGER;
    asbstart: BOOLEAN;
    i: INTEGER;


PROCEDURE LoadFailure();
BEGIN
    Error.WriteString("N2CPS_LoadFailure() - cannot load the assembly " + args[0]^); Error.WriteLn;
    HALT(2);
END LoadFailure;


PROCEDURE CheckCmdSyntax();
VAR
    argc, i: INTEGER;

    PROCEDURE Usage();
    BEGIN
        Error.WriteString("format: N2CPS [-m] [-v] [-w] [-h=size] <assembly_file> [<assembly_file>] [...]"); Error.WriteLn;
        Error.WriteString("               -h=size   run N2CPS with specified hash size"); Error.WriteLn;
        Error.WriteString("               -m        with method name mangling (gpcp doesn't handling method overload)"); Error.WriteLn;
        Error.WriteString("               -v        with verbose information"); Error.WriteLn;
        Error.WriteString("               -w        with warning information"); Error.WriteLn;
        Error.WriteLn;
        HALT(1);
    END Usage;

    PROCEDURE CheckOption(argno: INTEGER);
    (* firstly, all the possible switches, followed by all the assembly files *)
    VAR
        i, hs: INTEGER;
    BEGIN
        IF ~asbstart THEN
            IF (arg[0] = '-') THEN
                IF (arg[1] = 'w') & (arg[2] = 0X) THEN
                    MP.WarningOn(); warn := TRUE;
                ELSIF (arg[1] = 'v') & (arg[2] = 0X) THEN
                    verbose := TRUE;
                ELSIF (arg[1] = 'h') & (arg[2] = '=') THEN
                    i := 3; hs := 0;
                    WHILE ((arg[i] >= '0') & (arg[i] <= '9')) DO
                        hs := 10 * hs + (ORD(arg[i]) - ORD('0')); INC(i);
                    END; (* WHILE *)
                    IF (arg[i] # 0X) THEN Usage() END;
                    MS.SetHashSize(hs);
                ELSIF (arg[1] = 'r') & (arg[2] = 'c') & (arg[4] = 0X) THEN 
                    IF (arg[3] = '1') THEN
                        MP.SetVersion(MP.VerRC1);
                    ELSIF (arg[3] = '3') THEN
                        MP.SetVersion(MP.VerRC3);
                    END; (* IF *)
                ELSIF (arg[1] = 'b') & (arg[2] = 'e') & (arg[3] = 't') & (arg[4] = 'a') & (arg[6] = 0X) THEN 
                    IF (arg[5] = '1') THEN
                        MP.SetVersion(MP.VerBeta1);
                    ELSIF (arg[5] = '2') THEN
                        MP.SetVersion(MP.VerBeta2);
                    ELSE
                        Usage();
                    END; (* IF *)
                ELSIF (arg[1] = 'b') & (arg[2] = 'u') & (arg[3] = 'l') & (arg[4] = 'l') & (arg[6] = 0X) THEN 
                    IF (arg[5] = '1') THEN
                        MP.SetVersion(MP.VerFull1);
                    END; (* IF *)
                ELSIF (arg[1] = 'm') & (arg[2] = 0X) THEN
                    MS.SetMethodNameManglingOn();
             (* ELSIF (arg[1] = '?') & ... THEN *)
                    (* other possible switches here *)
                ELSE
                    Usage();
                END; (* IF *)
            ELSE
                (* assembly files *)
                asbstart := TRUE;
                (* stores the arg *)
                args[acount] := ST.ToChrOpen(arg);
                INC(acount);
            END;
        ELSE
            IF (arg[0] = '-') THEN
                (* should not have any more switches *)
                Usage();
            ELSE
                (* stores the arg *)
                args[acount] := ST.ToChrOpen(arg);
                INC(acount);
            END; (* IF *)
        END; (* IF *)
    END CheckOption;

BEGIN
    asbstart := FALSE;
    acount := 0;
    MP.SetVersion(MP.VerDefault);
    argc := ProgArgs.ArgNumber();
    IF argc = 0 THEN
        Usage();
    ELSE
        ProgArgs.GetArg(0, arg);
        IF argc = 1 THEN
            (* cannot have any switch for single argument *)
            IF arg[0] = '-' THEN Usage(); END;
            CheckOption(0);
        ELSE (* argc = 2, 3, ... *)
            FOR i := 0 TO argc-2 DO
                CheckOption(i);
                ProgArgs.GetArg(i+1, arg);
            END; (* FOR *)
            CheckOption(i);
        END; (* IF *)
    END; (* IF *)
END CheckCmdSyntax;


PROCEDURE CheckFiles(): BOOLEAN;
VAR
    i: INTEGER;
    status: BOOLEAN;
BEGIN
    status := TRUE;
    FOR i :=0 TO acount-1 DO
        IF ~GPFiles.exists(args[i]^) THEN
            Error.WriteString("N2CPS_CheckFiles - " + args[i]^ + " does not exist"); Error.WriteLn;
            status := FALSE;
        END; (* IF *)
    END; (* IF *)
    RETURN status;
END CheckFiles;


BEGIN
    MP.WarningOff(); warn := FALSE; verbose := FALSE; MS.SetMethodNameManglingOff();
    CheckCmdSyntax();
    pMain := MP.NewParser();

    IF CheckFiles() THEN

        i := 0;
        WHILE (i < acount) DO
    
            IF ~pMain.LoadAssembly(ST.ToChrOpen(args[i])) THEN LoadFailure() END;
IF verbose THEN
CS.WriteString("N2CPS - assembly " + args[i]^ + " loaded"); CS.WriteLn;
END; (* IF *)

            (* parse the assembly specified in the argument list *)
            main[i] := pMain.ParseAssembly();

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)
            (* turn off MetaParser warning messages after main assembly has been parsed *)
            MP.WarningOff();

            (* checkout what related assemblies is referenced by the specified assembly *)
            NEW(atv); atv.Initialize(MS.GetAllAssemblies());
            other := atv.GetNextAssembly();
            IF other # NIL THEN
                pOther := MP.NewParser();
                WHILE (other # NIL) DO
                    IF other # main[i] THEN

                        other.SnapShot();
                        NEW(stv); stv.Initialize(other.GetNamespaces());
                        fns := stv.GetNextNamespace();
                        missingSym := FALSE;
                        WHILE (fns # NIL) & ~missingSym DO
                            asbname := other.GetName();
                            nsname := fns.GetName();
                            modname := MS.MakeModuleName(asbname, nsname);
                            IF ST.StrCmp(modname, ST.NullString) # ST.Equal THEN
                                symfname := ST.StrCat(modname, ST.ToChrOpen(SW.SymbolExt)); 
IF verbose THEN
CS.WriteString("N2CPS - locating symbol file: " + symfname^); CS.WriteLn;
END; (* IF *)
                                symfile := GF.findOnPath("CPSYM", symfname);
                                IF symfile # NIL THEN
IF verbose THEN
CS.WriteString("N2CPS - parsing symbol file of module " + modname^); CS.WriteLn;
END; (* IF *)
                                    SR.ParseSymbolFile(symfile, modname);
                                    GF.CloseFile(symfile);
                                ELSE
IF verbose THEN
CS.WriteString("N2CPS - symbol file of module " + modname^ + " NOT exist"); CS.WriteLn;
END; (* IF *)
                                    missingSym := TRUE;
                                END; (* IF *)
                            END; (* IF *)
                            fns := stv.GetNextNamespace();
                        END; (* WHILE *)
                        IF missingSym THEN
                            (* should we parse the entire assembly for missing one symbol file *)
                            (* parse assemblies referenced by main assembly *)
                            name := other.GetFileName();

IF verbose THEN
CS.WriteString("N2CPS - locating assembly file of module: " + modname^); CS.WriteLn;
END; (* IF *)
                            (* (1) try name.dll at current path *)
                            dllfname := ST.StrCat(name,ST.ToChrOpen(MS.AssemblyExt));
                            fullname := ST.StrCat(pMain.GetAsmPath(), dllfname);
                            IF GPFiles.exists(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - (1) loading DLL assembly file: " + fullname^); CS.WriteLn;
END; (* IF *)
                                IF pOther.LoadAssembly(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - parsing assembly file: " + dllfname^); CS.WriteLn;
END; (* IF *)

                                    ao := pOther.ParseAssembly();
                                    ASSERT(ao = other);
                                END; (* IF *)
                            ELSE
                                (* (2) try name.mcl at current path *)
                                mclfname := ST.StrCat(name,ST.ToChrOpen(MS.TypeLibExt));
                                fullname := ST.StrCat(pMain.GetAsmPath(), mclfname);
                                IF GPFiles.exists(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - (2) loading MCL assembly file: " + fullname^); CS.WriteLn;
END; (* IF *)
                                    IF pOther.LoadAssembly(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - parsing assembly file: " + mclfname^); CS.WriteLn;
END; (* IF *)
                                        ao := pOther.ParseAssembly();
                                        ASSERT(ao = other);
                                    END; (* IF *)
                                ELSE
                                    (* (3) try name.dll at Default path *)
                                    fullname := ST.StrCat(ST.ToChrOpen(MP.DefaultPath), dllfname);
                                    IF GPFiles.exists(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - (3) loading DLL assembly file: " + fullname^); CS.WriteLn;
END; (* IF *)
                                        IF pOther.LoadAssembly(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - parsing assembly file: " + dllfname^); CS.WriteLn;
END; (* IF *)
                                            ao := pOther.ParseAssembly();
                                            ASSERT(ao = other);
                                        END; (* IF *)
                                    ELSE
                                        (* (4) try name.mcl at Default path *)
                                        fullname := ST.StrCat(ST.ToChrOpen(MP.DefaultPath), mclfname);
                                        IF GPFiles.exists(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - (4) loading MCL assembly file: " + fullname^); CS.WriteLn;
END; (* IF *)
                                            IF pOther.LoadAssembly(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - parsing assembly file: " + mclfname^); CS.WriteLn;
END; (* IF *)
                                                ao := pOther.ParseAssembly();
                                                ASSERT(ao = other);
                                            END; (* IF *)
                                        ELSE

                                            (* (5) try name.dll at Interop path *)
                                            fullname := ST.StrCat(ST.ToChrOpen(MP.InteropPath), dllfname);
                                            IF GPFiles.exists(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - (5) loading DLL assembly file: " + fullname^); CS.WriteLn;
END; (* IF *)
                                                IF pOther.LoadAssembly(fullname) THEN
IF verbose THEN
CS.WriteString("N2CPS - parsing assembly file: " + dllfname^); CS.WriteLn;
END; (* IF *)
                                                    ao := pOther.ParseAssembly();
                                                    ASSERT(ao = other);
                                                END; (* IF *)
                                            ELSE
                                                IF name^ # MS.DmyAssembly THEN  (* this is a dummy name *)
                                                    IF warn THEN
                                                        Error.WriteString("N2CPS - Warning !!! - symbol file or dll file of related assembly ");
                                                        Error.WriteString(dllfname);
                                                        Error.WriteString(" cannot be found"); Error.WriteLn;
                                                    END; (* IF *)
                                                END; (* IF *)
                                            END; (* IF *)
                                        END; (* IF *)
                                    END; (* IF *)
                                END; (* IF *)
                            END; (* IF *)
                        END; (* IF *)

                    ELSE
                    END; (* IF *)
                    other := atv.GetNextAssembly();
                END; (* WHILE *)
            ELSE
            END; (* IF *)

            INC(i);
        END; (* WHILE *)

(* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> *)
        i := 0;
        WHILE (i < acount) DO
            main[i].SnapShot();
IF verbose THEN
CS.WriteString("N2CPS - writing symbol file(s) of assembly: " + main[i].GetName()^); CS.WriteLn;
END; (* IF *)
            SW.EmitSymbolFiles(main[i]);
            INC(i);
        END; (* WHILE *)

    END; (* IF *)

END N2CPS.
