// copyright 2001-2002 by The Mind Electric

package electric.util;

import java.lang.reflect.*;
import electric.util.reflect.*;

/**
 * <tt>ArrayUtil</tt> defines a set of static methods for manipulating Java arrays.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public final class ArrayUtil
  {
  private static final Class BYTE_ARRAY = (new byte[ 0 ]).getClass();

  // ********** ARRAY OPERATIONS ********************************************

  /**
   * Return a new array that is a copy of the array plus a new element.
   * The component type of the array must be the same as that type of the element.
   * @param array An array
   * @param element The element to append.
   */
  public static Object addElement( Object array, Object element )
    {
    int length = Array.getLength( array );
    Object newarray = Array.newInstance( array.getClass().getComponentType(), length + 1 );
    System.arraycopy( array, 0, newarray, 0, length );
    Array.set( newarray, length, element );
    return newarray;
    }

  /**
   * Return a new array that is a copy of the array plus a new integer element.
   * The component type of the array must be the same as that type of the element.
   * @param array An array
   * @param element The element to append.
   */
  public static int[] addElement( int[] array, int element )
    {
    int length = array.length;
    int[] newarray = new int[ length + 1 ];
    System.arraycopy( array, 0, newarray, 0, length );
    newarray[ length ] = element;
    return newarray;
    }

  /**
   * Return a new array that is the union of the elements in array1 and array2.
   * The component types of both arrays must be the same.
   * @param array1 The first array.
   * @param array2 The second array.
   */
  public static Object addElements( Object array1, Object array2 )
    {
    int length1 = Array.getLength( array1 );
    int length2 = Array.getLength( array2 );
    Object newarray = Array.newInstance( array1.getClass().getComponentType(), length1 + length2 );
    System.arraycopy( array1, 0, newarray, 0, length1 );
    System.arraycopy( array2, 0, newarray, length1, length2 );
    return newarray;
    }

  /**
   * @param array
   * @param newLength
   */
  public static Object grow( Object array, int newLength )
    {
    int oldLength = Array.getLength( array );
    Object newarray = Array.newInstance( array.getClass().getComponentType(), newLength );
    System.arraycopy( array, 0, newarray, 0, oldLength );
    return newarray;
    }

  /**
   * Return a new array that is a copy of the array with the new element inserted at the beginning.
   * The component type of the array must be the same as that type of the element.
   * @param array An array
   * @param element The element to insert.
   */
  public static Object insertElement( Object array, Object element )
    {
    return insertElementAt( array, element, 0 );
    }

  /**
   * Return a new array that is a copy of the array with the newly inserted element.
   * The component type of the array must be the same as that type of the element.
   * @param array An array
   * @param element The element to insert.
   * @param index The index to insert the element before.
   */
  public static Object insertElementAt( Object array, Object element, int index )
    {
    int length = Array.getLength( array );
    Object newarray = Array.newInstance( array.getClass().getComponentType(), length + 1 );

    if( index > 0 )
      System.arraycopy( array, 0, newarray, 0, index );

    Array.set( newarray, index, element );
    System.arraycopy( array, index, newarray, index + 1, length - index );
    return newarray;
    }

  /**
   * @param array
   * @param element
   */
  public static Object removeElement( Object array, Object element )
    {
    int length = Array.getLength( array );

    for( int i = 0; i < length; i++ )
      if( element.equals( Array.get( array, i ) ) )
        return removeElementAt( array, i );

    return array;
    }

  /**
   * @param array1
   * @param array2
   */
  public static Object removeElements( Object array1, Object array2 )
    {
    int length = Array.getLength( array2 );

    for( int i = 0; i < length; i++ )
      array1 = removeElement( array1, Array.get( array2, i ) );

    return array1;
    }

  /**
   * @param array
   * @param element
   */
  public static Object removeElementIdentity( Object array, Object element )
    {
    int length = Array.getLength( array );

    for( int i = 0; i < length; i++ )
      if( element == Array.get( array, i ) )
        return removeElementAt( array, i );

    return array;
    }

  /**
   * @param oldarray
   * @param index
   */
  public static Object removeElementAt( Object oldarray, int index )
    {
    int length = Array.getLength( oldarray );
    Object newarray = Array.newInstance( oldarray.getClass().getComponentType(), length - 1 );
    System.arraycopy( oldarray, 0, newarray, 0, index );
    System.arraycopy( oldarray, index + 1, newarray, index, length - index - 1 );
    return newarray;
    }

  /**
   * @param value
   * @param array
   */
  public static int indexOf( int value, int[] array )
    {
    for( int i = 0; i < array.length; i++ )
      if( array[ i ] == value )
        return i;

    return -1;
    }

  /**
   * @param value
   * @param array
   */
  public static int indexOf( Object value, Object[] array )
    {
    for( int i = 0; i < array.length; i++ )
      if( equals( array[ i ], value ) )
        return i;

    return -1;
    }

  /**
   * @param value
   * @param array
   */
  public static int indexOfIdentity( Object value, Object[] array )
    {
    for( int i = 0; i < array.length; i++ )
      if( value == array[ i ] )
        return i;

    return -1;
    }

  /**
   * Returns true if the value object is contained within the array.  NOTE that
   * this is not a test for reference equality, but rather value equality.
   * @param value
   * @param array
   */
  public static boolean contains( Object value, Object[] array )
    {
    return (indexOf( value, array ) != -1);
    }

  /**
   * @param array1
   * @param array2
   */
  public static boolean equals( Object[] array1, Object[] array2 )
    {
    if( array1.length != array2.length )
      return false;

    for( int i = 0; i < array1.length; i++ )
      if( !equals( array1[ i ], array2[ i ] ) )
        return false;

    return true;
    }

  /**
   * @param object1
   * @param object2
   */
  public static boolean equals( Object object1, Object object2 )
    {
    if( object1 == null && object2 == null )
      return true;
    else if( object1 == null || object2 == null )
      return false;
    else
      return object1.equals( object2 );
    }

  /**
   * @param object
   */
  public static byte[] toBytes( Object object )
    {
    if( BYTE_ARRAY.isInstance( object ) )
      return (byte[]) object;
    else if( object == null )
      return "null".getBytes();
    else
      return object.toString().getBytes();
    }

  /**
   * @param possibleArray
   */
  public static boolean isPrimitiveArray( Object possibleArray )
    {
    return isPrimitiveArray( possibleArray.getClass() );
    }

  /**
   * @param arrayClass
   */
  public static boolean isPrimitiveArray( Class arrayClass )
    {
    while( arrayClass.isArray() )
      arrayClass = arrayClass.getComponentType();

    return arrayClass.isPrimitive();
    }

  /**
   * @param primitiveArray
   */
  public static Object convertToWrapperArray( Object primitiveArray )
    {
    int[] dimensions = new int[ 0 ];
    Class arrayClass = primitiveArray.getClass();

    while( arrayClass.isArray() )
      {
      dimensions = addElement( dimensions, 0 );
      arrayClass = arrayClass.getComponentType();
      }

    Object newArray = Array.newInstance( Wrappers.getWrapperClass( arrayClass ), dimensions );
    return copyArray( primitiveArray, newArray );
    }

  /**
   * @param primitiveArray
   */
  public static Object convertToPrimitiveArray( Object wrapperArray )
    {
    int[] dimensions = new int[ 0 ];
    Class arrayClass = wrapperArray.getClass();

    while( arrayClass.isArray() )
      {
      dimensions = addElement( dimensions, 0 );
      arrayClass = arrayClass.getComponentType();
      }

    Object newArray = Array.newInstance( Wrappers.getPrimitiveClass( arrayClass ), dimensions );
    return copyArray( wrapperArray, newArray );
    }

  /**
   * @param srcArray
   * @param destArray
   */
  private static Object copyArray( Object srcArray, Object destArray )
    {
    int length = Array.getLength( srcArray );

    if( length != Array.getLength( destArray ) )
      destArray = Array.newInstance( destArray.getClass().getComponentType(), length );

    for( int i = 0; i < length; i ++ )
      {
      Object member = Array.get( srcArray, i );

      if( member.getClass().isArray() )
        {
        Object destMember = Array.get( destArray, i );

        if( destMember == null )
          destMember = Array.newInstance( destArray.getClass().getComponentType().getComponentType(), Array.getLength( member ) );

        member = copyArray( member, destMember );
        }

      Array.set( destArray, i, member );
      }

    return destArray;
    }

  /**
   * @param array
   */
  public static int getNumberOfDimensions( Object array )
    {
    return getNumberOfDimensions( array.getClass() );
    }

  /**
   * @param arrayClass
   */
  public static int getNumberOfDimensions( Class arrayClass )
    {
    int dimensions = 0;

    while( arrayClass.isArray() )
      {
      arrayClass = arrayClass.getComponentType();
      dimensions++;
      }

    return dimensions;
    }

  /**
   * @param array
   */
  public static Class getBaseClass( Object array )
    {
    return getBaseClass( array.getClass() );
    }

  /**
   * @param arrayClass
   */
  public static Class getBaseClass( Class arrayClass )
    {
    while( arrayClass.isArray() )
      arrayClass = arrayClass.getComponentType();

    return arrayClass;
    }
  }