// copyright 2001-2002 by The Mind Electric

package electric.util.log;

import java.io.*;
import java.util.*;

/**
 * <tt>Log</tt> defines static methods for maintaining a collection of ILoggers
 * and sending an event to all of them. The collection is initialized to a
 * single default logger that logs ERROR and COMMENT events to System.out.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public final class Log
  {
  private static final Hashtable stringToCode = new Hashtable();
  private static final Hashtable codeToString = new Hashtable();
  private static final Hashtable nameToLogger = new Hashtable();
  private static final Hashtable disabledCategories = new Hashtable();
  private static final long COMMENT_EVENT = getCode( "COMMENT" );
  private static long masks; // union of masks of all loggers

  static
    {
    initLoggers();
    }

  /**
   *
   */
  private static void initLoggers()
    {
    addLogger( "default", new WriterLogger( 0 ) );
    }

  // ********** TESTING *****************************************************

  /**
   * Return true if at least one of the loggers is logging a
   * category defined by the mask.
   * @param mask The mask.
   */
  public static boolean isLogging( long mask )
    {
    return ((masks & mask) != 0);
    }

  /**
   * Return true if at least one of the loggers is logging the specified category.
   * @param category The category.
   */
  public static boolean isLogging( String category )
    {
    return isLogging( getCode( category ) );
    }

  // ********** MASKS *******************************************************

  /**
   * Return the code for the specified category.
   * @param category The category.
   */
  public synchronized static long getCode( String category )
    {
    Long code = (Long) stringToCode.get( category );

    if( code == null )
      {
      code = new Long( 1 << stringToCode.size() );
      stringToCode.put( category, code );
      codeToString.put( code, category );
      }

    return code.longValue();
    }

  /**
   * Return the category associated with the specified code.
   * @param code The code.
   */
  public synchronized static String getCategory( long code )
    {
    return (String) codeToString.get( new Long( code ) );
    }

  /**
   * Recalculate the master mask value.
   */
  static synchronized void recalcMasks()
    {
    masks = 0;

    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      masks |= ((ILogger) enum.nextElement()).getMask();
    }

  // ********** ADDING/REMOVING/GETTING LOGGERS *************************************

  /**
   * Returns a logger with the specified logger name.
   * @param name The name of the logger
   * @return named logger or null if no logger is mapped to the name
   */
  public synchronized static ILogger getLogger( String name )
    {
    return (ILogger) nameToLogger.get( name );
    }

  /**
   * Add the specified logger to my collection of loggers.
   * @param name The name of the logger.
   * @param logger The logger to add.
   */
  public synchronized static void addLogger( String name, ILogger logger )
    {
    nameToLogger.put( name, logger );
    recalcMasks();
    }

  /**
   * Remove the specified logger from my collection of loggers.
   * @param name The name of the logger.
   */
  public synchronized static void removeLogger( String name )
    {
    nameToLogger.remove( name );
    recalcMasks();
    }

  /**
   * Disable the specified logger from my collection of loggers.
   * @param name The name of the logger.
   */
  public synchronized static void disableLogger( String name )
    {
    ILogger logger = (ILogger) nameToLogger.get( name );

    if( logger != null )
      logger.disable();
    }

  /**
   * Reenable the specified logger in my collection of loggers.
   * @param name The name of the logger.
   */
  public synchronized static void enableLogger( String name )
    {
    ILogger logger = (ILogger) nameToLogger.get( name );

    if( logger != null )
      logger.enable();
    }

  /**
   * Return an enumeration over all the current loggers.
   */
  public synchronized static Enumeration getLoggers()
    {
    Vector loggers = new Vector();

    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      loggers.addElement( enum.nextElement() );

    return loggers.elements();
    }

  /**
   * Return an enumeration over all the current logger names.
   */
  public synchronized static Enumeration getLoggerNames()
    {
    Vector names = new Vector();

    for( Enumeration enum = nameToLogger.keys(); enum.hasMoreElements(); )
      names.addElement( enum.nextElement() );

    return names.elements();
    }

  // ********** CONTROLLING ALL LOGGERS *************************************

  /**
   * Instruct all loggers to start logging events of the specified category.
   * @param category The category.
   */
  public synchronized static void startLogging( String category )
    {
    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      ((ILogger) enum.nextElement()).startLogging( category );
    }

  /**
   * Instruct all loggers to stop logging events of the specified category.
   * @param category The category.
   */
  public synchronized static void stopLogging( String category )
    {
    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      ((ILogger) enum.nextElement()).stopLogging( category );
    }

  /**
   * Instruct all loggers to resume logging events of the specified category.
   * @param category The category.
   */
  public synchronized static void enableLogging( String category )
    {
    Vector v = (Vector) disabledCategories.remove( category );

    if( v == null )
      return;

    for( Enumeration enum = v.elements(); enum.hasMoreElements(); )
      {
      ILogger logger = (ILogger) enum.nextElement();
      logger.startLogging( category );
      }
    }

  /**
   * Instruct all loggers to stop logging events of the specified category.
   * @param category The category.
   */
  public synchronized static void disableLogging( String category )
    {
    long code = getCode( category );
    Vector logs = null;

    if( (logs = (Vector) disabledCategories.get( category )) == null )
      logs = new Vector();

    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      {
      ILogger logger = (ILogger) enum.nextElement();

      if( (logger.getMask() & code) != 0 )
        {
        logger.stopLogging( category );

        if( !logs.contains( logger ) )
          logs.addElement( logger );
        }
      }

    if( !logs.isEmpty() )
      disabledCategories.put( category, logs );
    }

  /**
   * Set the mask of all the loggers to the specified value.
   * @param mask The new mask.
   */
  public synchronized static void setMask( long mask )
    {
    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      ((ILogger) enum.nextElement()).setMask( mask );
    }

  // ********** LOG AN EVENT ************************************************

  /**
   * Log the event with category "COMMENT".
   * This is equivalent to log( COMMENT_EVENT, event );
   * @param event The event.
   */
  public static void log( Object event )
    {
    log( COMMENT_EVENT, event );
    }

  /**
   * @param category
   * @param throwable
   */
  public synchronized static void log( String category, Throwable throwable )
    {
    log( getCode( category ), throwable );
    }

  /**
   * @param code
   * @param throwable
   */
  public synchronized static void log( long code, Throwable throwable )
    {
    log( code, null, throwable );
    }

  /**
   * @param code
   * @param message
   * @param throwable
   */
  public synchronized static void log( long code, String message, Throwable throwable )
    {
    if( (masks & code) == 0 )
      return;

    StringWriter writer = new StringWriter();
    throwable.printStackTrace( new PrintWriter( writer ) );

    if( message != null )
      message += ("\n" + writer.toString());
    else
      message = writer.toString();

    log( code, message );
    }

  /**
   * Send event( category, message ) to each logger that is logging the specified category.
   * @param category The category.
   * @param event The event to log.
   */
  public synchronized static void log( String category, Object event )
    {
    long code = getCode( category );

    if( (masks & code) != 0 )
      log( code, category, event );
    }

  /**
   * Send event( category, message ) to each logger that is logging the specified category.
   * @param code The category code.
   * @param event The event to log.
   */
  public synchronized static void log( long code, Object event )
    {
    if( (masks & code) != 0 )
      log( code, getCategory( code ), event );
    }

  /**
   * @param code
   * @param category
   * @param event
   */
  private synchronized static void log( long code, String category, Object event )
    {
    Date timestamp = new Date();

    for( Enumeration enum = nameToLogger.elements(); enum.hasMoreElements(); )
      {
      ILogger logger = (ILogger) enum.nextElement();

      if( (logger.getMask() & code) != 0 )
        logger.event( category, event, timestamp );
      }
    }
  }
