// copyright 2001-2002 by The Mind Electric

package electric.util.log;

import java.io.*;
import java.util.*;
import java.text.*;
import electric.util.*;

/**
 * <tt>WriterLogger</tt> is an implementation of ILogger that extends Logger
 * and defines event() to write the event to a Writer.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class WriterLogger extends Logger
  {
  private static final String DEFAULT_FORMAT = "MMM:dd:yyyy hh:mm:ss";

  // line.separator ok for applets
  public final static String EOL = System.getProperty("line.separator");

  private Writer writer;
  private SimpleDateFormat dateFormatter;
  private boolean logThreadName = false;

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct a WriterLogger that writes to the specified writer with
   * an initial mask value of zero (i.e. does not log any events).
   * @param writer The writer.
   */
  public WriterLogger( Writer writer )
    {
    this( writer, 0 );
    }

  /**
   * Construct a WriterLogger that writes to System.out with an
   * initial mask value
   * @param mask The initial mask value.
   */
  public WriterLogger( long mask )
    {
    this( new OutputStreamWriter( System.out ), mask );
    }

  /**
   * Construct a WriterLogger that writes to the specified writer with an
   * initial mask value.
   * @param writer The writer.
   * @param mask The initial mask value.
   */
  public WriterLogger( Writer writer, long mask )
    {
    super( mask );
    this.writer = writer;
    }

  // ********** CONFIGURATION ***********************************************

  /**
   * @param format
   */
  public void logDateTime( String format )
    {
    if( format == null )
      format = DEFAULT_FORMAT;

    dateFormatter = new SimpleDateFormat( format );
    }

  /**
   * @param logThreadName
   */
  public void logThreadName( boolean logThreadName )
    {
    this.logThreadName = logThreadName;
    }

  // ********** WRITER ACCESSORS ********************************************

  /**
   * Set the writer.
   * @param writer The new writer.
   */
  public void setWriter( Writer writer )
    {
    this.writer = writer;
    }

  /**
   * Return the writer.
   */
  public Writer getWriter()
    {
    return writer;
    }

  // ********** EVENT HANDLER ***********************************************

  /**
   * If enabled, log an event of the specified category to the underlying Writer.
   * @param category The category.
   * @param event The event.
   * @param timestamp The time of the event was logged.
   */
  public void event( String category, Object event, Date timestamp )
    {
    try
      {
      StringBuffer buffer = new StringBuffer();
      buffer.append( '[' ).append( category );

      if( logThreadName )
        buffer.append( ", " ).append( Thread.currentThread().getName() );

      if( dateFormatter != null )
        buffer.append( ", " ).append( dateFormatter.format( timestamp ) );

      buffer.append( "] " );
      buffer.append( event.toString() ).append( EOL );
      writer.write( buffer.toString() );
      writer.flush();
      }
    catch( IOException exception )
      {
      System.out.println( "WriterLogger error: " + exception );
      }
    }
  }
