// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import java.util.*;
import electric.util.*;

/**
 * <tt>Element</tt> represents an XML element, and includes methods for
 * manipulating attributes and child nodes. Some of the accessors support
 * a useful subset of XPath.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class Element extends Parent implements IQNamed, INamespaceContext, org.w3c.dom.Element
  {
  private static final String UNDEFINED = "UNDEFINED";
  private static final String NOT_A_NUMBER = "NaN";
  private static final String POSITIVE_INFINITY = "INF";
  private static final String NEGATIVE_INFINITY = "-INF";

  String prefix;
  String namespace;
  String name;
  private NodeList attributes = new NodeList();

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct an Element with name set to "UNDEFINED" and no parent.
   */
  public Element()
    {
    this.name = UNDEFINED;
    }

  /**
   * Construct an Element with the specified name and no parent.
   * @param name The name.
   */
  public Element( String name )
    {
    this.name = name;
    }

  /**
   * Construct an Element with the specified prefix, name and namespace.
   * Care should be taken when using this constructor, because the validity
   * of the prefix cannot be checked since the Element is not part of a
   * valid XML document.
   * @param name The prefix.
   * @param name The name.
   * @param namespace The namespace.
   */
  public Element( String prefix, String name, String namespace )
    {
    this.prefix = prefix;
    this.name = name;
    this.namespace = namespace;
    }

  /**
   * Construct a copy of the specified Element.
   * @param element The Element to copy.
   */
  public Element( Element element )
    {
    super( element );
    this.prefix = element.prefix;
    this.namespace = element.namespace;
    this.name = element.name;

    for( Attribute attribute = (Attribute) element.attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      attributes.append( new Attribute( attribute ) );
    }

  /**
   * Construct an Element from the specified lexical analyzer.
   * @param lex The lexical analyzer.
   * @param parent The parent of this Element.
   * @param strip True if extra whitespace is removed during parsing.
   * @throws IOException If an error occurs during parsing.
   * @throws NamespaceException If a namespace prefix could not be resolved.
   */
  Element( Lex lex, Parent parent, boolean strip )
    throws IOException, NamespaceException
    {
    parent.addChild( this );
    lex.read(); // '<'
    name = lex.readToken();

    if( lex.peek() == ':' )
      {
      prefix = name;
      lex.read(); // ':'
      name = lex.readToken();
      }

    lex.skipWhitespace();
    int peek = lex.peek();

    if( peek != '>' && peek != '/' )
      parseAttributes( lex );

    if( prefix == null )
      {
      namespace = getNamespace( "" );
      }
    else
      {
      namespace = getNamespace( prefix );

      if( namespace == null )
        throw new NamespaceException( "could not find namespace with prefix " + prefix );
      }

    int ch = lex.read();

    if( ch == '/' ) // empty
      {
      lex.readChar( '>' );
      }
    else if( ch != '>' )
      {
      throw new IOException( "expected > or /" );
      }
    else // read >, parse children
      {
      parseChildren( lex, strip );
      lex.readChar( '<' );
      lex.readChar( '/' );

      if( prefix != null )
        {
        String endPrefix = lex.readToken();

        if( !prefix.equals( endPrefix ) )
          throw new IOException( "<" + prefix + ":...> does not match </" + endPrefix + ":...>" );

        lex.readChar( ':' );
        }

      String endName = lex.readToken();
      lex.readChar( '>' );

      if( !name.equals( endName ) )
        throw new IOException( "<..." + name + "> does not match </..." + endName + ">" );
      }
    }

  /**
   * Parse my attributes from the specified lexical analyzer.
   * @param lex The lexicial analyzer.
   * @throws IOException If an I/O exception occurs during parsing.
   * @throws NamespaceException If a name could not be resolved.
   */
  private void parseAttributes( Lex lex )
    throws IOException, NamespaceException
    {
    int peek = 0;

    do
      {
      attributes.append( new Attribute( lex, this ) );
      lex.skipWhitespace();
      peek = lex.peek();
      }
    while( peek != '>' && peek != '/' );

    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      attribute.resolve( this );
    }

  /**
   * Parse my children from the specified lexical analyzer
   * @param lex The lexical analyzer
   * @param strip True if extra whitespace is removed during parsing.
   * @throws IOException If an I/O exception occurs.
   * @throws NamespaceException If a name could not be resolved.
   */
  private void parseChildren( Lex lex, boolean strip )
    throws IOException, NamespaceException
    {
    while( true )
      {
      StringBuffer whitespace = lex.readWhitespace();
      lex.mark( 2 );
      int ch1 = lex.peekRead();
      int ch2 = lex.peekRead();
      lex.reset();

      if( ch1 == -1 ) // eof
        {
        break;
        }
      else if( ch1 != '<' )
        {
        new Text( lex, whitespace, this );
        }
      else if( ch2 == '/' ) // end tag "</"
        {
        if( whitespace != null && ((!strip) || children.isEmpty()) )
          new Text( lex, whitespace, this );

        break;
        }
      else
        {
        if( whitespace != null && !strip )
          new Text( this ).setString( whitespace.toString() );

        if( ch2 == '!' && lex.peekString( CData.START ) )
          new CData( lex, this );
        else if( ch2 == '!' && lex.peekString( Comment.START ) )
          new Comment( lex, this );
        else if( ch2 == '?' )
          new Instruction( lex, this );
        else // start tag "<"
          new Element( lex, this, strip );
        }
      }
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this Element.
   */
  public Object clone()
    {
    return new Element( this );
    }

  // ********** ROOT ********************************************************

  /**
   * Return the root of my parent, or null if I have no parent.
   */
  public Element getRoot()
    {
    return (parent == null ? this : parent.getRoot());
    }

  // ********** PARENT ******************************************************

  /**
   * Return my parent element.
   */
  public Element getParentElement()
    {
    return (Element) getParent();
    }

  // ********** NAMES *******************************************************

  /**
   * Return my name.
   */
  public String getName()
    {
    return name;
    }

  /**
   * Return my qualified name.
   */
  public String getQName()
    {
    if( namespace == null )
      return name;
    else if( namespace.endsWith( ":" ) )
      return namespace + name;
    else
      return namespace + ":" + name;
    }

  /**
   * Return my namespace prefix, or null if there is none.
   */
  public String getPrefix()
    {
    return prefix;
    }

  /**
   * Return my namespace, or null if there is none.
   */
  public String getNamespace()
    {
    return namespace;
    }

  /**
   * Set my name.
   * @param name The new name value.
   * @return Myself, to allow cascading.
   */
  public Element setName( String name )
    {
    this.name = name;
    this.namespace = getNamespace( "" );
    return this;
    }

  /**
   * Set my name.
   * @param prefix The namespace prefix.
   * @param name The new name value.
   * @return Myself, to allow cascading.
   * @throws NamespaceException If a namespace prefix could not be resolved.
   */
  public Element setName( String prefix, String name )
    throws NamespaceException
    {
    this.prefix = prefix;
    this.name = name;

    if( prefix == null )
      {
      namespace = getNamespace( "" );
      }
    else
      {
      namespace = getNamespace( prefix );

      if( namespace == null )
        throw new NamespaceException( "could not find namespace with prefix " + prefix );
      }

    return this;
    }

  /**
   * Return the fully qualified version of the specified name, with any prefix
   * expanded to the full namespace value.
   * @param name The original name.
   */
  public String getQName( String string )
    throws NamespaceException
    {
    int index = string.indexOf( ':' );
    String ns = null;

    if( index != -1 )
      {
      String prefix = string.substring( 0, index );
      string = string.substring( index + 1 );
      ns = getNamespace( prefix );

      if( ns == null )
        throw new NamespaceException( "could not resolve " + prefix + ":" + string );
      }
    else
      {
      ns = getNamespace( "" );
      }

    if( ns == null )
      return string;
    else if( ns.endsWith( ":" ) )
      return ns + string;
    else
      return ns + ":" + string;
    }

  // ********** MATCHING ****************************************************

  /**
   * Return true if my namespace and name match the specified values.
   * @param namespace The namespace.
   * @param name The name.
   */
  public boolean hasName( String namespace, String name )
    {
    if( !this.name.equals( name ) )
      return false;
    else if( namespace == null )
      return true;
    else if( this.namespace == null )
      return false;
    else
      return this.namespace.equals( namespace );
    }

  /**
   * Return true if my name match the specified value.
   * @param name The name.
   */
  public boolean hasName( String name )
    {
    return this.name.equals( name );
    }

  // ********** TEXT ********************************************************

  /**
   * Return true if I have at least one Text node.
   */
  public boolean hasText()
    {
    return (getText() != null);
    }

  /**
   * Return my first Text node, or null if I have none.
   */
  public Text getText()
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Text )
        return (Text) node;

    return null;
    }

  /**
   * Return my first Text node whose element has the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public Text getText( String name )
    {
    Element element = getElement( name );
    return (element == null ? null : element.getText());
    }

  /**
   * Return my first Text node whose element has the specified namespace and name,
   * or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Text getText( String namespace, String name )
    {
    Element element = getElement( namespace, name );
    return (element == null ? null : element.getText());
    }

  /**
   * Return my first Text node whose element has the specified xpath,
   * or null if I have none.
   * @param xpath The XPath.
   */
  public Text getText( IXPath xpath )
    {
    Element element = getElement( xpath );
    return (element == null ? null : element.getText());
    }

  /**
   * Return the string of my first Text node, or null if I have none.
   */
  public String getTextString()
    {
    Text text = getText();
    return (text == null ? null : text.string);
    }

  /**
   * Return the string of my first Text node whose element has the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public String getTextString( String name )
    {
    Text text = getText( name );
    return (text == null ? null : text.string);
    }

  /**
   * Return the string of my first Text node whose element has the specified
   * namespace and name, or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public String getTextString( String namespace, String name )
    {
    Text text = getText( namespace, name );
    return (text == null ? null : text.string);
    }

  /**
   * Return the string of my first Text node whose element has the specified xpath,
   * or null if I have none.
   * @param name The XPath.
   */
  public String getTextString( IXPath xpath )
    {
    Text text = getText( xpath );
    return (text == null ? null : text.string);
    }

  /**
   * Return the trimmed string of my first Text node, or null if I have none.
   */
  public String getTrimTextString()
    {
    Text text = getText();
    return (text == null ? null : text.string.trim());
    }

  /**
   * Return the trimmed string of my first Text node whose element has the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public String getTrimTextString( String name )
    {
    Text text = getText( name );
    return (text == null ? null : text.string.trim());
    }

  /**
   * Return the trimmed string of my first Text node whose element has the specified
   * namespace and name, or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public String getTrimTextString( String namespace, String name )
    {
    Text text = getText( namespace, name );
    return (text == null ? null : text.string.trim());
    }

  /**
   * Return the trimmed string of my first Text node whose element has the specified xpath,
   * or null if I have none.
   * @param name The XPath.
   */
  public String getTrimTextString( IXPath xpath )
    {
    Text text = getText( xpath );
    return (text == null ? null : text.string.trim());
    }

  /**
   * If text is not null, set my first Text node to have the specified text,
   * adding one if necessary. If text is null, remove my first Text node if I have one.
   * @param text The text.
   * @return Myself, to allow cascading.
   */
  public Element setText( String text )
    {
    return setText( text == null ? null : new Text( text ) );
    }

  /**
   * Apply setText( text ) to the named child, creating the child if necessary.
   * @param name The child element name.
   * @param text The text.
   */
  public void setText( String name, String text )
    {
    setElement( name ).setText( text );
    }

  /**
   * If text is not null, set my first Text node to have the specified text,
   * adding one if necessary. If text is null, remove my first Text node if I have one.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param text The text.
   */
  public void setText( String prefix, String name, String text )
    {
    setElement( prefix, name ).setText( text );
    }

  /**
   * If text is not null, set my first Text node to have the specified text,
   * adding one if necessary. If text is null, remove my first Text node if I have one.
   * @param text The Text.
   * @return Myself, to allow cascading.
   */
  public Element setText( Text text )
    {
    Text oldText = getText();

    if( text == null )
      {
      if( oldText != null )
        oldText.remove();
      }
    else
      {
      if( oldText == null )
        addChild( text );
      else
        oldText.replaceWith( text );
      }

    return this;
    }

  /**
   * Add a Text node with the specified text.
   * @param text The text.
   * @return Myself, to allow cascading.
   */
  public Element addText( String text )
    {
    return addText( new Text( text ) );
    }

  /**
   * Add a Text node with the specified text.
   * @param text The Text.
   * @return Myself, to allow cascading.
   */
  public Element addText( Text text )
    {
    addChild( text );
    return this;
    }

  /**
   * Insert a Text node with the specified text.
   * @param text The text.
   * @return Myself, to allow cascading.
   */
  public Element insertText( String text )
    {
    return insertText( new Text( text ) );
    }

  /**
   * Insert a Text node with the specified text.
   * @param text The Text.
   * @return Myself, to allow cascading.
   */
  public Element insertText( Text text )
    {
    insertChild( text );
    return this;
    }

  // ********** STRINGS *****************************************************

  /**
   * Return the trimmed string of my first Text node, or null if I have none.
   */
  public String getString()
    {
    Text text = getText();
    return (text == null ? null : text.string.trim());
    }

  /**
   * Return the trimmed string of my first Text node whose element has the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public String getString( String name )
    {
    Text text = getText( name );
    return (text == null ? null : text.string.trim());
    }

  /**
   * Return the trimmed string of my first Text node whose element has the specified
   * namespace and name, or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public String getString( String namespace, String name )
    {
    Text text = getText( namespace, name );
    return (text == null ? null : text.string.trim());
    }

  /**
   * If text is not null, set my first Text node to have the specified text,
   * adding one if necessary. If text is null, remove my first Text node if I have one.
   * @param value The value.
   */
   public void setString( String value )
     {
     setText( value );
     }

  // ********** BOOLEANS ****************************************************

  /**
   * Return the element text converted to a boolean.
   * @throws NumberFormatException If an error occurs.
   */
  public boolean getBoolean()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing boolean value" );
    else if( text.equalsIgnoreCase( "true" ) || text.equals( "1" ) || text.equals( "yes" ) )
      return true;
    else if( text.equalsIgnoreCase( "false" ) || text.equals( "0" ) || text.equals( "no" ) )
      return false;
    else
      throw new NumberFormatException( "invalid value for the boolean argument. expected \"true\" or \"false\", but received - " + text );
    }

  /**
   * Return the element text of the named child converted to a boolean.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public boolean getBoolean( String name )
    throws NumberFormatException
    {
    return needElement( name ).getBoolean();
    }

  /**
   * Return the element text of the named child converted to a boolean.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public boolean getBoolean( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getBoolean();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setBoolean( boolean value )
    {
    setText( (value ? "true" : "false") );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setBoolean( String name, boolean value )
    {
    setElement( name ).setBoolean( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setBoolean( String prefix, String name, boolean value )
    {
    setElement( prefix, name ).setBoolean( value );
    }

  // ********** BYTES *******************************************************

  /**
   * Return the element text converted to a byte.
   * @throws NumberFormatException If an error occurs.
   */
  public byte getByte()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing byte value" );

    return Byte.parseByte( text );
    }

  /**
   * Return the element text of the named child converted to a byte.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public byte getByte( String name )
    throws NumberFormatException
    {
    return needElement( name ).getByte();
    }

  /**
   * Return the element text of the named child converted to a byte.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public byte getByte( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getByte();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setByte( byte value )
    {
    setText( Byte.toString( value ) );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setByte( String name, byte value )
    {
    setElement( name ).setByte( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setByte( String prefix, String name, byte value )
    {
    setElement( prefix, name ).setByte( value );
    }

  // ********** CHARS *******************************************************

  /**
   * Return the element text converted to a char.
   * @throws NumberFormatException If an error occurs.
   */
  public char getChar()
    throws NumberFormatException
    {
    return (char) getShort();
    }

  /**
   * Return the element text of the named child converted to a char.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public char getChar( String name )
    throws NumberFormatException
    {
    return (char) getShort( name );
    }

  /**
   * Return the element text of the named child converted to a char.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public char getChar( String namespace, String name )
    throws NumberFormatException
    {
    return (char) getShort( namespace, name );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setChar( char value )
    {
    setShort( (short) value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setChar( String name, char value )
    {
    setShort( name, (short) value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setChar( String prefix, String name, char value )
    {
    setShort( prefix, name, (short) value );
    }

  // ********** SHORTS ******************************************************

  /**
   * Return the element text converted to a short.
   * @throws NumberFormatException If an error occurs.
   */
  public short getShort()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing short value" );

    return Short.parseShort( text );
    }

  /**
   * Return the element text of the named child converted to a short.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public short getShort( String name )
    throws NumberFormatException
    {
    return needElement( name ).getShort();
    }

  /**
   * Return the element text of the named child converted to a short.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public short getShort( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getShort();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setShort( short value )
    {
    setText( Short.toString( value ) );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setShort( String name, short value )
    {
    setElement( name ).setShort( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setShort( String prefix, String name, short value )
    {
    setElement( prefix, name ).setShort( value );
    }

  // ********** INTS ********************************************************

  /**
   * Return the element text converted to an int.
   * @throws NumberFormatException If an error occurs.
   */
  public int getInt()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing int value" );

    return Integer.parseInt( text );
    }

  /**
   * Return the element text of the named child converted to an int.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public int getInt( String name )
    throws NumberFormatException
    {
    return needElement( name ).getInt();
    }

  /**
   * Return the element text of the named child converted to an int.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public int getInt( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getInt();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setInt( int value )
    {
    setText( Integer.toString( value ) );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setInt( String name, int value )
    {
    setElement( name ).setInt( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setInt( String prefix, String name, int value )
    {
    setElement( prefix, name ).setInt( value );
    }

  // ********** LONGS *******************************************************

  /**
   * Return the element text converted to a long.
   * @throws NumberFormatException If an error occurs.
   */
  public long getLong()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing long value" );

    return Long.parseLong( text );
    }

  /**
   * Return the element text of the named child converted to a long.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public long getLong( String name )
    throws NumberFormatException
    {
    return needElement( name ).getLong();
    }

  /**
   * Return the element text of the named child converted to a long.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public long getLong( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getLong();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setLong( long value )
    {
    setText( Long.toString( value ) );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setLong( String name, long value )
    {
    setElement( name ).setLong( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setLong( String prefix, String name, long value )
    {
    setElement( prefix, name ).setLong( value );
    }

  // ********** FLOATS ******************************************************

  /**
   * Return the element text converted to a float.
   * @throws NumberFormatException If an error occurs.
   */
  public float getFloat()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing float value" );

    if( NOT_A_NUMBER.equals( text ) )
      return Float.NaN;
    else if( POSITIVE_INFINITY.equals( text ) )
      return Float.POSITIVE_INFINITY;
    else if( NEGATIVE_INFINITY.equals( text ) )
      return Float.NEGATIVE_INFINITY;
    else
      return new Float( text ).floatValue(); // JDK 1.1 forgot parseFloat()
    }

  /**
   * Return the element text of the named child converted to a float.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public float getFloat( String name )
    throws NumberFormatException
    {
    return needElement( name ).getFloat();
    }

  /**
   * Return the element text of the named child converted to a float.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public float getFloat( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getFloat();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setFloat( float value )
    {
    if( value == Float.POSITIVE_INFINITY )
      setText( POSITIVE_INFINITY );
    else if( value == Float.NEGATIVE_INFINITY )
      setText( NEGATIVE_INFINITY );
    else
      setText( Float.toString( value ) );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setFloat( String name, float value )
    {
    setElement( name ).setFloat( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setFloat( String prefix, String name, float value )
    {
    setElement( prefix, name ).setFloat( value );
    }

  // ********** DOUBLES *****************************************************

  /**
   * Return the element text converted to a double.
   * @throws NumberFormatException If an error occurs.
   */
  public double getDouble()
    throws NumberFormatException
    {
    String text = getString();

    if( text == null )
      throw new NumberFormatException( "missing double value" );

    if( NOT_A_NUMBER.equals( text ) )
      return Double.NaN;
    else if( POSITIVE_INFINITY.equals( text ) )
      return Double.POSITIVE_INFINITY;
    else if( NEGATIVE_INFINITY.equals( text ) )
      return Double.NEGATIVE_INFINITY;
    else
      return new Double( text ).doubleValue(); // JDK 1.1 forgot parseDouble()
    }

  /**
   * Return the element text of the named child converted to a double.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public double getDouble( String name )
    throws NumberFormatException
    {
    return needElement( name ).getDouble();
    }

  /**
   * Return the element text of the named child converted to a double.
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException If an error occurs.
   */
  public double getDouble( String namespace, String name )
    throws NumberFormatException
    {
    return needElement( namespace, name ).getDouble();
    }

  /**
   * Set the element text to the specified value as a string.
   * @param value
   */
  public void setDouble( double value )
    {
    if( value == Double.POSITIVE_INFINITY )
      setText( POSITIVE_INFINITY );
    else if( value == Double.NEGATIVE_INFINITY )
      setText( NEGATIVE_INFINITY );
    else
      setText( Double.toString( value ) );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param name The child element name.
   * @param value
   */
  public void setDouble( String name, double value )
    {
    setElement( name ).setDouble( value );
    }

  /**
   * Set the element text to the specified value as a string.
   * @param prefix The namespace prefix.
   * @param name The child element name.
   * @param value
   */
  public void setDouble( String prefix, String name, double value )
    {
    setElement( prefix, name ).setDouble( value );
    }

  // ********** ATTRIBUTES **************************************************

  /**
   * Return true if I have one or more Attributes.
   */
  public boolean hasAttributes()
    {
    return !attributes.isEmpty();
    }

  /**
   * Return an enumeration over my Attributes.
   */
  public Attributes getAttributeObjects()
    {
    return new Attributes( attributes );
    }

  /**
   * Return my first Attribute node with the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public Attribute getAttributeObject( String name )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.name.equals( name ) )
        return attribute;

    return null;
    }

  /**
   * Return my first Attribute node with the specified namespace and name,
   * or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Attribute getAttributeObject( String namespace, String name )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.hasName( namespace, name ) )
        return attribute;

    return null;
    }

  /**
   * Return my first Attribute node with the specified xpath,
   * or null if I have none.
   * @param xpath The XPath.
   */
  public Attribute getAttributeObject( IXPath xpath )
    {
    return xpath.getAttribute( this );
    }

  /**
   * Return my Attribute nodes with the specified xpath.
   * @param xpath The XPath.
   */
  public Attributes getAttributes( IXPath xpath )
    {
    return xpath.getAttributes( this );
    }

  /**
   * Return the value of my first Attribute node with the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public String getAttributeValue( String name )
    {
    Attribute attribute = getAttributeObject( name );
    return (attribute == null ? null : attribute.value);
    }

  /**
   * Return the value of my first Attribute node with the specified namespace and name,
   * or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public String getAttributeValue( String namespace, String name )
    {
    Attribute attribute = getAttributeObject( namespace, name );
    return (attribute == null ? null : attribute.value);
    }

  /**
   * Return the value of my first Attribute node with the specified xpath,
   * or null if I have none.
   * @param xpath The XPath.
   */
  public String getAttributeValue( IXPath xpath )
    {
    Attribute attribute = getAttributeObject( xpath );
    return (attribute == null ? null : attribute.value);
    }

  /**
   * Add an attribute with the specified name and value, overwriting
   * any previous value if present.
   * @param name The name.
   * @param value The value.
   */
  public void setAttribute( String name, String value )
    {
    setAttribute( new Attribute( name, value ) );
    }

  /**
   * Add an attribute with the specified namespace prefix, name and value, overwriting
   * any previous value if present.
   * @param prefix The namespace prefix.
   * @param name The name.
   * @param value The value.
   * @return Myself, to allow cascading.
   */
  public Element setAttribute( String prefix, String name, String value )
    {
    setAttribute( new Attribute( prefix, name, value ) );
    return this;
    }

  /**
   * Add the specified attribute, overwriting any previous value if present.
   * @param attribute The attribute.
   * @throws NamespaceException If the attribute name could not be resolved.
   * @return Myself, to allow cascading.
   */
  public Element setAttribute( Attribute attribute )
    throws NamespaceException
    {
    attribute.resolve( this );

    for( Attribute node = (Attribute) attributes.first; node != null; node = (Attribute) node.next )
      if( node.hasName( attribute.namespace, attribute.name ) )
        {
        attributes.replace( node, attribute );
        return this;
        }

    attributes.append( attribute );
    return this;
    }

  /**
   * Remove the attribute with the specified name if one exists.
   * @param name The name.
   */
  public void removeAttribute( String name )
    {
    Attribute attribute = getAttributeObject( name );

    if( attribute != null )
      attribute.remove();
    }

  /**
   * Remove and return the attribute with the specified namespace and name, or return
   * null if none was found.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Attribute removeAttribute( String namespace, String name )
    {
    Attribute attribute = getAttributeObject( namespace, name );

    if( attribute != null )
      attribute.remove();

    return attribute;
    }

  /**
   * Remove and return the attribute with the specified xpath, or return
   * null if none was found.
   * @param xpath The XPath.
   */
  public Attribute removeAttribute( IXPath xpath )
    {
    Attribute attribute = getAttributeObject( xpath );

    if( attribute != null )
      attribute.remove();

    return attribute;
    }

  /**
   * Remove and return the attributes with the specified xpath.
   * @param xpath The XPath.
   */
  public Attributes removeAttributes( IXPath xpath )
    {
    Attributes attributes = getAttributes( xpath );
    attributes.remove();
    attributes.reset();
    return attributes;
    }

  // ********** SIBLINGS ****************************************************

  /**
   * Set my next sibling to a new Element with the specified name.
   * @param name The name.
   * @return The new Element.
   */
  public Element setNextSibling( String name )
    {
    Element sibling = new Element();
    sibling.setParent( parent );
    sibling.setName( name );
    setNextSiblingChild( sibling );
    return sibling;
    }

  /**
   * Set my next sibling to a new Element with the specified namespace prefix and name.
   * @param prefix The namespace prefix.
   * @param name The name.
   * @return The new Element.
   */
  public Element setNextSibling( String prefix, String name )
    {
    Element sibling = new Element();
    sibling.setParent( parent );
    sibling.setName( prefix, name );
    setNextSiblingChild( sibling );
    return sibling;
    }

  /**
   * Set my next sibling to the specified element.
   * @param element The element.
   * @return The element.
   */
  public Element setNextSibling( Element element )
    {
    element.setParent( parent );
    setNextSiblingChild( element );
    return element;
    }

  /**
   * Set my previous sibling to a new Element with the specified name.
   * @param name The name.
   * @return The new Element.
   */
  public Element setPreviousSibling( String name )
    {
    Element sibling = new Element();
    sibling.setParent( parent );
    sibling.setName( name );
    setPreviousSiblingChild( sibling );
    return sibling;
    }

  /**
   * Set my previous sibling to a new Element with the specified name.
   * @param prefix The namespace prefix.
   * @param name The name.
   * @return The new Element.
   */
  public Element setPreviousSibling( String prefix, String name )
    {
    Element sibling = new Element();
    sibling.setParent( parent );
    sibling.setName( prefix, name );
    setPreviousSiblingChild( sibling );
    return sibling;
    }

  /**
   * Set my previous sibling to the specified element.
   * @param element The element.
   * @return The element.
   */
  public Element setPreviousSibling( Element element )
    {
    element.setParent( parent );
    setPreviousSiblingChild( element );
    return element;
    }

  // ********** NAMESPACES **************************************************

  /**
   * Return my namespaces as a Dictionary of prefix -> value.
   */
  public Dictionary getNamespaces()
    {
    Hashtable namespaces = new Hashtable();

    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isNamespace )
        namespaces.put( attribute.name, attribute.value );

    return namespaces;
    }

  /**
   * Return the value of the namespace with the specified prefix, or null if
   * there is none. This operation searches up the element hierarchy starting
   * at this element.
   * @param prefix The prefix.
   */
  public String getNamespace( String prefix )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isNamespace && attribute.name.equals( prefix ) )
        {
        String value = attribute.value;
        return (value.length() > 0 ? value : null); // support xmlns=""
        }

    return (parent == null ? null : parent.getNamespace( prefix ));
    }

  /**
   * Return a prefix that maps to a particular
   * namespace value, searching from the current element up through its parents.
   * Return null if none is found.
   * @param namespace The namespace to match.
   */
  public String getNamespacePrefix( String namespace )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isNamespace && attribute.value.equals( namespace ) )
        return attribute.name;

    if( parent != null )
      return parent.getNamespacePrefix( namespace );

    return null;
    }

  /**
   * Return an array of all the prefixes that map to a particular
   * namespace value, searching from the current element up through its parents.
   * @param namespace The namespace to match.
   */
  public String[] getNamespacePrefixes( String namespace )
    {
    Vector prefixes = new Vector();
    Vector matches = new Vector();
    addNamespacePrefixes( namespace, prefixes, matches );
    String[] results = new String[ matches.size() ];
    matches.copyInto( results );
    return results;
    }

  /**
   * Add all the prefixes that map to a particular namespace value in my context to
   * the specified vector.
   * @param namespace The namespace to match
   * @param prefixes All the prefixes so far.
   * @param matches All the prefixes that matched.
   */
  protected void addNamespacePrefixes( String namespace, Vector prefixes, Vector matches )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isNamespace )
        {
        if( attribute.value.equals( namespace ) && !prefixes.contains( attribute.name ) )
          matches.addElement( attribute.name );

        prefixes.addElement( attribute.name );
        }

    if( parent != null )
      parent.addNamespacePrefixes( namespace, prefixes, matches );
    }

  /**
   * Add a new namespace with the specified prefix and value, overwriting
   * any previous value if present.
   * To add a default namespace, set the prefix to an empty string.
   * This operation automatically sets a corresponding xmlns attribute.
   * @param prefix The prefix
   * @param value The value
   */
  public void setNamespace( String prefix, String value )
    throws NamespaceException
    {
    setAttribute( new Attribute( prefix, value, true ) );
    }

  /**
   * Remove the namespace with the specified prefix.
   * To remove a default namespace, set the prefix to an empty string.
   * @param prefix The namespace to remove.
   */
  public boolean removeNamespace( String prefix )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isNamespace && attribute.name.equals( prefix ) )
        {
        attribute.remove();
        return true ;
        }

    return false;
    }

  // ********** IDS *********************************************************

  /**
   * Return the element whose "id" attribute is equal to the specified value,
   * or null if none exists.
   * @param value
   */
  public synchronized Element getElementWithId( String value )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isId && attribute.value.equals( value ) )
        return this;

    return super.getElementWithId( value );
    }

  /**
   *
   */
  public Hashtable getIds()
    {
    Hashtable idToElement = new Hashtable();
    addIds( idToElement );
    return idToElement;
    }

  /**
   * @param idElement
   */
  synchronized void addIds( Hashtable idToElement )
    {
    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      if( attribute.isId )
        idToElement.put( attribute.value, this );

    super.addIds( idToElement );
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself to the specified writer starting at the specified indent level.
   * If the indent level is -1, no indentation will occur, otherwise the indent
   * level increases by two at each child node.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( NodeWriter writer )
    throws IOException
    {
    writer.writeIndent();
    writer.write( '<' );

    if( prefix != null && !prefix.equals( "" ) )
      {
      writer.write( prefix );
      writer.write( ':' );
      }

    writer.write( name );

    for( Attribute attribute = (Attribute) attributes.first; attribute != null; attribute = (Attribute) attribute.next )
      {
      writer.write( ' ' );
      writer.write( attribute );
      }

    if( children.isEmpty() )
      {
      if( writer.getExpandEmptyElements() )
        {
        writer.write( "></" );

        if( prefix != null && !prefix.equals( "" ) )
          {
          writer.write( prefix );
          writer.write( ':' );
          }

        writer.write( name );
        writer.write( '>' );
        }
      else
        {
        writer.write( "/>" );
        }
      }
    else
      {
      writer.write( '>' );
      writeChildren( writer );
      writer.write( "</" );

      if( prefix != null && !prefix.equals( "" ) )
        {
        writer.write( prefix );
        writer.write( ':' );
        }

      writer.write( name );
      writer.write( '>' );
      }
    }

  /**
   * Write my children to the specified writer starting at the specified indent level.
   * If the indent level is -1, no indentation will occur, otherwise the indent
   * level increases by two at each child node.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  private void writeChildren( NodeWriter writer )
    throws IOException
    {
    if( children.first == children.last && children.first instanceof Text )
      {
      int originalIndent = writer.getIndent();
      writer.setIndent( -1 );
      writer.write( children.first );
      writer.setIndent( originalIndent );
      }
    else if( writer.getIndent() == -1 )
      {
      for( Node child = children.first; child != null; child = child.next )
        writer.write( child );
      }
    else
      {
      writer.increaseIndent();

      for( Node node = children.first; node != null; node = node.next )
        {
        writer.writeEOL();
        writer.write( node );
        }

      writer.writeEOL();
      writer.decreaseIndent();
      writer.writeIndent();
      }
    }

  // ********** DOM *********************************************************

  /**
   * Return ELEMENT_NODE.
   */
  public short getNodeType()
    {
    return ELEMENT_NODE;
    }

  /**
   * Return my qualified name.
   */
  public String getNodeName()
    {
    return getQName();
    }

  /**
   * Set my namespace prefix to the specified value.
   * @param prefix The new namespace prefix.
   */
  public void setPrefix( String prefix )
    {
    this.prefix = prefix;
    }

  /**
   *
   */
  public String getTagName()
    {
    return (prefix == null ? name : prefix + ':' + name);
    }

  /**
   *
   */
  public String getLocalName()
    {
    return name;
    }

  /**
   *
   */
  public String getNamespaceURI()
    {
    return namespace;
    }

  /**
   *
   */
  public org.w3c.dom.NamedNodeMap getAttributes()
    {
    return attributes;
    }

  /**
   * @param name
   */
  public boolean hasAttribute( String name )
    {
    return (getAttributeObject( name ) != null);
    }

  /**
   * Return my first Attribute node with the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public String getAttribute( String name )
    {
    return getAttributeValue( name );
    }

  /**
   * @param name
   */
  public org.w3c.dom.Attr getAttributeNode( String name )
    {
    return getAttributeObject( name );
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  public org.w3c.dom.Attr getAttributeNodeNS( String namespaceURI, String localName )
    {
    return getAttributeObject( namespaceURI, localName );
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  public boolean hasAttributeNS( String namespaceURI, String localName )
    {
    return (getAttributeObject( namespaceURI, localName ) != null);
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  public String getAttributeNS( String namespaceURI, String localName )
    {
    return getAttributeValue( namespaceURI, localName );
    }

  /**
   * @param name
   */
  public org.w3c.dom.NodeList getElementsByTagName( String name )
    {
    NodeList nodes = new NodeList();

    for( Elements elements = getElements(); elements.hasMoreElements(); )
      elements.next().addElementsByTagName( name, nodes );

    return nodes;
    }

  /**
   * @param name
   * @param nodes
   */
  void addElementsByTagName( String name, NodeList nodes )
    {
    if( "*".equals( name ) || this.name.equals( name ) )
      nodes.append( new Selection( this ) );

    for( Elements elements = getElements(); elements.hasMoreElements(); )
      elements.next().addElementsByTagName( name, nodes );
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  public org.w3c.dom.NodeList getElementsByTagNameNS( String namespaceURI, String localName )
    {
    NodeList nodes = new NodeList();

    if( "*".equals( namespaceURI ) )
      for( Elements elements = getElements(); elements.hasMoreElements(); )
        elements.next().addElementsByTagName( localName, nodes );
    else
      for( Elements elements = getElements(); elements.hasMoreElements(); )
        elements.next().addElementsByTagNameNS( namespaceURI, localName, nodes );

    return nodes;
    }

  /**
   * @param namespaceURI
   * @param localName
   * @param nodes
   */
  void addElementsByTagNameNS( String namespaceURI, String localName, NodeList nodes )
    {
    if( "*".equals( localName ) || name.equals( localName ) )
      {
      if( namespaceURI == null && namespace == null )
        nodes.append( new Selection( this ) );
      else if( namespaceURI != null && namespaceURI.equals( namespace ) )
        nodes.append( new Selection( this ) );
      }

    for( Elements elements = getElements(); elements.hasMoreElements(); )
      elements.next().addElementsByTagNameNS( namespaceURI, localName, nodes );
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  public void removeAttributeNS( String namespaceURI, String localName )
    {
    removeAttribute( namespaceURI, localName );
    }

  /**
   * @param oldAttr
   */
  public org.w3c.dom.Attr removeAttributeNode( org.w3c.dom.Attr oldAttr )
    {
    // should check that attribute is in my list
    ((Attribute) oldAttr).remove();
    return oldAttr;
    }

  /**
   * @param newAttr
   */
  public org.w3c.dom.Attr setAttributeNodeNS( org.w3c.dom.Attr newAttr )
    {
    Attribute attribute = (Attribute) newAttr;
    attribute.resolve( this );

    for( Attribute node = (Attribute) attributes.first; node != null; node = (Attribute) node.next )
      if( node.hasName( attribute.namespace, attribute.name ) )
        {
        attributes.replace( node, attribute );
        return node;
        }

    attributes.append( attribute );
    return null;
    }

  /**
   * @param newAttr
   */
  public org.w3c.dom.Attr setAttributeNode( org.w3c.dom.Attr newAttr )
    {
    return setAttributeNodeNS( newAttr );
    }

  /**
   * @param namespaceURI
   * @param qualifiedName
   * @param value
   */
  public void setAttributeNS( String namespaceURI, String qualifiedName, String value )
    {
    String[] parts = getParts( qualifiedName );
    Attribute attribute = getAttributeObject( namespaceURI, parts[ 1 ] );

    if( attribute == null )
      {
      setAttribute( parts[ 0 ], parts[ 1 ], value );
      }
    else
      {
      attribute.setPrefix( parts[ 0 ] ); // seems odd
      attribute.setValue( value );
      }
    }

  /**
   * @param qualifiedName
   */
  public static String[] getParts( String qualifiedName )
    {
    int colon = qualifiedName.indexOf( ':' );

    if( colon != -1 )
      return new String[]{ qualifiedName.substring( 0, colon ), qualifiedName.substring( colon + 1 ) };
    else
      return new String[]{ null, qualifiedName };
    }

  // ********** STEP INDENT *************************************************

  /**
   * Set the printing indent level to the specified value.
   * @deprecated Use NodeWriter.setDefaultStep() instead.
   * @param indent The new indent value.
   */
  public static void setStepIndent( int indent )
    {
    NodeWriter.setDefaultStep( indent );
    }

  /**
   * Return the indentation value.
   * @deprecated Use NodeWriter.getDefaultStep() instead.
   */
  public static int getStepIndent()
    {
    return NodeWriter.getDefaultStep();
    }
  }