// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import electric.util.*;

/**
 * <tt>NodeList</tt> is a doubly-linked list of Nodes, and includes methods for
 * adding, replacing and removing nodes.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public final class NodeList implements Serializable, org.w3c.dom.NodeList, org.w3c.dom.NamedNodeMap
  {
  public Node first; // the first node in this list, or null if the list is empty
  public Node last; // the last node in this list, or null if the list is empty

  // ********** TESTING *****************************************************

  /**
   * Return true if this list is empty.
   */
  public boolean isEmpty()
    {
    return (first == null);
    }

  // ********** ACCESS ******************************************************

  /**
   * @param name
   */
  private Node getNode( String name )
    {
    for( Node node = first; node != null; node = node.next )
      if( ((INamed) node).getName().equals( name ) )
        return node;

    return null;
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  private Node getNode( String namespaceURI, String localName )
    {
    for( Node node = first; node != null; node = node.next )
      if( localName.equals( ((IQNamed) node).getName() ) && (namespaceURI == null || namespaceURI.equals( ((IQNamed) node).getNamespace() )) )
        return node;

    return null;
    }

  // ********** MANIPULATION ************************************************

  /**
   * Append the specified node to the end of this list.
   * If the new node was already in a document, remove it first.
   * @param node The node to append.
   */
  public void append( Node node )
    {
    node.remove();
    node.list = this;
    node.prev = last;

    if( first == null )
      first = node;
    else
      last.next = node;

    last = node;
    }

  /**
   * Insert the specified node at the beginning of this list.
   * If the new node was already in a document, remove it first.
   * @param node The node to insert.
   */
  public void insert( Node node )
    {
    node.remove();
    node.list = this;
    node.next = first;

    if( last == null )
      last = node;
    else
      first.prev = node;

    first = node;
    }

  /**
   * Remove the specified node from this list.
   * @param node The node to remove.
   */
  public void remove( Node node )
    {
    if( node.prev == null )
      first = node.next;
    else
      node.prev.next = node.next;

    if( node.next == null )
      last = node.prev;
    else
      node.next.prev = node.prev;

    node.prev = null;
    node.next = null;
    }

  /**
   * Replace the old node with the new node in this list.
   * If the new node was already in a document, remove it first.
   * @param oldNode The node to replace.
   * @param newNode The new node.
   */
  public void replace( Node oldNode, Node newNode )
    {
    newNode.remove();
    newNode.list = this;

    if( oldNode.prev == null )
      first = newNode;
    else
      oldNode.prev.next = newNode;

    if( oldNode.next == null )
      last = newNode;
    else
      oldNode.next.prev = newNode;

    newNode.prev = oldNode.prev;
    newNode.next = oldNode.next;
    }

  /**
   * Append the new node as a sibling of the existing node.
   * If the new node was already in a document, remove it first.
   * @param node The existing node.
   * @param newNode The new node.
   */
  public void addSiblingNode( Node node, Node newNode )
    {
    newNode.remove();
    newNode.list = this;

    newNode.next = node.next;
    newNode.prev = node;

    if( node.next == null )
      last = newNode;
    else
      node.next.prev = newNode;

    node.next = newNode;
    }

  /**
   * Insert the new node as a sibling of the existing node.
   * If the new node was already in a document, remove it first.
   * @param node The existing node.
   * @param newNode The new node.
   */
  public void insertSiblingNode( Node node, Node newNode )
    {
    newNode.remove();
    newNode.list = this;

    newNode.prev = node.prev;
    newNode.next = node;

    if( node.prev == null )
      first = newNode;
    else
      node.prev.next = newNode;

    node.prev = newNode;
    }

  // ********** SIZE ********************************************************

  /**
   * Return the number of nodes in this list.
   */
  public int size()
    {
    int count = 0;

    for( Node node = first; node != null; node = node.next )
      ++count;

    return count;
    }

  // ********** CLEAR *******************************************************

  /**
   * Remove all the nodes.
   */
  public void clear()
    {
    first = null;
    last = null;
    }

  // ********** DOM *********************************************************

  /**
   * Return the number of nodes in the list.
   * The range of valid child node indices is 0 to length - 1 inclusive.
   */
  public int getLength()
    {
    return size();
    }

  /**
   * Return the indexth item in the collection. If index is greater than or
   * equal to the number of nodes in the list, this returns null.
   * @param index The index of the node to return.
   */
  public org.w3c.dom.Node item( int index )
    {
    if( index < 0 )
      return null;

    Node node = first;

    for( int i = 0; i < index; i++ )
      if( node == null )
        return null;
      else
        node = node.next;

    return node.getNode();
    }

  /**
   * Return the node with the specified name, or null if not found
   * @param name The name of the node to return.
   */
  public org.w3c.dom.Node getNamedItem( String name )
    {
    return getNode( name );
    }

  /**
   * Adds a node using its nodeName attribute. If a node with that name is
   * already present in this map, it is replaced by the new one. As the nodeName
   * attribute is used to derive the name which the node must be stored under,
   * multiple nodes of certain types (those that have a "special" string value)
   * cannot be stored as the names would clash. This is seen as preferable to
   * allowing nodes to be aliased.
   * @param node The node to be added.
   */
  public org.w3c.dom.Node setNamedItem( org.w3c.dom.Node node )
    {
    Node oldNode = getNode( node.getNodeName() );

    if( oldNode == null )
      {
      append( (Node) node );
      return null;
      }
    else
      {
      replace( oldNode, (Node) node );
      return oldNode;
      }
    }

  /**
   * Removes a node specified by name. When this map contains the attributes
   * attached to an element, if the removed attribute is known to have a
   * default value, an attribute immediately appears containing the default
   * value as well as the corresponding namespace URI, local name, and prefix
   * when applicable.
   * @param name The name of the node to remove.
   */
  public org.w3c.dom.Node removeNamedItem( String name )
    {
    Node node = getNode( name );

    if( node != null )
      node.remove();

    return node;
    }

  /**
   * Retrieves a node specified by local name and namespace URI.
   * @param namespaceURI The namespace URI.
   * @param localName The local name.
   */
  public org.w3c.dom.Node getNamedItemNS( String namespaceURI, String localName )
    {
    return getNode( namespaceURI, localName );
    }

  /**
   * Adds a node using its namespaceURI and localName. If a node with that
   * namespace URI and that local name is already present in this map, it is
   * replaced by the new one.
   * @param node The node to add.
   */
  public org.w3c.dom.Node setNamedItemNS( org.w3c.dom.Node node )
    {
    Node oldNode = getNode( node.getNamespaceURI(), node.getLocalName() );

    if( oldNode == null )
      {
      append( (Node) node );
      return null;
      }
    else
      {
      replace( oldNode, (Node) node );
      return oldNode;
      }
    }

  /**
   * Removes a node specified by local name and namespace URI. A removed
   * attribute may be known to have a default value when this map contains
   * the attributes attached to an element, as returned by the attributes
   * attribute of the Node interface. If so, an attribute immediately appears
   * containing the default value as well as the corresponding namespace
   * URI, local name, and prefix when applicable.
   * @param namespaceURI The namespace URI.
   * @param localName The local name.
   */
  public org.w3c.dom.Node removeNamedItemNS( String namespaceURI, String localName )
    {
    Node node = getNode( namespaceURI, localName );

    if( node != null )
      node.remove();

    return node;
    }
  }