// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import java.util.*;
import electric.util.*;
import electric.xml.xpath.*;

/**
 * <tt>XPath</tt> provides a set of methods for selecting nodes from
 * an element based on an XPath. The XPath tokens that are currently
 * supported are ".", "..", "...", "/", "*", "@*", "[n]", "name[n]", "[@key='value']",
 * "name[@key='value']", "[text='value']", "name[text='value']" and "@key". To prevent
 * special characters from being treated as XPath directives, enclose the path
 * in single quotes.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

final public class XPath implements IXPath
  {
  private static IXPathFactory factory = new electric.xml.xpath.TMEXPathFactory();
  private static boolean cacheByDefault = true;

  private IXPath xpath;

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct an XPath with the specified expression and no namespace context.
   * @param xpath The xpath expression.
   * @throws XPathException If the path is malformed.
   */
  public XPath( String xpath )
    throws XPathException
    {
    this( xpath, null, cacheByDefault );
    }

  /**
   * Construct an XPath with the specified expression and no namespace context.
   * @param xpath The xpath expression.
   * @throws XPathException If the path is malformed.
   */
  public XPath( String xpath, boolean cache )
    throws XPathException
    {
    this( xpath, null, cache );
    }

  /**
   * Construct an XPath with the specified expression and namespace context.
   * @param xpath The xpath expression.
   * @param namespaceContext The namespace context.
   * @throws XPathException If the path is malformed.
   */
  public XPath( String xpath, INamespaceContext namespaceContext )
    throws XPathException
    {
    this( xpath, namespaceContext, cacheByDefault );
    }

  /**
   * Construct an XPath with the specified expression and namespace context.
   * @param xpath The xpath expression.
   * @param namespaceContext The namespace context.
   * @throws XPathException If the path is malformed.
   */
  public XPath( String xpath, INamespaceContext namespaceContext, boolean cache )
    throws XPathException
    {
    if( cache )
      this.xpath = factory.newCachedXPath( xpath, namespaceContext );
    else
      this.xpath = factory.newXPath( xpath, namespaceContext );
    }

  // ********** STANDARD METHODS ********************************************

  /**
   * Return the xpath expression.
   */
  public String toString()
    {
    return xpath.toString();
    }

  // ********** XPATH FACTORY ***********************************************

  /**
   *
   */
  public static IXPathFactory getFactory()
    {
    return factory;
    }

  /**
   * @param factory
   */
  public static void setFactory( IXPathFactory factory )
    {
    XPath.factory = factory;
    }

  // ********** CACHING BY DEFAULT ******************************************

  /**
   * @param flag
   */
  public static void setCacheByDefault( boolean flag )
    {
    cacheByDefault = flag;
    }

  /**
   *
   */
  public static boolean getCacheByDefault()
    {
    return cacheByDefault;
    }

  // ********** ATTRIBUTES **************************************************

  /**
   * @param element
   * @throws XPathException
   */
  public Attribute getAttribute( Element element )
    throws XPathException
    {
    return xpath.getAttribute( element );
    }

  /**
   * @param element
   * @throws XPathException
   */
  public Attributes getAttributes( Element element )
    throws XPathException
    {
    return xpath.getAttributes( element );
    }

  // ********** ELEMENTS ****************************************************

  /**
   * @param parent
   * @throws XPathException
   */
  public Element getElement( Parent parent )
    throws XPathException
    {
    return xpath.getElement( parent );
    }

  /**
   * @param parent
   * @throws XPathException
   */
  public Elements getElements( Parent parent )
    throws XPathException
    {
    return xpath.getElements( parent );
    }

  // ********** NAMESPACES **************************************************

  /**
   * @param prefix
   * @param namespace
   * @throws XPathException
   */
  public void setNamespace( String prefix, String namespace )
    throws XPathException
    {
    xpath.setNamespace( prefix, namespace );
    }
  }