/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */
package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.undo.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.text.rtf.*;
import javax.swing.text.html.*;
import java.util.*;
import java.io.*;

/**
 * Implements an editor with undo support, loading/saving capabilities.
 * Undo operations supports hot keys:
 * <ul>
 * <li><i>Undo</i> - Ctrl + Z
 * <li><i>Redo</i> - Ctrl + Shift + Z
 * </ul>
 */
public class CEditor extends JTextPane implements UndoableEditListener {
  /**
   * The list which containes undo actions.
   */
  public ArrayList undoList = new ArrayList();

  /**
   * The current position in the undo list.
   */
  public int undoPosition = -1;

  /**
   * The modified flag.
   */
  private boolean modified = false;

  /**
   * Constructs this editor with default parameters.
   */
  public CEditor() {
    this.getDocument().addUndoableEditListener(this);

    Keymap map = this.addKeymap("UndoMap", this.getKeymap());

    KeyStroke undo = KeyStroke.getKeyStroke(KeyEvent.VK_Z,
      InputEvent.CTRL_MASK, false);
    KeyStroke redo = KeyStroke.getKeyStroke(KeyEvent.VK_Z,
      InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK, false);

    map.addActionForKeyStroke(undo, new UndoAction());
    map.addActionForKeyStroke(redo, new RedoAction());

    this.setKeymap(map);
  }

  /**
   * Performs an event when text is edited. Defined in <code>UndoListener</code>
   * interface.
   * @param e an object which describe edit changes.
   */
  public void undoableEditHappened(UndoableEditEvent e) {
    UndoItem undoItem = new UndoItem(this.getCaretPosition(), e.getEdit());
    for (int i = undoList.size() - 1; i > undoPosition; i--)
      undoList.remove(i);
    undoList.add(undoItem);
    undoPosition++;
    modified = true;
  }

  /**
   * Clears all undo changes and set modified flag to <code>FALSE</code>.
   */
  public void discardUndo() {
    undoPosition = -1;
    undoList.clear();
    modified = false;
  }

  /**
   * Undos changes in the edited text.
   */
  public void undo() {
    if (undoPosition >= 0) {
      UndoItem undoItem = (UndoItem) undoList.get(undoPosition);
      this.setCaretPosition(undoItem.position);
      undoItem.item.undo();
      undoPosition--;
      while (undoPosition >= 0) {
        UndoItem current = (UndoItem) undoList.get(undoPosition);
        if (current.item.getPresentationName().equals(
          undoItem.item.getPresentationName()) &&
            ((current.item.getPresentationName().startsWith("add") &&
            undoItem.position - current.position == 1) ||
            (current.item.getPresentationName().startsWith("del") &&
            current.position - undoItem.position == 1))) {
          undoItem = current;
          undoItem.item.undo();
          undoPosition--;
        } else {
          break;
        }
      }
    }
    modified = (undoPosition >= 0);
  }

  /**
   * Redos changes in the edited text.
   */
  public void redo() {
    if (undoPosition + 1 < undoList.size()) {
      undoPosition++;
      UndoItem undoItem = (UndoItem) undoList.get(undoPosition);
      undoItem.item.redo();
      this.setCaretPosition(undoItem.position);
      while (undoPosition + 1 < undoList.size()) {
        UndoItem current = (UndoItem) undoList.get(undoPosition+1);
        if (current.item.getPresentationName().equals(
          undoItem.item.getPresentationName()) &&
            ((current.item.getPresentationName().startsWith("add") &&
            current.position - undoItem.position == 1) ||
            (current.item.getPresentationName().startsWith("del") &&
            undoItem.position - current.position == 1))) {
          undoItem = current;
          undoItem.item.redo();
          undoPosition++;
        } else {
          break;
        }
      }
    }
    modified = (undoPosition >= 0);
  }

  /**
   * Checks was made any modifications with this edited text.
   * @result <code>TRUE</code> if there are some modifications was made
   *   and <code>FALSE</code> otherwise.
   */
  public boolean isModified() {
    return modified;
  }

  /**
   * Sets the new modified status for the edited text.
   * @param modified a new modified status of this text.
   */
  public void setModified(boolean modified) {
    this.modified = modified;
  }

  /**
   * Class to store undo changes.
   */
  private class UndoItem {
    public int position;
    public UndoableEdit item;

    public UndoItem(int position, UndoableEdit item) {
      this.position = position;
      this.item = item;
    }
  }

  /**
   * Implements Undo editor action.
   */
  private class UndoAction extends AbstractAction {
    public UndoAction() {}

    public void actionPerformed(ActionEvent e) {
      undo();
    }
  }

  /**
   * Implements Redo editor action.
   */
  private class RedoAction extends AbstractAction {
    public RedoAction() {}

    public void actionPerformed(ActionEvent e) {
      redo();
    }
  }

  /**
   * Initializes from a stream. This creates a model of the type appropriate
   * for the component and initializes the model from the stream. By default
   * this will load the model as plain text. Previous contents of the model
   * are discarded.
   * @param in The stream to read from.
   * @param desc An object describing the stream. This might be a string,
   *   a File, a URL, etc. Some kinds of documents (such as html for example)
   *   might be able to make use of this information. If non-null, it is added
   *   as a property of the document.
   */
  public void read(Reader in, Object desc) throws IOException {
    discardUndo();
    super.read(in, desc);
    this.getDocument().addUndoableEditListener(this);
  }

  /**
   * Stores the contents of the model into the given stream. By default this
   * will store the model as plain text.
   * @param out the output stream.
   */
  public void write(Writer out) throws IOException {
    super.write(out);
    discardUndo();
    this.getDocument().addUndoableEditListener(this);
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("Comedia Editor Test");
    CEditor editor = new CEditor();
//    editor.setDocument(new HTMLDocument());
    editor.setText("xxxxxxxxxxxxxxxx xxxxxxxxxxxxxxxxxx xxxxxxxxxxxxxxx xxxxxxxxxx");
    JScrollPane scroll = new JScrollPane(editor);
//    JTextArea text = new JTextArea();
//    editor.setAutoscrolls(false);
    frame.getContentPane().add(scroll, BorderLayout.CENTER);
//    text.add(editor);
//    frame.getContentPane().add(text, BorderLayout.CENTER);
    frame.setLocation(100, 100);
    frame.setSize(300, 300);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }
}