/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide.propertyinspector;

import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.ui.propertyinspector.core.Clazz;
import ch.abacus.lib.ui.propertyinspector.core.Property;
import ch.abacus.lib.ui.propertyinspector.core.AccessorInterface;
import ch.abacus.lib.ui.propertyinspector.display.*;
import ch.abacus.lib.ui.propertyinspector.display.editor.EditableStringListEditor;
import ch.abacus.designcockpit.ide.SuperDesignCockpit;
import ch.abacus.designcockpit.ide.IDEPropertyInspectorController;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import java.util.HashMap;
import java.util.ArrayList;


/**
 * Created by IntelliJ IDEA.
 * User: Stuart Booth (Live Clue, inc)
 * Date: Jul 22, 2005
 * Time: 7:01:25 PM
 */
public class IDEAbacusPropertyInspector implements IDEPropertyInspectorInterface {

    private HashMap<String, Clazz> m_clazzes;

    private PropertyInspector m_propertyInspector;
    private PropertyRegistry m_propertyRegistry;
    private PropertyFactory m_propertyfactory;
    private SuperDesignCockpit m_cockpit;
    private IDEPropertyInspectorController m_controller;
    private MetadataDispenser m_metadataDispenser;

    public void init(IDEPropertyInspectorController controller) {
        m_controller = controller;
        m_cockpit = controller.getSuperDesignCockpit();
        m_metadataDispenser = m_cockpit.getMetadataDispenser();

        m_propertyInspector = new PropertyInspector(m_cockpit.getGlobalInterface(), false);
        m_propertyRegistry = new PropertyRegistry(m_propertyInspector, m_cockpit);
        m_propertyfactory = new PropertyFactory(m_propertyRegistry, m_controller);

        m_propertyInspector.getModel().setAccessor(new MyAccessorInterface(controller));

        m_propertyInspector.getModel().setSortAlphabetically(true);
        //m_propertyInspector.getModel().setGrouped(false);

        String sReadOnly = m_cockpit.getLanguageManager().getMessage("PropertyInspector.Property_ReadOnly.label", "This property cannot be changed, as it has been designated as read-only");
        String sSingleSelect = m_cockpit.getLanguageManager().getMessage("PropertyInspector.Property_SingleSelect.label", "This property cannot be changed when multiple components are selected");
        m_propertyInspector.getLanguageStrings().put("readonly", sReadOnly);
        m_propertyInspector.getLanguageStrings().put("single selection", sSingleSelect);

        m_clazzes = new HashMap<String, Clazz>();

        loadClasses();
    }

    public void editingCanceled(ChangeEvent event) {
        getPropertyInspector().getTable().editingCanceled(event);
    }

    public void updateSelectedProperties() {
        getPropertyInspector().getModel().changeSelection();
    }

    public void newObjectsSelected(MetaObject[] objects) {
        final PropertyModel model = m_propertyInspector.getModel();
        model.beginUpdate();
        for (int iMetaObject = 0; iMetaObject < objects.length; iMetaObject++) {
            final MetaObject metaObject = objects[iMetaObject];
            if (metaObject != null)
                model.addSelection(metaObject.theClass.theMetadata.sClassName, metaObject);
        }
        model.endUpdate();
    }

    public void clearSelectedObjects() {
        m_propertyInspector.getModel().getSelection().clear();
        m_propertyInspector.getModel().changeSelection();
    }

    public JComponent getPropertyInspectorComponent() {
        return m_propertyInspector;
    }

    public void loadClasses() {
        final MetaClass firstClass = m_metadataDispenser.getFirstClass();

        m_propertyInspector.getModel().getClazzes().clear();
        m_clazzes.clear();

        addClazzes(firstClass);
    }

    private void addClazz(MetaClass metaClass) {
        String sClassName = metaClass.getMetadata().sClassName;

        //Class realClass = getClass(metaClass);

        String sParentClassName = null;
        Clazz parentClazz = null;
        final MetaClass parentClass = metaClass.getParentClass();
        if (parentClass != null) {
            sParentClassName = parentClass.getMetadata().sClassName;
            parentClazz = m_clazzes.get(sParentClassName);
        }
        Clazz clazz = new Clazz(parentClazz, sClassName);
        //System.out.println("Adding: " + sClassName + (parentClass != null? " parent=" +parentClass.getMetadata().sClassName: null));
        m_propertyInspector.getModel().getClazzes().add(clazz);
        m_clazzes.put(sClassName, clazz);

        addProperties(metaClass, clazz);

    }

    private void addProperties(MetaClass metaClass, Clazz clazz) {
        MetaPropertyDescriptor metaPropertyDescriptor = metaClass.getMetadata().theFirstProperty;
        while (metaPropertyDescriptor != null) {
            //System.out.println(metaClass.getMetadata().sClassName + " " + metaPropertyDescriptor.getName());
            addProperty(clazz, metaPropertyDescriptor);
            metaPropertyDescriptor = metaPropertyDescriptor.getNextProperty();
        }

    }

    private void addProperty(Clazz clazz, MetaPropertyDescriptor metaPropertyDescriptor) {
        if (metaPropertyDescriptor.bVisible) {


            Property property = m_propertyfactory.getProperty(metaPropertyDescriptor);
            clazz.getProperties().add(property);
        } else { // see if sub properties exist...
            MetaPropertyDescriptor subProperty = metaPropertyDescriptor.theFirstSubproperty;
            if (subProperty != null) {
                if (subProperty.bVisible)
                    addProperty(clazz, subProperty);
                subProperty = subProperty.getNextProperty();
                if (subProperty != null)
                    if (subProperty.bVisible)
                        addProperty(clazz, subProperty);
            }

        }
    }


//    private Class getClass(MetaClass metaClass) {
//        String sFullClassName = metaClass.getMetadata().sFullClassName;
//
//        boolean bContainerClass = isContainerClass(metaClass);
//
//
//        Class realClass = null;
//
//        if (!bContainerClass) {
//
//            try {
//                realClass = forName(sFullClassName);
//            } catch (ClassNotFoundException e) {
//                System.out.println("Cannot add: " + sFullClassName);
//            }
//        } else {
//            // container classes do not have real names, so we need to provide "real" ones...
//            System.out.println("Cannot add Container: " + metaClass.getMetadata().sClassName);
//            return IDEAbacusPropertyInspector.class;
//
//        }
//
//        return realClass;
//    }

//    private boolean isContainerClass(MetaClass metaClass) {
//        boolean bHasVisibleChildren = false;
//
//        metaClass = metaClass.getFirstClass();
//        while (metaClass != null) {
//            if (metaClass.getMetadata().bHidden == false)
//                return true;
//            metaClass = metaClass.getNextSibling();
//        }
//
//        return bHasVisibleChildren;
//    }


    private void addClazzes(MetaClass currentClass) {
        while (currentClass != null) {
            final MetaClassDetail metadata = currentClass.getMetadata();
            //final boolean bHidden = metadata.bHidden;
            //if (bHidden == false)
            addClazz(currentClass);
            if (currentClass.getFirstClass() != null) {
                addClazzes(currentClass.getFirstClass());
            }
            currentClass = currentClass.getNextSibling();
        }


    }

    public PropertyInspector getPropertyInspector() {
        return m_propertyInspector;
    }

    private static enum PropertyIds {
        Dumy;
    }


    private class MyAccessorInterface implements AccessorInterface {
        private IDEPropertyInspectorController m_controller;

        public MyAccessorInterface(IDEPropertyInspectorController controller) {
            m_controller = controller;
        }

        public void beginUpdate() {
            //System.out.println("beginUpdate");
        }

        public void setValue(Property property, ObjectInstance instance, Object value) {

            if (value != null) {
                final MetaObject metaObject = (MetaObject) instance.getObject();
                final String propertyName = (String) property.getData();

                m_controller.setPropertyValue(metaObject, propertyName, value);
            }


        }

        public Object getValue(Property property, ObjectInstance instance) {

            final MetaObject metaObject = (MetaObject) instance.getObject();
            final String propertyName = (String) property.getData();
            Object value =null;

            value = m_controller.getPropertyValue(metaObject, propertyName);

            //MetaProperty mp = metaObject.getPropertyMetadata(propertyName);
            final MetaClass metaClass = metaObject.theClass;
            MetaPropertyDescriptor mpd = metaClass.getProperty(propertyName, true);

            // thePropertyInfo!=null is checked when copying objects since some props are not
            // initliazed yet.

            if(mpd!=null && m_cockpit.getOpenSourceState()==true)
            {
                if(mpd.bObjectReference==true)
                {
                    ArrayList refList = mpd.getObjectReferenceList();

                    ArrayList stringSources = new ArrayList();
                    for(int i=0;i < refList.size();i++)
                    {
                        MetaObject mo = (MetaObject)refList.get(i);
                        stringSources.add(mo.getName());
                    }

                    PropertyValueEditorInterface pve = m_propertyRegistry.getEditorRegistry().get("-ObjReferenceList-" +propertyName);
                    if(refList!=null)
                        ((ObjecReferenceListEditor)pve).refreshList(stringSources);

                    if(value  instanceof JComponent)
                        value = ((JComponent)value).getName();
                }
                else if(mpd.getDataAwareness()==true)
                {
                        ArrayList refList = new ArrayList();
                        int iChoiceCount = mpd.getValueChoiceCount();
                        for(int i=0; i < iChoiceCount;i++)
                        {
                            refList.add(mpd.getValueChoiceAsString(i));
                        }

                    if(refList!=null)
                    {
                        PropertyValueEditorInterface pve = m_propertyRegistry.getEditorRegistry().get("-dbFieldNameList-" +propertyName);
                        pve = m_propertyRegistry.getEditorRegistry().get("-dbFieldNameList-" +propertyName);
                        if(pve!=null)
                        ((ObjecReferenceListEditor)pve).refreshList(refList);
                    }
                }
            }
            return value;

        }

        public void endUpdate() {
            m_controller.newObjectsSelected();
        }
    }
}
