/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.metadataEditor;

import ch.abacus.lib.ui.*;
import ch.abacus.lib.ui.validation.ValidationStaticMethods;
import ch.abacus.lib.ui.renderer.common.*;

import javax.swing.*;
import javax.swing.BorderFactory;
import javax.swing.text.Position;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeNode;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Collections;

/**
 * Created by IntelliJ IDEA.
 * User: Stuart
 * Date: July 13, 2004
 * Time: 5:58:17 PM
 */
public class ManageClassesDialog extends JADialog {
    final MetadataEditor editor;
    JATree treClasses = null;
    ArrayList bufferedActionCollection = new ArrayList();
    MetaClass theInitialSelectedMetaClass;
    MetadataEditorPageClasses.TableModel tableModel = null;
    DefaultTreeModel theTreeModel;
    IDEComponentPaletteNode baseClass = null;
    private IDEComponentPaletteNode theInitialSelectedMetaClassNode;

    public ManageClassesDialog (MetadataEditorPageClasses pageClasses, MetaClass selectedMetaClass) {
        super(pageClasses.editor, "Manage Classes", true, false);
        ValidationStaticMethods.setValidationEnabled(false);
        treClasses = new JATree();
        this.editor = pageClasses.editor;
        theInitialSelectedMetaClass = selectedMetaClass;
        initInterface();
        populateInterface();
    }

    private void initInterface () {
        MetaClass theFirstClass = editor.theMetadataDispenser.getFirstClass();

        baseClass = new IDEComponentPaletteNode(theFirstClass);
        theInitialSelectedMetaClassNode = baseClass;

        theTreeModel = new DefaultTreeModel(baseClass);
        treClasses.setModel(theTreeModel);
        Container pane = getContentPane();
        pane.setLayout(new BorderLayout());
        JAPanel pnlConstants = new JAPanel(new BorderLayout());
        pnlConstants.setBorder(BorderFactory.createTitledBorder("Classes"));
        pnlConstants.add(new JScrollPane(treClasses), BorderLayout.CENTER);
        JAButton btnAdd = new JAButton("Add");
        JAButton btnRename = new JAButton("Rename");
        JAButton btnDelete = new JAButton("Delete");
        btnAdd.addActionListener(new AddClassAction(this));
        btnRename.addActionListener(new RenameClassAction(this));
        btnDelete.addActionListener(new DeleteClassAction(this));
        JAPanel pnlButtons = new JAPanel(new FlowLayout(FlowLayout.LEFT, 5, 5));
        pnlButtons.add(btnAdd);
        pnlButtons.add(btnRename);
        pnlButtons.add(btnDelete);
        pnlConstants.add(pnlButtons, BorderLayout.SOUTH);
        pane.add(pnlConstants, BorderLayout.CENTER);
        JAPanel pnlDialogButtons = new JAPanel(new FlowLayout(FlowLayout.RIGHT, 5, 5));

        JAButton btnOK = new JAButton("OK");
        JAButton btnCancel = new JAButton("Cancel");
        btnOK.addActionListener(new OkAction(this));
        btnCancel.addActionListener(new CancelAction(this));

        pnlDialogButtons.add(btnOK);
        pnlDialogButtons.add(btnCancel);
        pane.add(pnlDialogButtons, BorderLayout.SOUTH);

        setSize(640, 480);
        setLocationRelativeTo(null);
        setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            public void windowClosing (WindowEvent e) {
                CancelAction cancelAction = new CancelAction((ManageClassesDialog.this));
                cancelAction.actionPerformed(new ActionEvent(this, 0, ""));
            }
        });
    }

    boolean loadClass (MetaClass theFirstClass, IDEComponentPaletteNode thisNode) {
        // A node in the treeview for every class in the heirarchy.
        MetaClass currClass = theFirstClass;
        while (currClass != null) {
            IDEComponentPaletteNode theNewNode = new IDEComponentPaletteNode(currClass);
            if (currClass == theInitialSelectedMetaClass)
                theInitialSelectedMetaClassNode = theNewNode;
            if (thisNode == null)
                theTreeModel.insertNodeInto(theNewNode, baseClass, baseClass.getChildCount());
            else
                theTreeModel.insertNodeInto(theNewNode, thisNode, thisNode.getChildCount());
            if (currClass.getFirstClass() != null)
                loadClass(currClass.getFirstClass(), theNewNode);
            currClass = currClass.getNextSibling();
        }
        return true;
    }

    public TreePath getTreePath(IDEComponentPaletteNode node) {
        ArrayList list = new ArrayList();

        // Add all nodes to list
        while (node != null) {
            list.add(node);
            node = (IDEComponentPaletteNode) node.getParent();
        }
        Collections.reverse(list);

        // Convert array of nodes to TreePath
        return new TreePath(list.toArray());
    }


    private void populateInterface () {
        MetaClass theFirstClass = editor.theMetadataDispenser.getFirstClass();
        loadClass(theFirstClass.getFirstClass(), baseClass);

        // select the initial class...
        if (theInitialSelectedMetaClassNode == baseClass)
            treClasses.expandRow(0);

        TreePath treePath = getTreePath(theInitialSelectedMetaClassNode);
        treClasses.setSelectionPath(treePath);
        treClasses.makeVisible(treePath);
        treClasses.scrollPathToVisible(treePath);
    }

    class RenameClassAction implements ActionListener {
        private final ManageClassesDialog dialog;

        public RenameClassAction (ManageClassesDialog dialog) {
            this.dialog = dialog;
        }

        public void actionPerformed (ActionEvent e) {
            IDEComponentPaletteNode node = (IDEComponentPaletteNode) treClasses.getLastSelectedPathComponent();
            if (node == null) {
                JOptionPane.showMessageDialog(dialog, "You must select a class before you can rename it.", "No Class Selected", JOptionPane.ERROR_MESSAGE);
                return;
            }

            String sOldName = (String) node.getUserObject();
            String sNewName = JOptionPane.showInputDialog("Rename Class \"" + sOldName + "\"\n\nPlease enter new name:");
            if (sNewName != null) {
                // buffer this action until the dialog's OK button is clicked...
                addBufferedAction(sOldName, sNewName);

                // Update the list...
                node.setUserObject(sNewName);
                treClasses.updateUI();
            }
        }

        void addBufferedAction (String sOldName, String sNewName) {
            BufferedAction bufferedAction = new BufferedAction();
            bufferedAction.setActionRename(sOldName, sNewName);
            bufferedActionCollection.add(bufferedAction);
        }
    }

    class AddClassAction implements ActionListener {
        private final ManageClassesDialog dialog;

        public AddClassAction (ManageClassesDialog dialog) {
            this.dialog = dialog;
        }

        public void actionPerformed (ActionEvent e) {
            IDEComponentPaletteNode parentNode = (IDEComponentPaletteNode) treClasses.getLastSelectedPathComponent();
            if (parentNode == null) {
                JOptionPane.showMessageDialog(dialog, "You must select a Superclass before you can add a class.", "No Superclass Selected", JOptionPane.ERROR_MESSAGE);
                return;
            }

            String sSuperclass = (String) parentNode.getUserObject();

            String sNewName = JOptionPane.showInputDialog("Add New Subclass of " +sSuperclass +":");
            if (sNewName != null) {
                // buffer this action until the dialog's OK button is clicked...
                addBufferedAction(sSuperclass, sNewName);

                // Update the list...
                IDEComponentPaletteNode subclassNode = new IDEComponentPaletteNode(sNewName);
                theTreeModel.insertNodeInto(subclassNode, parentNode, parentNode.getChildCount());
                TreePath treePath = getTreePath(subclassNode);
                treClasses.setSelectionPath(treePath);
                treClasses.scrollPathToVisible(treePath);

                treClasses.updateUI();
            }
        }

        void addBufferedAction (String sSuperclass, String sSubclass) {
            BufferedAction bufferedAction = new BufferedAction();
            bufferedAction.setActionAdd(sSuperclass, sSubclass);
            bufferedActionCollection.add(bufferedAction);
        }
    }

    class DeleteClassAction implements ActionListener {
        private final ManageClassesDialog dialog;

        public DeleteClassAction (ManageClassesDialog dialog) {
            this.dialog = dialog;
        }

        public void actionPerformed (ActionEvent e) {
            IDEComponentPaletteNode node = (IDEComponentPaletteNode) treClasses.getLastSelectedPathComponent();
            if (node == null) {
                JOptionPane.showMessageDialog(dialog, "You must select a class before you can delete it.", "No Class Selected", JOptionPane.ERROR_MESSAGE);
                return;
            }


            //String sClassName = (String) node.getMetaClass().getMetadata().sClassName;
            String sClassName = (String) node.getUserObject();
            Object sConfirmationText = "Are you sure you want to delete the class  \"" + sClassName + "\" and all its subclasses ?";
            String sConfirmationTitle = "Delete Class";
            int optionType = JOptionPane.YES_NO_OPTION;
            int messageType = JOptionPane.QUESTION_MESSAGE;
            int iChoice = JOptionPane.showConfirmDialog(dialog, sConfirmationText, sConfirmationTitle, optionType, messageType);
            if (iChoice == JOptionPane.YES_OPTION) {
                // buffer this action until the dialog's OK button is clicked...
                addBufferedAction(sClassName);

                // Update the tree...
                IDEComponentPaletteNode nextNode = (IDEComponentPaletteNode) node.getNextSibling();
                node.removeFromParent();
                treClasses.updateUI();
                if (nextNode != null)
                    treClasses.setSelectionPath(getTreePath(nextNode));
                else
                    treClasses.setSelectionRow(0);
            }
        }

        void addBufferedAction (String sClassName) {
            BufferedAction bufferedAction = new BufferedAction();
            bufferedAction.setActionDelete(sClassName);
            bufferedActionCollection.add(bufferedAction);
        }
    }

    class CancelAction implements ActionListener {
        private final ManageClassesDialog dialog;

        public CancelAction (ManageClassesDialog dialog) {
            this.dialog = dialog;
        }

        public void actionPerformed (ActionEvent e) {
            if (dialog.bufferedActionCollection.size() > 0) {
                Object sConfirmationText = "All changes will be lost. Are you sure you want to cancel ?";
                String sConfirmationTitle = "Cancel Changes";
                int optionType = JOptionPane.YES_NO_OPTION;
                int messageType = JOptionPane.QUESTION_MESSAGE;
                int iChoice = JOptionPane.showConfirmDialog(dialog, sConfirmationText, sConfirmationTitle, optionType, messageType);
                if (iChoice == JOptionPane.YES_OPTION) {
                    dialog.hide();
                }
            } else
                hide();
        }
    }


    class OkAction implements ActionListener {
        private final ManageClassesDialog dialog;

        public OkAction (ManageClassesDialog dialog) {
            this.dialog = dialog;
        }

        public void actionPerformed (ActionEvent e) {
            MetadataDispenser theMetadataDispenser = dialog.editor.theMetadataDispenser;
            if (dialog.bufferedActionCollection.size() == 0)
                return; // nothing to do!

            editor.setMetadataModified(true);

            for (int iAction = 0; iAction < dialog.bufferedActionCollection.size(); iAction++) {
                BufferedAction bufferedAction = (BufferedAction) dialog.bufferedActionCollection.get(iAction);
                //System.out.println(iAction + ": " + bufferedAction.getActionType() + " " + bufferedAction.getOldName() + " " + bufferedAction.getNewName());

                switch (bufferedAction.getActionType()) {
                    case BufferedAction.ACTION_RENAME:
                        {
                            MetaClass metaClass = theMetadataDispenser.findClass(bufferedAction.getOldName());
                            if (metaClass != null) {
                                metaClass.getMetadata().sClassName = bufferedAction.getNewName();
                            } else
                                System.out.println("Cannot Rename Class " + bufferedAction.getOldName() + ", as it cannot be found");
                            break;
                        }
                    case BufferedAction.ACTION_ADD:
                        {
                            // create the new class...
                            MetaClass metaSuperclass = theMetadataDispenser.findClass(bufferedAction.getOldName());
                            if (metaSuperclass != null) {
                                createSubclass(metaSuperclass, bufferedAction.getNewName());
                            }

                            break;
                        }
                    case BufferedAction.ACTION_DELETE:
                        {
                            MetaClass metaClass = theMetadataDispenser.findClass(bufferedAction.getOldName());
                            if (metaClass != null) {
                                metaClass.DeleteClass();
                            } else
                                System.out.println("Cannot Delete Class \"" + bufferedAction.getOldName() + "\", as it cannot be found");
                            break;
                        }

                }
            }

            dialog.hide();

        }
        public MetaClass createSubclass(MetaClass metaSuperclass, String sClassName) {
            // Make a child class of this class.
            MetadataDispenser theMetadataDispenser = dialog.editor.theMetadataDispenser;
            String sNewFullClassName = metaSuperclass.theMetadata.sFullClassName.substring(0, metaSuperclass.theMetadata.sFullClassName.lastIndexOf(".")) + "." +sClassName;
            MetaClassDetail theClass = new MetaClassDetail(sClassName, sNewFullClassName);
            MetaClass theMetaClass = new MetaClass(theClass, metaSuperclass, theMetadataDispenser, true);
            String sSuperClassName = metaSuperclass.theMetadata.sClassName;
            String sFullPathName = metaSuperclass.theMetadata.sFullClassName;
            theClass.setSuperClass(sSuperClassName, sFullPathName);
            theClass.setDisplayClass(sNewFullClassName);
            theClass.containerType = metaSuperclass.theMetadata.containerType;
            metaSuperclass.addClass(theMetaClass);

            return theMetaClass;
        }

    }

}


