/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * Language.java
 *
 * Creator:
 * 25.02.2003 15:08 Nef
 *
 * Maintainer:
 * 25.02.2003 15:08 Pecar
 *
 * Last Modification:
 * $Id: Language.java,v 1.1 2005/10/14 04:34:57 agraham Exp $
 *
 * Copyright (c) 2003 ABACUS Research AG, All Rights Reserved
 */
package ch.abacus.lib.lang;


import java.util.Locale;
import java.util.Iterator;

/**
 * This class defines the available languages for Abacus applications.  This information
 * is is used by other Abacus components to identify the language for application names
 * and some user settings.<br><br>
 *
 * <a name="language_table"><!-- --></a>
 * <b>Table of Languages defined in the Langauge class :</b>
 *  <table border=0 cellspacing=3 cellpadding=3>
 *      <tr bgcolor="#ccccff">
 *           <th align=left>Number
 *           <th align=left>Language
 *           <th align=left>ISOCode
 *           <th align=left>Character
 *           <th align=left>Language
 *      <tr valign=top>
 *          <td>0    <td>Language.DE       <td>de      <td>D     <td>German
 *      <tr valign=top bgcolor="#eeeeff">
 *          <td>1     <td>Language.FR      <td>fr      <td>F     <td>French
 *      <tr valign=top>
 *          <td>2     <td>Language.IT      <td>it      <td>I     <td>Italian
 *      <tr valign=top bgcolor="#eeeeff">
 *          <td>3     <td>Language.EN      <td>en      <td>E     <td>English
 *      <tr valign=top>
 *          <td>4     <td>Language.ES      <td>es      <td>S     <td>Spanish
 *  </table>
 */
public class Language {

  /** Index number for German Language = 0 */
  public static final int GERMAN_ID = 0;

  /** Index number for French Language = 1 */
  public static final int FRENCH_ID = 1;

  /** Index number for Italian Language = 2 */
  public static final int ITALIAN_ID = 2;

  /** Index number for English Language = 3 */
  public static final int ENGLISH_ID = 3;

  /** Index number for Spanish Language = 4 */
  public static final int SPANISH_ID = 4;

  /** ISO Code for German language = "de" */
  public static final String GERMAN_ISOCODE = "de";

  /** ISO Code for French language = "fr" */
  public static final String FRENCH_ISOCODE = "fr";

  /** ISO Code for Italian language = "it" */
  public static final String ITALIAN_ISOCODE = "it";

  /** ISO Code for English language = "en" */
  public static final String ENGLISH_ISOCODE = "en";

  /** ISO Code for Spanish language = "es" */
  public static final String SPANISH_ISOCODE = "es";

  /** Character code for German language = 'D' */
  public static final char GERMAN_CHAR = 'D';
  /** Character code for French language = 'F' */
  public static final char FRENCH_CHAR = 'F';
  /** Character code for Italian language = 'I' */
  public static final char ITALIAN_CHAR = 'I';
  /** Character code for English language = 'E' */
  public static final char ENGLISH_CHAR = 'E';
  /** Character code for Spanish language = 'S' */
  public static final char SPANISH_CHAR = 'S';

  /** Character code for German language = "D" */
  public static final String GERMAN_CHAR_STRING = "D";
  /** Character code for French language = "F" */
  public static final String FRENCH_CHAR_STRING = "F";
  /** Character code for Italian language = "I" */
  public static final String ITALIAN_CHAR_STRING = "I";
  /** Character code for English language = "E" */
  public static final String ENGLISH_CHAR_STRING = "E";
  /** Character code for Spanish language = "S" */
  public static final String SPANISH_CHAR_STRING = "S";

  /** Pre-defined language for German */
  public static final Language DE = new Language(GERMAN_ID);
  /** Pre-defined language for French */
  public static final Language FR = new Language(FRENCH_ID);
  /** Pre-defined language for Italian */
  public static final Language IT = new Language(ITALIAN_ID);
  /** Pre-defined language for English */
  public static final Language EN = new Language(ENGLISH_ID);
  /** Pre-defined language for Spanish */
  public static final Language ES = new Language(SPANISH_ID);

  /** Internal member array for all defined languages */
  private static final Language[] ALL = new Language[] { DE, FR, IT, EN, ES };

  /** Internal member array for all defined language ISO Codes */
  private static final String[] ISO_CODES = { GERMAN_ISOCODE, FRENCH_ISOCODE, ITALIAN_ISOCODE, ENGLISH_ISOCODE, SPANISH_ISOCODE };
  /** Internal member array for all defined language charcater ids */
  private static final char[] CHARS = { GERMAN_CHAR, FRENCH_CHAR, ITALIAN_CHAR, ENGLISH_CHAR, SPANISH_CHAR };
  /** Internal member array for all defined language character strings */
  private static final String[] CHAR_STRINGS = { GERMAN_CHAR_STRING, FRENCH_CHAR_STRING, ITALIAN_CHAR_STRING, ENGLISH_CHAR_STRING, SPANISH_CHAR_STRING };
  /** Internal member array for all language related Locales */
  private static final Locale[] LOCALES = { Locale.GERMAN, Locale.FRENCH, Locale.ITALIAN, Locale.ENGLISH, new Locale(ISO_CODES[SPANISH_ID]) };

  /**
   * Internal Index number of language (see <a href="#language_table">language table</a>
   * defined for indexes)
   *
   * @see #fromNr
   */
  private int mLanguageNr;

  /**
   * Constructor for creating a Language with an language index number.  All Languages
   * are pre-defined within the class so it is not possible create a new language
   * externally (see the <a href="#language_table">language table</a> for the
   * defined languages).
   *
   * @param nLanguageId the index number of the class
   */
  private Language(int nLanguageId) { mLanguageNr = nLanguageId; }

  /**
   * Compares an <code>Object</code>, that must be a Language, to this Language to see
   * if they are equal.  The function returns true if the index number of the Language
   * is equal, otherwise false.
   *
   * @param other the Object to compare (must be a Language)
   * @return returns true if the Language index number is equal, otherwise false.
   */
  public boolean equals(Object other) { return mLanguageNr == ((Language) other).mLanguageNr; }

  /**
   * Returns the ISOCode of the Language (same as {@link #getISOCode getIsoCode}).
   * See the <a href="#language_table">language table</a> for the
   * defined languages and associated ISO Codes.
   *
   * @return returns the ISOCode of the Language (same as {@link #getISOCode getIsoCode}).
   *
   * @see #getISOCode
   */
  public String toString() { return getISOCode(); }

  /**
   * Returns the Language index number.  See the <a href="#language_table">language table</a>
   * for the defined languages and associated index numbers.
   *
   * @return returns the Language index number.
   */
  public int getId() { return mLanguageNr; }

  /**
   * Returns the ISOCode of the Language. See the <a href="#language_table">language table</a>
   * for the defined languages and associated ISO Codes.
   *
   * @return returns the ISOCode of the Language.
   *
   * @see #toString
   */
  public String getISOCode() { return ISO_CODES[mLanguageNr]; }

  /**
   * Returns the unique identification character for the Language.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated identification characters.
   *
   * @return returns the unique identification character for the Language
   *
   * @see #getCharString
   */
  public char getCharacter() { return CHARS[mLanguageNr]; }

  /**
   * Returns the unique identification character for the Language.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated identification characters.
   *
   * @return returns the unique identification character for the Language
   *
   * @see #getCharacter
   */
  public String getCharString() { return CHAR_STRINGS[mLanguageNr]; }

  /**
   * Returns the <code>Locale</code> defined for the Language.  The <code>Locale</code>
   * defines some country and language specific settings.
   *
   * @return returns the <code>Locale</code> defined for the Language
   *
   * @see Locale
   */
  public Locale getLocale() { return LOCALES[mLanguageNr]; }

  /**
   * Retrieves a Language by the specified ISO Code.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated ISO Codes.
   *
   * @param isoCode the ISO code defining the language (e.g. "de", "fr", "it", "en", "es")
   * @return returns the Language according to the specified ISO Code
   *
   * @see #getISOCode
   */
  public static Language fromISOCode(String isoCode) { return fromISOCode(isoCode, Language.DE); }

  /**
   * Retrieves a Language by the specified index number.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated index number (id).
   *
   * <b>Note :</b> If the associated language if not found the <code>Language.DE</code>
   * is returned by default.
   *
   * @param nr the index number defining the language (e.g. 0, 1, 2, 3, 4)
   * @return returns the Language according to the specified index number
   *
   * @see #fromNr(int, Language)
   * @see #getId
   */
  public static Language fromNr(int nr) { return fromNr(nr, Language.DE); }

  /**
   * Retrieves a Language by the specified unique identification character.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated identification character.<br><br>
   *
   * <b>Note :</b> If the associated language if not found the <code>Language.DE</code>
   * is returned by default.
   *
   * @param ch the language identification character that specifies the language
   * @return returns the language acciated with the character identifier or the Language.DE as the default
   *
   * @see #fromChar(char, Language)
   */
  public static Language fromChar(char ch) { return fromChar(ch, Language.DE); }

  /**
   * Retrieves a Language by the specified unique identification character.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated identification character.<br><br>
   *
   * @param ch the language identification character that specifies the language
   * @param defaultLanguage the default language to return if language it not found (can be null)
   * @return returns the language acciated with the character identifier or the specified defaultLanguage.
   *
   * @see #fromChar(char)
   */
  public static Language fromChar(char ch, Language defaultLanguage) {
    for(int i = 0; i < CHARS.length; i++)
      if(CHARS[i] == ch)
        return ALL[i];
    return defaultLanguage;
  }

  /**
   * Retrieves a Language by the specified ISO Code.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated ISO Codes.<br><br>
   *
   * @param isoCode the language ISO Code that specifies the language
   * @param defaultLanguage the default language to return if language it not found (can be null)
   * @return returns the language acciated with the ISO Code or the specified defaultLanguage.
   */
  public static Language fromISOCode(String isoCode, Language defaultLanguage) {
    for(int i = 0; i < ISO_CODES.length; i++)
      if(ISO_CODES[i].equals(isoCode))
        return ALL[i];
    return defaultLanguage;
  }

  /**
   * Returns an iterator for all defined languages in the abacus environment.
   *
   * Example:
   * <pre>
   * Iterator i = Language.iterator();
   * while(i.hasNext()) {
   *   Language lang = (Language) i.next();
   *   System.out.println(lang);
   * }
   * </pre>
   *
   * @return
   */
  public static final Iterator iterator() {
    return new Iterator() {
      private int counter = 0;
      public boolean hasNext() { return counter < ALL.length; }
      public Object next() { return counter < ALL.length ? ALL[counter++] : null; }
      public void remove() { } //it's not possible to remove a language
    };
  }

  /**
   * Retrieves a Language by the specified index number.  See the
   * <a href="#language_table">language table</a> for the defined languages and
   * associated index number (id).
   *
   * @param nr the index number defining the language (e.g. 0, 1, 2, 3, 4)
   * @param defaultLanguage the default language to return if language it not found (can be null)
   * @return returns the Language according to the specified index number
   *
   * @see #getId
   * @see #fromNr(int)
   */
  public static Language fromNr(int nr, Language defaultLanguage) {
    if(nr < 0 || nr >= ALL.length) return defaultLanguage;
    return ALL[nr];
  }

  /**
   * Test function for testing Language class
   *
   * @param args command line arguments
   */
  public static void main(String[] args) {
    System.out.println("---- Language ----");
    for(int i = 0; i < ALL.length; i++) {
      Language language = Language.ALL[i];
      System.out.println(language.getId() + ", " + language.getISOCode() + ", " + language.getCharacter());
        for(int il = 0; il < ALL.length; il++) {
            //System.out.println("  " + ALL[il].getISOCode() + ": " + language.getName(ALL[il]));
        }
    }
    Language language = Language.DE;
    System.out.println(language.getId() + ", " + language.getISOCode() + ", " + language.getCharacter());
    language = Language.FR;
    System.out.println(language.getId() + ", " + language.getISOCode() + ", " + language.getCharacter());
    language = Language.IT;
    System.out.println(language.getId() + ", " + language.getISOCode() + ", " + language.getCharacter());
    language = Language.EN;
    System.out.println(language.getId() + ", " + language.getISOCode() + ", " + language.getCharacter());
    language = Language.ES;
    System.out.println(language.getId() + ", " + language.getISOCode() + ", " + language.getCharacter());
  }
}
