/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * VerticalFlowLayout2.java  
 *
 * Creator:
 * 04.07.2005 14:44:41 Pecar
 *
 * Maintainer:
 * 04.07.2005 14:44:41 Pecar
 *
 * Last Modification:
 * $Id: VerticalFlowLayout2.java,v 1.7 2005/07/13 08:24:37 pecar Exp $
 *
 * Copyright (c) 2005 ABACUS Research AG, All Rights Reserved
 */
package ch.abacus.lib.ui.layout;

import ch.abacus.lib.util.math.IntMath;

import java.awt.*;

public class VerticalFlowLayout2 implements LayoutManager2, java.io.Serializable {

    public static enum HAlignment { Left, Center, Right }
    public static enum VAlignment { Top, Middle, Bottom }

    private HAlignment mHAlignment;
    private VAlignment mVAlignment;
    private int mFillComponent = -1;
    private int mHgap;
    private int mVgap;
    private boolean mHfill;
    private boolean mVfill;

    public VerticalFlowLayout2() {
      this(HAlignment.Left, VAlignment.Top, 0, 0, true, false);
    }
    public VerticalFlowLayout2(boolean hfill, boolean vfill){
      this(HAlignment.Left, VAlignment.Top, 0, 0, hfill, vfill);
    }
    public VerticalFlowLayout2(HAlignment hAlign, VAlignment vAlign) {
        this(hAlign, vAlign, 0, 0, true, false);
    }
    public VerticalFlowLayout2(HAlignment hAlign, VAlignment vAlign, int hgap, int vgap, boolean hfill, boolean vfill) {
        mHAlignment = hAlign;
        mVAlignment = vAlign;
        setHgap(hgap);
        setVgap(vgap);
        mHfill = hfill;
        mVfill = vfill;
    }
    public int getFillComponent() {
        return mFillComponent;
    }
    public void setFillComponent(int fillComponent) {
        mFillComponent = fillComponent;
    }
    public int getHgap() {
        return mHgap;
    }
    public void setHgap(int hgap) {
        if(hgap < 0)
            throw new IllegalArgumentException("hgap must be positive");
        mHgap = hgap;
    }
    public int getVgap() {
        return mVgap;
    }
    public void setVgap(int vgap) {
        if(vgap < 0)
            throw new IllegalArgumentException("vgap must be positive");
        mVgap = vgap;
    }
    public boolean isHfill() {
        return mHfill;
    }
    public void setHfill(boolean hfill) {
        mHfill = hfill;
    }
    public boolean isVfill() {
        return mVfill;
    }
    public void setVfill(boolean vfill) {
        mVfill = vfill;
    }

    public void addLayoutComponent(Component comp, Object constraints) { }
    public void addLayoutComponent(String name, Component comp) { }
    public void removeLayoutComponent(Component comp) { }
    public float getLayoutAlignmentX(Container parent) { return 0.5f; }
    public float getLayoutAlignmentY(Container parent) { return 0.5f; }
    public void invalidateLayout(Container target) { }

    public Dimension maximumLayoutSize(Container target) {
        if(mHfill && mVfill)
            return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
        else {
            synchronized(target.getTreeLock()) {
                final Insets insets = target.getInsets();
                int height = mVfill ? Integer.MAX_VALUE : 0;
                int width = mHfill ? Integer.MAX_VALUE : 0;
                for(int i = 0; i < target.getComponentCount(); i++) {
                    Component c = target.getComponent(i);
                    if(c.isVisible()) {
                        Dimension d = c.getMaximumSize();
                        height = IntMath.add(height, d.height, mVgap);
                        if(width < d.width)
                            width = d.width;
                    }
                }
                height = IntMath.add(height, insets.top, insets.bottom, mVgap);
                width = IntMath.add(width, insets.left, insets.right, 2 * mHgap);
                return new Dimension(width, height);
            }
        }
    }

    public void layoutContainer(Container target) {
        synchronized(target.getTreeLock()) {
            final Insets insets = target.getInsets();

            int fillComp = -1;
            int fillHeight = target.getHeight() - insets.top - insets.bottom - mVgap;
            int width = target.getWidth() - insets.left - insets.right - 2 * mHgap;

            for(int i = 0; i < target.getComponentCount(); i++) {
                Component c = target.getComponent(i);
                if(c.isVisible()) {
                    Dimension d = c.getPreferredSize();
                    if((i == mFillComponent) || ((i == (target.getComponentCount() - 1)) && fillComp == -1) && mVfill) {
                        fillComp = i;
                        fillHeight -= mVgap;
                    } else
                        fillHeight -= d.height + mVgap;
                }
            }
            if(fillComp == -1 && mVfill)
                fillComp = target.getComponentCount();

            int top = insets.top + mVgap;
            int left = insets.left + mHgap;
            if(!mVfill) {
                switch(mVAlignment) {
                    case Middle:
                        top = insets.top + mVgap + fillHeight / 2;
                    break;
                    case Bottom:
                        top = insets.top + mVgap + fillHeight;
                    break;
                }
            }
            for(int i = 0; i < target.getComponentCount(); i++) {
                Component c = target.getComponent(i);
                if(c.isVisible()) {
                    Dimension d = c.getPreferredSize();
                    int h = i == fillComp ? fillHeight : d.height;
                    int w = mHfill ? width : d.width;
                    int l = left;
                    if(!mHfill) {
                        switch(mHAlignment) {
                            case Center:
                                l = left + (width - d.width) / 2;
                            break;
                            case Right:
                                l = left + width - d.width;
                            break;
                        }
                    }
                    c.setBounds(l, top, w, h);
                    top += h;
                    top += mVgap;
                }
            }
        }
    }
    public Dimension minimumLayoutSize(Container target) {
        synchronized(target.getTreeLock()) {
            final Insets insets = target.getInsets();
            int height = 0;
            int width = 0;
            for(int i = 0; i < target.getComponentCount(); i++) {
                Component c = target.getComponent(i);
                if(c.isVisible()) {
                    Dimension d = c.getMinimumSize();
                    height = IntMath.add(height, d.height, mVgap);
                    if(width < d.width)
                        width = d.width;
                }
            }
            height = IntMath.add(height, insets.top, insets.bottom, mVgap);
            width = IntMath.add(width, insets.left, insets.right, 2 * mHgap);
            return new Dimension(width < 0 ? 0 : width, height < 0 ? 0 : height);
        }
    }
    public Dimension preferredLayoutSize(Container target) {
        synchronized(target.getTreeLock()) {
            final Insets insets = target.getInsets();
            int height = 0;
            int width = 0;
            for(int i = 0; i < target.getComponentCount(); i++) {
                Component c = target.getComponent(i);
                if(c.isVisible()) {
                    Dimension d = c.getPreferredSize();
                    height = IntMath.add(height, d.height, mVgap);
                    if(width < d.width)
                        width = d.width;
                }
            }
            height = IntMath.add(height, insets.top, insets.bottom, mVgap);
            width = IntMath.add(width, insets.left, insets.right, 2 * mHgap);
            return new Dimension(width < 0 ? 0 : width, height < 0 ? 0 : height);
        }
    }
}
