/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * UserPropertiesInterface.java
 *
 * Open source stub
 */
package ch.abacus.lib.util;

import ch.abacus.lib.lang.Language;

import java.util.TimeZone;
import java.util.Iterator;

/**
 * The following interface defines the wrapper functions for obtaining the
 * user properties from the AbaKernel.<br><br>
 *
 * Additional information about the
 * <a href="http://www.nef.ch/abakernel.aspx" target="_blank">AbaKernel</a>
 * is available from
 * <a href="http://www.nef.ch" target="_blank">The NEF Zone !</a>.
 */
public interface UserPropertiesInterface {

  /**
   * This function returns the Language of the particular user.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/GetUserLanguage.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @return returns the language of the user
   */
  public Language getLanguage();

  /**
   * This function returns the current TimeZone set by the user.<br><br>
   *
   * <b>Note :</b> Normally this this is the current time zone of the computer. In
   * the user can set a specific time zone (comment date: Sept, 2003)<br><br>
   *
   * <b>Note :</b> This function is not connected to the AbaKernel
   *
   * @return returns the current TimeZone used by the user
   */
  public TimeZone getTimeZone();

  /**
   * This function returns the FormatPatterns defined for the user.<br><br>
   *
   * <b>Note :</b> Currently this returns a standard FormatPatterns object for the
   * user. Future implementation will return a specific FormatPatterns defined
   * by the user. (comment date: Sept, 2003)<br><br>
   *
   * <b>Note :</b> This function is not connected to the AbaKernel
   *
   * @return returns the FormatPatterns defined for the user
   */
  public Object getFormatPatterns();

  /**
   * This function returns a unique AbaGUID for the current user.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/UserNumberToUID.html" target="_blank">
   * UserNumberToUID</a> and <a href="http://www.nef.ch/abakernel/GetUserNumber.html" target="_blank">
   * GetUserNumber</a> in the AbaKernel Documentation).  This funtion uses a
   * mix of these 2 AbaKernel functions.
   *
   * @return returns a unique AbaGUID for the current user.
   */
  public Object getUID();

  /**
   * This function returns the name of the user (with a maximum length of
   * 20 characters).<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/QueryUserName.html" target="_blank">
   * AbaKernel Documentation</a>.
   *
   * @return returns the name of the user (maximum length 20 characters)
   */
  public String getName();

  /**
   * This function returns the full name of the user (with a maximum length of
   * 260 characters).<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/QueryFullUserName.html" target="_blank">
   * AbaKernel Documentation</a>.
   *
   * @return returns the full name of the user (maximum length 260 characters)
   */
  public String getFullName();

  /**
   * This function returns the email address of the user (with a maximum length of
   * 260 characters).<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/QueryUserMailAddress.html" target="_blank">
   * AbaKernel Documentation</a>.
   *
   * @return returns the email address of the user (maximum length 260 characters)
   */
  public String getMailAddress();

  /**
   * This function returns a user number, which can be between 1 and 9999.
   * A return value of zero(0) indicates and error.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/GetUserNumber.html" target="_blank">
   * AbaKernel Documentation</a>.
   *
   * @return returns the user number (between 1 and 9999), zero is an error
   */
  public int getNumber();

  /**
   * This function detemines whether the user is a member of the
   * user group <i>Administrator</i>.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/IsUserAdministrator.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @return returns true if the user is a member of the user group <i>Administrator</i>, otherwise false
   */
  public boolean isAdministrator();

  /**
   * This function detemines whether the user is a member of the
   * user group <i>Supervisor</i> or <i>Administrator</i> for a particular
   * application.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/IsUserSupervisor.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @param application the specified application to check
   * @return returns true if the user is a member of the user group <i>Supervisor</i> or <i>Administrator</i>, otherwise false
   */
  public boolean isSupervisor(Object application);

  /**
   * This function detemines whether the user is a member of the
   * user group <i>AbaAgent-Benutzer</i> or <i>Administrator</i>.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel.aspx?IsUserAbaAgentUser" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @return returns true if the user is a member of the user group <i>AbaAgent-Benutzer</i> or <i>Administrator</i>, otherwise false
   */
  public boolean isAbaAgentUser();

  /**
   * This function returns the Web-Mandant of the user, which can be between
   * 1 and 9999.  A return of zero(0) means that no Web-Mandant is defined
   * for the user.  The Web-Mandant is used by the AbaAgent.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/GetUserWebClient.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @return returns the Web-Mandant of the user (1 to 9999), zero(0) if not defined
   */
  public int getWebClient();

  /**
   * This function returns the default user category of the user.  A user
   * can have more than one user category that are simultaneously active.
   * Therefore a user has a defined default user category.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/QueryDefaultUserCategory.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @return returns the default user category of the user
   */
  public Object getDefaultUserCategory();

  /**
   * This function returns an iterator for the UserCategoryInterface objects
   * to which the the user belongs.  A user can belong to more than one
   * user category.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/GetNumberOfUserCategories.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @return returns an iterator for the user categories to which the the user belongs.
   */
  public Iterator getUserCategories();

  /**
   * This function checks if a user has the rights to access a particular
   * application.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel.aspx?CheckApplicationAccess" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @param application the specified application to check for access
   * @return returns true if the user has access rights to the application, otherwise false.
   */
  public boolean hasApplicationAccess(Object application);

  /**
   * This function checks if a user has the rights to access a particular
   * client Mandant within an application.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/CheckClientAccess.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @param application the specified application to check for access
   * @param client the specified Mandant to check for access
   * @return returns true if the user has access rights to the client mandant, otherwise false.
   */
  public boolean hasClientAccess(Object application, Object client);

  /**
   * This function checks if a user has the rights to access a particular
   * program within an application.<br><br>
   *
   * (See also <a href="http://www.nef.ch/abakernel/CheckProgramAccess.html" target="_blank">
   * AbaKernel Documentation</a>).
   *
   * @param application the specified application to check for access
   * @param programNr the number of the specified program within the application
   * @return returns true if the user has access rights to the application program, otherwise false.
   */
  public boolean hasProgramAccess(Object application, int programNr);
}
