/**
 * Title:        Comedia Utils
 * Description:  Project contains some general purpose non-visual beans.
 * Beans do not require any special libraies.
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.util.xml;

import java.util.*;
import java.io.*;
import org.comedia.util.scanner.*;

/**
 * Presents a default XML element which can contain strings or other XML
 * elements. This class does not perform any XML validation.
 */
public class CDefaultXmlElement implements CXmlElement {
  /**
   * The element tag name.
   */
  protected String tagName = "";

  /**
   * The element properties.
   */
  protected Hashtable props = new Hashtable();

  /**
   * The child element objects.
   */
  protected ArrayList childs = new ArrayList();

  /**
   * Constructs this class with default properties.
   */
  public CDefaultXmlElement() {
  }

  /**
   * Constructs this class and assignes tag name.
   * @param tagName a name of element tag.
   */
  public CDefaultXmlElement(String tagName) {
    this.tagName = tagName.toUpperCase();
  }

  /**
   * Clears the content of this element and restores an initial state.
   */
  public void clear() {
    props.clear();
    childs.clear();
  }

  /**
   * Parses this element from input stream.
   * @param scanner lexical scanner to parse XML.
   */
  public void parseElement(CXmlScanner scanner) throws CXmlException {

    String token, param, value;

    // Parses the rest of tag.
    while (true) {
      if (scanner.getTokenType() == scanner.BRACE) {
        token = scanner.getToken();
        scanner.gotoNextToken();
        if (token.equals("/>")) return;
        if (token.endsWith(">")) break;
        throw new CXmlException("Incorrent XML structure at line: " +
          scanner.getLineNo());
      }

      if (scanner.getTokenType() != scanner.KEYWORD)
        throw new CXmlException("Incorrent XML structure at line: " +
          scanner.getLineNo());

      param = scanner.getToken();
      scanner.gotoNextToken();
      if (scanner.getTokenType() == scanner.OPERATOR) {
        scanner.gotoNextToken();
        value = scanner.unwrapString(scanner.getToken());
        scanner.gotoNextToken();
      } else
        value = null;
      props.put(param.toUpperCase(), value);
    }
    // Parses all child elements and end tag.
    parseChildElements(scanner);
  }

  /**
   * Parses child elements from input stream.
   * @param scanner lexical scanner to parse XML.
   */
  public void parseChildElements(CXmlScanner scanner) throws CXmlException {
    String line = "", token;

    // Parses content
    while (scanner.getTokenType() != scanner.EOF) {
      token = scanner.getToken();
      // Skips special word
      if (token.equals("<?")) {
        while (!scanner.getToken().equals("?>")) {
          if (scanner.getTokenType() == scanner.EOF)
            throw new CXmlException("Incorrent XML structure at line: " +
              scanner.getLineNo());
          scanner.gotoNextToken();
        }
        scanner.gotoNextToken();
        continue;
      }

      // Skips special word
      if (token.equals("<!")) {
        while (!scanner.getToken().equals(">")) {
          if (scanner.getTokenType() == scanner.EOF)
            throw new CXmlException("Incorrent XML structure at line: " +
              scanner.getLineNo());
          scanner.gotoNextToken();
        }
        scanner.gotoNextToken();
        continue;
      }

      // Process start element
      if (token.equals("<")) {
        if (line.length() > 0) {
          childs.add(line);
          line = "";
        }

        scanner.gotoNextToken();
        token = scanner.getToken();
        if (scanner.getTokenType() != scanner.KEYWORD)
          throw new CXmlException("Incorrent XML structure at line: " +
            scanner.getLineNo());
        scanner.gotoNextToken();
        CXmlElement element = this.createChildElement(token.toUpperCase());
        childs.add(element);
        element.parseElement(scanner);
        continue;
      }

      // Process end element
      if (token.equals("</")) {
        scanner.gotoNextToken();
        token = scanner.getToken().toUpperCase();
        if (token.equals(tagName)) {
          scanner.gotoNextToken();
          if (scanner.getToken().equals(">")) {
            scanner.gotoNextToken();
            break;
          }
        }
        throw new CXmlException("Incorrent XML structure at line: " +
          scanner.getLineNo());
      }

      if (scanner.getTokenType() == scanner.BRACE)
        throw new CXmlException("Incorrent XML structure at line: " +
          scanner.getLineNo());

      if (line.length() > 0)
        line += " ";
      line += scanner.unwrapValue(token);
      scanner.gotoNextToken();
    }
    if (line.length() > 0)
      childs.add(line);
  }

  /**
   * Creates an indent string.
   * @param indent number of white spaces in the indent.
   */
  protected String createIndent(int indent) {
    String result = "";
    for (int i = 0; i < indent; i++)
      result += " ";
    return result;
  }

  /**
   * Writes this element from output stream.
   * @param output stream to write this element and all contents.
   * @param indent number of indent spaces.
   */
  public void writeElement(Writer out, int indent) throws IOException {
    out.write(createIndent(indent) + "<" + tagName);
    // Writes element properties.
    Enumeration params = props.keys();
    while (params.hasMoreElements()) {
      Object param = params.nextElement();
      Object value = props.get(param);
      out.write(" " + param.toString());
      if (value != null)
        out.write("=" + CXmlScanner.wrapString(value.toString()));
    }
    // Writes end of tag
    if (childs.size() == 0)
      out.write("/>\n");
    else {
      out.write(">\n");
      // Writes element content.
      writeChildElements(out, indent);
      out.write(createIndent(indent) + "</" + tagName + ">\n");
    }
  }

  /**
   * Writes child elements of this element to output stream.
   * @param output stream to write child elemens.
   * @param indent number of indent spaces.
   */
  public void writeChildElements(Writer out, int indent) throws IOException {
    for (int i = 0; i < childs.size(); i++) {
      Object current = childs.get(i);
      if (current instanceof CXmlElement)
        ((CXmlElement) current).writeElement(out, indent + 2);
      else
        out.write(createIndent(indent) + "  "
          + CXmlScanner.wrapValue(current.toString()) + "\n");
    }
  }

  /**
   * Creates a child element to this one.
   * @param tagName a name of element tag.
   */
  public CXmlElement createChildElement(String tagName) {
    return new CDefaultXmlElement(tagName.toUpperCase());
  }
}