/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.ArrayList;

/**
 * A form container represents a <code>java.awt.Container</code> in the form model.
 * In addition to the inherited functionality, it has a form layout manager and
 * a list of child form components.
 * The relationship between {@link FormContainer}, {@link FormLayoutManager} and
 * {@link FormLayoutConstraints} is the same as in Swing. The form container has
 * a form layout manager. The form layout manager holds the form constrains for
 * the children of the form container.
 * <p>
 * Take a look at the example source code (FormSaverExamples.java) for details.
 * <p>
 * Example using JGoodies FormLayout:
 * <pre>
 *   FormLayoutManager layout = new FormLayoutManager(FormLayout.class);
 *   layout.setProperty("$columnSpecs", "default, labelcompgap, default:grow");
 *   layout.setProperty("$rowSpecs", "default, linegap, default");
 *
 *   FormContainer panel = new FormContainer("javax.swing.JPanel", layout);
 *   panel.setName("this");
 *   panel.setProperty("$size", new Dimension(300, 200)); // size in design view
 *
 *   // create "name" label
 *   FormComponent nameLabel = new FormComponent("javax.swing.JLabel");
 *   nameLabel.setName("nameLabel");
 *   nameLabel.setProperty("text", "Name:");
 *
 *   // add "name" label to panel
 *   FormLayoutConstraints cons = new FormLayoutConstraints( CellConstraints.class );
 *   cons.setPropertyInt("gridX", 1);
 *   cons.setPropertyInt("gridY", 1);
 *   panel.add(nameLabel, cons);
 *
 *   // create "name" text field
 *   FormComponent nameField = new FormComponent("javax.swing.JTextField");
 *   nameField.setName("nameField");
 *   nameField.setPropertyInt("columns", 20);
 *
 *   // add "name" text field to panel
 *   cons = new FormLayoutConstraints( CellConstraints.class );
 *   cons.setPropertyInt("gridX", 3);
 *   cons.setPropertyInt("gridY", 1);
 *   panel.add(nameField, cons);
 * </pre>
 *
 * @author Karl Tauber
 */
public class FormContainer
	extends FormComponent
{
	public static final String FIELD_MENU_BAR = "menuBar";

	private FormLayoutManager layout;
	private final ArrayList<FormComponent> components = new ArrayList<FormComponent>();

	private FormContainer menuBar;
	private FormContainer menuBarOwner;

	/**
	 * Constructs a form container for the specified class.
	 */
	public FormContainer( String className ) {
		this( className, null );
	}

	/**
	 * Constructs a form container for the specified class and form layout manager.
	 */
	public FormContainer( String className, FormLayoutManager layout ) {
		super( className );
		setLayout( layout );
	}

	FormContainer( FormContainer obj, int dummy ) {
		super( obj, dummy );

		// clone layout manager
		if( obj.layout != null )
			setLayout( (FormLayoutManager) obj.layout.clone() );

		// clone components and constraints
		int count = obj.components.size();
		components.ensureCapacity( count );
		for( int i = 0; i < count; i++ ) {
			FormComponent c = obj.getComponent( i );
			FormLayoutConstraints cons = null;
			if( obj.layout != null )
				cons = obj.layout.getConstraints( c );
			if( cons != null )
				cons = (FormLayoutConstraints) cons.clone();
			add( (FormComponent) c.clone(), cons );
		}

		// clone menu bar
		if( obj.menuBar != null )
			setMenuBar( (FormContainer) obj.menuBar.clone() );
	}

	/**
	 * Clones this form container.
	 */
	@Override
	public Object clone() {
		return new FormContainer( this, 0 );
	}

	/**
	 * Returns the form layout manager used by this form container.
	 */
	public FormLayoutManager getLayout() {
		return layout;
	}

	/**
	 * Sets the form layout manager for this form container.
	 */
	public void setLayout( FormLayoutManager layout ) {
		if( this.layout == layout )
			return;

		FormLayoutManager oldLayout = this.layout;
		if( oldLayout != null )
			oldLayout.setContainer( null );

		this.layout = layout;

		if( layout != null )
			layout.setContainer( this );

		if( model != null && model.eventProvider != null )
			model.eventProvider.fireLayoutChanged( this, oldLayout, layout );
	}

	/**
	 * Returns the number of form components in this form container.
	 */
	public int getComponentCount() {
		return components.size();
	}

	/**
	 * Returns the form component at <code>index</code>.
	 */
	public FormComponent getComponent( int index ) {
		return components.get( index );
	}

	/**
	 * Returns all form components in this form container.
	 */
	public FormComponent[] getComponents() {
		return components.toArray( new FormComponent[components.size()] );
	}

	/**
	 * Returns the index of <code>comp</code> in this form container;
	 * or -1 if the form component is not a child of this form container.
	 */
	public int getComponentIndex( FormComponent comp ) {
		return components.indexOf( comp );
	}

	/**
	 * Adds a form component to the end of this form container.
	 */
	public void add( FormComponent comp ) {
		add( comp, null );
	}

	/**
	 * Adds a form component to this form container at the specified position.
	 */
	public void add( FormComponent comp, int index ) {
		add( comp, null, index );
	}

	/**
	 * Adds a form component to the end of this form container and
	 * sets the specified constraints in the form layout manager of this container.
	 */
	public void add( FormComponent comp, FormLayoutConstraints constraints ) {
		add( comp, constraints, -1 );
	}

	/**
	 * Adds a form component to this form container at the specified position and
	 * sets the specified constraints in the form layout manager of this container.
	 *
	 * @param comp The form component to be added.
	 * @param constraints The form layout constraints for the form component.
	 * @param index The position in the container's list at which to insert the
	 * 		component; or -1 to insert at the end
	 */
	public void add( FormComponent comp, FormLayoutConstraints constraints, int index ) {
		if( comp == this )
			throw new IllegalArgumentException(); // can not add component to itself

		FormContainer compParent = comp.getParent();
		if( compParent == this )
			return;
		else if( compParent != null )
			compParent.remove( comp );

		if( index < 0 || index == components.size() ) {
			index = components.size();
			components.add( comp );
		} else
			components.add( index, comp );

		comp.setParent( this );
		comp.setModel( model );

		if( layout != null )
			layout.setConstraints( comp, constraints, false );

		if( model != null ) {
			if( model.nameRegistry != null )
				model.nameRegistry.componentAdded( comp );
			if( model.eventProvider != null )
				model.eventProvider.fireComponentAdded( comp, this, index, constraints );
		}
	}

	/**
	 * Removes the specified form component from this form container.
	 */
	public void remove( FormComponent comp ) {
		int index = components.indexOf( comp );
		if( index >= 0 )
			remove( index );
	}

	/**
	 * Removes the form component at the specified index from this form container.
	 */
	public void remove( int index ) {
		FormComponent comp = components.remove( index );
		comp.setParent( null );
		comp.setModel( null );

		FormLayoutConstraints constraints = null;
		if( layout != null )
			constraints = layout.setConstraints( comp, null, false );

		if( model != null ) {
			if( model.nameRegistry != null )
				model.nameRegistry.componentRemoved( comp );
			if( model.eventProvider != null )
				model.eventProvider.fireComponentRemoved( comp, this, index, constraints );
		}
	}

	/**
	 * Removes the form component with the specified name from this form container.
	 */
	public void remove( String name ) {
		int count = components.size();
		for( int i = 0; i < count; i++ ) {
			if( name.equals( getComponent(i).getName() ) ) {
				remove( i );
				break;
			}
		}
	}

	/**
	 * Removes add children from this form container.
	 */
	public void removeAll() {
		for( int i = components.size() - 1; i >= 0; i-- )
			remove( i );
	}

	/**
	 * Returns the menu bar of this form container.
	 *
	 * @since 2.0
	 */
	public FormContainer getMenuBar() {
		return menuBar;
	}

	/**
	 * Sets the menu bar of this form container.
	 *
	 * @since 2.0
	 */
	public void setMenuBar( FormContainer menuBar ) {
		if( this.menuBar == menuBar )
			return;

		if( this.menuBar != null ) {
			this.menuBar.menuBarOwner = null;
			this.menuBar.setModel( null );
		}

		FormComponent oldMenuBar = this.menuBar;
		this.menuBar = menuBar;

		if( menuBar != null ) {
			menuBar.menuBarOwner = this;
			menuBar.setModel( model );
		}

		if( model != null ) {
			if( model.nameRegistry != null )
				model.nameRegistry.componentFieldChanged( this, FIELD_MENU_BAR, oldMenuBar, menuBar );
			if( model.eventProvider != null )
				model.eventProvider.fireComponentFieldChanged( this, FIELD_MENU_BAR, oldMenuBar, menuBar );
		}
	}

	/**
	 * Returns the owner of a menu bar.
	 *
	 * @since 2.0
	 */
	public FormContainer getMenuBarOwner() {
		return menuBarOwner;
	}

	@Override
	void setModel( FormModel model ) {
		super.setModel( model );

		if( menuBar != null )
			menuBar.setModel( model );

		// set model also in all children
		int count = components.size();
		for( int i = 0; i < count; i++ )
			components.get(i).setModel( model );
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormComponentVisitor#visit} is called
	 * with this form container, with the menu bar of this form container and
	 * with all children of this form container.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormComponentVisitor#visit}.
	 * @since 3.0
	 */
	@Override
	public boolean accept( FormComponentVisitor visitor ) {
		if( !super.accept( visitor ) )
			return false;

		if( menuBar != null && !menuBar.accept( visitor ) )
			return false;

		int count = components.size();
		for( int i = 0; i < count; i++ ) {
			if( !components.get(i).accept( visitor ) )
				return false;
		}
		return true;
	}

	/**
	 * Returns the menu bar and all form components in this form container.
	 *
	 * @since 2.0
	 */
	public FormComponent[] getMenuBarAndComponents() {
		FormComponent[] comps = new FormComponent[components.size() + (menuBar != null ? 1 : 0)];
		components.toArray( comps );
		if( menuBar != null ) {
			System.arraycopy( comps, 0, comps, 1, components.size() );
			comps[0] = menuBar;
		}
		return comps;
	}

	@Override
	void updateReferences( String oldName, String newName ) {
		super.updateReferences( oldName, newName );

		if( menuBar != null )
			menuBar.updateReferences( oldName, newName );

		int count = components.size();
		for( int i = 0; i < count; i++ )
			components.get(i).updateReferences( oldName, newName );
	}
}
