/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.ArrayList;

/**
 * A form root is a container for all top-level components in a form model.
 *
 * @author Karl Tauber
 */
public class FormRoot
	extends FormContainer
{
	private static final FormBindingGroup[] EMPTY_BINDING_GROUPS = {};

	private ArrayList<FormBindingGroup> bindingGroups;

	/**
	 * Constructs a form root.
	 */
	public FormRoot() {
		super( (String) null, new FormLayoutManager( null ) );
	}

	FormRoot( FormRoot obj, int dummy ) {
		super( obj, dummy );
	}

	/**
	 * Clones this form root.
	 */
	@Override
	public Object clone() {
		return new FormRoot( this, 0 );
	}

	/**
	 * Returns the number of form binding groups in this form root.
	 *
	 * @since 5.0
	 */
	public int getBindingGroupCount() {
		return (bindingGroups != null) ? bindingGroups.size() : 0;
	}

	/**
	 * Returns the form binding group at <code>index</code>.
	 *
	 * @since 5.0
	 */
	public FormBindingGroup getBindingGroup( int index ) {
		return bindingGroups.get( index );
	}

	/**
	 * Returns the form binding group for the given <code>name</code>.
	 *
	 * @since 5.0
	 */
	public FormBindingGroup getBindingGroup( String name ) {
		if( bindingGroups == null )
			return null;

		for( FormBindingGroup bindingGroup : bindingGroups ) {
			if( name.equals( bindingGroup.getName() ) )
				return bindingGroup;
		}
		return null;
	}

	/**
	 * Returns all form binding groups in this form root.
	 *
	 * @since 5.0
	 */
	public FormBindingGroup[] getBindingGroups() {
		return (bindingGroups != null)
			? bindingGroups.toArray( new FormBindingGroup[bindingGroups.size()] )
			: EMPTY_BINDING_GROUPS;
	}

	/**
	 * Returns the index of <code>bindingGroup</code> in this form;
	 * or -1 if the binding group is not a child of this form.
	 */
	public int getBindingGroupIndex( FormBindingGroup bindingGroup ) {
		return bindingGroups.indexOf( bindingGroup );
	}

	/**
	 * Adds a form binding group to the end of this form root.
	 *
	 * @since 5.0
	 */
	public void addBindingGroup( FormBindingGroup bindingGroup ) {
		addBindingGroup( bindingGroup, -1 );
	}

	/**
	 * Adds a form binding group to this form root at the specified position.
	 *
	 * @param bindingGroup The form binding group to be added.
	 * @param index The position in the binding group list at which
	 * 		to insert the binding group; or -1 to insert at the end
	 * @since 5.0
	 */
	public void addBindingGroup( FormBindingGroup bindingGroup, int index ) {
		if( bindingGroups == null )
			bindingGroups = new ArrayList<FormBindingGroup>();

		if( index < 0 || index == bindingGroups.size() ) {
			index = bindingGroups.size();
			bindingGroups.add( bindingGroup );
		} else
			bindingGroups.add( index, bindingGroup );

		bindingGroup.setRoot( this );

		if( model != null && model.eventProvider != null )
			model.eventProvider.fireBindingGroupAdded( this, bindingGroup, index );
	}

	/**
	 * Removes the specified form binding group from this form root.
	 *
	 * @since 5.0
	 */
	public void removeBindingGroup( FormBindingGroup bindingGroup ) {
		if( bindingGroups == null )
			return;

		int index = bindingGroups.indexOf( bindingGroup );
		if( index >= 0 )
			removeBindingGroup( index );
	}

	/**
	 * Removes the form binding group at the specified index from this form root.
	 *
	 * @since 5.0
	 */
	public void removeBindingGroup( int index ) {
		if( bindingGroups == null )
			return;

		FormBindingGroup bindingGroup = bindingGroups.remove( index );
		bindingGroup.setRoot( null );

		if( model != null && model.eventProvider != null )
			model.eventProvider.fireBindingGroupRemoved( this, bindingGroup, index );
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormBindingVisitor#visit} is called
	 * with all form bindings of this form root.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormBindingVisitor#visit}.
	 * @since 5.0
	 */
	public boolean accept( FormBindingVisitor visitor ) {
		if( bindingGroups == null )
			return true;

		int count = bindingGroups.size();
		for( int i = 0; i < count; i++ ) {
			if( !bindingGroups.get(i).accept( visitor ) )
				return false;
		}
		return true;
	}

	@Override
	void updateReferences( String oldName, String newName ) {
		super.updateReferences( oldName, newName );

		if( bindingGroups != null ) {
			int count = bindingGroups.size();
			for( int i = 0; i < count; i++ )
				bindingGroups.get(i).updateReferences( oldName, newName );
		}
	}
}
