/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.Image;
import java.beans.*;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Karl Tauber
 */
public class BeanInfoEx
	implements BeanInfo
{
	private static final HashMap<String, PropertyDescriptor> EMPTY_MAP = new HashMap<String, PropertyDescriptor>( 1 );
	private static final HashMap<Class<?>, PersistenceDelegate> EMPTY_MAP2 = new HashMap<Class<?>, PersistenceDelegate>( 1 );

	private final BeanInfo beanInfo;
	private HashMap<String, PropertyDescriptor> propertiesMap;
	private HashMap<Class<?>, PersistenceDelegate> propertyPersistenceDelegatesMap;

	public BeanInfoEx( BeanInfo beanInfo ) {
		this.beanInfo = beanInfo;
	}

	public BeanDescriptor getBeanDescriptor() {
		return beanInfo.getBeanDescriptor();
	}

	public EventSetDescriptor[] getEventSetDescriptors() {
		return beanInfo.getEventSetDescriptors();
	}

	public int getDefaultEventIndex() {
		return beanInfo.getDefaultEventIndex();
	}

	public PropertyDescriptor[] getPropertyDescriptors() {
		return beanInfo.getPropertyDescriptors();
	}

	public int getDefaultPropertyIndex() {
		return beanInfo.getDefaultPropertyIndex();
	}

	public MethodDescriptor[] getMethodDescriptors() {
		return beanInfo.getMethodDescriptors();
	}

	public BeanInfo[] getAdditionalBeanInfo() {
		return beanInfo.getAdditionalBeanInfo();
	}

	public Image getIcon( int iconKind ) {
		return beanInfo.getIcon( iconKind );
	}

	public PropertyDescriptor getPropertyDescriptor( String name ) {
		if( propertiesMap == null ) {
			PropertyDescriptor[] descs = getPropertyDescriptors();
			if( descs != null && descs.length > 0 ) {
				propertiesMap = new HashMap<String, PropertyDescriptor>( descs.length );
				for( int i = 0; i < descs.length; i++ )
					propertiesMap.put( descs[i].getName(), descs[i] );
			} else
				propertiesMap = EMPTY_MAP;
		}
		return propertiesMap.get( name );
	}

	public EventSetDescriptor getEventSetDescriptor( String listener ) {
		EventSetDescriptor[] descs = getEventSetDescriptors();
		if( descs == null )
			return null;
		for( int i = 0; i < descs.length; i++ ) {
			if( listener.equals( descs[i].getListenerType().getName() ) )
				return descs[i];
		}
		return null;
	}

	public Map<Class<?>, PersistenceDelegate> getPersistenceDelegateMap() {
		if( propertyPersistenceDelegatesMap == null ) {
			PropertyDescriptor[] descs = getPropertyDescriptors();
			if( descs != null && descs.length > 0 ) {
				for( int i = 0; i < descs.length; i++ ) {
					Class<?> type = descs[i].getPropertyType();
					if( type == null )
						continue;
					Object delegate = descs[i].getValue( "persistenceDelegate" );
					Object extraDelegates = descs[i].getValue( "extraPersistenceDelegates" );
					if( delegate == null && extraDelegates == null )
						continue;

					// create map lazy
					if( propertyPersistenceDelegatesMap == null )
						propertyPersistenceDelegatesMap = new HashMap<Class<?>, PersistenceDelegate>();

					if( delegate != null ) {
						if( !(delegate instanceof PersistenceDelegate) )
							throwPropDescException( descs[i].getName(),
								"persistenceDelegate",
								"is not an instance of java.beans.PersistenceDelegate." );
						propertyPersistenceDelegatesMap.put( type, (PersistenceDelegate) delegate );
					}
					if( extraDelegates != null ) {
						if( !(extraDelegates instanceof Object[]) )
							throwPropDescException( descs[i].getName(),
								"extraPersistenceDelegates",
								"is not an Object[]." );

						Object[] extraArray = (Object[]) extraDelegates;
						if( extraArray.length % 2 != 0 )
							throwPropDescException( descs[i].getName(),
								"extraPersistenceDelegates",
								"size of Object[] is not a multiple of two." );

						for( int j = 0; j < extraArray.length; j += 2 ) {
							if( !(extraArray[j] instanceof Class<?>) )
								throwPropDescException( descs[i].getName(),
									"extraPersistenceDelegates",
									"Object[" + j + "] is not an instance of java.lang.Class." );
							if( !(extraArray[j+1] instanceof PersistenceDelegate) )
								throwPropDescException( descs[i].getName(),
									"extraPersistenceDelegates",
									"Object[" + (j+1) + "] is not an instance of java.beans.PersistenceDelegate." );

							propertyPersistenceDelegatesMap.put( (Class<?>) extraArray[j], (PersistenceDelegate) extraArray[j+1] );
						}
					}
				}
			}
			if( propertyPersistenceDelegatesMap == null )
				propertyPersistenceDelegatesMap = EMPTY_MAP2;
		}
		return propertyPersistenceDelegatesMap;
	}

	private void throwPropDescException( String propName, String attrName, String message ) {
		BeanDescriptor beanDesc = getBeanDescriptor();
		String className = (beanDesc != null) ? beanDesc.getBeanClass().getName() : "";
		throw new IllegalArgumentException( className + "." + propName
			+ ": Property descriptor attribute value \"" + attrName + "\" "
			+ message );
	}
}
