/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.util.StringTokenizer;

/**
 * @author Karl Tauber
 */
public class GridBagSpec
{
	public static final int CENTER	= AlignmentUtil.CENTER;
	public static final int TOP		= AlignmentUtil.TOP;
	public static final int LEFT	= AlignmentUtil.LEFT;
	public static final int BOTTOM	= AlignmentUtil.BOTTOM;
	public static final int RIGHT	= AlignmentUtil.RIGHT;
	public static final int FILL	= AlignmentUtil.FILL;
	public static final int BASELINE		= AlignmentUtil.BASELINE;
	public static final int ABOVE_BASELINE	= AlignmentUtil.ABOVE_BASELINE;
	public static final int BELOW_BASELINE	= AlignmentUtil.BELOW_BASELINE;

	private static final int DEFAULT_ALIGN = FILL;

    private int defaultAlignment;
	private int minimumSize;
	private double resizeWeight;

	public GridBagSpec() {
		this( DEFAULT_ALIGN, 0, 0.0 );
	}

	public GridBagSpec( int defaultAlignment, int minimumSize, double resizeWeight ) {
		this.defaultAlignment = defaultAlignment;
		this.minimumSize = minimumSize;
		this.resizeWeight = resizeWeight;
	}

	public GridBagSpec( String encodedDescription ) {
		this();

		StringTokenizer st = new StringTokenizer( encodedDescription, ":" );
		if( !st.hasMoreTokens() )
			throw new IllegalArgumentException( encodedDescription );

		int tokenCount = st.countTokens();
		String token = st.nextToken();

		if( tokenCount >= 2 ) {
			int alignment = AlignmentUtil.toValue( token );
			if( alignment >= 0 ) {
				defaultAlignment = alignment;
				token = st.nextToken();
			}
		}

		minimumSize = Integer.parseInt( token );

		if( st.hasMoreTokens() )
			resizeWeight = Double.parseDouble( st.nextToken() );
	}

	public int getDefaultAlignment() {
		return defaultAlignment;
	}

	public int getMinimumSize() {
		return minimumSize;
	}

	public double getResizeWeight() {
		return resizeWeight;
	}

	@Override
	public String toString() {
		StringBuffer buf = new StringBuffer();
		if( defaultAlignment != DEFAULT_ALIGN ) {
			buf.append( AlignmentUtil.toName( defaultAlignment ) );
			buf.append( ':' );
		}
		buf.append( minimumSize );
		if( resizeWeight != 0.0 ) {
			buf.append( ':' );
			buf.append( resizeWeight );
		}
		return buf.toString();
	}

	public static String toString( int alignment ) {
		return AlignmentUtil.toName( alignment );
	}

	public static GridBagSpec[] decodeSpecs( String encodedSpecs ) {
		if( encodedSpecs == null )
			return new GridBagSpec[0];

		StringTokenizer st = new StringTokenizer( encodedSpecs, ", " );
		GridBagSpec[] specs = new GridBagSpec[st.countTokens()];
		for( int i = 0; i < specs.length; i++ )
			specs[i] = new GridBagSpec( st.nextToken() );
		return specs;
	}

	public static String encodeSpecs( GridBagSpec[] specs ) {
		StringBuffer buf = new StringBuffer( specs.length * 9 );
		for( int i = 0; i < specs.length; i++ ) {
			if( i > 0 )
				buf.append( ", " );
			buf.append( specs[i].toString() );
		}
		return buf.toString();
	}
}
