/*
 * Copyright (C) 2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.Color;
import java.awt.Font;
import java.awt.Insets;
import java.lang.reflect.Field;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.border.MatteBorder;
import javax.swing.border.TitledBorder;

/**
 * @author Karl Tauber
 * @since 5.0.2
 */
public class RuntimeUtils
{
	public static final int defaultTitleJustification;
	public static final int defaultTitlePosition;

	static {
		// the default title position has changed in Java 6
		TitledBorder titledBorder = new TitledBorder( "" );
		defaultTitleJustification = titledBorder.getTitleJustification();
		defaultTitlePosition = getRawTitledBorderPosition( titledBorder );
	}

	public static TitledBorder cloneTitledBorder( TitledBorder titledBorder, String newTitle ) {
		return new TitledBorder( getRawTitledBorderBorder( titledBorder ), newTitle,
			titledBorder.getTitleJustification(), getRawTitledBorderPosition( titledBorder ),
			getRawTitledBorderFont( titledBorder ), getRawTitledBorderColor( titledBorder ) );
	}

	/**
	 * Handle default value returned by TitledBorder.getBorder().
	 */
	public static Border getRawTitledBorderBorder( TitledBorder border ) {
		Border titleBorder = border.getBorder();
		if( titleBorder == UIManager.getBorder( "TitledBorder.border" ) )
			titleBorder = null;
		return titleBorder;
	}

	/**
	 * Directly access protected field TitledBorder.titlePosition because
	 * TitledBorder.getTitlePosition() does some default handling in Java 6.
	 */
	public static int getRawTitledBorderPosition( TitledBorder border ) {
		try {
			return getFieldValueInt( border, TitledBorder.class, "titlePosition" );
		} catch( Exception ex ) {
			// fallback
			return border.getTitlePosition();
		}
	}

	/**
	 * Handle default value returned by TitledBorder.getTitleFont() in Java 6 and earlier.
	 */
	public static Font getRawTitledBorderFont( TitledBorder border ) {
		Font titleFont = border.getTitleFont();
		if( titleFont == UIManager.getFont( "TitledBorder.font" ) )
			titleFont = null;
		return titleFont;
	}

	/**
	 * Handle default value returned by TitledBorder.getTitleColor() in Java 6 and earlier.
	 */
	public static Color getRawTitledBorderColor( TitledBorder border ) {
		Color titleColor = border.getTitleColor();
		if( titleColor == UIManager.getColor( "TitledBorder.titleColor" ) )
			titleColor = null;
		return titleColor;
	}

	/**
	 * Directly access protected fields of EmptyBorder because MatteBorder
	 * overrides getBorderInsets() and modifies the insets.
	 */
	public static Insets getRawMatteBorderInsets( MatteBorder border ) {
		try {
			int top = getFieldValueInt( border, EmptyBorder.class, "top" );
			int left = getFieldValueInt( border, EmptyBorder.class, "left" );
			int bottom = getFieldValueInt( border, EmptyBorder.class, "bottom" );
			int right = getFieldValueInt( border, EmptyBorder.class, "right" );
			return new Insets( top, left, bottom, right );
		} catch( Exception ex ) {
			// fallback
			return border.getBorderInsets();
		}
	}

	private static int getFieldValueInt( Object object, Class<?> type, String fieldName )
		throws Exception
	{
		Field field = type.getDeclaredField( fieldName );
		field.setAccessible( true );
		return field.getInt( object );
	}
}
