/**
SpinButton component
Copyright IBM 1996, All rights reserved
@version 1.0
@author Philip Taunton
Created: 02/12/96
*/

import java.awt.*;

/**
A custom Java component that provides a numerical spin button. The contents
of the TextField can be incremented or decremented by selecting the up or
down arrows. The current value can be overtyped if the component is not
set to be readonly.
<p>
IMPORTANT NOTE: On Windows platforms there is currently a problem with
updating the TextField while spinning. The spin is still occurring, however
and the display will be updated when the mouse button is released (although
sometimes even this doesn't happen in which case a single click on the up
or down button will refresh the display). It is anticipated that this
problem will be resolved in Java version 1.1.
*/
public class SpinButton extends Panel
{
/**
Use this value to set the input style of the SpinButton to be numeric.
@see SpinButton#setInputType
*/
  public static final int SB_NUMERIC_ONLY=1;
//  public static final int SB_ALPHANUMERIC=2;


/**
Construct a SpinButton object. It will create a SpinButton with an
SB_NUMERIC_ONLY input style by default.
@see SpinButton#SB_NUMERIC_ONLY
*/
  public SpinButton()
  {
    input_mode=SB_NUMERIC_ONLY;           // default to numeric type
    real_constructor();
  }

/**
Construct a SpinButton object with the specified input type.
@param input_type The input type to set for the SpinButton.
@exception IllegalArgumentException Invalid SpinButton input type.
*/
  public SpinButton(int input_type)
  {
    setInputType(input_type);             // set the input type
    real_constructor();                   // call the common constructor
  }

  private void real_constructor()
  {
    read_only=false;                      // default to editable
    int_value=0;                          // initial spin value is zero
    min_int=-2147483648;                  // initial minimum value
    max_int=2147483647;                   // initial maximum value
  	edit_area = new TextField();					// create the text field
    scroll=new Scrollbar(Scrollbar.VERTICAL,5,1,0,10);
    this.setLayout(null);
    this.add(edit_area);
    this.add(scroll);
    showValue();
    show();
  }

/**
Changes the permitted input type for the SpinButton.
@see SpinButton#SB_NUMERIC_ONLY
@param input_type The input type to set for the SpinButton.
@exception IllegalArgumentException Invalid SpinButton input type.
*/
  public void setInputType(int input_type)
  {
    if (input_type!=SB_NUMERIC_ONLY)      // validate the input type
      throw new IllegalArgumentException("Invalid SpinButton input type");
    input_mode=input_type;                // set mode to that supplied
  }

/**
Returns the data type permitted in the SpinButton.
@see SpinButton#SB_NUMERIC_ONLY
@return The current data type.
*/
  public int inputTypeAllowed()
  {
    return(input_mode);                   // just return the value
  }

/**
Changes the current editable mode of the SpinButton. A value of true will
make the field read only. A value of false will make it editable.
@param r_only The edit mode to set.
*/
  public void setReadOnly(boolean r_only)
  {
    read_only=r_only;                     // set local variable
    edit_area.setEditable(read_only);     // now set the text field
  }

/**
Returns the current editable setting for the SpinButton. A value of true
means the field is read only. A value of false means the field is editable.
@return The current edit mode.
*/
  public boolean isReadOnly()
  {
    return(read_only);                    // just return the local value
  }

/**
Changes the current minimum and maximum values for the spin range. This method
is only valid for integer SpinButtons of the type SB_NUMERIC_ONLY.
@param lower The lower limit of the spin range.
@param upper The upper limit of the spin range.
@exception RuntimeException SpinButton not set to valid input type.
*/
  public void setRange(int lower, int upper)
  {
    if (input_mode!=SB_NUMERIC_ONLY)
      throw new RuntimeException("Cannot set SpinButton range if not numeric");
    min_int=lower;
    max_int=upper;
  }

/**
Returns the current setting for the lower range value. This method is only
valid for integer SpinButtons of the type SB_NUMERIC_ONLY.
@exception RuntimeException SpinButton not set to valid input type.
@return The current lower range value.
*/
  public int getLower()
  {
    if (input_mode!=SB_NUMERIC_ONLY)
      throw new RuntimeException("Cannot get SpinButton range if not numeric");
    return(min_int);                      // just return the value
  }

/**
Returns the current setting for the upper range value. This method is only
valid for integer SpinButtons of the type SB_NUMERIC_ONLY.
@exception RuntimeException SpinButton not set to valid input type.
@return The current upper range value.
*/
  public int getUpper()
  {
    if (input_mode!=SB_NUMERIC_ONLY)
      throw new RuntimeException("Cannot get SpinButton range if not numeric");
    return(max_int);                      // just return the value
  }

/**
Spins the SpinButton up by the specified amount. If the spin up value would
cause the spin value to be larger than the maximum value, then the
spin button will be set to the maximum value. Negative numbers
are ignored.
@param spin_by The number to spin the button up by.
*/
  public void spinUp(int spin_by)
  {
    if (spin_by>0)
    {
      if (key_press==1)
      {                                   // check for manual update first
        Integer it=new Integer(edit_area.getText()); // get new value
        int_value=it.intValue();
        key_press=0;                      // reset flag
      }
      int_value+=spin_by;
      if (int_value>max_int)
        int_value=max_int;                // spin maxes out the value
      showValue();                        // update spin
    }
  }

/**
Spins the SpinButton down by the specified amount. If the spin down value would
cause the spin value to be lower than the minimum value, then the
spin button will be set to the minimum value. Negative numbers
are ignored.
@param spin_by The number to spin the button down by.
*/
  public void spinDown(int spin_by)
  {
    if (spin_by>0)
    {
      if (key_press==1)
      {                                   // check for manual update first
        Integer it=new Integer(edit_area.getText()); // get new value
        int_value=it.intValue();
        key_press=0;                      // reset flag
      }
      int_value-=spin_by;
      if (int_value<min_int)
        int_value=min_int;                // spin minimises the value
      showValue();                        // update spin
    }
  }

/**
Enables the SpinButton for input.
*/
  public void enable()
  {
    edit_area.enable();                   // ensure all sub-objects get request
    scroll.enable();
  }

/**
Enables or disables the SpinButton for input depending on the boolean
condition. A Value of true will enable the component, a value of false
will disable the component.
@param cond Enable (true) or disable (false) the component.
*/
  public void enable(boolean cond)
  {
    edit_area.enable(cond);               // ensure all sub-objects get request
    scroll.enable(cond);
  }

/**
Disables the SpinButton from input.
*/
  public void disable()
  {
    edit_area.disable();                  // ensure all sub-objects get request
    scroll.disable();
  }

/**
Changes the current position and size of the SpinButton. The two buttons
will expand vertically to fill any space, but are of fixed width. The
TextField will expand both horizontally and vertically to fill the remaining
space.
@param x The new position on the x axis.
@param y The new position on the y axis.
@param width The new width.
@param height The new height.
*/
  public synchronized void reshape(
  	int x,
    int y,
    int width,
    int height)
  {
  	super.reshape(x,y,width,height);      // let the panel reshape first
    edit_area.reshape(0,0,width-BUTTON_WIDTH,height); // edit area on left
    scroll.reshape(width-BUTTON_WIDTH,0,BUTTON_WIDTH,height);  // buttons on right
  }

/**
Returns the recommended minimum size for the SpinButton. Used by layout
managers to determine the screen dimensions.
@return A Dimension containing the minimum width and height.
*/
  public Dimension minimumSize()
  {
    return new Dimension(2*BUTTON_WIDTH,2*MIN_BUTTON_HEIGHT);
  }

/**
Returns the preferred size for the SpinButton. Used by layout
managers to determine the screen dimensions.
@return A Dimension containing the preferred width and height.
*/
  public Dimension preferredSize()
  {
    return(super.preferredSize());
  }

/**
Sets the current numeric value of the SpinButton regardless of the
validity of the number.
@param new_value The new number to put in the field.
*/
  public void setValue(int new_value)
  {
    int_value=new_value;                  // store new value
    key_press=0;                          // reset any key press flag
    showValue();                          // and display it
  }

/**
Sets the current text value of the SpinButton regardless of the
validity of the data. NOTE: This is a placeholder function until
alphanumeric support is added.
@param new_value The new text to put in the field.
*/
  public void setValue(String new_value)
  {
    key_press=0;                          // reset any key press flag
    edit_area.setText(new_value);
  }

/**
Get the current numeric value of the SpinButton.
@return the current value of the SpinButton
*/
  public int intValue()
  {
    if (key_press==1)
    {                                 // check for manual update first
      Integer it=new Integer(edit_area.getText()); // get new value
      int_value=it.intValue();
      key_press=0;                    // reset flag
    }
    return(int_value);
  }

/** 
Handles scroll and keyboard events for the SpinButton.
@return True or false depending on whether the event has been handled or not.
*/
  public boolean handleEvent(Event event)
  {
    int updated;
    if (event.id==Event.SCROLL_LINE_UP)
    {
      if (input_mode==SB_NUMERIC_ONLY)
      {
        if (key_press==1)
        {
          Integer it=new Integer(edit_area.getText()); // get new value
          int_value=it.intValue();
          key_press=0;                    // reset flag
        }
        int_value+=1;                     // increment spin value
        if (int_value>max_int)
          int_value=min_int;              // loop back to minimum value
        showValue();                      // update spin
        scroll.setValue(5);               // make sure scroll bar doesn't run out of space
      }
      return true;
    }
    else if (event.id==Event.SCROLL_LINE_DOWN)
    {
      if (input_mode==SB_NUMERIC_ONLY)
      {
        if (key_press==1)
        {
          Integer it=new Integer(edit_area.getText()); // get new value
          int_value=it.intValue();
          key_press=0;                    // reset flag
        }
        int_value-=1;                     // decrement spin value
        if (int_value<min_int)
          int_value=max_int;              // loop back to maximum value
        showValue();                      // update spin
        scroll.setValue(5);               // make sure scroll bar doesn't run out of space
      }
      return true;
    }
    else if (event.target==edit_area && event.id==Event.KEY_PRESS)
    {
      if (event.key>=0x20)
      {                                   // only interested in non-space chars
        if (input_mode==SB_NUMERIC_ONLY)
        {
          if (valid_numeric.indexOf(event.key)==-1)
            return true;                  // not a valid key - ignore
          else
          {
            if (event.key=='-' && edit_area.getText().length()>0)
              return true;                // minus not valid if not first char
            else
              key_press=1;                // flag a valid keypress
          }
        }
      }
    }
    return super.handleEvent(event);
  }

  private void showValue()
  {
    String value=Integer.toString(int_value);
    edit_area.setText(value);
  }

	private TextField edit_area;            // the viewing area
  private static final int BUTTON_WIDTH=15;   // actual fixed button width
  private static final int MIN_BUTTON_HEIGHT=10;  // minimum height
  private static final int PREFERRED_HEIGHT=25;   // recommended height
  private int input_mode;                 // the current spin type
  private int int_value;                  // integer value for numeric spin
  private int min_int, max_int;           // minimum and maximum spin values
  private boolean read_only;              // editable status
  private Scrollbar scroll;
  private String valid_numeric="-0123456789";
  private int key_press=0;                // flag if valid keypress

} // end class SpinButton

