/* ContainerItem: This class represents an item which may be displayed */
/*    with the DataContainer class.                                    */

package extend.awt;

import java.awt.Image;

public class ContainerItem {

   public ContainerItem (Image picture, String text) {

      parent = null;
      nextSibling = null;
      previousSibling = null;
      firstChild = null;
      data = null;
      label = text;
      columns = null;
      expanded = false;
      icon = picture;
   }

   public void addBefore (ContainerItem item) {

      this.nextSibling = item;
      this.previousSibling = item.previousSibling;
      item.previousSibling = this;
      
      if (this.previousSibling != null)
        this.previousSibling.nextSibling = this;
   }

   public void moveBefore (ContainerItem item) {

      remove ();
      addBefore (item);
   }

   public void addAfter (ContainerItem item) {

      this.previousSibling = item;
      this.nextSibling = item.nextSibling;
      item.nextSibling = this;

      if (this.nextSibling != null)
         this.nextSibling.previousSibling = this;
   }

   public void moveAfter (ContainerItem item) {

      remove ();
      addAfter (item);
   }

   public void addToFront (ContainerItem item) {

      ContainerItem first = item.getFirst ();

      first.previousSibling = this;
      this.nextSibling = first;
      this.previousSibling = null;
   }

   public void moveToFront (ContainerItem item) {

      remove ();
      addToFront (item);
   }

   public void addToEnd (ContainerItem item) {

      ContainerItem last = item.getLast ();

      last.nextSibling = this;
      this.previousSibling = last;
      this.nextSibling = null;
   }

   public void moveToEnd (ContainerItem item) {

      remove ();
      addToEnd (item);
   }

   public void setChildList (ContainerItem first) {

      firstChild = first.getFirst ();

      first.parent = this;
   }

   public void remove () {

      ContainerItem item;
      ContainerItem hold;

      if (firstChild != null) {

         item = firstChild.getFirst ();

         while (item != null) {

            hold = item.getNext ();
            item.remove ();
            item = hold;
         }
      }

      hold = null;

      if (nextSibling != null) {

         nextSibling.previousSibling = previousSibling;
         hold = nextSibling;
      }

      if (previousSibling != null) {

         previousSibling.nextSibling = nextSibling;
         hold = previousSibling;
      }

      if (hold != null)
         hold = hold.getFirst ();

      if (parent != null)
         parent.firstChild = hold;

      nextSibling = null;
      previousSibling = null;
   }

   public String textFor () {return (label);}

   public void setData (Object ref) {data = ref;}

   public Object getData () {return (data);}

   public void setColumn (int which, Object data) {

      Object [] newOne = null;

      if (columns == null)
         newOne = new Object [which + 1];
      else if (which >= columns.length)
         newOne = new Object [which + 1];

      if (newOne != null) {

         if (columns != null)
            System.arraycopy (columns, 0, newOne, 0, columns.length);

         columns = newOne;
      }

      columns [which] = data;
   }

   public int itemCount () {

      int count = 0;
      ContainerItem work = getFirst ();

      while (work != null) {

         count++;
         work = work.nextSibling;
      }

      return count;
   }

   public int columnCount () {

      return ((columns == null) ? 0 : columns.length);
   }

   public Object getColumn (int which) {

      if (columns == null)
         throw new IllegalArgumentException ("No such column: " + which);

      if (which < 0 || which >= columns.length)
         throw new IllegalArgumentException ("No such column: " + which);

      return columns [which];
   }

   public Object [] getColumns () {return columns;}

   public ContainerItem getFirst () {

      ContainerItem work = this;

      while (work.previousSibling != null)
         work = work.previousSibling;

      return work;
   }

   public ContainerItem getNext () {return nextSibling;}

   public ContainerItem getPrevious () {return previousSibling;}

   public ContainerItem getLast () {

      ContainerItem work = this;

      while (work.nextSibling != null)
         work = work.nextSibling;

      return work;
   }

   public ContainerItem getItem (int which) {

      ContainerItem work = getFirst ();

      while (which > 0 && work != null) {

         work = work.nextSibling;
         which--;
      }

      if (work == null)
         throw new IllegalArgumentException ("No such item");

      return work;
   }

   public ContainerItem getParent () {

      ContainerItem result = getFirst ();
      boolean notDone = true;

      while (notDone) {
         
         if (result.parent == null) {

            result = result.nextSibling;
            notDone = (result != null);

         } else {

            result = result.parent;
            notDone = false;
         }
      }

      return result;
   }

   public ContainerItem getFirstChild () {

      ContainerItem result = firstChild;

      if (firstChild != null)
         result = result.getFirst ();

      return result;
   }

   public void setExpanded (boolean newValue) {expanded = newValue;}

   public boolean isExpanded () {return expanded;}

   public void setSelected (boolean newValue) {selected = newValue;}

   public boolean isSelected () {return selected;}

   public void setImage (Image newIcon) {icon = newIcon;}

   public Image getImage () {return icon;}

   public void traverse (ItemProcessor processor, int tag) {

      ContainerItem item = this.getFirst ();
      int level = 0;

      while (item != null) {

         if (processor.processItem (item, tag)) {

            if (item.getFirstChild () != null) {

               item = item.getFirstChild ();
               level++;

            } else if (item.getNext () != null)
               item = item.getNext ();
            else {
               item = item.getParent ();
               level--;
               boolean notDone = (item != null && level >= 0);

               while (notDone) {

                  if (item.getNext () == null) {

                     item = item.getParent ();
                     level--;
                     notDone = (item != null && level >= 0);

                  } else {
        
                     item = item.getNext ();
                     notDone = false;
                  }
               }

               if (level < 0)
                  item = null;
            }

         } else
            item = null;
      }
   }

   private ContainerItem parent;
   private ContainerItem nextSibling;
   private ContainerItem previousSibling;
   private ContainerItem firstChild;
   private Object data;
   private String label;
   private Object [] columns;
   private boolean expanded;
   private boolean selected;
   private Image icon;
}
