/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Gravity.SINK;
import static javoids.ImageMap.Name.NO_IMAGE;
import static javoids.Shapes.Shape.POINT;

import java.awt.Color;
import java.awt.geom.Area;
import java.util.Vector;

/* GravityPoint--------------- */
/**
 * A single point of gravity that attract's other sprites.
 * @author mallette
 */
public class GravityPoint extends Sprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long   serialVersionUID        = 1L;
  /** The maximum number of these sprites allowed in the game */
  protected final static int  MAX_NUMBER              = 20;
  /** the maximum numbewr of lives */
  protected final static int  MAX_DEATHS              = 1;
  /** the maximum damage amount */
  protected final static int  MAXIMUM_DAMAGE_CAPACITY = Integer.MAX_VALUE;
  /** the maximum duration the sprite will be in the game */
  protected final static int  MAX_DURATION            = 2000;
  /** the default duration */
  protected final static int  DEFAULT_DURATION        = 1000;
  /** the default aging rate */
  protected final static int  DEFAULT_AGING_RATE      = 1;
  /** the sprite's maximum size */
  protected final static int  MAX_SIZE                = 100;
  /** the sprite's minimum size */
  protected final static int  MIN_SIZE                = 10;
  /** the sprite's default size */
  protected final static int  DEFAULT_SIZE            = 20;               // This should match the size of the image!
  /** the mass of the singularity (different than a normal sprite) */
  private final static double mass                    = 5.0 / GravityPoint.MAX_NUMBER;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   */
  public GravityPoint(BasicSprite parent,Health health,Move move,int _size)
  {
    super(parent,health,move,_size,NO_IMAGE,new Vector<Area>(),new Vector<Color>());
    this.setGravity(SINK);
    this.getMove().setMaxVelocity(0);
    this.getAreas().clear();
    this.getAreas().addAll(Shapes.getAreas(POINT,this.getSize(),GravityPoint.MAX_SIZE));
    this.setDisplayAreas(true);
    this.getColors().clear();
    if (Debug.debug)
      this.getColors().add(new Color(255,255,255,127));
    else
      this.getColors().add(new Color(127,127,127,127));
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return GravityPoint.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return GravityPoint.MIN_SIZE;
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return GravityPoint.DEFAULT_SIZE;
  }

  /**
   * @param _size the size to set the sprite's size to
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    super.setSize(_size >= GravityPoint.MIN_SIZE ? _size <= GravityPoint.MAX_SIZE ? _size : GravityPoint.MAX_SIZE : GravityPoint.MIN_SIZE);
  }

  /**
   * @return the name of the image this sprite uses
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? NO_IMAGE : this.getOriginalImage();
  }

  /**
   * @return the mass of the singularity
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#getMass()
   */
  @Override
  public double getMass()
  {
    return GravityPoint.mass;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("GravityPoint.ToString"),super.toString(),Double.valueOf(GravityPoint.mass)); //$NON-NLS-1$
  }
}
/* GravityPoint--------------- */
