/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.ImageMap.Name.NO_IMAGE;
import static javoids.SoundMap.Sound.NO_SOUND;

import java.io.Serializable;
import java.util.HashMap;

/* ItemBase------------------- */
/**
 * A class to represent an item in the game (weapon, or other usable tool)
 * @author mallette
 */
public class Item implements Serializable
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;

  /**
   * An enumeration of directions that you can travel along the weapon list (left, right or choose best).
   * @author mallette
   */
  public enum Direction
  {
    /** Cycle left through the list */
    LEFT,
    /** Cycle right through the list */
    RIGHT,
    /** choose the best from the list */
    BEST;
  }

  /**
   * The type of item. This is currently just the name of the item.
   * @author mallette
   */
  public enum Type implements Serializable
  {
    /** placeholder when no item is selected */
    NO_TOOL,
    /** the weakest gun */
    GUN1,
    /** a stronger gun, but fires less shots */
    GUN2,
    /** the strongest regular gun, but fires the fewest shots */
    GUN3,
    /** weakest machine gun */
    MACHINEGUN1,
    /** regular machine gun */
    MACHINEGUN2,
    /** strongest machine gun */
    MACHINEGUN3,
    /** weakest machine gun that fires multiple bullets two at a time (in front) */
    MULTIGUN1,
    /** regular machine gun that fires multiple bullets three at a time */
    MULTIGUN2,
    /** strongest front firing machine gun that fires multiple bullets five at a time */
    MULTIGUN3,
    /** weak machinegun gun that fires multiple bullets at a time in the 4 cardinal directions */
    MULTIGUN4,
    /** strongest machinegun gun that fires multiple bullets at a time in the 8 cardinal directions */
    MULTIGUN5,
    /** strongest machine gun that fires multiple bullets at a time in all directions but at reduced strength */
    MULTIGUN6,
    /** fires a ssmall number of rockets */
    ROCKET1,
    /** fires a lot of rockets in rapid succession */
    ROCKET2,
    /** a shield */
    SHIELD,
    /** extra movement power for a sprite */
    AFTERBURNER,
    /** hyperspace jump (teleport to new location) */
    JUMP,
    /** bomb that breaks apart larger javoids into smaller parts (or destroys the smallest) */
    BOMB1,
    /** bomb that destroyes everything except the player */
    BOMB2;
  }

  /** the type of item (the item name) */
  private Type           type;
  /** the text version of the item name */
  private String         name;
  /** is the item limitless? */
  private boolean        limitless;
  /** the default image to use for drawing */
  private ImageMap.Name  defaultImage = NO_IMAGE;
  /** the current image to use for drawing */
  private ImageMap.Name  image        = NO_IMAGE;
  /** the sound to play when used */
  private SoundMap.Sound sound        = NO_SOUND;
  /** the maximum number of uses */
  private int            maximumCount;
  /** the number of uses left */
  private int            count;
  /** the default number of uses */
  private int            defaultCount;
  /** the size of the item (for drawing) */
  private int            size;

  /**
   * Constructor (default)
   */
  public Item()
  {
    this.reset();
  }

  /**
   * Constructor (copy)
   * @param item the item to copy
   */
  public Item(Item item)
  {
    this.reset();
    this.setType(item.getType());
    this.setName(item.getName());
    this.setLimitless(item.isLimitless());
    this.setDefaultImage(item.getDefaultImage());
    this.setImage(item.getImage());
    this.setSound(item.getSound());
    this.setMaximumCount(item.getMaximumCount());
    this.setCount(item.getCount());
    this.setDefaultCount(item.getDefaultCount());
    this.setSize(item.getSize());
  }

  /**
   * @return the text representation of the item name
   */
  public String getName()
  {
    return this.name;
  }

  /**
   * @return the item type
   */
  public Type getType()
  {
    return this.type;
  }

  /**
   * @return is the item limitless?
   */
  public boolean isLimitless()
  {
    return this.limitless;
  }

  /**
   * @return the current item the item is using
   */
  public ImageMap.Name getImage()
  {
    return this.image;
  }

  /**
   * @return the default image this item uses
   */
  public ImageMap.Name getDefaultImage()
  {
    return this.defaultImage;
  }

  /**
   * @return the sound that this item makes when used
   */
  public SoundMap.Sound getSound()
  {
    return this.sound;
  }

  /**
   * @return the current number of uses left for this item
   */
  public int getCount()
  {
    return this.count;
  }

  /**
   * @return the default number of uses for this item
   */
  public int getDefaultCount()
  {
    return this.defaultCount;
  }

  /**
   * @return the maximum number of uses for this item
   */
  public int getMaximumCount()
  {
    return this.maximumCount;
  }

  /**
   * @return the size of the image used to represent this item
   */
  public int getSize()
  {
    return this.size;
  }

  /**
   * @param _name set the item name to this
   */
  public void setName(String _name)
  {
    this.name = _name;
  }

  /**
   * @param _type set the type of the item
   */
  public void setType(Type _type)
  {
    if (_type.ordinal() >= Type.GUN1.ordinal() && _type.ordinal() <= Type.BOMB2.ordinal())
      this.type = _type;
    else
      System.out.printf(Messages.getString("Item.ItemMissing"),_type); //$NON-NLS-1$
  }

  /**
   * @param _limitless set the item to limitless (true/false)
   */
  public void setLimitless(boolean _limitless)
  {
    this.limitless = _limitless;
  }

  /**
   * @param _image set the current image this item is using
   */
  public void setImage(ImageMap.Name _image)
  {
    this.image = _image;
  }

  /**
   * @param _defaultImage set the default image this item uses
   */
  public void setDefaultImage(ImageMap.Name _defaultImage)
  {
    this.defaultImage = _defaultImage;
  }

  /**
   * @param _sound set the sound this item makes
   */
  public void setSound(SoundMap.Sound _sound)
  {
    this.sound = _sound;
  }

  /**
   * @param _count set the current number of uses left to this value
   */
  public void setCount(int _count)
  {
    this.count = _count > 0 ? _count <= this.maximumCount ? _count : this.maximumCount : 0;
  }

  /**
   * @param _defaultCount set the default number of uses to this value
   */
  public void setDefaultCount(int _defaultCount)
  {
    this.defaultCount = _defaultCount > 0 ? _defaultCount : 0;
  }

  /**
   * @param _maximumCount set the maximum number of uses to this value
   */
  public void setMaximumCount(int _maximumCount)
  {
    this.maximumCount = _maximumCount > 0 ? _maximumCount : 0;
  }

  /**
   * @param _size set the size of the image to represent this item
   */
  public void setSize(int _size)
  {
    this.size = _size > 0 ? _size : 0;
  }

  /**
   * @param _count modify the number of uses by this amount (if not limitless)
   */
  public void modifyCount(int _count)
  {
    if (!this.isLimitless())
      this.setCount(this.getCount() + _count);
  }

  /**
   * reset the item's information to limited and a default count
   */
  public void reset()
  {
    this.limitless = false;
    this.count = this.defaultCount;
  }

  /**
   * @return true if the current item is a weapon
   */
  public boolean isWeapon()
  {
    return Item.isWeapon(this.type);
  }

  /**
   * @param item the item to test for weapon status
   * @return whether the item is a weapon or not
   */
  public static boolean isWeapon(Type item)
  {
    return item.ordinal() >= Type.GUN1.ordinal() && item.ordinal() <= Type.ROCKET2.ordinal();
  }

  /**
   * Determine what weapon to select next.
   * @param items a mapping of item types to items
   * @param currentItem the currently selected item
   * @param direction the direction to go through the list (left or right)
   * @return the newly selected item
   */
  public static Item.Type cycleWeapon(HashMap<Item.Type,Item> items,Item.Type currentItem,Direction direction)
  {
    Item.Type item = currentItem;
    switch (direction)
    {
      case RIGHT :
        do
        {
          item = Type.ROCKET2.equals( item ) ? Type.GUN1 : Type.values()[item.ordinal() + 1];
        }
        while (items.get(item).getCount() == 0 && !item.equals( currentItem )); // there must be at least two types of weapons to work
        break;
      case LEFT :
        do
        {
          item = Type.GUN1.equals( item ) ? Type.ROCKET2 : Type.values()[item.ordinal() - 1];
        }
        while (items.get(item).getCount() == 0 && !item.equals( currentItem )); // there must be at least two types of weapons to work
        break;
      default :
        break;
    }
    return Item.setWeapon(item);
  }

  /**
   * @param item the weapon item type to set
   * @return the item set, or NO_ITEM if it is anything other than a weapon item.
   */
  public static Item.Type setWeapon(Item.Type item)
  {
    Item.Type returnValue;
    if (item.ordinal() >= Type.GUN1.ordinal() && item.ordinal() <= Type.ROCKET2.ordinal())
      returnValue = item;
    else
    {
      System.out.printf(Messages.getString("Item.ItemMissingSetWeapon"),item); //$NON-NLS-1$
      returnValue = Type.GUN1;
    }
    return returnValue;
  }

  /**
   * @param item the non-weapon item type to set
   * @return the item set, or NO_ITEM if it is anything other than a non-weapon item.
   */
  public Item.Type setItem(Item.Type item)
  {
    Item.Type returnValue;
    if (item.ordinal() >= Type.SHIELD.ordinal() && item.ordinal() <= Type.BOMB2.ordinal())
      returnValue = item;
    else
    {
      System.out.printf(Messages.getString("Item.ItemMissingSetWeapon"),item); //$NON-NLS-1$
      returnValue = Type.NO_TOOL;
    }
    return returnValue;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Item.ToString"),this.type,this.name,Boolean.valueOf(this.limitless),this.defaultImage,this.image,this.sound,Integer.valueOf(this.defaultCount),Integer.valueOf(this.count),Integer.valueOf(this.maximumCount),Integer.valueOf(this.size)); //$NON-NLS-1$
  }
}
/* ItemBase------------------- */
