/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.ImageMap.Name.NO_IMAGE;

import java.awt.Image;
import java.io.Serializable;
import java.util.HashMap;

import javax.sound.sampled.Clip;
import javax.sound.sampled.FloatControl;
import javax.swing.ImageIcon;

import javoids.ImageMap.Name;

/* Media--------------- */
/**
 * A class to load the media files required for this game. The files include, the images, sounds, liscence, readme, and version information.
 * @author mallette
 */
public final class Media implements Serializable
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long       serialVersionUID = 1L;
  /** the readme file name */
  private final static String     README           = Messages.getString("Media.FileNameReadme"); //$NON-NLS-1$
  /** the version file name */
  private final static String     VERSION          = Messages.getString("Media.FileNameVersion"); //$NON-NLS-1$
  /** the license file name */
  private final static String     LICENSE          = Messages.getString("Media.FileNameLicense"); //$NON-NLS-1$
  /** the item file name */
  private final static String     ITEM             = Messages.getString("Media.FileNameItem");   //$NON-NLS-1$
  /** an image map of all images */
  private final static ImageMap   imageMap         = new ImageMap();
  /** an sound map of all sounds */
  private final static SoundMap   soundMap         = new SoundMap();
  /** the readme file's text */
  private final static TextLoader readme           = new TextLoader(Media.README);
  /** the version file's text */
  private final static TextLoader version          = new TextLoader(Media.VERSION);
  /** the license file's text */
  private final static TextLoader license          = new TextLoader(Media.LICENSE);
  /** the item file's text */
  private final static ItemLoader item             = new ItemLoader(Media.ITEM);
  /** is the sound enabled */
  private static boolean          isSoundEnabled   = true;
  /** the volume level (0..100%) */
  private static int              volumePercent    = 50;

  static
  {
    System.out.printf(Messages.getString("Media.WaitingForFiles")); //$NON-NLS-1$
    Media.item.run();
    Media.imageMap.start();
    Media.soundMap.start();
    Media.readme.start();
    Media.version.start();
    Media.license.start();
    try
    {
      Media.imageMap.join();
      Media.soundMap.join();
      Media.readme.join();
      Media.version.join();
      Media.license.join();
    }
    catch (InterruptedException interruptedException)
    {
      System.out.printf(Messages.getString("Media.ErrorWaitingToJoin"),interruptedException.getMessage()); //$NON-NLS-1$
    }
    System.out.printf(Messages.getString("Media.FinishedLoadingMedia")); //$NON-NLS-1$
  }

  /**
   * @return a mapping of the image name to the image.
   */
  public static HashMap<Name,Image> getImages()
  {
    return Media.imageMap.getImages();
  }

  /**
   * @param name the desired image's name
   * @return the image associated with that image
   */
  public static Image getImage(ImageMap.Name name)
  {
    return Media.imageMap.getImage(NO_IMAGE.equals( name ) ? NO_IMAGE : name);
  }

  /**
   * @param name the desired icon's name
   * @return the icon associated with that name
   */
  public static ImageIcon getImageIcon(ImageMap.Name name)
  {
    return Media.imageMap.getImageIcon(NO_IMAGE.equals( name ) ? NO_IMAGE : name);
  }

  /**
   * @param name the name of the desired image
   * @param width the width of the image
   * @param height the height of the image
   * @param hints rendering hints
   * @return the scaled image
   */
  public static Image getScaledImage(ImageMap.Name name,int width,int height,int hints)
  {
    return NO_IMAGE.equals( name ) ? null : Media.imageMap.getScaledImage(name,width,height,hints);
  }

  /**
   * @return the number of images that were loaded
   */
  public static int getNumberImages()
  {
    return Media.imageMap.size();
  }

  /**
   * @return a mapping of a name to a sound clip
   */
  public static HashMap<SoundMap.Sound,Clip> getSounds()
  {
    return Media.soundMap.getSounds();
  }

  /**
   * @return mapping of an item name to an item
   */
  public static HashMap<Item.Type,Item> getItems()
  {
    return Media.item.copyItems();
  }

  /**
   * @return the number of sounds that were loaded
   */
  public static int getNumberSounds()
  {
    return Media.soundMap.size();
  }

  /**
   * @return the text of the readme file
   */
  public static String getReadme()
  {
    return Media.readme.getText();
  }

  /**
   * @return the text of the version file
   */
  public static String getVersion()
  {
    return Media.version.getText();
  }

  /**
   * @return the text of the license file
   */
  public static String getLicense()
  {
    return Media.license.getText();
  }

  /**
   * @param _isSoundEnabled set if the sound is enabled
   */
  public static void setIsSound(boolean _isSoundEnabled)
  {
    Media.isSoundEnabled = _isSoundEnabled;
  }

  /**
   * @param name the image name to set
   * @param image the image to associate with the image name
   */
  public static void setImage(ImageMap.Name name,Image image)
  {
    if (image != null)
      Media.imageMap.putImage(name,image);
  }

  /**
   * @param sound the name of the sound to play
   */
  public static void play(SoundMap.Sound sound)
  {
    if (Media.isSoundEnabled && Media.soundMap != null)
    {
      final Clip clip = Media.soundMap.getSound(sound);
      Media.setVolume(clip,Media.getVolume());
      clip.setFramePosition(0);
      clip.start();
    }
  }

  /**
   * @param clip the sound clip to set the volume for
   * @param _volumePercent the volume level to set
   */
  public static void setVolume(Clip clip,int _volumePercent)
  {
    try
    {
      if (clip.isControlSupported(FloatControl.Type.VOLUME))
      {
        FloatControl volumeControl = (FloatControl)clip.getControl(FloatControl.Type.VOLUME);
        volumeControl.setValue((float)(_volumePercent / 100.0));
      }
      else if (clip.isControlSupported(FloatControl.Type.MASTER_GAIN))
      {
        float value = (float)(_volumePercent / 100.0);
        FloatControl gainControl = (FloatControl)clip.getControl(FloatControl.Type.MASTER_GAIN);
        float dB = (float)(Math.log(value == 0.0 ? 0.0001 : value) / Math.log(10.0) * 20.0);
        gainControl.setValue(dB);
      }
    }
    catch (IllegalArgumentException illegalArgumentException)
    {
      System.out.printf(Messages.getString("Media.ErrorSetVolume"),illegalArgumentException.getMessage()); //$NON-NLS-1$
    }
  }

  /**
   * @return the volume level (0..100%)
   */
  public static int getVolume()
  {
    return Media.volumePercent;
  }

  /**
   * @param value the percent value to set the volume level
   */
  public static void setVolume(int value)
  {
    if (value >= 0)
      Media.volumePercent = value;
  }

  /**
   * stop all sounds from playing
   */
  public static void stopAll()
  {
    for (Clip clip : Media.soundMap.getSounds().values())
      if (clip != null)
        clip.stop();
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Media.ToString"),Media.imageMap,Media.soundMap); //$NON-NLS-1$
  }
}
/* Media--------------- */
