/*  Saper Game.
    Copyright (C) 2007 - Pawel Bednarek
    <bednarek.pawel@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/> 
 */
package saper;

import java.awt.Point;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import settings.Settings;

/**
 * Logika gry. 
 */
public final class SaperGame {

    public static final String SETTINGS_WIDTH = "width";
    public static final String SETTINGS_HEIGHT = "height";
    public static final String SETTINGS_MINES = "mines";
    public static final String SETTINGS_TIP = "tip";
    
    // oznacza ze na polu jest mina
    public static final int MINE = -1;
    // oznacza ze user moze pole zaznaczyc z mozliwoscia wybuchu miny
    public static final int FIELD_ENABLED = 0;
    // oznacza ze user wybral pole
    public static final int FIELD_DISABLED = 1;
    
    // szerokosc planszy
    private int width = -1;
    // wysokosc planszy
    private int height = -1;
    // ilosc min
    private int mines = -1;
    // ilosc odkrytych min
    private int minesUncovered = 0;
    // popelniono blad
    private boolean error = false;
    // rozpoczeta juz gra
    private boolean started = false;
    // gra skonczona
    private boolean ended = false;
    
    // pole z minami oraz iloscia sasiadujacych min
    private int[][] minesField = null;
    // pole odpowiadajace powyzszemu ze statusem
    private int[][] fieldsStatus = null;
    
    // nasluchiwacze zdarzen gdy
    private List listeners = new ArrayList();
    // bledne zaznaczenia min (Point(x,y))
    private List markerErrors = new ArrayList();
    
    private boolean tipDone = false;
    
    /**
     * Stworzenie nowej gry o zadanych parametrach
     */
    public SaperGame(int width, int height, int mines) throws IllegalArgumentException {
        // sprawdzenie poprawnosci poarametrow wejsciowych
        if(width < 0 || height < 0 || mines >= width * height) throw new IllegalArgumentException("Nieprawidowe dane dla stworzenia gry");
        
        // utworzenie tablicy min
        this.minesField = SaperUtils.generateManeField(width,height,mines);
        this.width = width;
        this.height = height;
        this.mines = mines;
        // ustawienie poczatkowego statusu min
        this.fieldsStatus = new int[width][height];
        for (int i = 0; i < this.fieldsStatus.length; i++)
            for (int j = 0; j < this.fieldsStatus[i].length; j++)
                this.fieldsStatus[i][j] = FIELD_ENABLED;
    }
    
    public void addSaperGameListener(SaperGameListener sgl){
        if(!this.listeners.contains(sgl)) this.listeners.add(sgl);
    }
    
    public void removeSaperGameListener(SaperGameListener sgl){
        this.listeners.remove(sgl);
    }
    
    public synchronized void start(){
        if(!this.started){
            this.started = true;
            fireGameStarted();
        }
    }
    
    public synchronized void end(){
        this.ended = true;
        fireGameEnd();
    }
    /**
     * Sprawdzenie czy mozna grac
     * @return
     */
    public boolean allowPlay(){
        return this.started && !this.ended;
    }
    
    /**
     * Odsloniecie pola w wyniku akcji uzytkownika
     * @param x
     * @param y
     * @param boom true gdy znalezienie miny ma powodowac koniec gry
     * @throws IndexOutOfBoundsException
     */
    public void uncoverField(final int x, final int y, boolean boom) throws IndexOutOfBoundsException {
        if(!allowPlay()) return;
        // jesli pole jest odkryte a obslugujemy ewentualny wybuch
        // to nic nie robimy
        if(this.fieldsStatus[x][y] != FIELD_ENABLED && boom){
            return;
        }
        // jesli pole ma zero to nie ma sensu go zakrywac
        if(this.fieldsStatus[x][y] == FIELD_DISABLED && this.minesField[x][y] == 0){
        	return;
        }
        // wywolany zostanie wybuch bomby w razie bledu
        if(boom){
            // zaznaczenie, ze pole bylo wybrane
            this.fieldsStatus[x][y] = FIELD_DISABLED;
	        // sprawdzenie, czy mina
	        if(this.minesField[x][y] == SaperGame.MINE) {
	            this.error = true;
	            List list = uncoverAll();
	            list.add(new SaperPoint(x,y,this.minesField[x][y]));
	            fireFieldsUncovered(list);
                this.markerErrors.add(new Point(x,y));
	            end();
	        // nie ma miny
	        } else {
                // odsloniecie wybranego pola 
	            List list = uncover(x,y,null,true,false);
	            fireFieldsUncovered(list);
	        }
	    // zanzaczamy ze mina lub zaznaczamy ze nie mina (zmiana statusu)
        } else {
            // gdy na polu jest zaznaczona ilosc nic nie robimy
            if(this.fieldsStatus[x][y] == FIELD_DISABLED && this.minesField[x][y] > 0 && !this.markerErrors.contains(new Point(x,y))) return;
            
            // odznaczenie miny
            if(this.fieldsStatus[x][y] == FIELD_DISABLED) {
                this.fieldsStatus[x][y] = FIELD_ENABLED;
                // usuniecie ewentualne z listy bledow
                Point point = new Point(x,y);
                if(this.markerErrors.contains(point)) this.markerErrors.remove(point);
                this.minesUncovered--;
                fireFieldMineStatusChaneged(new SaperPoint(x,y,Integer.MIN_VALUE));
            // zaznaczenie miny
            } else {
                this.fieldsStatus[x][y] = FIELD_DISABLED;
                if(this.minesField[x][y] != SaperGame.MINE) this.markerErrors.add(new Point(x,y));
                this.minesUncovered++;
                fireFieldMineStatusChaneged(new SaperPoint(x,y,SaperGame.MINE));
            }
        }
        // sprawdzenie czy koniec gry
        if(this.mines == this.minesUncovered && allFieldsUncovered()) end();
    }
    
    /**
     * Odsloniecie wszystkich pozostalych pol w wyniku wybuchu miny
     * @return lista punktow ktore zostaly odsloniete
     */
    private List uncoverAll(){
        List list = new ArrayList(this.mines);
        for (int i = 0; i < this.width; i++) {
            for (int j = 0; j < this.height; j++) {
                if(this.fieldsStatus[i][j] == FIELD_ENABLED && this.minesField[i][j] == MINE) {
                    SaperPoint sp = new SaperPoint(i,j,MINE);
                    list.add(sp);
                }
            }
        }
        return list;
    }
    /**
     * Ilosc zaznaczonych min dookola zadanego pola x,y
     * @param x
     * @param y
     * @return
     */
    private int uncoveredMines(int x, int y){
        int count = 0;
        for(int i = x-1; i <= x+1; i++){
            if(i < 0 || i == this.width) continue;
            for(int j = y-1; j <= y+1; j++){
                if(j < 0 || j == this.height) continue;
                if(this.minesField[i][j] == MINE && this.fieldsStatus[i][j] == FIELD_DISABLED) count++;
                if(this.markerErrors.contains(new Point(i,j))) count++;
            }
        }
        return count;
    }
    /**
     * Ilosc pol ktore sa jeszcze zasloniete
     * @param x
     * @param y
     * @return
     */
    private int countHidden(int x, int y){
        int count = 0;
        for(int i = x-1; i <= x+1; i++){
            if(i < 0 || i == this.width) continue;
            for(int j = y-1; j <= y+1; j++){
                if(j < 0 || j == this.height) continue;
                if(this.fieldsStatus[i][j] == FIELD_ENABLED) count++;
            }
        }
        return count;
    }
    /**
     * Ustawienie pol ktore pozostaly dookola jako miny, mozna, bo wczesniej
     * sprawdzono ze ilosc min do odkrycia rowna sie ilosci postych pol 
     * @param x
     * @param y
     * @return
     */
    private List setAllAsMine(int x, int y){
        List list = new ArrayList();
        for(int i = x-1; i <= x+1; i++){
            if(i < 0 || i == this.width) continue;
            for(int j = y-1; j <= y+1; j++){
                if(j < 0 || j == this.height) continue;
                if(this.fieldsStatus[i][j] == FIELD_ENABLED){
                    SaperPoint sp = new SaperPoint(i,j,this.minesField[i][j]);
                    list.add(sp);
                    this.fieldsStatus[i][j] = FIELD_DISABLED;
                    this.minesUncovered++;
                }
            }
        }
        
        return list;
    }
    /**
     * Sprawdzenie czy wszystkie pola zostaly juz sprawdzone
     * @return
     */
    private boolean allFieldsUncovered(){
        for (int i = 0; i < this.fieldsStatus.length; i++)
            for (int j = 0; j < this.fieldsStatus[i].length; j++)
                if(this.fieldsStatus[i][j] == FIELD_ENABLED) return false;
        return true;
    }
    
    /**
     * Odsloniecie sasiadow
     * @param x
     * @param y
     */
    public void uncoverNeighbours(int x, int y){
        if(!allowPlay()) return;
        int uncoveredMines = uncoveredMines(x,y);
        int hidden = countHidden(x,y);
        if(
             // warunek ze zaznaczona jest liczba min taka jak powinna byc
             (this.fieldsStatus[x][y] != FIELD_ENABLED && this.minesField[x][y]== uncoveredMines) ||
             // warunek ze liczba niezaznaczonych pol wynosi tyle ile liczba brakujacych min
             (this.fieldsStatus[x][y] != FIELD_ENABLED && this.minesField[x][y] - uncoveredMines == hidden)   
        ){
            List list = null;
            if(this.minesField[x][y] - uncoveredMines == hidden)
                list = setAllAsMine(x,y); // wszystkie pozostale pola musza byc minami
             else
                list = uncover(x,y,null,false,true); // wszystkie pozostale pole "musza" byc nieminami
            
            fireFieldsUncovered(list);
            // ale mogl sie zdarzyc ze jednak blad jest wiec koniec gry
            if(this.error){
                List list_all = uncoverAll();
                fireFieldsUncovered(list_all);
                end();
            } else {
                // sprawdzenie czy koniec gry
                if(this.mines == this.minesUncovered && allFieldsUncovered()) end();
            }
        } else {
            fireFieldsUncovered(null);
        }
    }
    /**
     * Odznaczenie pola i ewentualnie pol sasiadujacych.
     * Metoda rekurencyjna
     * @param x
     * @param y
     * @param points lista punktow SaperPoint ktore zostaly juz odsloniete w trakcie akcji iodslanianai
     * @param addxy czy dodac do listy punkt ktory jest wskazany
     * @param forceNeighbours czy wymuszac odlanianie sasiadow - dla odslaniania hurtowego
     * @return
     */
    private List uncover(int x, int y, final List points_in, boolean addxy, boolean forceNeighbours){
        // stworzenie liczby punktow jesli nie zostala zadana
    	List points = points_in;
        if(points == null) points = new ArrayList();
        
        // dodanie lokalnego punktu jesli trzeba
        if(addxy) points.add(new SaperPoint(x,y,this.minesField[x][y]));
        
        // jesli nie ma powodowac odloniecia sasiadow
        if(this.minesField[x][y] > 0 && !forceNeighbours){
            return points; 
        }
        
        // odslaniani sasiedzi w tej metodzie tylko
        List uncoveredList = new ArrayList();
        
        // odslanianie
        for(int i = x-1; i <= x+1; i++){
            if(i < 0 || i == this.width) continue;
            for(int j = y-1; j <= y+1; j++){
                if(j < 0 || j == this.height) continue;
                // nic nie robimy z polem juz odslonietym
                if(this.fieldsStatus[i][j] == FIELD_DISABLED) continue;
                // odslaniamy pole i dodajemy do tablic
                SaperPoint sp = new SaperPoint(i,j,this.minesField[i][j]);
                points.add(sp);
                uncoveredList.add(sp);
                this.fieldsStatus[i][j] = FIELD_DISABLED;
            }
        }
        
        // jesli odslonieci sasiedzi sa odslaniamy, ale niekoniecznie - parametry
        // metody uncover
        if(uncoveredList.size() != 0){
            for (int i = 0, n = uncoveredList.size(); i < n; i++) {
                SaperPoint element = (SaperPoint) uncoveredList.get(i);
                if(element.status == MINE) this.error = true;
                uncover(element.x,element.y,points,false,false);
                // ostatni parametr false wiec nie jest odslanianie na sile
            }
        }
        
        return points;
    }
    
    public int getHeight() {
        return this.height;
    }
    
    public int getMines() {
        return this.mines;
    }
    
    public int getWidth() {
        return this.width;
    }
    
    public int getMinesLeft(){
        return this.mines - this.minesUncovered;
    }
    
    public String getMinesLeftString(){
        return Integer.toString(getMinesLeft());
    }
    
    public boolean isStarted(){
        return this.started;
    }
    
    public boolean isEnded(){
        return this.ended;
    }
    
    private void fireGameStarted(){
        SaperGameEvent sge = new SaperGameEvent(this,null);
        for (int i = 0, n = this.listeners.size(); i < n; i++) {
            SaperGameListener element = (SaperGameListener) this.listeners.get(i);
            element.gameStarted(sge);
        }
    }
    
    private void fireGameEnd(){
        List list = new ArrayList();
        for(int i = 0, n = this.markerErrors.size(); i < n; i++){
            Point p = (Point) this.markerErrors.get(i);
            list.add(new SaperPoint(p.x,p.y,this.minesField[p.x][p.y]));
        }
        SaperGameEvent sge = new SaperGameEvent(this,list);
        for (int i = 0, n = this.listeners.size(); i < n; i++) {
            SaperGameListener element = (SaperGameListener) this.listeners.get(i);
            if(!this.error && this.minesUncovered == this.mines) element.gameVictory(sge);
            else element.gameLost(sge);
        }
        
    }
    
    private void fireFieldsUncovered(List list){
        SaperGameEvent sge = new SaperGameEvent(this,list);
        for (int i = 0, n = this.listeners.size(); i < n; i++) {
            SaperGameListener element = (SaperGameListener) this.listeners.get(i);
            element.gameFieldsUncovered(sge);
        }
    }
    
    private void fireFieldMineStatusChaneged(SaperPoint point){
        List list = new ArrayList(1);
        list.add(point);
        SaperGameEvent event = new SaperGameEvent(this,list);
        for (int i = 0, n = this.listeners.size(); i < n; i++) {
            SaperGameListener element = (SaperGameListener) this.listeners.get(i);
            element.gameFieldMineStatusChanged(event);
        }
    }

    public boolean isError() {
        return this.error;
    }
    
    public Point getGameTip(){
    	Point result = null;
    	String doTipString = Settings.getSettings(SaperFrame.SETTINGS_NAME).getProperty(SETTINGS_TIP, "true");
    	boolean doTip = Boolean.valueOf(doTipString).booleanValue();
    	if(!this.tipDone && doTip){
//	    	for (int i = 0; i < this.minesField.length; i++) {
//	    		for(int j = 0; j < this.minesField[i].length; j++){
//	    			if(this.minesField[i][j] == 0){
//	    				this.tipDone = true;
//	    				return new Point(i,j); }
//	    		}
//			}
    		int counter = 1000;
    		int x = 0, y = 0;
    		Random random = new Random();
    		do {
    			x = random.nextInt(this.width);
    			y = random.nextInt(this.height);
    			if(this.minesField[x][y] == 0) {
    				result = new Point(x, y);
    			}
    		} while(result == null && counter > 0);
    	}
    	
		return result;
    }
}
