#ifndef TAGS_CONTEXT_H
#define TAGS_CONTEXT_H

#include <vsapi.h>


extern "C" {

//////////////////////////////////////////////////////////////////////
// "C" style API for tracking current tagging context, including
// all tags in the current file, as well as tracking the heirarchy
// of nested tag contexts.
//
// The context data is inheritently transient.  Since it is relatively
// inexpensive to compute, it is best if not stored, but simply
// recalculated when a new buffer is opened.
//

//////////////////////////////////////////////////////////////////////
// flags used by tag_get_detail2 for retrieving specific context
// information very quickly.
//
#define VS_TAGDETAIL_max 256

#define VS_TAGDETAIL_context (VS_TAGDETAIL_max*1)
#define VS_TAGDETAIL_local   (VS_TAGDETAIL_max*2)
#define VS_TAGDETAIL_match   (VS_TAGDETAIL_max*3)

#define VS_TAGDETAIL_context_tag_file      (VS_TAGDETAIL_context+0)
#define VS_TAGDETAIL_context_name          (VS_TAGDETAIL_context+1)
#define VS_TAGDETAIL_context_type          (VS_TAGDETAIL_context+2)
#define VS_TAGDETAIL_context_file          (VS_TAGDETAIL_context+3)
#define VS_TAGDETAIL_context_line          (VS_TAGDETAIL_context+4)
#define VS_TAGDETAIL_context_start_linenum (VS_TAGDETAIL_context+4)
#define VS_TAGDETAIL_context_start_seekpos (VS_TAGDETAIL_context+5)
#define VS_TAGDETAIL_context_scope_linenum (VS_TAGDETAIL_context+6)
#define VS_TAGDETAIL_context_scope_seekpos (VS_TAGDETAIL_context+7)
#define VS_TAGDETAIL_context_end_linenum   (VS_TAGDETAIL_context+8)
#define VS_TAGDETAIL_context_end_seekpos   (VS_TAGDETAIL_context+9)
#define VS_TAGDETAIL_context_class         (VS_TAGDETAIL_context+10)
#define VS_TAGDETAIL_context_flags         (VS_TAGDETAIL_context+11)
#define VS_TAGDETAIL_context_args          (VS_TAGDETAIL_context+12)
#define VS_TAGDETAIL_context_return        (VS_TAGDETAIL_context+13)
#define VS_TAGDETAIL_context_outer         (VS_TAGDETAIL_context+14)
#define VS_TAGDETAIL_context_parents       (VS_TAGDETAIL_context+15)

#define VS_TAGDETAIL_local_tag_file        (VS_TAGDETAIL_local+0)
#define VS_TAGDETAIL_local_name            (VS_TAGDETAIL_local+1)
#define VS_TAGDETAIL_local_type            (VS_TAGDETAIL_local+2)
#define VS_TAGDETAIL_local_file            (VS_TAGDETAIL_local+3)
#define VS_TAGDETAIL_local_line            (VS_TAGDETAIL_local+4)
#define VS_TAGDETAIL_local_start_linenum   (VS_TAGDETAIL_local+4)
#define VS_TAGDETAIL_local_start_seekpos   (VS_TAGDETAIL_local+5)
#define VS_TAGDETAIL_local_scope_linenum   (VS_TAGDETAIL_local+6)
#define VS_TAGDETAIL_local_scope_seekpos   (VS_TAGDETAIL_local+7)
#define VS_TAGDETAIL_local_end_linenum     (VS_TAGDETAIL_local+8)
#define VS_TAGDETAIL_local_end_seekpos     (VS_TAGDETAIL_local+9)
#define VS_TAGDETAIL_local_class           (VS_TAGDETAIL_local+10)
#define VS_TAGDETAIL_local_flags           (VS_TAGDETAIL_local+11)
#define VS_TAGDETAIL_local_args            (VS_TAGDETAIL_local+12)
#define VS_TAGDETAIL_local_return          (VS_TAGDETAIL_local+13)
#define VS_TAGDETAIL_local_outer           (VS_TAGDETAIL_local+14)
#define VS_TAGDETAIL_local_parents         (VS_TAGDETAIL_local+15)

#define VS_TAGDETAIL_match_tag_file        (VS_TAGDETAIL_match+0)
#define VS_TAGDETAIL_match_name            (VS_TAGDETAIL_match+1)
#define VS_TAGDETAIL_match_type            (VS_TAGDETAIL_match+2)
#define VS_TAGDETAIL_match_file            (VS_TAGDETAIL_match+3)
#define VS_TAGDETAIL_match_line            (VS_TAGDETAIL_match+4)
#define VS_TAGDETAIL_match_start_linenum   (VS_TAGDETAIL_match+4)
#define VS_TAGDETAIL_match_start_seekpos   (VS_TAGDETAIL_match+5)
#define VS_TAGDETAIL_match_scope_linenum   (VS_TAGDETAIL_match+6)
#define VS_TAGDETAIL_match_scope_seekpos   (VS_TAGDETAIL_match+7)
#define VS_TAGDETAIL_match_end_linenum     (VS_TAGDETAIL_match+8)
#define VS_TAGDETAIL_match_end_seekpos     (VS_TAGDETAIL_match+9)
#define VS_TAGDETAIL_match_class           (VS_TAGDETAIL_match+10)
#define VS_TAGDETAIL_match_flags           (VS_TAGDETAIL_match+11)
#define VS_TAGDETAIL_match_args            (VS_TAGDETAIL_match+12)
#define VS_TAGDETAIL_match_return          (VS_TAGDETAIL_match+13)
#define VS_TAGDETAIL_match_outer           (VS_TAGDETAIL_match+14)
#define VS_TAGDETAIL_match_parents         (VS_TAGDETAIL_match+15)

// Tag match types for speed insert of tag matches
#define VS_TAGMATCH_tag      0
#define VS_TAGMATCH_context  1
#define VS_TAGMATCH_local    2
#define VS_TAGMATCH_match    3

// Options flags used by various functions in context.e
// for example _MatchSymbolInContext()
#define VS_TAGCONTEXT_ALLOW_locals         0x00000001
#define VS_TAGCONTEXT_ALLOW_private        0x00000002
#define VS_TAGCONTEXT_ALLOW_protected      0x00000004
#define VS_TAGCONTEXT_ALLOW_package        0x00000008
#define VS_TAGCONTEXT_ONLY_volatile        0x00000010
#define VS_TAGCONTEXT_ONLY_const           0x00000020
#define VS_TAGCONTEXT_ONLY_static          0x00000040
#define VS_TAGCONTEXT_ONLY_non_static      0x00000080
#define VS_TAGCONTEXT_ONLY_data            0x00000100
#define VS_TAGCONTEXT_ONLY_funcs           0x00000200
#define VS_TAGCONTEXT_ONLY_classes         0x00000400
#define VS_TAGCONTEXT_ONLY_packages        0x00000800
#define VS_TAGCONTEXT_ONLY_inclass         0x00001000
#define VS_TAGCONTEXT_ONLY_constructors    0x00002000
#define VS_TAGCONTEXT_ONLY_this_class      0x00004000
#define VS_TAGCONTEXT_ONLY_parents         0x00008000
#define VS_TAGCONTEXT_FIND_derived         0x00010000
#define VS_TAGCONTEXT_ALLOW_anonymous      0x00020000
#define VS_TAGCONTEXT_ACCESS_private       0x0000000E
#define VS_TAGCONTEXT_ACCESS_protected     0x0000000C
#define VS_TAGCONTEXT_ACCESS_package       0x00000008
#define VS_TAGCONTEXT_ACCESS_public        0x00000000
#define VS_TAGCONTEXT_ANYTHING             VS_TAGCONTEXT_ACCESS_public


//////////////////////////////////////////////////////////////////////
// Context tracking related functions.
//

// Add a tag and its context information to the context list.
// The context for the current tag includes all tag information, 
// as well as the ending line number and begin/scope/end seek 
// positions in the file.  If unknown, the end line number/seek 
// position may be deferred, see tag_end_context().
//    outer_context -- context ID for the outer context (eg. class/struct)
//    tag_name      -- tag string
//    tag_type      -- string specifying tag_type
//    file_name     -- full path of file the tag is located in
//    start_line_no -- start line number of tag within file
//    start_seekpos -- start seek position of tag within file
//    scope_line_no -- start line number of start of tag inner scope
//    scope_seekpos -- start seek position of tag inner scope
//    end_line_no   -- (optional) ending line number of tag within file
//    end_seekpos   -- (optional) end seek position of tag within file
//    class_name    -- (optional) name of class that tag is present in,
//                     use concatenation (as defined by language rules)
//                     to specify names of inner classes.
//    tag_flags     -- (optional) see VS_TAGFLAG_* above.
//    signature     -- (optional) tag signature (return type, arguments, etc)
// Returns sequence number (context_id) of tag context on success, 
// or <0 on error.
//
int VSAPI tag_insert_context(int outer_context, 
                             VSPSZ tag_name, VSPSZ tag_type, 
                             VSPSZ file_name, 
                             int start_line_no, int start_seekpos,
                             int scope_line_no, int scope_seekpos,
                             int end_line_no, int end_seekpos,
                             VSPSZ class_name, int tag_flags,
                             VSPSZ signature);

// Set the end positions of the context at with the given context ID.
//    context_id    -- id for the context to modify
//    end_line_no   -- ending line number of tag within file
//    end_seekpos   -- end seek position of tag within file
// Returns 0 on success, <0 on error.   
//
int VSAPI tag_end_context(int context_id, int end_line_no, int end_seekpos);

// Set the class inheritance for the given context tag.
//    context_id    -- id for the context to modify
//    parents       -- parents of the context item
// Returns 0 on success, <0 on error.
int VSAPI tag_set_context_parents(int context_id, VSPSZ parents);

// Clear all context information.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_clear_context();

// Return the total number of context tags
int VSAPI tag_get_num_of_context();

// Retrieve information about the given context ID.
//    context_id    -- context ID to look up (from tag_insert_context)
//    tag_name      -- (reference) tag string (native case)
//    tag_type      -- (reference) string specifying tag_type
//                     (see above for list of standard type names).
//    file_name     -- (reference) full path of file the tag is located in
//    start_line_no -- (reference) start line number of tag within file
//    start_seekpos -- (reference) start seek position of tag within file
//    scope_line_no -- (reference)start line number of start of tag inner scope
//    scope_seekpos -- (reference) start seek position of tag inner scope
//    end_line_no   -- (optional) ending line number of tag within file
//    end_seekpos   -- (optional) end seek position of tag within file
//    class_name    -- (reference) name of class that tag is present in,
//                     uses concatenation (as defined by language rules)
//                     to specify names of inner classes (see insert, above).
//                     set to empty string if not defined.
//    tag_flags     -- (reference) see VS_TAGFLAG_* above.
//    signature     -- (reference) arguments or formal parameters
//    return_type   -- (reference) constant value or return type
//
int VSAPI tag_get_context(int context_id,
                          VSHREFVAR tag_name, VSHREFVAR type_name,
                          VSHREFVAR file_name, 
                          VSHREFVAR start_line_no, VSHREFVAR start_seekpos, 
                          VSHREFVAR scope_line_no, VSHREFVAR scope_seekpos, 
                          VSHREFVAR end_line_no,   VSHREFVAR end_seekpos,
                          VSHREFVAR class_name, VSHREFVAR tag_flags,
                          VSHREFVAR signature, VSHREFVAR return_type);

// Check if the current buffer position is still within the current context
// Returns one of the following codes:
//     0 -- the context is not set or totally wrong
//    -1 -- context info loaded, but the cursor is out of context
//     1 -- the context is within the tag definition
//     2 -- the context is within the scope of the tag/function
//
int VSAPI tag_check_context();

// Return the index of the current context item.
// Returns <0 on error, 0 if no current context.
//
int VSAPI tag_current_context();

// Return the index of the nearest context item.
//    linenum -- line number to check context on
// Returns <0 on error, 0 if no such context.
//
int VSAPI tag_nearest_context(int linenum);

// Find a the first context entry with the given tag prefix, or if
// 'exact', with the exact tag name.  Use case-sensitive match if
// case_sensitive != 0.
//    tag_prefix -- tag name or prefix of tag name
//    exact      -- search for exact match or prefix match
//    case_sens  -- case sensitive string comparison?
//    allow_anon -- (optional) pass through anonymous classes
//    class_name -- (optional) class to find item in
// Returns context ID of tag if found, <0 on error or not found.
//
int VSAPI tag_find_context(VSPSZ tag_prefix, int exact, int case_sensitive);

// Find a the next context entry with the given tag prefix, or if
// 'exact', with the exact tag name.  Use case-sensitive match if
// case_sensitive != 0.
//    tag_prefix -- tag name or prefix of tag name
//    exact      -- search for exact match or prefix match
//    case_sens  -- case sensitive string comparison?
//    allow_anon -- (optional) pass through anonymous classes
//    class_name -- (optional) class to find item in
// Returns context ID of tag if found, <0 on error or not found.
//
int VSAPI tag_next_context(VSPSZ tag_prefix, int exact, int case_sensitive);

// Insert all the context items into the given tree.
//    treeWID       -- tree widget to load info into
//    treeIndex     -- tree index to insert into
//    include_tab   -- include tab when creating caption
//    force_leaf    -- force item to be inserted as a leaf item
//    tree_flags    -- tree flags to set for this item
//    pushtag_flags -- PUSHTAG_*, see slick.sh
// Returns 0 on success, <0 on error.
//    
int VSAPI tag_tree_insert_context(int treeWID, int treeIndex, 
                                  int include_tab, int force_leaf, 
                                  int tree_flags, int pushtag_flags);

// Insert all the tags from the current context into the currently
// open tag file.  Assumes that the context is up-to-date at the
// time that this function is called.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_transfer_context(VSPSZ file_name);


//////////////////////////////////////////////////////////////////////
// Local declarations related tracking functions
//

// Add a local variable tag and its information to the locals list.
//    tag_name      -- tag string
//    tag_type      -- string specifying tag_type
//    file_name     -- full path of file the tag is located in
//    line_no       -- start line number of tag within file
//    class_name    -- (optional) name of class that tag is present in,
//                     use concatenation (as defined by language rules)
//                     to specify names of inner classes.
//    tag_flags     -- (optional) see VS_TAGFLAG_* above.
//    signature     -- (optional) tag signature (return type, arguments, etc)
// Returns sequence number (local_id) of local variable on success, 
// or <0 on error.
//
int VSAPI tag_insert_local(VSPSZ tag_name, VSPSZ tag_type, 
                           VSPSZ file_name, int line_no,
                           VSPSZ class_name, int tag_flags,
                           VSPSZ signature);

// Add a local variable tag and its information to the locals list.
//    tag_name      -- tag string
//    tag_type      -- string specifying tag_type
//    file_name     -- full path of file the tag is located in
//    start_linenum -- start line number of tag within file
//    start_seekpos -- start seek position of tag within file
//    scope_linenum -- start line number of start of tag inner scope
//    scope_seekpos -- start seek position of tag inner scope
//    end_linenum   -- (optional) ending line number of tag within file
//    end_seekpos   -- (optional) end seek position of tag within file
//    class_name    -- (optional) name of class that tag is present in,
//                     use concatenation (as defined by language rules)
//                     to specify names of inner classes.
//    tag_flags     -- (optional) see VS_TAGFLAG_* above.
//    signature     -- (optional) tag signature (return type, arguments, etc)
// Returns sequence number (local_id) of local variable on success, 
// or <0 on error.
//
int VSAPI tag_insert_local2(VSPSZ tag_name, VSPSZ tag_type, VSPSZ file_name,
                            int start_linenum, int start_seekpos,
                            int scope_linenum, int scope_seekpos,
                            int end_linenum,   int end_seekpos,
                            VSPSZ class_name, int tag_flags, VSPSZ signature);

// Return the index of the current local variable.
// Returns <0 on error, 0 if no current local variable.
//
int VSAPI tag_current_local();

// Set the class inheritance for the given local tag.
//    context_id    -- id for the context to modify
//    parents       -- parents of the context item
// Returns 0 on success, <0 on error.
int VSAPI tag_set_local_parents(int context_id, VSPSZ parents);

// Return the total number of local variables
int VSAPI tag_get_num_of_locals();

// Kill all locals after 'local_id', not including 'local_id'
//    local_id    -- id for the local variable to start removing at
//                   Use '0' to remove all locals.
// Returns 0 on success, <0 on error.   
//
int VSAPI tag_clear_locals(int local_id);

// Retrieve information about the given local variable or parameter.
//    local_id      -- local ID to look up (from tag_insert_local)
//    tag_name      -- (reference) tag string (native case)
//    tag_type      -- (reference) string specifying tag_type
//                     (see above for list of standard type names).
//    file_name     -- (reference) full path of file the tag is located in
//    line_no       -- (reference) start line number of tag within file
//    class_name    -- (reference) name of class that tag is present in,
//                     uses concatenation (as defined by language rules)
//                     to specify names of inner classes (see insert, above).
//                     set to empty string if not defined.
//    tag_flags     -- (reference) see VS_TAGFLAG_* above.
//    signature     -- (reference) arguments or formal parameters
//    return_type   -- (reference) constant value or return type
//
int VSAPI tag_get_local(int local_id,
                        VSHREFVAR tag_name, VSHREFVAR type_name,
                        VSHREFVAR file_name, VSHREFVAR line_no,
                        VSHREFVAR class_name, VSHREFVAR tag_flags,
                        VSHREFVAR signature, VSHREFVAR return_type);

// Retrieve complete information about the given local ID.
//    local_id      -- local ID to look up (from tag_insert_local)
//    tag_name      -- (reference) tag string (native case)
//    tag_type      -- (reference) string specifying tag_type
//                     (see above for list of standard type names).
//    file_name     -- (reference) full path of file the tag is located in
//    start_linenum -- (reference) start line number of tag within file
//    start_seekpos -- (reference) start seek position of tag within file
//    scope_linenum -- (reference)start line number of start of tag inner scope
//    scope_seekpos -- (reference) start seek position of tag inner scope
//    end_linenum   -- (optional) ending line number of tag within file
//    end_seekpos   -- (optional) end seek position of tag within file
//    class_name    -- (reference) name of class that tag is present in,
//                     uses concatenation (as defined by language rules)
//                     to specify names of inner classes (see insert, above).
//                     set to empty string if not defined.
//    tag_flags     -- (reference) see VS_TAGFLAG_* above.
//    signature     -- (reference) arguments or formal parameters
//    return_type   -- (reference) constant value or return type
//    
int VSAPI tag_get_local2(int context_id,
                         VSHREFVAR tag_name, VSHREFVAR type_name,
                         VSHREFVAR file_name, 
                         VSHREFVAR start_linenum, VSHREFVAR start_seekpos, 
                         VSHREFVAR scope_linenum, VSHREFVAR scope_seekpos, 
                         VSHREFVAR end_linenum,   VSHREFVAR end_seekpos,
                         VSHREFVAR class_name, VSHREFVAR tag_flags,
                         VSHREFVAR signature, VSHREFVAR return_type);

// Find a the first local tag with the given tag prefix, or if
// 'exact', with the exact tag name.  Use case-sensitive match if
// case_sensitive != 0.
//    tag_prefix -- tag name or prefix of tag name
//    exact      -- search for exact match or prefix match
//    case_sens  -- case sensitive string comparison?
//    allow_anon -- (optional) pass through anonymous classes
//    class_name -- (optional) class to find item in
// Returns local ID of tag if found, <0 on error or not found.
//
int VSAPI tag_find_local(VSPSZ tag_prefix, int exact, int case_sensitive);

// Find a the next local tag with the given tag prefix, or if
// 'exact', with the exact tag name.  Use case-sensitive match if
// case_sensitive != 0.
//    tag_prefix -- tag name or prefix of tag name
//    exact      -- search for exact match or prefix match
//    case_sens  -- case sensitive string comparison?
//    allow_anon -- (optional) pass through anonymous classes
//    class_name -- (optional) class to find item in
// Returns local ID of tag if found, <0 on error or not found.
//
int VSAPI tag_next_local(VSPSZ tag_prefix, int exact, int case_sensitive);


//////////////////////////////////////////////////////////////////////
// Search match tracking related functions
//

// Add a search match tag and its information to the matches list.
//    tag_name      -- tag string
//    tag_type      -- string specifying tag_type
//    file_name     -- full path of file the tag is located in
//    line_no       -- start line number of tag within file
//    class_name    -- (optional) name of class that tag is present in,
//                     use concatenation (as defined by language rules)
//                     to specify names of inner classes.
//    tag_flags     -- (optional) see VS_TAGFLAG_* above.
//    signature     -- (optional) tag signature (return type, arguments, etc)
// Returns sequence number (match_id) of matching tag on success, 
// or <0 on error.
//
int VSAPI tag_insert_match(VSPSZ tag_file,
                           VSPSZ tag_name, VSPSZ tag_type, 
                           VSPSZ file_name, int line_no,
                           VSPSZ class_name, int tag_flags,
                           VSPSZ signature);

// Speedy version of tag_insert_match that simply clones a context,
// local, or current tag match
int VSAPI tag_insert_match_fast(int match_type, int local_or_context_id);

// Return the total number of context matches
int VSAPI tag_get_num_of_matches();

// Kill all search matches.
// Returns 0 on success, <0 on error.   
//
int VSAPI tag_clear_matches();

// Retrieve information about the given search match.
//    match_id      -- match ID to look up (from tag_insert_match)
//    tag_name      -- (reference) tag string (native case)
//    tag_type      -- (reference) string specifying tag_type
//                     (see above for list of standard type names).
//    file_name     -- (reference) full path of file the tag is located in
//    line_no       -- (reference) start line number of tag within file
//    class_name    -- (reference) name of class that tag is present in,
//                     uses concatenation (as defined by language rules)
//                     to specify names of inner classes (see insert, above).
//                     set to empty string if not defined.
//    tag_flags     -- (reference) see VS_TAGFLAG_* above.
//    signature     -- (reference) arguments or formal parameters
//    return_type   -- (reference) constant value or return type
//
int VSAPI tag_get_match(int match_id, VSHREFVAR tag_file,
                        VSHREFVAR tag_name, VSHREFVAR type_name,
                        VSHREFVAR file_name, VSHREFVAR line_no,
                        VSHREFVAR class_name, VSHREFVAR tag_flags,
                        VSHREFVAR signature, VSHREFVAR return_type);


// Get the given detail type for the given item either as a
// context tag, local tag, or part of a match set.
//    detail_id   -- VS_TAGDETAIL_* (see tagsdb.h)
//    item_id     -- context_id, local_id, or match_id
//    result      -- (reference) value of detail is returned here
// Returns 0 on success, <0 on error.
void VSAPI tag_get_detail2(int detail_id, int item_id, VSHREFVAR result);


//////////////////////////////////////////////////////////////////////////////
// Utility functions for implementing context tagging functions,
// optimized components of context.e
//

// Returns true if at the current access level, we have access to the member
// with the given flags at our access level.
//    context_flags  -- VS_TAGCONTEXT_*
//    tag_flags      -- VS_TAGFLAG_* (from tag details)
// Returns 1 if we have access to the tag, 0 if not
//
int VSAPI tag_check_access_level(int context_flags, int tag_flags);

// Returns true if the tag having the given tag_flags and type matches the
// requirements set by the given context flags.
//    context_flags -- VS_TAGCONTEXT_*
//    tag_flags     -- VS_TAGFLAG_* (from tag details)
//    type_name     -- tag type (from tag details)
// Returns 1 if the tag with the given type and flags passes the context
// flags, otherwise, returns 0.
//
int VSAPI tag_check_context_flags(int context_flags, int tag_flags, VSPSZ type_name);

//////////////////////////////////////////////////////////////////////////////
// Decompose a class name into its outer component and 
// inner class name only.
//    class_name  -- class name to decompose
//    inner_name  -- (reference) 'inner' class name (class name only)
//    outer_name  -- (reference) 'outer' class name (class_name - inner_name)
// This is strictly a string function, no file I/O or searching is involved.
//
void VSAPI tag_split_class_name(VSPSZ class_name, VSHREFVAR inner_name, VSHREFVAR outer_name);

// Determine whether the class name and outer class name should be
// joined using a package seperator or class seperator and return
// the resulting string.  This involves searching for the outer
// class name as a package, if found, then use package string.
// The current object must be an editor control or current buffer.
//    class_name     -- name of class to join (inner_class)
//    outer_class    -- name of outer class or package to join
//    tag_files      -- (reference, read-only array of strings)
//                      list of tag files to search
//    case_sensitive -- 1/0 for case-sensitive or case-insensitve search
// Returns static string if outer_class :: class_name was found in
// the context or tag database, otherwise returns '';
// 
VSPSZ VSAPI tag_join_class_name(VSPSZ class_name, VSPSZ outer_class,
                                VSHREFVAR tag_files, int case_sensitive);

// Determine if 'parent_class' is a parent of 'child_class', that is, does
// 'child_class' derive either directly or transitively from 'parent_class'?
//    parent_class   -- class to check if it is a parent (base class)
//    child_class    -- class to check if it is derived from parent
//    tag_files      -- (reference to _str[]) tag files to search
//    case_sensitive -- case sensitive (1) or case insensitive (0)
//    normalize      -- attempt to normalize class name or take as-is?
// Returns 1 if 'child_class' derives from 'search_class', otherwise 0.
//
int VSAPI tag_is_parent_class(VSPSZ parent_class, VSPSZ child_class,
                              VSHREFVAR tag_files, int case_sensitive, int normalize);

//////////////////////////////////////////////////////////////////////////////
// Lookup 'symbol' and see if it could be a typedef symbol.  If so,
// return 1, otherwise return 0.
//    symbol         -- symbol to investigate
//    tag_files      -- (reference to _str[]) tag files to search
//    case_sensitive -- use case-sensitive comparisons?
// This function ignore class scope.  Simply put, if 'symbol' is a typedef,
// anywhere, this function may return true.  Thus, it's should really be used
// only as an arbiter prior to attempting to match 'symbol' in context as a
// typedef.  The reason that this function behaves in this was is for speed
// and simplicity.  Otherwise, it would have to search all parent and outer
// class scopes, in addition to locals, context and tag files.
//
int VSAPI tag_check_for_typedef(VSPSZ symbol, VSHREFVAR tag_files, int case_sensitive);

// Look up 'symbol' and see if it is a simple preprocessing symbol.
// If so, return the value of symbol in alt_symbol.
//    symbol     -- current symbol to look for
//    line_no    -- if found in current context, define must be before 'line'
//    tag_files      -- (reference to _str[]) list of tag files to search
//    alt_symbol -- (reference) returns value of #define
// Returns the number of matches was found, 0 of none found.
//
int VSAPI tag_check_for_define(VSPSZ symbol, int line_no,
                               VSHREFVAR tag_files, VSHREFVAR alt_symbol);

// Look up 'symbol' and see if it is a template class.
// If so, return the signature of the template in template_sig.
//    symbol         -- current symbol to look for
//    outer_class    -- class that 'symbol' must be in
//    case_sensitive -- case_sensitive symbol comparison?
//    tag_files      -- (reference to _str[]) tag files to search
//    template_sig   -- (reference) returns signature of template_sig
// Returns 1 if a match was found and sets template_sig, 0 of none found.
// The current object must be an editor control or current buffer.
//
int VSAPI tag_check_for_template(VSPSZ symbol, VSPSZ outer_class, int case_sensitive,
                                 VSHREFVAR tag_files, VSHREFVAR template_sig);

// Look up 'symbol' and see if it is a package, namespace, module or unit.
//    symbol         -- current symbol to look for
//    tag_files      -- (reference to _str[]) list of tag files to search
//    exact_match    -- look for exact match rather than prefix match
//    case_sensitive -- case sensitive comparison?
// Returns 1 if 'symbol' or prefix of matches package, otherwise returns 0.
//
int VSAPI tag_check_for_package(VSPSZ symbol, VSHREFVAR tag_files,
                                int exact_match, int case_sensitive);

// Check if the given class is in the same package as the other class.
//    class1    -- first class name, qualified
//    class2    -- second class name, qualified
//    case_sens -- case sensitive comparison for package names?
// Returns true of so, false otherwise.
//
int VSAPI tag_is_same_package(VSPSZ class1, VSPSZ class2, int case_sensitive);

//////////////////////////////////////////////////////////////////////////////
// List the global symbols of the given type
//    treewid      -- window id of tree control to insert into,
//                    0 indicates to insert into a match set
//    tree_index   -- tree index to insert matches under
//    tag_files    -- (reference to _str[]) tag files to search
//    type_id      --  first type id (see VS_TAGTYPE_*, above)
//                     if (type_id<0), returns tags with ID>VS_TAGTYPE_LASTID
//    mask         --  flag mask (see VS_TAGFLAG_*, above)
//    nzero        --  if 1, succeed if mask & tag.flags != 0
//                     if 0, succeed if mask & tag.flags == 0
//    vnum_matches -- (reference) number of matches
//    max_matches  -- maximum number of matches allowed
// Returns nothing.
//
void VSAPI tag_list_globals_of_type(int treewid, int tree_index, VSHREFVAR tag_files,
                                    int type_id, int mask, int nonzero,
                                    VSHREFVAR vnum_matches, int max_matches);

// List the packages matching the given prefix expression.
//    treewid      -- window id of tree control to insert into,
//                    0 indicates to insert into a match set
//    tree_index   -- tree index to insert matches under
//    prefix       -- symbol prefix to match
//    tag_files    -- (reference to _str[]) tag files to search
//    vnum_matches -- (reference) number of matches
//    max_matches  -- maximum number of matches allowed
//    exact_match  -- exact match or prefix match (0)
//    case_sens    -- case sensitive (1) or case insensitive (0)
// Nothing is returned.
//
void VSAPI tag_list_context_packages(int treewid, int tree_index,
                                     VSPSZ prefix, VSHREFVAR tag_files,
                                     VSHREFVAR vnum_matches, int max_matches,
                                     int exact_match, int case_sensitive);

//////////////////////////////////////////////////////////////////////////////
// List any symbols, reguardless of context or scope (excluding locals)
// matching the given prefix expression.
//    treewid      -- window id of tree control to insert into,
//                    0 indicates to insert into a match set
//    tree_index   -- tree index to insert matches under
//    prefix       -- symbol prefix to match
//    tag_files    -- (reference to _str[]) tag files to search
//    pushtag_flags -- VS_TAGFILTER_*, tag filter flags
//    context_flags -- VS_TAGCONTEXT_*, tag context filter flags
//    vnum_matches -- (reference) number of matches
//    max_matches  -- maximum number of matches allowed
//    exact_match  -- exact match or prefix match (0)
//    case_sens    -- case sensitive (1) or case insensitive (0)
// Nothing is returned.
//
void VSAPI tag_list_any_symbols(int treewid,int tree_index, VSPSZ prefix,
                                VSHREFVAR tag_files, int pushtag_flags,int context_flags,
                                VSHREFVAR vnum_matches,int max_matches,
                                int exact_match,int case_sensitive);

// List the symbols found in files having the given 'base' filename
// and passing the given pushtag and context flags.
//    treewid       -- window id of tree control to insert into,
//                     0 indicates to insert into a match set
//    tree_index    -- tree index to insert matches under
//    prefix        -- symbol prefix to match
//    tag_files     -- (reference to _str[]) tag files to search
//    pushtag_flags -- VS_TAGFILTER_*, tag filter flags
//    context_flags -- VS_TAGCONTEXT_*, tag context filter flags
//    vnum_matches  -- (reference) number of matches
//    max_matches   -- maximum number of matches allowed
//    exact_match   -- exact match or prefix match (0)
//    case_sens     -- case sensitive (1) or case insensitive (0)
// Nothing is returned.
//
void VSAPI tag_list_in_file(int treewid, int tree_index, VSPSZ prefix,
                            VSHREFVAR tag_files, VSPSZ search_file_name,
                            int pushtag_flags, int context_flags,
                            VSHREFVAR vnum_matches, int max_matches,
                            int exact_match, int case_sensitive);

// List the global symbols visible in the given list of tag files
// matching the given tag filters and context flags.
//    treewid       -- window id of tree control to insert into,
//                     0 indicates to insert into a match set
//    tree_index    -- tree index to insert matches under
//    prefix        -- symbol prefix to match
//    check_context -- check for symbols in the current context?
//    tag_files     -- (reference to _str[]) tag files to search
//    pushtag_flags -- VS_TAGFILTER_*, tag filter flags
//    context_flags -- VS_TAGCONTEXT_*, tag context filter flags
//    vnum_matches  -- (reference) number of matches
//    max_matches   -- maximum number of matches allowed
//    exact_match   -- exact match or prefix match (0)
//    case_sens     -- case sensitive (1) or case insensitive (0)
// Nothing is returned.
//
void VSAPI tag_list_context_globals(int treewid, int tree_index, VSPSZ prefix,
                                    int check_context, VSHREFVAR tag_files,
                                    int pushtag_flags, int context_flags,
                                    VSHREFVAR vnum_matches, int max_matches,
                                    int exact_match, int case_sensitive);

//////////////////////////////////////////////////////////////////////////////
// Attempt to locate the given symbol in the given class by searching
// local variables.  Recursively looks for symbols in enumerated
// types and anonymous unions (designated by having *both* the anonymous
// and 'maybe_var' tag flags).
//    treewid       -- window id of tree control to insert into,
//                     0 indicates to insert into a match set
//    tree_index    -- tree index to insert matches under
//    prefix        -- symbol prefix to match
//    search_class  -- name of class to search for matches
//    tag_files     -- (reference to _str[]) tag files to search
//    pushtag_flags -- VS_TAGFILTER_*, tag filter flags
//    context_flags -- VS_TAGCONTEXT_*, tag context filter flags
//    vnum_matches  -- (reference) number of matches
//    max_matches   -- maximum number of matches allowed
//    exact_match   -- exact match or prefix match (0)
//    case_sens     -- case sensitive (1) or case insensitive (0)
// Look at num_matches to see if any matches were found.  Generally
// if (num_matches >= max_matches) there may be more matches, but
// the search terminated early.  Returns 1 if the definition of the given
// class 'search_class_name' is found, othewise returns 0, indicating
// that no matches were found.
//
int VSAPI tag_list_class_locals(int treewid, int tree_index, VSHREFVAR tag_files,
                                VSPSZ prefix, VSPSZ search_class,
                                int pushtag_flags, int context_flags,
                                VSHREFVAR vnum_matches, int max_matches,
                                int exact_match, int case_sensitive);

//////////////////////////////////////////////////////////////////////////////
// Attempt to locate the given symbol in the given class by searching
// the current context.  Recursively looks for symbols in enumerated
// types and anonymous unions (designated by having *both* the anonymous
// and 'maybe_var' tag flags).
//    treewid       -- window id of tree control to insert into,
//                     0 indicates to insert into a match set
//    tree_index    -- tree index to insert matches under
//    prefix        -- symbol prefix to match
//    search_class  -- name of class to search for matches
//    tag_files     -- (reference to _str[]) tag files to search
//    pushtag_flags -- VS_TAGFILTER_*, tag filter flags
//    context_flags -- VS_TAGCONTEXT_*, tag context filter flags
//    vnum_matches  -- (reference) number of matches
//    max_matches   -- maximum number of matches allowed
//    exact_match   -- exact match or prefix match (0)
//    case_sens     -- case sensitive (1) or case insensitive (0)
// Look at num_matches to see if any matches were found.  Generally
// if (num_matches >= max_matches) there may be more matches, but
// the search terminated early.  Returns 1 if the definition of the given
// class 'search_class_name' is found, othewise returns 0, indicating
// that no matches were found.
//
int VSAPI tag_list_class_context(int treewid, int tree_index, VSHREFVAR tag_files,
                                 VSPSZ prefix, VSPSZ search_class,
                                 int pushtag_flags, int context_flags,
                                 VSHREFVAR vnum_matches, int max_matches,
                                 int exact_match, int case_sensitive);

//////////////////////////////////////////////////////////////////////////////
// Attempt to locate the given symbol in the given class by searching
// the given tag files.  Recursively looks for symbols in enumerated
// types and anonymous unions (designated by having *both* the anonymous
// and 'maybe_var' tag flags).
//    treewid       -- window id of tree control to insert into,
//                     0 indicates to insert into a match set
//    tree_index    -- tree index to insert matches under
//    prefix        -- symbol prefix to match
//    search_class  -- name of class to search for matches
//    tag_files     -- (reference to _str[]) tag files to search
//    pushtag_flags -- VS_TAGFILTER_*, tag filter flags
//    context_flags -- VS_TAGCONTEXT_*, tag context filter flags
//    vnum_matches  -- (reference) number of matches
//    max_matches   -- maximum number of matches allowed
//    exact_match   -- exact match or prefix match (0)
//    case_sens     -- case sensitive (1) or case insensitive (0)
// Look at num_matches to see if any matches were found.  Generally
// if (num_matches >= max_matches) there may be more matches, but
// the search terminated early.  Returns 1 if the definition of the given
// class 'search_class_name' is found, othewise returns 0, indicating
// that no matches were found.
//
int VSAPI tag_list_class_tags(int treewid, int tree_index, VSHREFVAR tag_files,
                              VSPSZ prefix, VSPSZ search_class,
                              int pushtag_flags, int context_flags,
                              VSHREFVAR vnum_matches, int max_matches,
                              int exact_match, int case_sensitive);


} // extern "C"


#endif
// TAGS_CONTEXT_H
