#ifndef TAGS_DB_H
#define TAGS_DB_H

#include <vsapi.h>


extern "C" {

//////////////////////////////////////////////////////////////////////
// "C" style API for creating and accessing BTREE tag database.
//

//////////////////////////////////////////////////////////////////////
// Flags associated with tags, denoting access restrictions and
// and other attributes of class members (proc's, proto's, and var's)
//    NOT virtual and NOT static implies normal class method
//    NOT protected and NOT private implies public
//    NOT const implies normal read/write access
//    NOT volatile implies normal optimizations are safe
//    NOT template implies normal class definition
//
#define VS_TAGFLAG_virtual      0x0001  // virtual function (instance)
#define VS_TAGFLAG_static       0x0002  // static method / member (class)
#define VS_TAGFLAG_access       0x000C  // public/protected/private/package
#define VS_TAGFLAG_public       0x0000  // test equality with (flags & access)
#define VS_TAGFLAG_protected    0x0004  // protected access
#define VS_TAGFLAG_private      0x0008  // private access
#define VS_TAGFLAG_package      0x000C  // package access (for Java)
#define VS_TAGFLAG_const        0x0010  // const
#define VS_TAGFLAG_final        0x0020  // final
#define VS_TAGFLAG_abstract     0x0040  // abstract/deferred method
#define VS_TAGFLAG_inline       0x0080  // inline / out-of-line method
#define VS_TAGFLAG_operator     0x0100  // overloaded operator
#define VS_TAGFLAG_constructor  0x0200  // class constructor
#define VS_TAGFLAG_volatile     0x0400  // volatile method
#define VS_TAGFLAG_template     0x0800  // template class
#define VS_TAGFLAG_inclass      0x1000  // part of class interface?
#define VS_TAGFLAG_destructor   0x2000  // class destructor
#define VS_TAGFLAG_const_destr  0x2200  // class constructor or destructor
#define VS_TAGFLAG_synchronized 0x4000  // synchronized (thread safe)
#define VS_TAGFLAG_transient    0x8000  // transient / persistent data

#define VS_TAGFLAG_native       0x00010000  // Java native method?
#define VS_TAGFLAG_macro        0x00020000  // Tag was part of macro expansion?
#define VS_TAGFLAG_extern       0x00040000  // "extern" C prototype (not local)
#define VS_TAGFLAG_maybe_var    0x00080000  // Prototype which could be a variable.  Anonymous union.
#define VS_TAGFLAG_anonymous    0x00100000  // Anonymous structure or class
#define VS_TAGFLAG_mutable      0x00200000  // mutable C++ class member


//////////////////////////////////////////////////////////////////////
// Flags passed to tag to extract specific information about the
// current tag, using tag_get_detail(), below
//
#define VS_TAGDETAIL_name           0  // (string) tag name
#define VS_TAGDETAIL_type           1  // (string) tag type
#define VS_TAGDETAIL_type_id        2  // (int) unique id for tag type (VS_TAGTYPE_*)
#define VS_TAGDETAIL_file_name      3  // (string) full path of file the tag is located in
#define VS_TAGDETAIL_file_date      4  // (string) modification data of file when tagged
#define VS_TAGDETAIL_file_line      5  // (int) line number of tag within file
#define VS_TAGDETAIL_file_id        6  // (int) unique id for file the tag is located in
#define VS_TAGDETAIL_class_simple   7  // (string) name of class the tag is present in
#define VS_TAGDETAIL_class_name     8  // (string) name of class with outer classes
#define VS_TAGDETAIL_class_package  9  // (string) package/module/namespace tag belongs to
#define VS_TAGDETAIL_class_id      10  // (int) unique id for class tag belongs to
#define VS_TAGDETAIL_flags         11  // (int) tag flags (see VS_TAGFLAG_* above)
#define VS_TAGDETAIL_return        12  // (string) return type for functions, type of variables
#define VS_TAGDETAIL_arguments     13  // (string) function arguments
#define VS_TAGDETAIL_tagseekloc    14  // (int) PRIVATE
#define VS_TAGDETAIL_fileseekloc   15  // (int) PRIVATE
#define VS_TAGDETAIL_classseekloc  16  // (int) PRIVATE
#define VS_TAGDETAIL_num_tags      17  // (int) number of tags in database
#define VS_TAGDETAIL_num_classes   18  // (int) number of classes in database
#define VS_TAGDETAIL_num_files     19  // (int) number of files in database
#define VS_TAGDETAIL_num_types     20  // (int) number of types in database
#define VS_TAGDETAIL_num_refs      21  // (int) number of references in database
#define VS_TAGDETAIL_LASTID        21  // last tag detail id

//////////////////////////////////////////////////////////////////////
// Characters used as seperators when storing compound strings
// in the database.
//
#define VS_TAGSEPARATOR_class      ':'  // seperates nested class names
#define VS_TAGSEPARATOR_package    '/'  // seperates class from package name
#define VS_TAGSEPARATOR_args       '\1' // seperators arguments from return type
#define VS_TAGSEPARATOR_parents    ';'  // seperators list of class parents

#define VS_TAGSEPARATOR_zclass     ":"  // seperates nested class names
#define VS_TAGSEPARATOR_zpackage   "/"  // seperates class from package name
#define VS_TAGSEPARATOR_zargs      "\1" // seperators arguments from return type
#define VS_TAGSEPARATOR_zparents   ";"  // seperators list of class parents

//////////////////////////////////////////////////////////////////////
// NOTE: administrative functions are now in tagsmain.h
//
// insertion and removal of tags

// Insert the given tag with accompanying information into the
// the database.  This is the easiest to use version of insert,
// since you do not need to know the ID of the tag type, you
// simply pass a string.
//    tag_name      -- tag string
//    tag_type      -- string specifying tag_type (see above for
//                     list of standard type names).  If the string
//                     is not a standard type, a new type will be
//                     created and inserted in the tag database.
//    file_name     -- full path of file the tag is located in
//    line_no       -- (optional) line number of tag within file
//    class_name    -- (optional) name of class that tag is present in,
//                     use concatenation (as defined by language rules)
//                     to specify names of inner classes.
//    tag_flags     -- (optional) see VS_TAGFLAG_* above.
//    signature     -- (optional) tag signature (return type, arguments, etc)
// If a record with the same tag name, type, file, *and* class
// already exists in the database, the line number will be updated.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_insert_tag(VSPSZ tag_name, VSPSZ tag_type,
                         VSPSZ file_name, int line_no,
                         VSPSZ class_name, int tag_flags,
                         VSPSZ signature);

// Insert the given tag with the accompanying information into the
// database.  This function is identical to vsTagInsert, except
// that rather than passing in a tag type, you pass in an int, using
// one of the standard types defined above (see VS_TAGTYPE_*).
//    tag_name      -- tag string
//    tag_type      -- tag type (see VS_TAGTYPE_*), will return an error
//                     if (tag_type <= 0 || tag_type > VSTAGTYPE_LASTID).
//    file_name     -- full path of file the tag is located in
//    line_no       -- (optional) line number of tag within file
//    class_name    -- (optional) name of class that tag is present in,
//                     use concatenation (as defined by language rules)
//                     to specify names of inner classes.
//    tag_flags     -- (optional) see VS_TAGFLAG_* above.
//    signature     -- (optional) tag signature (return type, arguments, etc)
// If a record with the same tag name, type, file, *and* class
// already exists in the database, the line number will be updated.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_insert_simple(VSPSZ tag_name, int type_id,
                            VSPSZ file_name, int line_no,
                            VSPSZ class_name, int tag_flags,
                            VSPSZ signature);

// Remove the current tag (most recently retrieved tag) from the database.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_remove_tag();

// Remove all tags associated with the given class from the database.
//    class_name    -- full name of the class the tag is associated with
//                     if NULL, all non-class tags are removed.
//    remove_class  -- if non-zero, the class is removed from the database,
//                     in addition to the tags associated with the class.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_remove_from_class(VSPSZ class_name /*, int remove_class */);

// Modify the set of parent classes for a given class.
//    class_name    -- class to modify parent relationships for
//    parents       -- classes that 'class_name' inherits from, semicolon separated
// Use the NULL or empty string to indicate that class_name is a base class.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_set_inheritance(VSPSZ class_name, VSPSZ parents);

// Retrieve the set of parent classes for a given class.
//    class_name    -- class to modify parent relationships for
//    parents       -- classes that 'class_name' inherits from, semicolon separated
// This function has the side effect of position the class iterator on the
// given class.  Returns BT_RECORD_NOT_FOUND_RC if class_name is not
// in the database.  Returns 0 on success, <0 on error.
//
int VSAPI tag_get_inheritance(VSPSZ class_name, VSHREFVAR parents);


//////////////////////////////////////////////////////////////////////
// Retrieval functions

// Retrieve first tag with the given tag name, type, and class
// (all are necessary to uniquely identify a tag).  If class_name
// is unknown, simply use NULL.
// Use vsTagGetInfo (below) to extract the details about the tag.
//    tag_name   -- name of tag to search for
//    tag_type   -- tag type name (see VS_TAGTYPE_*)
//    class_name -- name of class that tag is present in,
//                  use concatenation (as defined by language rules)
//                  to specify names of inner classes.
//    arguments  -- (optional) function arguments to attempt to
//                  match.  Ignored if they result in no matches.
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_find_tag(VSPSZ tag_name, VSPSZ tag_type, VSPSZ class_name /*,VSPSZ arguments*/);

// Retrieve next tag with the given tag name, type, and class
// (all are necessary to uniquely identify a tag).  If class_name
// is unknown, simply use NULL.
// Use vsTagGetInfo (below) to extract the details about the tag.
// Should be called only after calling tag_find_tag.
//    tag_name   -- name of tag to search for
//    tag_type   -- tag type name (see VS_TAGTYPE_*)
//    class_name -- name of class that tag is present in,
//                  use concatenation (as defined by language rules)
//                  to specify names of inner classes.
//    arguments  -- (optional) function arguments to attempt to
//                  match.  Ignored if they result in no matches.
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_next_tag(VSPSZ tag_name, VSPSZ tag_type, VSPSZ class_name /*,VSPSZ arguments*/);

// Retrieve first with the given tag name.
// Use tag_get_info (below) to extract the details about the tag.
//    tag_name   -- name of tag to search for
//    file_name  -- full path to file containing tag
//    line_no    -- line that tag is expected to be present on
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_find_closest(VSPSZ tag_name, VSPSZ file_name, int line_no);

// Retrieve first tag with the given tag name (case-insensitive).
// Use vsTagGetInfo (below) to extract the details about the tag.
//    tag_name   -- name of tag to search for
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_find_equal(VSPSZ tag_name /*,int case_sensitive, VSPSZ class_name*/);

// Retrieve the next tag with the same tag name as the last one retrieved.
// Should be called only after calling tag_find_equal or tag_find_tag.
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_next_equal(/*int case_sensitive, VSPSZ class_name*/);

// Retrieve the first tag with the given tag name (case-insensitive).
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_find_prefix(VSPSZ tag_prefix /*,int case_sensitive, VSPSZ class_name*/);

// Retrieve the next tag with the given prefix (case-insensitive).
// Should be called only after calling vsTagGetTagPrefix().
//    tag_prefix -- name of tag to search for
// Returns 0 on success, <0 on error, or if no such tag.
//
int VSAPI tag_next_prefix(VSPSZ tag_prefix /*,int case_sensitive, VSPSZ class_name*/);

// Retrieve information about the current tag (as defined by calls
// to getTagEQ, getTagPrefix, getNextEQ, getNextPrefix).  If no such
// tag, all strings will be set to the empty string, and line_no will
// be set to 0.
//    tag_name   -- (reference) tag string (native case)
//    tag_type   -- (reference) string specifying tag_type
//                  (see above for list of standard type names).
//    file_name  -- (reference) full path of file the tag is located in
//    line_no    -- (reference) line number of tag within file
//                  set to 0 if not defined.
//    class_name -- (reference) name of class that tag is present in,
//                  uses concatenation (as defined by language rules)
//                  to specify names of inner classes (see insert, above).
//                  set to empty string if not defined.
//    tag_flags  -- (reference) see VS_TAGFLAG_* above.
//
void VSAPI tag_get_info(VSHREFVAR tag_name, VSHREFVAR type_name,
                        VSHREFVAR file_name, VSHREFVAR line_no,
                        VSHREFVAR class_name, VSHREFVAR tag_flags);

// Retrieve specific details about the current tag (as defined by calls
// to getTagEQ, getTagPrefix, getNextEQ, getNextPrefix).  If no such
// tag, all strings will be set to the empty string, and ints to 0.
// See VS_TAGDETAIL_*, above.
//    tag_detail -- ID of detail to extract (VS_TAGDETAIL_*)
//    result     -- (reference) set to value of requested tag detail
//
void VSAPI tag_get_detail(int tag_detail, VSHREFVAR result);


//////////////////////////////////////////////////////////////////////
// file-name based retrieval functions

// Find the first tag in the given file.
//    file_name  -- full path of file containing tags
// Tags will be returned in lexicographic, case-insensitive order,
// with no preference to tag type or class.
//
int VSAPI tag_find_in_file(VSPSZ file_name);

// Find the next tag in the current file.
// Tags will be returned in lexicographic, case-insensitive order,
// with no preference to tag type or class.
//
int VSAPI tag_next_in_file();


//////////////////////////////////////////////////////////////////////
// class-name based retrieval functions

// Retreive the name of the first class included in this tag database.
//    class_id      -- id of class/package, from tag_get_detail()
//    class_name    -- (reference) name of class
// Class names are ordered lexicographically, case-sensitive.
//
int VSAPI tag_get_class(int class_id, VSHREFVAR class_name);

// Retreive the name of the first class included in this tag database.
//    class_name    -- (reference) name of class
//    search_for    -- (optional) specific class to search for (prefix search)
//    normalize     -- normalize the class name (find what package it belongs to)
//    ignore_case   -- perform case-insensitive search? (default is case-sensitive)
// Class names are ordered lexicographically, case-sensitive.
//
int VSAPI tag_find_class(VSHREFVAR class_name /*, search_for, normalize, ignore_case*/);

// Retreive the name of the next class included in this tag database.
//    class_name    -- (reference) name of class
//    search_for    -- (optional) specific class to search for (prefix search)
//    normalize     -- normalize the class name (find what package it belongs to)
//    ignore_case   -- perform case-insensitive search? (default is case-sensitive)
// Class names are ordered lexicographically, case-sensitive.
//
int VSAPI tag_next_class(VSHREFVAR class_name /*,search_for, normalize, ignore_case*/);

// Find the first class in the given class.
//    class_name    -- name of class, containing tags
// Tags will be returned in lexicographic, case-insensitive order,
// with no preference to tag type or file path.
//
int VSAPI tag_find_in_class(VSPSZ class_name);

// Find the next tag in the given class.
// Tags will be returned in lexicographic, case-insensitive order,
// with no preference to tag type or file path.
//
int VSAPI tag_next_in_class();


//////////////////////////////////////////////////////////////////////
// global identifier retrieval functions

// Retrieve the first tag included in this tag database with global
// scope that is one of the given type (type_id) and that
// matches the given tag flag mask (mask & tag.mask != 0).
//    type_id --  first type id (see VS_TAGTYPE_*, above)
//                if (type_id<0), returns tags with ID>VS_TAGTYPE_LASTID
//    mask    --  flag mask (see VS_TAGFLAG_*, above)
//    nzero   --  if 1, succeed if mask & tag.flags != 0
//               if 0, succeed if mask & tag.flags == 0
// Tag names are ordered lexicographically, case insensitive
//
int VSAPI tag_find_global(int type_id, int mask, int nzero);

// Retrieve the next tag included in this tag database with global
// scope that is one of the given type (type_id) and that
// matches the given tag flag mask (mask & tag.mask != 0).
//    type_id --  first type id (see VS_TAGTYPE_*, above)
//                if (type_id<0), returns tags with ID>VS_TAGTYPE_LASTID
//    mask    --  flag mask (see VS_TAGFLAG_*, above)
//    nzero   --  if 1, succeed if mask & tag.flags != 0
//                if 0, succeed if mask & tag.flags == 0
// Tag names are ordered lexicographically, case insensitive
//
int VSAPI tag_next_global(int type_id, int mask, int nzero);


//////////////////////////////////////////////////////////////////////
// private functions used only by tagsdb.c and tagstree.c

// Get pointers to essential information for current tag.
// The pointers returned are pointers to staticly allocated strings.
// In contrast with tag_get_info, this function does not return
// the class name.  This is because the context in which this
// function is used does not normally require a class name.
//    tag_name      -- tag string (native case)
//    tag_type      -- string specifying tag_type
//                     (see above for list of standard type names).
//    type_id       -- ID for tag type (see VS_TAGTYPE_*)
//    file_name     -- full path of file the tag is located in
//    file_id       -- unique ID of file within tag database
//    line_no       -- line number of tag within file
//                     set to 0 if not defined.
//    tag_flags     -- see VS_TAGFLAG_* above.
//    signature     -- function/macro/class arguments
//    class_id      -- unique ID of class tag belongs to
//    class_name    -- 0 if not interested in class name, otherwise
//                     the class name will be copied into this pointer
//                     which must have at least VS_TAG_MAX_CLASSNAME
//    raw_signature -- 0 if not interested in raw signature & return type
//                     the raw signature (return type '\1' arguments)
//                     will be copied into this pointer, which must have
//                     VS_TAG_MAX_SIGNATURE characters allocated
//
void vsTagGetInfo(const char *&tag_name, const char *&tag_type, int &type_id,
                  const char *&file_name, int &file_id, int &line_no,
                  int &tag_flags, const char *&signature,
                  int &class_id, char *class_name=0, char *raw_signature=0);

// Fast "C" style entry point for retrieving only the tag type and flags
// information for the current tag.
//    tag_type      -- string specifying tag_type
//                     (see above for list of standard type names).
//    type_id       -- ID for tag type (see VS_TAGTYPE_*)
//    tag_flags     -- see VS_TAGFLAG_* above.
//
void vsTagGetTypeInfo(const char *&type_name, int &type_id, int &flags);

// Fast "C" style entry point for retrieving only the tag name and class
// information for the current tag.
//    member_name   -- tag name
//    class_id      -- ID (foreign key) for class name
//    class_name    -- class name
//    max_class_len -- size of buffer allocated for class name
//
void vsTagGetClassInfo(const char *&member_name, int &class_id,
                       char *class_name=0, int max_class_len=0);

// Returns 1 if the current tag name, interpreted as a class
// is not derived from another class.
//    class_name -- class to try looking up
int vsTagIsBaseClass(const char *class_name);

// Generic implementation of for tag_find_equal, tag_find_next,
// tag_find_prefix, and tag_next_prefix.
// Do initial search or start with next tag after current and
// compare tag name and class name, possibly case_sensitive or
// prefix match until we find a match.  If the class_name
// is NULL, do not compare class names, if it is the
// empty string, search specificly for globals.
//    tag_name       -- tag name or prefix
//    class_name     -- class name to check for
//    exact_match    -- nonzero for exact match, zero for prefix
//    case_sensitive -- case sensitive comparison?
//    find_first     -- do initial search first?
//    skip_duplicates-- get next record not equal to current
// Returns 0 if a match is found, <0 on error or not found.
//
int vsTagFindEqual(const char *tag_name, const char *class_name=0, 
                   int exact_match=1, int case_sensitive=0,
                   int find_first=1, int skip_duplicates=0);

// Find the first tag with the given, name, type and class.
//    tag_name    -- tag name to find, exact match
//    type_id     -- tag type to search for tag_name having (VS_TAGTYPE_*)
//    class_name  -- class name to search for tag_name having
//    find_first  -- find first match, or next match
//    signature   -- function signature, ignored if not match found
// Returns 0 on success, <0 on error or not found.
//
int vsTagFindTag(const char *tag_name, int type_id, const char *class_name,
                 int find_first=1, const char *signature=0);

// Internal function to return the first class name in the database,
// matching 'search_class', normalize the class name if necessary
// (find the class/package scope that this class belongs to).
//    search_class   -- name of class to look for, can be NULL
//    class_name     -- (output) output buffer for full class name
//    max_len        -- size of output buffer (class_name)
//    normalize      -- attempt to normalize class name found?
//    case_sensitive -- case sensitive search?
//    find-first     -- find first match or next match
// If given a non-zero second argument, will attempt to
// normalize the class name, finding a package scope that this
// class could belong to.  Returns 0 on success, <0 on error.
//
int vsTagFindClass(const char *search_class, char *class_name,
                   int max_len, int normalize_class=0,
                   int case_sensitve=1, int find_first=1);

// API function to retrieve the name of the class with the
// given class ID.
int vsTagGetClass(int class_id, char *class_name, int max_len);

// Internal function for retrieving the list of parents of
// the given class.
//    class_name  -- name of class to look up
//    parents     -- (output) output array
//    max_len     -- size of output buffer (parents)
// Returns 0 on success, <0 on error.
//
int vsTagGetInheritance(VSPSZ class_name, char *parents, int max_len);


} // extern "C"


#endif
// TAGS_DB_H
