// ---------------------------------------------------------------------------
//  M88 - PC-8801 emulator
//  Copyright (C) cisc 1998, 1999.
// ---------------------------------------------------------------------------
//  $Id: 88config.cpp,v 1.14 1999/06/30 14:07:02 cisc Exp $

#include "headers.h"
#include <stdlib.h>
#include "resource.h"
#include "88config.h"
#include "misc.h"
#include "messages.h"

static const char* AppName = "M88p2 for Windows";

#define BSTATE(b) (b ? BST_CHECKED : BST_UNCHECKED)

namespace PC8801
{
#ifdef __OS2__
inline int Limit(int v, int l, int s) { return v > l ? l : (v < s ? s : v); }
#endif

// ---------------------------------------------------------------------------
//  LoadConfigEntry
//
#ifndef __OS2__
static bool LoadConfigEntry
(const char* inifile, const char* entry, int* value, int def, bool applydefault)
{
    int n = GetPrivateProfileInt(AppName, entry, -1, inifile);

    if (n == -1 && applydefault)
        n = def;
    if (n != -1)
    {
        *value = n;
        return true;
    }
    return false;
}
#else
static bool LoadConfigEntry
(const char* inifile, const char* entry, int* value, int def, bool applydefault)
{
    int n = -1;

    if (n == -1 && applydefault)
        n = def;
    if (n != -1)
    {
        *value = n;
        return true;
    }
    return false;
}
#endif

// ---------------------------------------------------------------------------
//  LoadConfigDirectory
//
#ifndef __OS2__
void LoadConfigDirectory(Config* cfg, const char* inifile, const char* entry)
{
    if (cfg->flags & Config::savedirectory)
    {
        char path[_MAX_PATH];
        if (GetPrivateProfileString(AppName, entry, ";", path, _MAX_PATH, inifile))
        {
            if (path[0] != ';')
                SetCurrentDirectory(path);
        }
    }
}
#else
void LoadConfigDirectory(Config* cfg, const char* inifile, const char* entry)
{
    if (cfg->flags & Config::savedirectory)
    {
    }
}
#endif

// ---------------------------------------------------------------------------
//  M88Config::LoadConfig
//
#define VOLUME_BIAS 100

#define LOADVOLUMEENTRY(key, vol) \
    if (LoadConfigEntry(inifile, key, &n, VOLUME_BIAS, applydefault)) \
    vol = n - VOLUME_BIAS;

void LoadConfig(Config* cfg, const char* inifile, bool applydefault)
{
    int n;

    n = Config::subcpucontrol | Config::savedirectory;
    LoadConfigEntry(inifile, "Flags", &cfg->flags, n, applydefault);

    LoadConfigEntry(inifile, "Flag2", &cfg->flag2, 0, applydefault);

    if (LoadConfigEntry(inifile, "CPUClock", &n, 40, applydefault))
        cfg->clock = Limit(n, 1000, 1);

    if (LoadConfigEntry(inifile, "Speed", &n, 1000, applydefault))
        cfg->speed = Limit(n, 2000, 500);

    if (LoadConfigEntry(inifile, "RefreshTiming", &n, 3, applydefault))
        cfg->refreshtiming = Limit(n, 4, 1);

    if (LoadConfigEntry(inifile, "BASICMode", &n, Config::N88V2, applydefault))
    {
        if (   n == Config::N80    || n == Config::N88V1
            || n == Config::N88V1H || n == Config::N88V2 || n == Config::N802)
            cfg->basicmode = Config::BASICMode(n);
        else
            cfg->basicmode = Config::N88V2;
    }

    if (LoadConfigEntry(inifile, "Sound", &n, 2, applydefault))
        cfg->sound = Limit(n, 5, 0);

    if (LoadConfigEntry(inifile, "OPNClock", &n, 3993600, applydefault))
        cfg->opnclock = Limit(n, 10000000, 1000000);

    if (LoadConfigEntry(inifile, "ERAMBank", &n, 4, applydefault))
        cfg->nerams = Limit(n, 256, 4);

    if (LoadConfigEntry(inifile, "KeyboardType", &n, 0, applydefault))
        cfg->keytype = Config::KeyType(n);

    if (LoadConfigEntry(inifile, "Switches", &n, 1829, applydefault))
        cfg->dipsw = n;

    if (LoadConfigEntry(inifile, "SoundBuffer", &n, 200, applydefault))
        cfg->soundbuffer = Limit(n, 1000, 100);

    if (LoadConfigEntry(inifile, "MouseSensibility", &n, 4, applydefault))
        cfg->mousesensibility = Limit(n, 10, 1);

    if (LoadConfigEntry(inifile, "CPUMode", &n, Config::msauto, applydefault))
        cfg->cpumode = Limit(n, 2, 0);

    LOADVOLUMEENTRY("VolumeFM", cfg->volfm);
    LOADVOLUMEENTRY("VolumeSSG", cfg->volssg);
    LOADVOLUMEENTRY("VolumeADPCM", cfg->voladpcm);
    LOADVOLUMEENTRY("VolumeRhythm", cfg->volrhythm);
    LOADVOLUMEENTRY("VolumeBD", cfg->volbd);
    LOADVOLUMEENTRY("VolumeSD", cfg->volsd);
    LOADVOLUMEENTRY("VolumeTOP", cfg->voltop);
    LOADVOLUMEENTRY("VolumeHH", cfg->volhh);
    LOADVOLUMEENTRY("VolumeTOM", cfg->voltom);
    LOADVOLUMEENTRY("VolumeRIM", cfg->volrim);
}

// ---------------------------------------------------------------------------
//  SaveEntry
//
#ifndef __OS2__
static bool SaveEntry
(const char* inifile, const char* entry, int value, bool applydefault)
{
    char buf[_MAX_PATH];
    if (applydefault || -1 != GetPrivateProfileInt(AppName, entry, -1, inifile))
    {
        wsprintf(buf, "%d", value);
        WritePrivateProfileString(AppName, entry, buf, inifile);
        return true;
    }
    return false;
}
#else
static bool SaveEntry
(const char* inifile, const char* entry, int value, bool applydefault)
{
    return true;
}
#endif

#ifndef __OS2__
static bool SaveEntry
(const char* inifile, const char* entry, const char* value, bool applydefault)
{
    bool apply = applydefault;
    if (!applydefault)
    {
        char buf[8];
        GetPrivateProfileString(AppName, entry, ";", buf, sizeof(buf), inifile);
        if (buf[0] != ';')
            apply = true;
    }
    if (apply)
        WritePrivateProfileString(AppName, entry, value, inifile);
    return apply;
}
#else
static bool SaveEntry
(const char* inifile, const char* entry, const char* value, bool applydefault)
{
    return false;
}
#endif

// ---------------------------------------------------------------------------
//  SaveConfig
//
#ifdef __OS2__
void SaveConfig(Config* cfg, const char* inifile, bool writedefault)
{
}
#else
void SaveConfig(Config* cfg, const char* inifile, bool writedefault)
{
    char buf[_MAX_PATH];
    GetCurrentDirectory(_MAX_PATH, buf);
    SaveEntry(inifile, "Directory", buf, writedefault);

    SaveEntry(inifile, "Flags", cfg->flags, writedefault);
    SaveEntry(inifile, "Flag2", cfg->flag2, writedefault);
    SaveEntry(inifile, "CPUClock", cfg->clock, writedefault);
    SaveEntry(inifile, "Speed", cfg->speed, writedefault);
    SaveEntry(inifile, "RefreshTiming", cfg->refreshtiming, writedefault);
    SaveEntry(inifile, "BASICMode", cfg->basicmode, writedefault);
    SaveEntry(inifile, "Sound", cfg->sound, writedefault);
    SaveEntry(inifile, "Switches", cfg->dipsw, writedefault);
    SaveEntry(inifile, "SoundBuffer", cfg->soundbuffer, writedefault);
    SaveEntry(inifile, "MouseSensibility", cfg->mousesensibility, writedefault);
    SaveEntry(inifile, "CPUMode", cfg->cpumode, writedefault);
    SaveEntry(inifile, "KeyboardType", cfg->keytype, writedefault);

    SaveEntry(inifile, "VolumeFM", cfg->volfm + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeSSG", cfg->volssg + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeADPCM", cfg->voladpcm + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeRhythm", cfg->volrhythm + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeBD", cfg->volbd + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeSD", cfg->volsd + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeTOP", cfg->voltop + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeHH", cfg->volhh + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeTOM", cfg->voltom + VOLUME_BIAS, writedefault);
    SaveEntry(inifile, "VolumeRIM", cfg->volrim + VOLUME_BIAS, writedefault);
}
#endif

// ---------------------------------------------------------------------------
//  \z/
//
#ifdef __OS2__
WinConfig::WinConfig()
{
}
#else
WinConfig::WinConfig()
{
    propproc.SetDestination(PropProcGate, this);
    cpupage.SetDestination(CPUPageGate, this);
    screenpage.SetDestination(ScreenPageGate, this);
    soundpage.SetDestination(SoundPageGate, this);
    soundvolpage.SetDestination(SoundVolumePageGate, this);
    functionpage.SetDestination(FunctionPageGate, this);
    switchpage.SetDestination(SwitchPageGate, this);
    envpage.SetDestination(EnvPageGate, this);

    page = 0;
    hwndps = 0;
}
#endif

WinConfig::~WinConfig()
{

}

// ---------------------------------------------------------------------------
//  ݒ_CAO̎s
//  ݒ̔f̓bZ[Wōs\
//
#ifdef __OS2__
bool WinConfig::Show(HWND hwnd, Config* conf)
{
    return false;
}
#else
bool WinConfig::Show(HINSTANCE hinstance, HWND hwnd, Config* conf)
{
    if (!hwndps)
    {
        orgconfig = config = *conf;
        hinst = hinstance;
        hwndparent = hwnd;

        PROPSHEETPAGE psp[term];
        PROPSHEETHEADER psh;

        FillInPropPage(&psp[cpu     ], IDD_CONFIG_CPU, cpupage);
        FillInPropPage(&psp[screen  ], IDD_CONFIG_SCREEN, screenpage);
        FillInPropPage(&psp[sound   ], IDD_CONFIG_SOUND, soundpage);
        FillInPropPage(&psp[soundvol], IDD_CONFIG_SOUNDVOL, soundvolpage);
        FillInPropPage(&psp[function], IDD_CONFIG_FUNCTION, functionpage);
        FillInPropPage(&psp[switches], IDD_CONFIG_SWITCHES, switchpage);
        FillInPropPage(&psp[env     ], IDD_CONFIG_ENV, envpage);

        memset(&psh, 0, sizeof(psh));
        psh.dwSize = sizeof(psh);
        psh.dwFlags = PSH_PROPSHEETPAGE | PSH_MODELESS;
        psh.hwndParent = hwndparent;
        psh.hInstance = hinst;
        psh.pszCaption = "ݒ";
        psh.nPages = sizeof(psp) / sizeof(PROPSHEETPAGE);
        psh.nStartPage = page;
        psh.ppsp = psp;
        psh.pfnCallback = (PFNPROPSHEETCALLBACK) (void*) propproc;

        hwndps = (HWND) PropertySheet(&psh);
    }
    else
    {
        SetFocus(hwndps);
    }
    return false;
}
#endif

#ifdef __OS2__
void WinConfig::Close()
{
}
#else
void WinConfig::Close()
{
    if (hwndps)
    {
        DestroyWindow(hwndps);
        hwndps = 0;
    }
}
#endif

// ---------------------------------------------------------------------------
//  PropSheetProc
//
#ifdef __OS2__
#else
bool WinConfig::ProcMsg(MSG& msg)
{
    if (hwndps)
    {
        if (PropSheet_IsDialogMessage(hwndps, &msg))
        {
            if (!PropSheet_GetCurrentPageHwnd(hwndps))
                Close();
            return true;
        }
    }
    return false;
}
#endif

#ifdef __OS2__
#else
int WinConfig::PropProc(HWND hwnd, UINT m, LPARAM)
{
    switch (m)
    {
    case PSCB_INITIALIZED:
        hwndps = hwnd;
        break;
    }
    return 0;
}
#endif

#ifdef __OS2__
#else
int _cdecl WinConfig::PropProcGate
(WinConfig* config, HWND hwnd, UINT m, LPARAM l)
{
    return config->PropProc(hwnd, m, l);
}
#endif

// ---------------------------------------------------------------------------
//  Fill in psp
//
#ifdef __OS2__
#else
void WinConfig::FillInPropPage(PROPSHEETPAGE * psp, int iddlg, void* dlgproc)
{
    memset(psp, 0, sizeof(PROPSHEETPAGE));
    psp->dwSize = sizeof(PROPSHEETPAGE);
    psp->dwFlags = 0;
    psp->hInstance = hinst;
    psp->pszTemplate = MAKEINTRESOURCE(iddlg);
    psp->pszIcon = 0;
    psp->pfnDlgProc = (DLGPROC) dlgproc;
    psp->lParam = 0;
}
#endif

// ---------------------------------------------------------------------------
//  CPU Page
//
#ifdef __OS2__
#else
BOOL WinConfig::CPUPage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {
    case WM_INITDIALOG:
        config.clock = orgconfig.clock;
        config.speed = orgconfig.speed;
        SendDlgItemMessage(hdlg, IDC_CPU_SPEED, TBM_SETLINESIZE, 0, 1);
        SendDlgItemMessage(hdlg, IDC_CPU_SPEED, TBM_SETPAGESIZE, 0, 2);
        SendDlgItemMessage(hdlg, IDC_CPU_SPEED, TBM_SETRANGE, TRUE, MAKELONG(2, 20));
        SendDlgItemMessage(hdlg, IDC_CPU_SPEED, TBM_SETPOS, FALSE, config.speed / 100);
        break;

    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
            SetFocus(GetDlgItem(hdlg, config.flags & Config::fullspeed ? IDC_CPU_NOSUBCPUCONTROL : IDC_CPU_CLOCK));
            SendDlgItemMessage(hdlg, IDC_CPU_CLOCK_SPIN, UDM_SETRANGE, 0, MAKELONG(100, 1));
            SendDlgItemMessage(hdlg, IDC_CPU_CLOCK, EM_SETLIMITTEXT, 3, 0);
            CPUPageUpdate(hdlg);
            SendDlgItemMessage(hdlg, IDC_CPU_SPEED, TBM_SETRANGE, TRUE, MAKELONG(2, 20));
            page = cpu;
            break;

        case PSN_APPLY:
            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;
        }
        break;

    case WM_COMMAND:
        return CPUPageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    case WM_HSCROLL:
        CPUPageUpdateSlider(hdlg);
        PropSheet_Changed( hwndps, hdlg );
        break;

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::CPUPageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        switch (id)
        {
        case IDC_CPU_NOWAIT:
            config.flags ^= Config::fullspeed;
            if (config.flags & Config::fullspeed)
                config.flags &= ~Config::cpuburst;
            CPUPageUpdate(hdlg);
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_BURST:
            config.flags ^= Config::cpuburst;
            if (config.flags & Config::cpuburst)
                config.flags &= ~Config::fullspeed;
            CPUPageUpdate(hdlg);
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_CLOCKMODE:
            config.flags ^= Config::cpuclockmode;
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_NOSUBCPUCONTROL:
            config.flags ^= Config::subcpucontrol;
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_MS11:
            config.cpumode = Config::ms11;
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_MS21:
            config.cpumode = Config::ms21;
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_MSAUTO:
            config.cpumode = Config::msauto;
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;

        case IDC_CPU_ENABLEWAIT:
            config.flags ^= Config::enablewait;
            PropSheet_Changed( hwndps, hdlg );
            return TRUE;
        }
    }
    switch (id)
    {
    case IDC_CPU_CLOCK:
        if (nc == EN_CHANGE)
        {
            int clock = Limit(GetDlgItemInt(hdlg, IDC_CPU_CLOCK, 0, false), 100, 1)*10;
            if (clock != config.clock)
                PropSheet_Changed( hwndps, hdlg );
            config.clock = clock;
            return TRUE;
        }
        break;
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::CPUPageUpdate(HWND hdlg)
{
    SetDlgItemInt(hdlg, IDC_CPU_CLOCK, config.clock/10, false);
    CheckDlgButton(hdlg, IDC_CPU_NOWAIT, BSTATE(config.flags & Config::fullspeed));

    EnableWindow(GetDlgItem(hdlg, IDC_CPU_CLOCK), !(config.flags & Config::fullspeed));

    EnableWindow(GetDlgItem(hdlg, IDC_CPU_SPEED), !(config.flags & Config::cpuburst));
    EnableWindow(GetDlgItem(hdlg, IDC_CPU_SPEED_TEXT), !(config.flags & Config::cpuburst));

    CheckDlgButton(hdlg, IDC_CPU_NOSUBCPUCONTROL, BSTATE(!(config.flags & Config::subcpucontrol)));
    CheckDlgButton(hdlg, IDC_CPU_CLOCKMODE, BSTATE(config.flags & Config::cpuclockmode));
    CheckDlgButton(hdlg, IDC_CPU_BURST, BSTATE(config.flags & Config::cpuburst));
    CPUPageUpdateSlider(hdlg);

    static const int item[4] =
    { IDC_CPU_MS11, IDC_CPU_MS21, IDC_CPU_MSAUTO, IDC_CPU_MSAUTO };
    CheckDlgButton(hdlg, item[config.cpumode & 3], BSTATE(true));
    CheckDlgButton(hdlg, IDC_CPU_ENABLEWAIT, BSTATE(config.flags & Config::enablewait));
}
#endif

#ifdef __OS2__
#else
void WinConfig::CPUPageUpdateSlider(HWND hdlg)
{
    char buf[8];
    config.speed = SendDlgItemMessage(hdlg, IDC_CPU_SPEED, TBM_GETPOS, 0, 0) * 100;
    wsprintf(buf, "%d%%", config.speed/10);
    SetDlgItemText(hdlg, IDC_CPU_SPEED_TEXT, buf);
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::CPUPageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->CPUPage(hwnd, m, w, l);
}
#endif

// ---------------------------------------------------------------------------
//  Screen Page
//
#ifdef __OS2__
#else
BOOL WinConfig::ScreenPage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {
    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
            ScreenPageUpdate(hdlg);
            page = screen;
            break;

        case PSN_APPLY:
            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;
        }
        break;

    case WM_COMMAND:
        return ScreenPageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::ScreenPageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        PropSheet_Changed( hwndps, hdlg );
        switch (id)
        {
        case IDC_SCREEN_REFRESH_1:
            config.refreshtiming = 1;
            return 0;

        case IDC_SCREEN_REFRESH_2:
            config.refreshtiming = 2;
            return 0;

        case IDC_SCREEN_REFRESH_3:
            config.refreshtiming = 3;
            return 0;

        case IDC_SCREEN_REFRESH_4:
            config.refreshtiming = 4;
            return 0;

        case IDC_SCREEN_ENABLEPCG:
            config.flags ^= Config::enablepcg;
            return 0;

        case IDC_SCREEN_FV15K:
            config.flags ^= Config::fv15k;
            return 0;

        case IDC_SCREEN_DIGITALPAL:
            config.flags ^= Config::digitalpalette;
            return 0;

        case IDC_SCREEN_FORCE480:
            config.flags ^= Config::force480;
            return 0;

        case IDC_SCREEN_LOWPRIORITY:
            config.flags ^= Config::drawprioritylow;
            return 0;

        case IDC_SCREEN_FULLLINE:
            config.flags ^= Config::fullline;
            return 0;
        }
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::ScreenPageUpdate(HWND hdlg)
{
    static const int item[4] =
    { IDC_SCREEN_REFRESH_1, IDC_SCREEN_REFRESH_2, IDC_SCREEN_REFRESH_3, IDC_SCREEN_REFRESH_4 };
    CheckDlgButton(hdlg, item[(config.refreshtiming-1) & 3], BSTATE(true));

    // misc. option
    CheckDlgButton(hdlg, IDC_SCREEN_ENABLEPCG, BSTATE(config.flags & Config::enablepcg));
    CheckDlgButton(hdlg, IDC_SCREEN_FV15K, BSTATE(config.flags & Config::fv15k));
    CheckDlgButton(hdlg, IDC_SCREEN_DIGITALPAL, BSTATE(config.flags & Config::digitalpalette));
    CheckDlgButton(hdlg, IDC_SCREEN_FORCE480, BSTATE(config.flags & Config::force480));
    CheckDlgButton(hdlg, IDC_SCREEN_LOWPRIORITY, BSTATE(config.flags & Config::drawprioritylow));
    CheckDlgButton(hdlg, IDC_SCREEN_FULLLINE, BSTATE(config.flags & Config::fullline));
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::ScreenPageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->ScreenPage(hwnd, m, w, l);
}
#endif

// ---------------------------------------------------------------------------
//  Sound Page
//
#ifdef __OS2__
#else
BOOL WinConfig::SoundPage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {
    case WM_INITDIALOG:
        config.soundbuffer = orgconfig.soundbuffer;
        CheckDlgButton(hdlg,
            config.flag2 & Config::disableopn44 ? IDC_SOUND44_NONE :
            (config.flags & Config::enableopna ? IDC_SOUND44_OPNA : IDC_SOUND44_OPN),
            BSTATE(true));
        CheckDlgButton(hdlg,
            config.flags & Config::opnaona8 ? IDC_SOUNDA8_OPNA :
            (config.flags & Config::opnona8 ? IDC_SOUNDA8_OPN : IDC_SOUNDA8_NONE),
            BSTATE(true));
        break;

    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
            {
                UDACCEL ud[2] = { { 0, 10 }, { 1, 100 } };
                SendDlgItemMessage(hdlg, IDC_SOUND_BUFFERSPIN, UDM_SETRANGE, 0, MAKELONG(1000, 100));
                SendDlgItemMessage(hdlg, IDC_SOUND_BUFFERSPIN, UDM_SETACCEL, 2, (LPARAM) ud);
                SendDlgItemMessage(hdlg, IDC_SOUND_BUFFER, EM_SETLIMITTEXT, 4, 0);
                SoundPageUpdate(hdlg);
                page = sound;       // ݑIĂy[W
            }
            break;

        case PSN_APPLY:
            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;
        }
        break;

    case WM_COMMAND:
        return SoundPageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::SoundPageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        PropSheet_Changed( hwndps, hdlg );
        switch (id)
        {
        case IDC_SOUND44_OPN:
            config.flags &= ~Config::enableopna;
            config.flag2 &= ~Config::disableopn44;
            return 0;

        case IDC_SOUND44_OPNA:
            config.flags |= Config::enableopna;
            config.flag2 &= ~Config::disableopn44;
            return 0;

        case IDC_SOUND44_NONE:
            config.flags &= ~Config::enableopna;
            config.flag2 |= Config::disableopn44;
            return 0;

        case IDC_SOUNDA8_OPN:
            config.flags = (config.flags & ~Config::opnaona8) | Config::opnona8;
            return 0;

        case IDC_SOUNDA8_OPNA:
            config.flags = (config.flags & ~Config::opnona8) | Config::opnaona8;
            return 0;

        case IDC_SOUNDA8_NONE:
            config.flags = config.flags & ~(Config::opnaona8 | Config::opnona8);
            return 0;

        case IDC_SOUND_CMDSING:
            config.flags ^= Config::disablesing;
            return 0;

        case IDC_SOUND_MIXALWAYS:
            config.flags ^= Config::mixsoundalways;
            return 0;

        case IDC_SOUND_PRECISEMIX:
            config.flags ^= Config::precisemixing;
            return 0;

        case IDC_SOUND_WAVEOUT:
            config.flag2 ^= Config::usewaveoutdrv;
            return 0;

        case IDC_SOUND_NOSOUND:
            config.sound = 0;
            return 0;

        case IDC_SOUND_11K:
            config.sound = 1;
            return 0;

        case IDC_SOUND_22K:
            config.sound = 2;
            return 0;

        case IDC_SOUND_44K:
            config.sound = 3;
            return 0;

        case IDC_SOUND_55K:
            config.sound = 4;
            return 0;
        }
    }
    switch (id)
    {
    case IDC_SOUND_BUFFER:
        if (nc == EN_CHANGE)
        {
            uint buf;
            buf = Limit(GetDlgItemInt(hdlg, IDC_SOUND_BUFFER, 0, false), 1000, 100) / 10 * 10;
            if (buf != config.soundbuffer)
                PropSheet_Changed( hwndps, hdlg );
            config.soundbuffer = buf;
            return TRUE;
        }
        break;
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::SoundPageUpdate(HWND hdlg)
{
    static const int itemsr[5] =
    { IDC_SOUND_NOSOUND, IDC_SOUND_11K, IDC_SOUND_22K, IDC_SOUND_44K, IDC_SOUND_55K, };

    CheckDlgButton(hdlg, itemsr[Limit(config.sound, 4, 0)], BSTATE(true));

    CheckDlgButton(hdlg, IDC_SOUND_CMDSING, BSTATE(!(config.flags & Config::disablesing)));
    CheckDlgButton(hdlg, IDC_SOUND_MIXALWAYS, BSTATE(config.flags & Config::mixsoundalways));
    CheckDlgButton(hdlg, IDC_SOUND_PRECISEMIX, BSTATE(config.flags & Config::precisemixing));
    CheckDlgButton(hdlg, IDC_SOUND_WAVEOUT, BSTATE(config.flag2 & Config::usewaveoutdrv));

    SetDlgItemInt(hdlg, IDC_SOUND_BUFFER, config.soundbuffer, false);
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::SoundPageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->SoundPage(hwnd, m, w, l);
}
#endif

// ---------------------------------------------------------------------------
//  ʐݒy[W
//
#ifdef __OS2__
#else
void WinConfig::InitVolumeSlider(HWND hdlg, UINT id, int val)
{
    SendDlgItemMessage(hdlg, id, TBM_SETLINESIZE, 0, 1);
    SendDlgItemMessage(hdlg, id, TBM_SETPAGESIZE, 0, 5);
    SendDlgItemMessage(hdlg, id, TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
    SendDlgItemMessage(hdlg, id, TBM_SETPOS, FALSE, val);
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::SoundVolumePage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {
    case WM_INITDIALOG:
        InitVolumeSlider(hdlg, IDC_SOUND_VOLFM,     orgconfig.volfm);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLSSG,    orgconfig.volssg);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLADPCM,  orgconfig.voladpcm);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLRHYTHM, orgconfig.volrhythm);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLBD,     orgconfig.volbd);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLSD,     orgconfig.volsd);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLTOP,    orgconfig.voltop);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLHH,     orgconfig.volhh);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLTOM,    orgconfig.voltom);
        InitVolumeSlider(hdlg, IDC_SOUND_VOLRIM,    orgconfig.volrim);
        break;

    case WM_HSCROLL:
        SoundVolumePageUpdateSlider(hdlg);
        PropSheet_Changed( hwndps, hdlg );
        return 0;


    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
//          SoundVolumePageUpdate(hdlg);
            SoundVolumePageUpdateSlider(hdlg);
            page = soundvol;        // ݑIĂy[W
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLFM,     TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLSSG,    TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLADPCM,  TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLRHYTHM, TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLBD,     TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLSD,     TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOP,    TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLHH,     TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOM,    TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLRIM,    TBM_SETRANGE, TRUE, MAKELONG(-40, 20));
            break;

        case PSN_APPLY:
            config.volfm     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLFM,     TBM_GETPOS, 0, 0);
            config.volssg    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLSSG,    TBM_GETPOS, 0, 0);
            config.voladpcm  = SendDlgItemMessage(hdlg, IDC_SOUND_VOLADPCM,  TBM_GETPOS, 0, 0);
            config.volrhythm = SendDlgItemMessage(hdlg, IDC_SOUND_VOLRHYTHM, TBM_GETPOS, 0, 0);
            config.volbd     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLBD,     TBM_GETPOS, 0, 0);
            config.volsd     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLSD,     TBM_GETPOS, 0, 0);
            config.voltop    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOP,    TBM_GETPOS, 0, 0);
            config.volhh     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLHH,     TBM_GETPOS, 0, 0);
            config.voltom    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOM,    TBM_GETPOS, 0, 0);
            config.volrim    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLRIM,    TBM_GETPOS, 0, 0);

            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;

        case PSN_QUERYCANCEL:
            PostMessage(hwndparent, WM_M88_CHANGEVOLUME, (WPARAM) &orgconfig, 0);
            return FALSE;
        }
        break;

    case WM_COMMAND:
        return SoundVolumePageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::SoundVolumePageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        PropSheet_Changed( hwndps, hdlg );
        switch (id)
        {
        case IDC_SOUND_RESETVOL:
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLFM,     TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLSSG,    TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLADPCM,  TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLRHYTHM, TBM_SETPOS, TRUE, 0);
            SoundVolumePageUpdateSlider(hdlg);
            break;

        case IDC_SOUND_RESETRHYTHM:
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLBD,     TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLSD,     TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOP,    TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLHH,     TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOM,    TBM_SETPOS, TRUE, 0);
            SendDlgItemMessage(hdlg, IDC_SOUND_VOLRIM,    TBM_SETPOS, TRUE, 0);
            SoundVolumePageUpdateSlider(hdlg);
            break;
        }
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::SoundVolumePageUpdateSlider(HWND hdlg)
{
    config.volfm     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLFM,     TBM_GETPOS, 0, 0);
    config.volssg    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLSSG,    TBM_GETPOS, 0, 0);
    config.voladpcm  = SendDlgItemMessage(hdlg, IDC_SOUND_VOLADPCM,  TBM_GETPOS, 0, 0);
    config.volrhythm = SendDlgItemMessage(hdlg, IDC_SOUND_VOLRHYTHM, TBM_GETPOS, 0, 0);
    config.volbd     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLBD,     TBM_GETPOS, 0, 0);
    config.volsd     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLSD,     TBM_GETPOS, 0, 0);
    config.voltop    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOP,    TBM_GETPOS, 0, 0);
    config.volhh     = SendDlgItemMessage(hdlg, IDC_SOUND_VOLHH,     TBM_GETPOS, 0, 0);
    config.voltom    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLTOM,    TBM_GETPOS, 0, 0);
    config.volrim    = SendDlgItemMessage(hdlg, IDC_SOUND_VOLRIM,    TBM_GETPOS, 0, 0);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTFM,     config.volfm);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTSSG,    config.volssg);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTADPCM,  config.voladpcm);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTRHYTHM, config.volrhythm);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTBD,     config.volbd);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTSD,     config.volsd);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTTOP,    config.voltop);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTHH,     config.volhh);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTTOM,    config.voltom);
    SoundSetVolumeText(hdlg, IDC_SOUND_VOLTXTRIM,    config.volrim);
    PostMessage(hwndparent, WM_M88_CHANGEVOLUME, (WPARAM) &config, 0);
}
#endif

#ifdef __OS2__
#else
void WinConfig::SoundSetVolumeText(HWND hdlg, int id, int val)
{
    if (val > -40)
        SetDlgItemInt(hdlg, id, val, TRUE);
    else
        SetDlgItemText(hdlg, id, "Mute");
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::SoundVolumePageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->SoundVolumePage(hwnd, m, w, l);
}
#endif

// ---------------------------------------------------------------------------
//  Function Page
//
#ifdef __OS2__
#else
BOOL WinConfig::FunctionPage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {
    case WM_INITDIALOG:
        config.mousesensibility = orgconfig.mousesensibility;
        SendDlgItemMessage(hdlg, IDC_FUNCTION_MOUSESENSE, TBM_SETLINESIZE, 0, 1);
        SendDlgItemMessage(hdlg, IDC_FUNCTION_MOUSESENSE, TBM_SETPAGESIZE, 0, 4);
        SendDlgItemMessage(hdlg, IDC_FUNCTION_MOUSESENSE, TBM_SETRANGE, TRUE, MAKELONG(1, 10));
        SendDlgItemMessage(hdlg, IDC_FUNCTION_MOUSESENSE, TBM_SETPOS, FALSE, config.mousesensibility);
        break;

    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
            FunctionPageUpdate(hdlg);
            SendDlgItemMessage(hdlg, IDC_FUNCTION_MOUSESENSE, TBM_SETRANGE, TRUE, MAKELONG(1, 10));
            page = function;
            break;

        case PSN_APPLY:
            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;
        }
        break;

    case WM_COMMAND:
        return FunctionPageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    case WM_HSCROLL:
        config.mousesensibility = SendDlgItemMessage(hdlg, IDC_FUNCTION_PADSENSE, TBM_GETPOS, 0, 0);
        PropSheet_Changed( hwndps, hdlg );
        break;

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::FunctionPageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        PropSheet_Changed( hwndps, hdlg );
        switch (id)
        {
        case IDC_FUNCTION_SAVEDIR:
            config.flags ^= Config::savedirectory;
            return 0;

        case IDC_FUNCTION_ASKBEFORERESET:
            config.flags ^= Config::askbeforereset;
            return 0;

        case IDC_FUNCTION_SUPPRESSMENU:
            config.flags ^= Config::suppressmenu;
            if (config.flags & Config::suppressmenu)
                config.flags &= ~Config::enablemouse;
            FunctionPageUpdate(hdlg);
            return 0;

        case IDC_FUNCTION_USEARROWFOR10:
            config.flags ^= Config::usearrowfor10;
            return 0;

        case IDC_FUNCTION_SWAPPADBUTTONS:
            config.flags ^= Config::swappadbuttons;
            return 0;

//      case IDC_FUNCTION_WATCHREGISTER:
//          config.flags ^= Config::watchregister;
//          return 0;

//      case IDC_FUNCTION_USEQPC:
//          config.flags ^= Config::useqpc;
//          return 0;

        case IDC_FUNCTION_ENABLEPAD:
            config.flags ^= Config::enablepad;
            if (config.flags & Config::enablepad)
                config.flags &= ~Config::enablemouse;
            FunctionPageUpdate(hdlg);
            return 0;

        case IDC_FUNCTION_ENABLEMOUSE:
            config.flags ^= Config::enablemouse;
            if (config.flags & Config::enablemouse)
                config.flags &= ~(Config::enablepad | Config::suppressmenu);
            FunctionPageUpdate(hdlg);
            return 0;

        case IDC_FUNCTION_RESETF12:
            config.flags ^= Config::disablef12reset;
            return 0;

        case IDC_FUNCTION_MOUSEJOY:
            config.flags ^= Config::mousejoymode;
            return 0;
        }
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::FunctionPageUpdate(HWND hdlg)
{
    CheckDlgButton(hdlg, IDC_FUNCTION_SAVEDIR, BSTATE(config.flags & Config::savedirectory));
    CheckDlgButton(hdlg, IDC_FUNCTION_ASKBEFORERESET, BSTATE(config.flags & Config::askbeforereset));
    CheckDlgButton(hdlg, IDC_FUNCTION_SUPPRESSMENU, BSTATE(config.flags & Config::suppressmenu));
    CheckDlgButton(hdlg, IDC_FUNCTION_USEARROWFOR10, BSTATE(config.flags & Config::usearrowfor10));
    CheckDlgButton(hdlg, IDC_FUNCTION_ENABLEPAD, BSTATE(config.flags & Config::enablepad) != 0);
    EnableWindow(GetDlgItem(hdlg, IDC_FUNCTION_SWAPPADBUTTONS), (config.flags & Config::enablepad));
    CheckDlgButton(hdlg, IDC_FUNCTION_SWAPPADBUTTONS, BSTATE(config.flags & Config::swappadbuttons));
//  CheckDlgButton(hdlg, IDC_FUNCTION_WATCHREGISTER, BSTATE(config.flags & Config::watchregister));
//  CheckDlgButton(hdlg, IDC_FUNCTION_USEQPC, BSTATE(config.flags & Config::useqpc));
    CheckDlgButton(hdlg, IDC_FUNCTION_RESETF12, BSTATE(!(config.flags & Config::disablef12reset)));
    CheckDlgButton(hdlg, IDC_FUNCTION_ENABLEMOUSE, BSTATE(config.flags & Config::enablemouse));
    CheckDlgButton(hdlg, IDC_FUNCTION_MOUSEJOY, BSTATE(config.flags & Config::mousejoymode));
    EnableWindow(GetDlgItem(hdlg, IDC_FUNCTION_MOUSEJOY), (config.flags & Config::enablemouse) != 0);
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::FunctionPageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->FunctionPage(hwnd, m, w, l);
}
#endif

// ---------------------------------------------------------------------------
//  Switch Page
// ---------------------------------------------------------------------------

#ifdef __OS2__
#else
BOOL WinConfig::SwitchPage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {

    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
            SwitchPageUpdate(hdlg);
            page = switches;
            break;

        case PSN_APPLY:
            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;
        }
        break;

    case WM_COMMAND:
        return SwitchPageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::SwitchPageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        if (IDC_DIPSW_1H <= id && id <= IDC_DIPSW_CL)
        {
            int n = (id - IDC_DIPSW_1H) / 2;
            int s = (id - IDC_DIPSW_1H) & 1;

            if (!s) // ON
                config.dipsw &= ~(1 << n);
            else
                config.dipsw |= 1 << n;
            PropSheet_Changed( hwndps, hdlg );
            return 0;
        }

        switch (id)
        {
        case IDC_DIPSW_DEFAULT:
            config.dipsw = 1829;
            PropSheet_Changed( hwndps, hdlg );
            SwitchPageUpdate(hdlg);
            return 0;
        }
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::SwitchPageUpdate(HWND hdlg)
{
    for (int i=0; i<12; i++)
    {
        CheckDlgButton(hdlg, IDC_DIPSW_1L + i*2, BSTATE(0 != (config.dipsw & (1 << i))));
        CheckDlgButton(hdlg, IDC_DIPSW_1H + i*2, BSTATE(0 == (config.dipsw & (1 << i))));
    }
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::SwitchPageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->SwitchPage(hwnd, m, w, l);
}
#endif

// ---------------------------------------------------------------------------
//  Env Page
//
#ifdef __OS2__
#else
BOOL WinConfig::EnvPage(HWND hdlg, UINT msg, WPARAM wp, LPARAM lp)
{
    switch (msg)
    {
    case WM_INITDIALOG:
        break;

    case WM_NOTIFY:
        switch (((NMHDR*) lp)->code)
        {
        case PSN_SETACTIVE:
            EnvPageUpdate(hdlg);
            page = env;
            break;

        case PSN_APPLY:
            orgconfig = config;
            PostMessage(hwndparent, WM_M88_APPLYCONFIG, (WPARAM) &config, 0);
            return PSNRET_NOERROR;
        }
        break;

    case WM_COMMAND:
        return EnvPageCommand(hdlg, HWND(lp), HIWORD(wp), LOWORD(wp));

    default:
        return FALSE;
    }
    return TRUE;
}
#endif

#ifdef __OS2__
#else
BOOL WinConfig::EnvPageCommand(HWND hdlg, HWND hwctl, UINT nc, UINT id)
{
    if (nc == BN_CLICKED)
    {
        PropSheet_Changed( hwndps, hdlg );
        switch (id)
        {
        case IDC_ENV_KEY98:
            config.keytype = Config::PC98;
            return 0;

        case IDC_ENV_KEY106:
            config.keytype = Config::AT106;
            return 0;
        }
    }
    return FALSE;
}
#endif

#ifdef __OS2__
#else
void WinConfig::EnvPageUpdate(HWND hdlg)
{
    static const int item[2] =
    { IDC_ENV_KEY106, IDC_ENV_KEY98 };
    CheckDlgButton(hdlg, item[(config.keytype) & 1], BSTATE(true));
}
#endif

#ifdef __OS2__
#else
BOOL _cdecl WinConfig::EnvPageGate
(WinConfig* config, HWND hwnd, UINT m, WPARAM w, LPARAM l)
{
    return config->EnvPage(hwnd, m, w, l);
}
#endif
}
