## Copyright (C) 1995, 1996, 1997  Kurt Hornik
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## usage:  logn_inv (x [, a, v])
##
## For each element of x, compute the quantile (the inverse of the CDF)
## at x of the lognormal distribution with parameters a and v. If a
## random variable follows this distribution, its logarithm is normally
## distributed with mean log (a) and variance v.
##
## Default values are a = 1, v = 1.

## Author:  KH <Kurt.Hornik@ci.tuwien.ac.at>
## Description:  Quantile function of the log normal distribution

function inv = logn_inv (x, a, v)

  if !((nargin == 1) || (nargin == 3))
    usage ("logn_inv (x [, a, v])");
  endif

  if (nargin == 1)
    a = 1;
    v = 1;
  endif

  ## The following "straightforward" implementation unfortunately does
  ## not work (because exp (Inf) -> NaN):
  ## inv = exp (norm_inv (x, log (a), v));
  ## Hence ...
  
  [retval, x, a, v] = common_s (x, a, v);
  if (retval > 0)
    error (["logn_inv:  ", ...
	    "x, a and v must be of common size or scalars"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  a = reshape (a, 1, s);
  v = reshape (v, 1, s);
  inv = zeros (1, s);

  k = find (!(x >= 0) | !(x <= 1) | !(a > 0) | !(a < Inf) ...
      | !(v > 0) | !(v < Inf)); 
  if any (k)
    inv(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x == 1) & (a > 0) & (a < Inf) & (v > 0) & (v < Inf));
  if any (k)
    inv(k) = Inf * ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < 1) & (a > 0) & (a < Inf) ...
      & (v > 0) & (v < Inf));
  if any (k)
    inv(k) = a(k) .* exp (sqrt (v(k)) .* stdnorm_inv (x(k)));
  endif

  inv = reshape (inv, r, c);
  
endfunction
