//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxLib filesystem functions, Critical-error handling
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface
#include <txtpriv.h>                            // private interface


#if defined (DOS32)
   ULONG            TxAutoFail   = 1;           // start in AutoFail for DOS
#else
   ULONG            TxAutoFail   = 0;           // but NOT for OS/2 and WIN!
#endif                                          // to assure first autofail
                                                // call to succeeed


/*****************************************************************************/
// Set critical error handling to auto-fail (versus interactive Abort, retry .)
// Calls to this may be nested, autofail is ON until matching fail==false call
/*****************************************************************************/
void TxFsAutoFailCriticalErrors
(
   BOOL                fail                     // IN    autofail on CritErr
)
{
   ENTER();
   TRACES(( "TxAutoFail: %lu, fail: %s\n", TxAutoFail, (fail) ? "TRUE" : "NO"));

   if ((fail == FALSE) && (TxAutoFail != 0))
   {
      if (--TxAutoFail == 0)                    // 0 after decrement
      {
         #if defined   (WIN32)
            SetErrorMode( 0);
         #elif defined (DEV32)
            DosError( FERR_ENABLEHARDERR);
         #endif
      }
   }
   else if (fail)
   {
      if (TxAutoFail++ == 0)                    // 0 before increment
      {
         #if defined   (WIN32)
            SetErrorMode( SEM_FAILCRITICALERRORS);
         #elif defined (DEV32)
            DosError( FERR_DISABLEHARDERR);
         #endif
      }
   }
   //- Note: DOS TX-handler will obey the TxAutoFail level directly ...
   VRETURN();
}                                           // end 'TxFsAutoFailCriticalErrors'
/*---------------------------------------------------------------------------*/

