//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX display functions for HEX and ASCII data structures
// file-logging facilities
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  24-07-2005 Initial version, split off from TXCON.C

#include <txlib.h>

/*****************************************************************************/
// Make ascii-dump of data area on TxPrint output
/*****************************************************************************/
void TxDisplAscDump
(
   char               *lead,                    // IN    leading string
   BYTE               *data,                    // IN    data area
   ULONG               size                     // IN    size to dump
)
{
   BYTE               *s;
   ULONG               i;

   if ((data != NULL) && (size != 0) && (!TxAbort()))
   {
      TxPrint("%s", lead);
      for (s = data, i = 0; i < size && (*s != 0x1a) && !TxAbort(); s++, i++)
      {
         TxPrint("%c", (char ) *s);
      }
   }
}                                               // end 'TxDisplAscDump'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make hex-dump of data area on TxPrint output
/*****************************************************************************/
void TxDisplHexDump
(
   BYTE               *data,                    // IN    data area
   ULONG               size                     // IN    size to dump
)
{
   TxDisplayHex( "", data, size, 0);
}                                               // end 'TxDisplHexDump'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make hex-dump of based data area on TxPrint output, with leading text
/*****************************************************************************/
void TxDisplayHex
(
   char               *lead,                    // IN    leading text
   BYTE               *data,                    // IN    data area
   ULONG               size,                    // IN    size to dump
   ULONG               base                     // IN    base for display
)
{
   BYTE                *s;
   BYTE                c;
   ULONG               i;
   ULONG               n = 0;
   TXTM                ascii;                   // ascii part buffer
   TXTM                line;                    // accumulating buffer
   TXTS                hex;                     // temporary hex buffer

   if ((data != NULL) && (size != 0))
   {
      TxPrint("%s", lead);
      for (s=data, i=0; i < size; s++, i++)
      {
         n = i % 16;
         switch (n)
         {
            case 0:
               if (i)
               {
                  TxPrint("%s %s[%s%s%s]%s\n", line, CNC, CNN, ascii, CNC, CNN);
               }
               memset(ascii, 0, TXMAXTM);
               if (i && (i % 1024) == 0)        // every 64 lines a formfeed
               {
                  TxPrint("Byte offset: %s%lu%s\n", CBG, i, CNN);
               }
               sprintf( line, "%s-%5.5X-%s ", CBZ, i + base, CNN);
               break;

            case 8:
               strcat( line, " ");
               break;

            default:
               break;
         }
         c = *s;
         ascii[n] = TxPrintable(c);
         sprintf( hex, " %2.2x", (int) c & 0xff);
         strcat( line, hex);
      }
      if (n < 8)
      {
         strcat( line, " ");                    // middle column
      }
      for (; n < 15; n++)
      {
         strcat( line, "   ");                  // fill rest with spaces
      }
      TxPrint( "%s %s[%s%s%s]%s\n", line, CNC, CNN, ascii, CNC, CNN);
   }
}                                               // end 'TxDisplayHex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make 'LONG' hex-dump of data area on TxPrint output, with 'C' comment ASCII
/*****************************************************************************/
void TxDispLongHex
(
   BYTE               *data,                    // IN    data area
   ULONG               size                     // IN    size to dump
)
{
   BYTE                *s;
   ULONG               value = 0;
   BYTE                c;
   ULONG               i;
   ULONG               n = 0;
   TXTM                ascii;                   // ascii part buffer
   TXTM                line;                    // accumulating buffer
   TXTS                hex;                     // temporary hex buffer

   if ((data != NULL) && (size != 0))
   {
      for (s=data, i=0; i < size; s++, i++)
      {
         if ((n = i % 16) == 0)
         {
            if (i)
            {
               TxPrint("%s%s// %s%s%s%s\n", line, CNC, CNN, ascii, CNC, CNN);
            }
            memset(ascii, 0, TXMAXTM);
            strcpy( line, "   ");
         }
         c = *s;
         ascii[n] = TxPrintable(c);

         switch (i % 4)
         {
            case 0: value  = c;       break;
            case 1: value |= c <<  8; break;
            case 2: value |= c << 16; break;
            case 3: value |= c << 24;
               sprintf( hex, "0x%8.8x%c ", value, (i < (size -1)) ? ',' : ' ');
               strcat( line, hex);
               break;
         }
      }
      for (; n < 15; n += 4)
      {
         strcat( line, "            ");         // fill rest with spaces
      }
      TxPrint("%s%s// %s%s%s%s\n", line, CNC, CNN, ascii, CNC, CNN);
   }
}                                               // end 'TxDispLongHex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create Ascii or hex-dump string for a stream of bytes, with one leading space
/*****************************************************************************/
char *TxData2AutoHexAscii                       // RET   output stream (buffer)
(
   BYTE               *data,                    // IN    data area
   ULONG               size,                    // IN    size to dump
   ULONG               bufSize,                 // IN    size output buffer
   char               *buffer                   // INOUT output buffer
)
{
   int                 i;
   BYTE               *s;
   BOOL                isAscii = TRUE;          // data is Pure ASCII

   if ((data != NULL) && (size != 0))
   {
      for (s=data, i=0; i < (size - 1); s++, i++) // allow 0x00 for LAST char!
      {
         if (!isprint(*s))
         {
            isAscii = FALSE;
            break;
         }
      }
      if (isAscii == TRUE)                      // print as ASCII string
      {
         sprintf( buffer, " %*.*s", size, size, data);
      }
      else                                      // print as HEX stream
      {
         TxData2HexStream( data, size, bufSize, buffer);
      }
   }
   return (buffer);
}                                               // end 'TxData2AutoHexAscii'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create hex-dump format string for a stream of bytes, with one leading space
/*****************************************************************************/
char *TxData2HexStream                          // RET   output stream (buffer)
(
   BYTE               *data,                    // IN    data area
   ULONG               size,                    // IN    size to dump
   ULONG               bufSize,                 // IN    size output buffer
   char               *buffer                   // INOUT output buffer
)
{
   int                 i;
   BYTE               *s;
   TXTS                hex;                     // temporary hex buffer

   if ((data != NULL) && (size != 0) && (buffer != NULL))
   {
      *buffer = 0;                              // start empty
      for (s=data, i=0; (i < size) && (i < (bufSize / 4)); s++, i++)
      {
         if (i > 0)
         {
            if ((i % 8) == 0)
            {
               strcat( buffer, " ");            // make groups of 8 bytes
            }
         }
         sprintf( hex, " %2.2hhx", *s);
         strcat(  buffer, hex);
      }
   }
   return (buffer);
}                                               // end 'TxData2HexStream'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return ascii printable character, conservative small reange
/*****************************************************************************/
char TxPrintable                                // RET   printable character
(
   char                c                        // IN    character
)
{
   if (isprint(c))
   {
      return (c);
   }
   else
   {
      return ('.');
   }
}                                               // end 'TxPrintable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return ascii printable character, widest safe range
/*****************************************************************************/
char TxPrintSafe                                // RET   printable character
(
   char                c                        // IN    character
)
{
   if (TxIsSafeAscii(c))
   {
      return (c);
   }
   else
   {
      return ('');
   }
}                                               // end 'TxPrintSafe'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test character, to see if it is ascii printable, widest safe range
/*****************************************************************************/
BOOL TxIsSafeAscii                              // RET   printable character
(
   char                c                        // IN    character
)
{
   if ((c!=0x00) && (c!=0x07) && (c!=0x08) && (c!=0x09) &&
       (c!=0x0a) && (c!=0x0c) && (c!=0x0d) && (c!=0x1a) && (c!=0x1b))
   {
      return( TRUE);
   }
   else
   {
      return (FALSE);
   }
}                                               // end 'TxIsSafeAscii'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test character to see if it is printable, pure ASCII range, with whitespace
/*****************************************************************************/
BOOL TxIsPureAscii                              // RET   pure ASCII character
(
   char                c                        // IN    character
)
{
   if (((c >= 0x20) && (c < 0x7f)) || (c == 0x0d) || (c == 0x0a) || (c == 0x09))
   {
      return( TRUE);
   }
   else
   {
      return (FALSE);
   }
}                                               // end 'TxIsPureAscii'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Output textual info in array of char-pointers, add newlines in-between
/*****************************************************************************/
void TxShowTxt
(
   char               *txt[]                    // IN    text to display
)
{
   char              **s;

   for (s  = txt; s && (*s != NULL); s++)
   {
      TxPrint( "%s%s", (s!= txt) ? "\n": "", *s);
   }
}                                               // end 'TxShowTxt'
/*---------------------------------------------------------------------------*/

