//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// file-logging facilities
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  26-12-96   Split-off from ansilog

#include <txlib.h>

static  FILE      *log_handle  = 0;
static  TXLN       log_fname   = "logfile";
static  BOOL       log_7bit    = TRUE;
static  BOOL       log_reopen  = TRUE;          // reopen on each line

/*****************************************************************************/
// Close existing and if specified, open new logfile
// Can NOT have any internal tracing! Used in TxPrint to scrollbuffer stream!
// 20170729 JvW: Store full path for log_fname to prevent 'wandering' logfile
/*****************************************************************************/
BOOL TxAppendToLogFile                          // RET   logfile opened
(
   char               *fname,                   // IN    name of (new) logfile
   BOOL                verbose                  // IN    Show the action
)
{
   BOOL                fExist = TRUE;           // logfile exists (will append)
   TXLN                old_fname;

   TxRepl( fname, FS_PALT_SEP, FS_PATH_SEP);    // fixup ALT separators
   strcpy( old_fname, log_fname);               // remember for close message
   if (fname && strlen(fname))
   {
      if (strcmp( fname, ".") != 0)             // not 'current one' ?
      {
         TXLN          numbered;
         char         *relative = numbered;

         //- make fname an absolute path (avoiding log wandering on CD)
         //- and honor the -dir:path data dir when no absolute spec given
         fExist = TxAutoNumberedFname( fname, "log", numbered);
         if ((numbered[0] != FS_PATH_SEP) &&   // not absolute from root
             (numbered[1] != ':'        )  )   // or with a drive component
         {
            if (numbered[0] == '.')             // explicit relative to current
            {
               if (numbered[1] != '.')          // explicit relative to parent
               {
                  relative += 2;                // skip . and seperator
               }
               getcwd( log_fname, TXMAXLN);
            }
            else
            {
               if (TxaExeSwitchSet(TXA_O_DIR))  // explicit files dir given
               {
                  strcpy( log_fname, TxaExeSwitchStr(TXA_O_DIR, "", ""));
               }
               else                             // make relative to current
               {
                  getcwd( log_fname, TXMAXLN);
               }
            }
            if (log_fname[ strlen( log_fname) - 1] != FS_PATH_SEP)
            {
               strcat( log_fname, FS_PATH_STR);
            }
            strcat( log_fname, relative);
         }
         else                                   // already absolute
         {
            strcpy( log_fname, numbered);
         }
      }
      if (log_handle != 0)                      // close old one AFTER
      {                                         // AutoNumbered for trace!
         if (verbose)
         {
            TxPrint("\nClosing logfile   : '%s'\n", old_fname);
         }
         fclose(log_handle);
         log_handle = 0;
      }
      if (((log_handle = fopen(log_fname, "a" TXFMODE)) == 0) && (verbose))
      {
         TxPrint("Error opening log : '%s'\n", log_fname);
      }
      else
      {
         TxLogfileState( DEVICE_ON);            // always ON for new file
         if (verbose)                           // avoid any TxPrint when
         {                                      // working in quiet mode
            if (TxaOptUnSet('7'))               // NO strip to 7-bit ASCII
            {
               log_7bit = FALSE;
            }
            if (TxaExeArgc() != 0)              // Switches already parsed ?
            {                                   // (-q = quiet switch to come)
               TxPrint("%s log  : '%s' (%s-bit ASCII)\n",
                         (fExist) ? "Appending to" : "Creating new",
                         log_fname, (log_7bit) ? "7" : "8");
            }
         }
      }
   }
   else                                         // just close it
   {
      if (log_handle != 0)                      // close old one
      {
         if (verbose)
         {
            TxPrint("\nClosing logfile   : '%s'\n", old_fname);
         }
         fclose(log_handle);
         log_handle = 0;
      }
   }
   return( log_handle != 0);
}                                               // end 'TxAppendToLogFile'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set log close/reopen on each TxPrint on or off
/*****************************************************************************/
void TxSetLogReOpen
(
   BOOL                reopen                   // IN    log reopen on
)
{
   log_reopen = reopen;
}                                               // end 'TxSetLogReOpen'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close existing and reopen (implementing a brute-force flush)
/*****************************************************************************/
void TxCloseReopenLogFile
(
   void
)
{
   static BOOL         warn = TRUE;

   if (log_handle != 0)
   {
      fclose(log_handle);
      if ((log_handle = fopen(log_fname, "a" TXFMODE)) == 0)
      {
         if (warn)
         {
            TxPrint("\nError re-opening  : '%s'\n", log_fname);
            warn = FALSE;                       // I will say this only once!
         }
      }
   }
}                                               // end 'TxCloseReopenLogFile'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Signal logfile active
/*****************************************************************************/
FILE *TxQueryLogFile                            // RET   logfile active
(
   BOOL               *ascii7bit,               // OUT   Use 7-bit ASCII only
   BOOL               *reopenlog                // OUT   reopen logfile
)                                               //       NULL if not wanted
{
   if (ascii7bit != NULL)
   {
      *ascii7bit  = log_7bit;
   }
   if (reopenlog != NULL)
   {
      *reopenlog  = log_reopen;
   }
   return( log_handle);
}                                               // end 'TxQueryLogFile'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get name for current logfile, if any
/*****************************************************************************/
char *TxQueryLogName                            // RET   filename or NULL
(
   void
)                                               //       (NULL if not wanted)
{
   char               *rc = NULL;

   if (log_handle != 0)
   {
      rc = log_fname;
   }
   return (rc);
}                                               // end 'TxQueryLogName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Construct name for new logfile based upon a sequence number 0..n
// Can NOT have any internal tracing! Used in TxPrint to scrollbuffer stream!
/*****************************************************************************/
char *TxBuildLogName                            // RET   filename or NULL
(
   ULONG               seq,                     // IN    sequence number
   ULONG               retain,                  // IN    nr of files kept
   TXLN                buf                      // IN    filename buffer
)
{
   char               *rc = buf;
   TXTS                ext;

   if (seq == 0)                                // first will be .log
   {
      strcpy( ext, "log");
   }
   else
   {
      if ((seq >= 100) && (retain < 100))       // OK to use modulo seq ?
      {
         seq %= 100;                            // reduce to two digits, to
      }                                         // limit extension to three
      sprintf( ext, "l%02u", seq);
   }
   strcpy( rc, log_fname);

   TxStripExtension( rc);                       // remove current extension
   TxFnameExtension( rc, ext);                  // and add new one

   return (rc);
}                                               // end 'TxBuildLogName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Substitue specific date/time fragments for ~1..9 and YYYYMMDD for ~ at end
// Substitute first unused auto-nr for FINAL ^ or @ in filename; add extension
// Can NOT have any internal tracing! Used in TxPrint to scrollbuffer stream!
/*****************************************************************************/
BOOL TxAutoNumberedFname                        // RET   TRUE when not unique
(
   char               *spec,                    // IN    (path +) file specification
   char               *ext,                     // IN    file extension to be used
   char               *anName                   // OUT   auto-numbered name
)
{
   BOOL                rc = TRUE;               // function return
   ULONG               fnum;
   TXLN                subst;                   // spec with substituted ~ fragments
   FILE               *fp;                      // temp handle for EXIST test
   char               *pseq;                    // pointer to sequence-nr position ^ or @

   TxDateTimeSubst( spec, subst);               // create subst fileName from spec
   pseq = strrchr( subst, '^');
   if (pseq == NULL)
   {
      pseq = strrchr( subst, '@');
   }
   if (pseq != NULL)
   {
      *pseq  = 0;                               // terminate first part of name
      pseq++;                                   // and point to second part now
      for (fnum = 1; fnum <= 1000; fnum++)
      {
         sprintf( anName, "%s%03u%s", subst, (fnum % 1000), pseq);
         TxFnameExtension( anName, ext);        // add extension (has no TRACE!)
         if ((fp = fopen( anName, "r")) != NULL)
         {
            fclose( fp);                        // exists, close again
         }
         else                                   // numbered file does not exist yet
         {
            rc = FALSE;                         // found one that does NOT exist
            break;                              // last checked will be 000
         }
      }
   }
   else
   {
      strcpy(  anName, subst);                  // default, substitued name as is
      TxFnameExtension( anName, ext);           // add extension (has no TRACE!)
      if ((fp = fopen(  anName, "r")) != NULL)
      {
         fclose( fp);                           // exists, close again
      }
      else                                      // file does not exist yet
      {
         rc = FALSE;                            // resulting filename not unique
      }
   }
   return (rc);
}                                               // end 'TxAutoNumberedFname'
/*---------------------------------------------------------------------------*/

