//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// Generic TX string functions
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  24-07-2005 Initial version, split off from TXUTIL

#include <txlib.h>

/*****************************************************************************/
// Replace specfied character by another in whole string, count replacements
/*****************************************************************************/
int TxRepl                                      // RET   nr of replacements
(
   char               *str,                     // INOUT string to convert
   char                old,                     // IN    old char, to replace
   char                new                      // IN    new char
)
{
   int                 rc = 0;

   while (*str)
   {
      if (*str == old)
      {
         *str = new;
         rc++;
      }
      str++;
   }
   return (rc);
}                                               // end 'TxRepl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy string, and replace specfied character by a string
/*****************************************************************************/
int TxRepStr                                    // RET   nr of replacements
(
   char               *str,                     // IN    string to convert
   char                old,                     // IN    old char, to replace
   char               *new,                     // IN    new string
   char               *dest,                    // OUT   destination string
   int                 len                      // IN    max length destination
)
{
   int                 rc = 0;
   char               *s;
   int                 l;

   for (s = dest, l = 0; *str && (l < len); str++)
   {
      if (*str == old)
      {
         l += strlen(    new);                  // length to be
         if (l < len)
         {
            *s = '\0';                          // terminate for concat
            strcat(  s,  new);                  // and concat new str
            s += strlen( new);                  // at string end ...
            rc++;
         }
      }
      else
      {
         *s++ = *str;                           // just copy one char
         l++;                                   // and count ...
      }
   }
   *s = '\0';                                   // make sure it is terminated
   return (rc);
}                                               // end 'TxRepStr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Strip leading/trailing characters from a string, dest and source can be same
/*****************************************************************************/
int TxStrip                                     // RET   nr of stripped chars
(
   char               *dest,                    // OUT   destination string
   char               *source,                  // IN    source string
   char                lead,                    // IN    leading  chars to strip
   char                trail                    // IN    trailing chars to strip
)
{
   int                 rc = 0;
   char               *s = dest;

   while ((*source) && (*source == lead))
   {
      source++;
      rc++;
   }
   while (*source)
   {
      *s++ = *source++;
   }
   *s = 0;
   while ((s-- > dest) && (*s == trail))
   {
      *s = 0;
      rc++;
   }
   return (rc);
}                                               // end 'TxStrip'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Pad/Clip a string to specified length, concatenating or removing characters
/*****************************************************************************/
int TxPClip                                     // RET   +:padded -:removed
(
   char               *str,                     // INOUT destination string
   int                 size,                    // IN    requested length
   char                pad                      // IN    trailing chars to pad
)
{
   int                 pos = strlen(str);
   int                 rc  = size - pos;

   str[ size] = 0;                              // terminate at wanted length

   while (pos < size)
   {
      str[pos] = pad;
      pos++;
   }
   return (rc);
}                                               // end 'TxPClip'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert string to all uppercase (like non-possix strupr function)
/*****************************************************************************/
char *TxStrToUpper                              // RET   Updated string
(
   char               *str                      // INOUT destination string
)
{
   char               *s;

   for (s = str; *s; s++)
   {
      *s = toupper( *s);
   }
   return (str);
}                                               // end 'TxStrToUpper'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert string to all lowercase (like non-possix strlwr function)
/*****************************************************************************/
char *TxStrToLower                              // RET   Updated string
(
   char               *str                      // INOUT destination string
)
{
   char               *s;

   for (s = str; *s; s++)
   {
      *s = tolower( *s);
   }
   return (str);
}                                               // end 'TxStrToLower'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create string with DATE-TIME substitution based on '~' spec format (ZOC compatible)
// 1=monthday 2=month 3=year 4=hour 5-minute 6=second 7=dname 8=mname 9=weeknr
/*****************************************************************************/
char *TxDateTimeSubst                           // RET   Destination string
(
   char               *fmt,                     // IN    format specification
   char               *str                      // INOUT destination string
)                                               //       of sufficient length!
{
   char               *s;                       // date/time specifier finder
   TXTS                dtFragment;              // date/time fragment string
   time_t              tt = time( &tt);         // current date/time
   struct tm          *tm = localtime( &tt);    // NOW + offset in TM format

   *str = 0;                                    // start empty
   for (s = fmt; *s; s++)
   {
      dtFragment[0] = *s;                       // start with copy of current char
      dtFragment[1] = 0;
      if (*s == '~')                            // substitute a fragment
      {
         s++;                                   // point to format spec number
         switch (*s)
         {
            case '-': case '_': case '+': case '=': case '(': case ')': case '!': case '.':
                      strftime( dtFragment, TXMAXTS, "%Y%m%d ", tm); // ~ before delimitter
                      dtFragment[8] = *s;   break; // Replace space by delimitter character

            case  0 :                                                      // ~  at END of spec
            case '0': strftime( dtFragment, TXMAXTS, "%Y%m%d", tm); break; // ~0 before the END
            case '1': strftime( dtFragment, TXMAXTS, "%d",     tm); break;
            case '2': strftime( dtFragment, TXMAXTS, "%m",     tm); break;
            case '3': strftime( dtFragment, TXMAXTS, "%Y",     tm); break;
            case '4': strftime( dtFragment, TXMAXTS, "%H",     tm); break;
            case '5': strftime( dtFragment, TXMAXTS, "%M",     tm); break;
            case '6': strftime( dtFragment, TXMAXTS, "%S",     tm); break;
            case '7': strftime( dtFragment, TXMAXTS, "%A",     tm); break;
            case '8': strftime( dtFragment, TXMAXTS, "%B",     tm); break;
            case '9': strftime( dtFragment, TXMAXTS, "%V",     tm); break;
            default:            dtFragment[1] = *s; // keep fmt-char and next as literal
                                dtFragment[2] = 0;                  break;
         }
      }
      strcat( str, dtFragment);                 // append next char/fragment
   }
   return (str);
}                                               // end 'TxDateTimeSubst'
/*---------------------------------------------------------------------------*/
