//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// Author: J. van Wijk
//
// TX simple event hook implementation, for multi-thread communication
// like passing status or progress information to UI dialogs
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  02-12-2005   Initial version

#include <txlib.h>                              // public interface
#include <txwpriv.h>                            // private window interface


#if defined (USEWINDOWING)

// Note: can be refined later to have a list of hooks per event-type

// Simple event hook data structure (single hook per event-type)
static TXWHANDLE      txwEventHookHandle[TXHK_EVENTS] = {TXHWND_NULL};


/*****************************************************************************/
// Attach specified hook-event to a window-handle
/*****************************************************************************/
ULONG txwAttachEventHook                        // RET   result
(
   ULONG               hookid,                  // IN    TXHK identifier
   TXWHANDLE           hwnd                     // IN    window handle
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (hookid < TXHK_EVENTS)
   {
      if (txwEventHookHandle[ hookid] == TXHWND_NULL)
      {
         TRACES(( "Attaching hwnd: %8.8lx for event: %lu\n", hwnd, hookid));
         txwEventHookHandle[ hookid] = hwnd;
      }
      else                                      // slot is in use
      {
         rc = TX_FAILED;
      }
   }
   else
   {
      rc = TX_INVALID_DATA;
   }
   RETURN (rc);
}                                               // end 'txwAttachEventHook'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Detach specified hook-event from a window-handle
/*****************************************************************************/
ULONG txwDetachEventHook                        // RET   result
(
   ULONG               hookid,                  // IN    TXHK identifier
   TXWHANDLE           hwnd                     // IN    window handle
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (hookid < TXHK_EVENTS)
   {
      if (txwEventHookHandle[ hookid] == hwnd)
      {
         TRACES(( "Detaching hwnd: %8.8lx for event: %lu\n", hwnd, hookid));
         txwEventHookHandle[ hookid] = TXHWND_NULL;
      }
      else                                      // hwnd not attached
      {
         rc = TX_FAILED;
      }
   }
   else
   {
      rc = TX_INVALID_DATA;
   }
   RETURN (rc);
}                                               // end 'txwDetachEventHook'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Query specified hook-event for a window-handle
/*****************************************************************************/
TXWHANDLE txwQueryEventHook                     // RET   window handle
(
   ULONG               hookid                   // IN    TXHK identifier
)
{
   TXWHANDLE           rc = TXHWND_NULL;           // function return

   ENTER();

   if (hookid < TXHK_EVENTS)
   {
      rc = txwEventHookHandle[ hookid];
   }
   RETURN (rc);
}                                               // end 'txwQueryEventHook'
/*---------------------------------------------------------------------------*/

#if defined (HAVETHREADS)

/*****************************************************************************/
// Signal specified hook-event, pass optional data (data must stay valid!)
/*****************************************************************************/
TXWHANDLE txwSignalEventHook                    // RET   window signalled
(
   ULONG               hookid,                  // IN    TXHK identifier (MP1)
   void               *data                     // IN    optional data   (MP2)
)
{
   TXWHANDLE           rc = TXHWND_NULL;        // function return

   ENTER();

   if (hookid < TXHK_EVENTS)
   {
      if ((rc = txwEventHookHandle[ hookid]) != TXHWND_NULL)
      {
         //- to be refined, may add boolean parameter for multiple messages
         //- otherwise, don't add multiple to avoid msg pile-up in queue

         if (txwQueueMsgPresent( rc, TXWM_HOOKEVENT) == FALSE)
         {
            if (txwPostMsg( rc, TXWM_HOOKEVENT, (TXWMPARAM) hookid, (TXWMPARAM) data))
            {
               txwNotifyAsyncInput();           // signal to synchronious
            }                                   // input procedures ...
            else
            {
               rc = TXHWND_NULL;                // Post failed, MSGQ full ?
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'txwSignalEventHook'
/*---------------------------------------------------------------------------*/

#endif                                          // HAVETHREADS
#endif                                          // USEWINDOWING
