//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxLib generic wildcard handling
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface


/*****************************************************************************/
// Function : count number of non-wildcard chars in string
// 20190810 JvW: Changed output for empty string from 1 to 0 (no wildcard)
/*****************************************************************************/
int TxStrWcnt
(
    char              *s
)
{
    int                result = 0;

    if ((s) && (*s))                            // input, non-empty
    {
       do                                       // start of DO loop
       {
           if ((*s != '*') && (*s != '?'))
           {
               result ++;
           }
       } while ( *( ++s) );                     /* until '\0'                                    */
    }
    return result;
}                                               /* end TxStrWcnt             */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
/* Function : wildcard compare of candidate to template string               */
/*            check if template is a valid description of the candidate      */
/*            wildcard characters are : '*' : matches zero or more of any    */
/*                                            character in the candidate     */
/*                                      '?' : matches exactly one of any     */
/*                                            character in the candidate     */
/*                                            except the '*' character       */
/*            return value : negative if template is no valid description    */
/*                                                                           */
/* Note that the routine is case sensitive on candidate and template         */
/* Note that trailing spaces are considered as significant                   */
/*****************************************************************************/
int TxStrWcmp
(
    char              *cs,                      // IN    candidate string
    char              *ts                       // IN    template string
)
{
    int                dc = 0;                  // declare result variable
    TXLN               sub;                     // substring buffer
    int                l;                       // length of sub string

    while ((*ts != '\0') && (dc == 0))
    {
        switch (*ts)
        {
            case '*':
                do                              // skip adjacent wildcard
                {                               // characters in template
                    if (*ts++ == '?')           // need one candidate char
                    {
                        if (*cs++ == '\0')      // end of candidate
                        {
                            dc--;               // difference found !
                        }
                    }                           // end if single wildcard
                } while ((*ts == '*') ||
                         (*ts == '?') );        // until non-wildcard

                if (dc == 0)
                {
                    if (*ts == '\0')
                    {                           // signal '*' at end of
                        dc++;                   // template
                    }
                    else
                    {
                        l = strcspn(ts, "*?");  // next wildcard position
                        strncpy( sub, ts, l);
                        sub[ l] = '\0';         // terminate it

                        if ((cs = strstr(cs, sub)) != NULL)
                        {
                            ts += l;            // skip common substring
                            cs += l;
                        }
                        else
                        {
                            dc--;               // difference found !
                        }                       // end if
                    }                           // end if
                }                               // end if no difference yet
                break;

            case '?':
                if (*cs++ != '*')
                {
                    ts++;
                }
                else
                {
                    dc--;                       // difference found !
                }
                break;

            default:
                if (*cs++ != *ts++)
                {
                    dc--;                       // difference found !
                }
                break;
        }                                       // end switch  *template
    }                                           // end while in template
    switch (dc)
    {
        case 0:
            if ((*ts == '\0') && (cs != NULL) && (*cs != '\0'))
            {
                dc--;                           // candidate too long
            }
            break;

        case 1:
            dc--;                               // template ends in '*'
            break;                              // result is no-difference

        default:
            break;
    }                                           // end switch
    return dc;
}                                               /* end TxStrWcmp             */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Function : See FlStrwcmp; case-insensitive
/*****************************************************************************/
int TxStrWicmp
(
    char              *cs,                      // IN    candidate string
    char              *ts                       // IN    template string
)
{
    TX1K               candi;                   // candidate string
    TXLN               templ;                   // template string

    TxStrToUpper( strcpy( candi, cs));
    TxStrToUpper( strcpy( templ, ts));
    return( TxStrWcmp( candi, templ));
}                                               /* end TxStrWicmp            */
/*---------------------------------------------------------------------------*/

