
// First include all needed standard header files
#include <stdlib.h>
#include <stdio.h>
#include <process.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <time.h>

#ifndef M_PI
#define M_PI 3.14159
#endif

// Then the needed OS/2 API stuffs
#define INCL_TYPES
#define INCL_WIN
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_GPI
#include <os2.h>

#include <cairo.h>
#include <cairo-os2.h>

#define DEBUG_BUILD


int bAnalogueClock = 0;

// --------------- Functions: -----------------------------------------------

#ifdef DEBUG_BUILD
void MorphToPM()
{
  PPIB pib;
  PTIB tib;

  DosGetInfoBlocks(&tib, &pib);

  // Change flag from VIO to PM:
  if (pib->pib_ultype==2) pib->pib_ultype = 3;
}
#endif

cairo_surface_t *surface = NULL;
cairo_t *cr = NULL;
double line_width;
cairo_font_extents_t font_extents;

static void inline DisableFPUException()
{
  unsigned short usCW;

  // Some OS/2 PM API calls modify the FPU Control Word,
  // but forget to restore it.

  // This can result in XCPT_FLOAT_INVALID_OPCODE exceptions,
  // so to be sure, we always disable Invalid Opcode FPU exception
  // before using FPU stuffs with Cairo or from this application.

  usCW = _control87(0, 0);
  usCW = usCW | EM_INVALID | 0x80;
  _control87(usCW, MCW_EM | 0x80);
}

void DrawImage(struct tm *pTime, int iWidth, int iHeight)
{
  double x, y, x2, y2;
  double dAlpha;
  int iUniSize;
  double dXAspect, dYAspect;
  int i;

  DisableFPUException();

  cairo_save(cr);
  if (iWidth<iHeight)
  {
    iUniSize = iWidth;
    dYAspect = iHeight;
    dYAspect /= iWidth;
    dXAspect = 1;
  }
  else
  {
    iUniSize = iHeight;
    dXAspect = iWidth;
    dXAspect /= iHeight;
    dYAspect = 1;
  }
  cairo_scale(cr, iUniSize, iUniSize);

  // Clear background with black
  cairo_set_source_rgb (cr, 0, 0 ,0);
  cairo_rectangle (cr,
                   0, 0,
                   dXAspect, dYAspect);
  cairo_fill(cr);

  // Draw markers
  for (i=0; i<60; i++)
  {
    dAlpha = 360.0 * i / 60.0;

    if (i % 5==0)
    {
      cairo_set_line_width(cr, 0.018);
      x = 0.32 * sin(dAlpha*M_PI/180.0);
      y = 0.32 * cos(dAlpha*M_PI/180.0);
    } else
    {
      cairo_set_line_width(cr, 0.005);
      x = 0.37 * sin(dAlpha*M_PI/180.0);
      y = 0.37 * cos(dAlpha*M_PI/180.0);
    }
    x2 = 0.41 * sin(dAlpha*M_PI/180.0);
    y2 = 0.41 * cos(dAlpha*M_PI/180.0);

    cairo_set_source_rgb (cr, 0.8, 0.8, 0.8);
    cairo_move_to (cr, 0.5 * dXAspect + x, 0.5 * dYAspect - y);
    cairo_line_to(cr, 0.5*dXAspect + x2, 0.5 * dYAspect - y2);
    cairo_close_path (cr);
    cairo_stroke (cr);
  }

  // Draw hour marker
  dAlpha = 360.0 * (pTime->tm_hour%12) / 12.0;
  if (bAnalogueClock)
  {
    dAlpha+= (pTime->tm_min%60)*0.5 + (pTime->tm_sec%60)*0.0083;
  }
  x = 0.25 * sin(dAlpha*M_PI/180.0);
  y = 0.25 * cos(dAlpha*M_PI/180.0);
  cairo_set_line_width(cr, 0.015);
  cairo_set_source_rgb (cr, 1, 1, 1);
  cairo_arc(cr, 0.5 * dXAspect, 0.5 * dYAspect, 0.020, 0, 2*M_PI);
  cairo_fill(cr);
  cairo_move_to (cr, 0.5 * dXAspect, 0.5 * dYAspect);
  cairo_line_to(cr, 0.5*dXAspect + x, 0.5 * dYAspect - y);
  cairo_close_path (cr);
  cairo_stroke (cr);

  // Draw minute marker
  dAlpha = 360.0 * pTime->tm_min / 60.0;
  if (bAnalogueClock)
  {
    dAlpha+= (pTime->tm_sec%60)/10.0;
  }
  x = 0.35 * sin(dAlpha*M_PI/180.0);
  y = 0.35 * cos(dAlpha*M_PI/180.0);
  cairo_set_line_width(cr, 0.015);
  cairo_set_source_rgb (cr, 1, 1, 1);
  cairo_arc(cr, 0.5 * dXAspect, 0.5 * dYAspect, 0.017, 0, 2*M_PI);
  cairo_fill(cr);
  cairo_move_to (cr, 0.5 * dXAspect, 0.5 * dYAspect);
  cairo_line_to(cr, 0.5*dXAspect + x, 0.5 * dYAspect - y);
  cairo_close_path (cr);
  cairo_stroke (cr);

  // Draw second marker
  dAlpha = 360.0 * pTime->tm_sec / 60.0;
  x = 0.35 * sin(dAlpha*M_PI/180.0);
  y = 0.35 * cos(dAlpha*M_PI/180.0);
  cairo_set_line_width(cr, 0.01);
  cairo_set_source_rgb (cr, 0.8, 0.1, 0.1);
  cairo_arc(cr, 0.5 * dXAspect, 0.5 * dYAspect, 0.014, 0, 2*M_PI);
  cairo_fill(cr);
  cairo_move_to (cr, 0.5 * dXAspect, 0.5 * dYAspect);
  cairo_line_to(cr, 0.5*dXAspect + x, 0.5 * dYAspect - y);
  cairo_close_path (cr);


  cairo_stroke (cr);

  cairo_restore(cr);
}

void DrawItOnce(HWND hwndClientWindow)
{
  HPS hpsClientWindow;
  SWP swpTemp;
  time_t Time;
  struct tm *pTime;
  int iOldSec;

  // Create Cairo surface
  hpsClientWindow = WinGetPS(hwndClientWindow);
  WinQueryWindowPos(hwndClientWindow, &swpTemp);

  surface = cairo_os2_surface_create(hpsClientWindow,
                                     swpTemp.cx,
                                     swpTemp.cy);
  cr = cairo_create(surface);

  // Get current time
  Time = time(NULL);
  pTime = localtime(&Time);
  iOldSec = pTime->tm_sec;

  // Draw image into cairo surface
  WinQueryWindowPos(hwndClientWindow, &swpTemp);
  DrawImage(pTime, swpTemp.cx, swpTemp.cy);

  // Destroy Cairo surface
  if (cr)
    cairo_destroy (cr);
  if (surface)
    cairo_surface_destroy (surface);

  WinReleasePS(hpsClientWindow);

}


MRESULT EXPENTRY MainWindowProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch (msg)
  {
    case WM_PAINT:
      {
        HPS hpsBeginPaint;
        RECTL rclRect;

        hpsBeginPaint = WinBeginPaint(hwnd, NULL, &rclRect);

        DrawItOnce(hwnd);

        // All done!
        WinEndPaint(hpsBeginPaint);
        return (MRESULT) FALSE;
      }
    case WM_SIZE:
        DrawItOnce(hwnd);
        break;
  }
  return WinDefWindowProc(hwnd, msg, mp1, mp2);
}

MRESULT EXPENTRY NewFrameProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  PFNWP pOldFrameProc;
  MRESULT result;
  PTRACKINFO ti;

  pOldFrameProc = (PFNWP) WinQueryWindowULong(hwnd, QWL_USER);

  result = (*pOldFrameProc)(hwnd, msg, mp1, mp2);

  if (msg == WM_QUERYTRACKINFO)
  {
    ti = (PTRACKINFO) mp2;
    ti->ptlMinTrackSize.x = 300;
    ti->ptlMinTrackSize.y = 200;
  }
  return result;
}

void PMTest()
{
  HAB hab;
  HMQ hmq;
  QMSG msg;
  ULONG flCreaFlags, flFrameStyle, flClientStyle;
  SWP swap;
  PFNWP pOldFrameProc;
  HWND hwndFrame;
  HWND hwndClient;

#ifdef DEBUG_BUILD
  printf("[PMThread] : Starting\n");
#endif
  // Setup this thread for using PM (sending messages etc...)
  hab = WinInitialize(0);
  hmq = WinCreateMsgQueue(hab, 0);

  // Create main window
  WinRegisterClass(hab, "PrivateClassName",  // Register a private class, so
		   MainWindowProc,           // we can use our own window procedure for frame
		   0, 0);

  // Set defaults for main window:
  flCreaFlags = 

    FCF_TASKLIST | FCF_SYSMENU | FCF_TITLEBAR | FCF_SIZEBORDER |
    FCF_DLGBORDER | FCF_MINMAX |
    FCF_SHELLPOSITION | FCF_NOBYTEALIGN | FCF_AUTOICON;
  flFrameStyle = 
    FS_NOBYTEALIGN | FS_BORDER;
  flClientStyle =
      WS_CLIPCHILDREN;

  hwndFrame = WinCreateStdWindow(HWND_DESKTOP,
				 flFrameStyle,       // Frame window style
				 &flCreaFlags,
                                 "PrivateClassName",
                                 "TestApp for CAIRO.DLL",
				 flClientStyle,//0,   // Client window style
				 NULLHANDLE,
				 0L,
                                 &hwndClient);

  pOldFrameProc = WinSubclassWindow(hwndFrame, NewFrameProc); // Subclass window to be able
                                                              // to handle resize events
  // Store old window proc in frame window's ULONG
  WinSetWindowULong(hwndFrame, QWL_USER, (ULONG) pOldFrameProc);

  // Move the window more down if it would be out of the screen
  WinQueryWindowPos(hwndFrame, &swap);
  if (swap.y+swap.cy > WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN))
  {
    WinSetWindowPos(hwndFrame, HWND_TOP, swap.x,
		    WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN) - swap.cy, 0, 0, SWP_MOVE);
    WinQueryWindowPos(hwndFrame, &swap);
  }
  if (swap.x+swap.cx > WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN))
  {
    if (WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - swap.cx >= 0)
      WinSetWindowPos(hwndFrame, HWND_TOP, 
		      WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - swap.cx, swap.y,  0, 0, SWP_MOVE);
  }

  WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_SHOW | SWP_ACTIVATE);

#ifdef DEBUG_BUILD
  printf("[PMThread] : Reached WinProcessDlg\n");
#endif
  while (WinGetMsg(hab, &msg, 0, 0, 0))
    WinDispatchMsg(hab, &msg);
#ifdef DEBUG_BUILD
  printf("[PMThread] : Shutting down!\n");
#endif


  WinDestroyWindow(hwndFrame); // Destroy windows

  // Uninitialize PM
  WinDestroyMsgQueue(hmq);
  WinTerminate(hab);

#ifdef DEBUG_BUILD
  printf("[PMThread] : Stopped\n");
#endif
}



// ---------- Main function -------------------------------------------------

int main(int argc, char *argv[])
{
#ifdef DEBUG_BUILD
  MorphToPM();
  setbuf(stdout, NULL);
  setbuf(stderr, NULL);
#ifdef __WATCOMC__
#if __WATCOMC__<1200
  printf("Compiled with WatcomC v%d.%02d\n", (__WATCOMC__)/100, (__WATCOMC__)%100);
#else
  printf("Compiled with OpenWatcom v%d.%02d\n", (__WATCOMC__-1100)/100, (__WATCOMC__-1100)%100);
#endif
#endif
  fflush(stdout);
#endif

  if (argc>1)
    bAnalogueClock = 1;
  else
    bAnalogueClock = 0;

  PMTest();

#ifdef DEBUG_BUILD
  printf("Everything done! Bye!\n");
#endif

  return 0;
}
