
// First include all needed standard header files
#include <stdlib.h>
#include <stdio.h>
#include <process.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <time.h>

#ifndef M_PI
#define M_PI 3.14159
#endif

// Then the needed OS/2 API stuffs
#define INCL_TYPES
#define INCL_WIN
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_GPI
#include <os2.h>

#include <cairo.h>
#include <cairo-os2.h>

#define DEBUG_BUILD

int bShutdown = 0;
int bNeedRedraw = 0;
int bAnalogueClock = 0;

// --------------- Functions: -----------------------------------------------

#ifdef DEBUG_BUILD
void MorphToPM()
{
  PPIB pib;
  PTIB tib;

  DosGetInfoBlocks(&tib, &pib);

  // Change flag from VIO to PM:
  if (pib->pib_ultype==2) pib->pib_ultype = 3;
}
#endif

cairo_surface_t *surface = NULL;
cairo_t *cr = NULL;
double line_width;
cairo_font_extents_t font_extents;

static void inline DisableFPUException()
{
  unsigned short usCW;

  // Some OS/2 PM API calls modify the FPU Control Word,
  // but forget to restore it.

  // This can result in XCPT_FLOAT_INVALID_OPCODE exceptions,
  // so to be sure, we always disable Invalid Opcode FPU exception
  // before using FPU stuffs with Cairo or from this application.

  usCW = _control87(0, 0);
  usCW = usCW | EM_INVALID | 0x80;
  _control87(usCW, MCW_EM | 0x80);
}

void DrawMan()
{
  cairo_move_to(cr, 0.5, 0.5);


  cairo_set_source_rgba(cr, 0.3, 0.8, 0.4, 0.8);
  cairo_set_line_width(cr, 0.015);
  cairo_move_to(cr, 0.5, 0.5);
  cairo_rel_line_to(cr, 0, +0.05);
  cairo_rel_line_to(cr, -0.03, +0.05);
  cairo_move_to(cr, 0.5, 0.50);
  cairo_rel_move_to(cr, 0, +0.05);
  cairo_rel_line_to(cr, +0.03, +0.05);
  cairo_move_to(cr, 0.5, 0.5);
  cairo_rel_line_to(cr, +0.03, +0.05);
  cairo_move_to(cr, 0.5, 0.5);
  cairo_rel_line_to(cr, -0.03, +0.05);
  cairo_move_to(cr, 0.5, 0.5);
  cairo_rel_line_to(cr, 0, -0.02);
  cairo_stroke(cr);

  cairo_new_path(cr);
  cairo_set_source_rgba(cr, 0.8, 0.3, 0.3, 0.8);
  cairo_set_line_width(cr, 0.03);
  cairo_arc(cr, 0.5, 0.5-0.03, 0.01, 0, 2*3.1415);
  cairo_stroke(cr);
  cairo_close_path(cr);
}

void DrawPage_ImagePattern(int iDrawCounter)
{
  cairo_surface_t *image;
  int width, height;


  image = cairo_image_surface_create_from_png("image.png");
  width = cairo_image_surface_get_width(image);
  height= cairo_image_surface_get_height(image);


  /* Scale */
  cairo_save(cr);

  cairo_translate(cr,
                  0.01,
                  0.13);
  cairo_scale(cr,
              1.0/width * (0.2 + (0.1*sin(3.1415/180.0*iDrawCounter*10))),
              1.0/height * (0.2 + (0.1*cos(3.1415/180.0*iDrawCounter*10)))
              );


  cairo_set_source_surface(cr, image, 0, 0);
  cairo_paint(cr);

  cairo_restore(cr);

  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.05);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Scaling");


  /* Rotate */
  cairo_save(cr);

  cairo_translate(cr,
                  0.6,
                  0.43);

  cairo_rotate(cr, 3.1415/180.0*((iDrawCounter*10)%360));

  cairo_scale(cr,
              1.0/width / 5.0,
              1.0/height / 5.0
             );

  cairo_set_source_surface(cr, image, 0, 0);
  cairo_paint(cr);

  cairo_restore(cr);

  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.05);

  cairo_move_to(cr, 0.52, 0.96);
  cairo_show_text(cr, "Rotating");


  /* Draw and rotate man */
  cairo_save(cr);

  cairo_scale(cr,
              1.01+sin(iDrawCounter*3.1415/180),
              1.01+sin(iDrawCounter*3.1415/180)
             );

  cairo_translate(cr,
                  0.3,
                  0.5);
  cairo_rotate(cr, -3.1415/180.0*((iDrawCounter*10)%360));

  cairo_translate(cr,
                  -0.3,
                  -0.5);

  DrawMan();
  cairo_restore(cr);


  cairo_surface_destroy(image);

}

void DrawImage(int iDrawCounter, int iWidth, int iHeight)
{
  int x, y;
  DisableFPUException();

  cairo_save(cr);
  cairo_scale(cr, iWidth, iHeight);

  // Clear background with black
  cairo_set_source_rgb(cr, 0, 0, 0);
  cairo_rectangle(cr,
                  0, 0,
                  1, 1);
  cairo_fill(cr);
  // Draw checkers
  for (y=0; y<10; y++)
    for (x=0; x<5; x++)
    {
      cairo_set_source_rgb(cr, 0.3, 0.3, 0.3);
      cairo_rectangle(cr,
                      x*0.2 + (y%2)*0.1, y*0.1,
                      0.1, 0.1);
      cairo_fill(cr);
    }


  // Draw the real meat
  DrawPage_ImagePattern(iDrawCounter);
  
  // Example group text
  cairo_set_source_rgba(cr, 0.8, 0.7, 0.9, 0.5);
  cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.10);

  cairo_move_to(cr, 0.02, 0.15);
  cairo_text_path(cr, "Transformations");
  cairo_fill_preserve(cr);
  cairo_set_source_rgba(cr, 0.5, 0.5, 0.5, 0.4);
  cairo_set_line_width(cr, 0.01);
  cairo_close_path (cr);
  cairo_stroke (cr);

  cairo_restore(cr);
}


MRESULT EXPENTRY MainWindowProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch (msg)
  {
    case WM_PAINT:
      {
        HPS hpsBeginPaint;
        RECTL rclRect;

        hpsBeginPaint = WinBeginPaint(hwnd, NULL, &rclRect);

        if (surface)
        {
#ifdef DEBUG_BUILD
//          printf("[MainWindowProc] : Repaint\n");
#endif
          if (!bNeedRedraw)
          {
            // Repaint the stuffs
            cairo_os2_surface_refresh_window(surface, hpsBeginPaint, &rclRect);
          } else
          {
            // It was resized or something, so we better show black image
            WinFillRect(hpsBeginPaint, &rclRect, CLR_BLACK);
          }
        }

        // All done!
        WinEndPaint(hpsBeginPaint);
        return (MRESULT) FALSE;
      }
    case WM_SIZE:
#ifdef DEBUG_BUILD
//        printf("[MainWindowProc] : In WM_SIZE\n");
#endif
        if (surface)
        {
          cairo_os2_surface_set_size(surface,
                                     SHORT1FROMMP(mp2), // New CX
                                     SHORT2FROMMP(mp2), // New CY
                                     6000);
          bNeedRedraw = 1;
        }
        break;
  }
  return WinDefWindowProc(hwnd, msg, mp1, mp2);
}

MRESULT EXPENTRY NewFrameProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  PFNWP pOldFrameProc;
  MRESULT result;
  PTRACKINFO ti;

  pOldFrameProc = (PFNWP) WinQueryWindowULong(hwnd, QWL_USER);

  result = (*pOldFrameProc)(hwnd, msg, mp1, mp2);

  if (msg == WM_QUERYTRACKINFO)
  {
    ti = (PTRACKINFO) mp2;
    ti->ptlMinTrackSize.x = 300;
    ti->ptlMinTrackSize.y = 200;
  }
  return result;
}

void fnDrawThread(void *pParm)
{
  HWND hwndClientWindow = (HWND) pParm;
  HPS hpsClientWindow;
  SWP swpTemp;
  int iDelayCounter;
  int iDrawCounter;

  // Create Cairo surface
  hpsClientWindow = WinGetPS(hwndClientWindow);
  WinQueryWindowPos(hwndClientWindow, &swpTemp);

  surface = cairo_os2_surface_create(hpsClientWindow,
                                     swpTemp.cx,
                                     swpTemp.cy);
  cr = cairo_create(surface);
  cairo_os2_surface_set_manual_window_refresh(surface,
                                              TRUE);

  iDrawCounter = 0;

  while (!bShutdown)
  {
    // Draw image into cairo surface
    WinQueryWindowPos(hwndClientWindow, &swpTemp);
    DrawImage(iDrawCounter, swpTemp.cx, swpTemp.cy);
    iDrawCounter++;
    // Make changes visible
    bNeedRedraw = 0;
    WinInvalidateRect(hwndClientWindow, NULL, FALSE);

    // Wait a bit
    iDelayCounter = 0;
//    while ((iDelayCounter<90) && (!bNeedRedraw) && (!bShutdown))
//    {
      DosSleep(32);
//      iDelayCounter++;
//    }
  }

  // Destroy Cairo surface
  if (cr)
    cairo_destroy (cr);
  if (surface)
    cairo_surface_destroy (surface);

  WinReleasePS(hpsClientWindow);

  _endthread();
}

void PMTest()
{
  HAB hab;
  HMQ hmq;
  QMSG msg;
  ULONG flCreaFlags, flFrameStyle, flClientStyle;
  SWP swap;
  PFNWP pOldFrameProc;
  HWND hwndFrame;
  HWND hwndClient;
  unsigned long tidDrawThread;


#ifdef DEBUG_BUILD
  printf("[PMThread] : Starting\n");
#endif
  // Setup this thread for using PM (sending messages etc...)
  hab = WinInitialize(0);
  hmq = WinCreateMsgQueue(hab, 0);

  // Create main window
  WinRegisterClass(hab, "PrivateClassName",  // Register a private class, so
		   MainWindowProc,           // we can use our own window procedure for frame
		   0, 0);

  // Set defaults for main window:
  flCreaFlags = 

    FCF_TASKLIST | FCF_SYSMENU | FCF_TITLEBAR | FCF_SIZEBORDER |
    FCF_DLGBORDER | FCF_MINMAX |
    FCF_SHELLPOSITION | FCF_NOBYTEALIGN | FCF_AUTOICON;
  flFrameStyle = 
    FS_NOBYTEALIGN | FS_BORDER;
  flClientStyle =
      WS_CLIPCHILDREN;

  hwndFrame = WinCreateStdWindow(HWND_DESKTOP,
				 flFrameStyle,       // Frame window style
				 &flCreaFlags,
                                 "PrivateClassName",
                                 "TestApp for CAIRO.DLL",
				 flClientStyle,//0,   // Client window style
				 NULLHANDLE,
				 0L,
                                 &hwndClient);

  pOldFrameProc = WinSubclassWindow(hwndFrame, NewFrameProc); // Subclass window to be able
                                                              // to handle resize events
  // Store old window proc in frame window's ULONG
  WinSetWindowULong(hwndFrame, QWL_USER, (ULONG) pOldFrameProc);

  // Move the window more down if it would be out of the screen
  WinQueryWindowPos(hwndFrame, &swap);
  if (swap.y+swap.cy > WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN))
  {
    WinSetWindowPos(hwndFrame, HWND_TOP, swap.x,
		    WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN) - swap.cy, 0, 0, SWP_MOVE);
    WinQueryWindowPos(hwndFrame, &swap);
  }
  if (swap.x+swap.cx > WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN))
  {
    if (WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - swap.cx >= 0)
      WinSetWindowPos(hwndFrame, HWND_TOP, 
		      WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - swap.cx, swap.y,  0, 0, SWP_MOVE);
  }

  WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_SHOW | SWP_ACTIVATE);

  // Start drawing
  bShutdown = 0;
  tidDrawThread = _beginthread(fnDrawThread,
                               0,
                               1024*1024,
                               (void *) hwndClient);

#ifdef DEBUG_BUILD
  printf("[PMThread] : Reached WinProcessDlg\n");
#endif
  while (WinGetMsg(hab, &msg, 0, 0, 0))
    WinDispatchMsg(hab, &msg);
#ifdef DEBUG_BUILD
  printf("[PMThread] : Shutting down!\n");
#endif

  // Stop drawing
  bShutdown = 1;
  DosWaitThread(&tidDrawThread, DCWW_WAIT);

  WinDestroyWindow(hwndFrame); // Destroy windows

  // Uninitialize PM
  WinDestroyMsgQueue(hmq);
  WinTerminate(hab);

#ifdef DEBUG_BUILD
  printf("[PMThread] : Stopped\n");
#endif
}



// ---------- Main function -------------------------------------------------

int main(int argc, char *argv[])
{
#ifdef DEBUG_BUILD
  MorphToPM();
  setbuf(stdout, NULL);
  setbuf(stderr, NULL);
#ifdef __WATCOMC__
#if __WATCOMC__<1200
  printf("Compiled with WatcomC v%d.%02d\n", (__WATCOMC__)/100, (__WATCOMC__)%100);
#else
  printf("Compiled with OpenWatcom v%d.%02d\n", (__WATCOMC__-1100)/100, (__WATCOMC__-1100)%100);
#endif
#endif
  fflush(stdout);
#endif

  if (argc>1)
    bAnalogueClock = 1;
  else
    bAnalogueClock = 0;

  PMTest();

#ifdef DEBUG_BUILD
  printf("Everything done! Bye!\n");
#endif

  return 0;
}
