/**
 * Dev32lib.h
 *
 * Author: David Azarewicz <david@88watts.net>
 *
 * Copyright (c) 2013-2018 David Azarewicz david@88watts.net
 *
 * LICENSE
 *
 * The following source code is provided to you solely for the purpose of
 * assisting you in developing your own OS/2 device drivers. You may use
 * this source code as a reference and to build your own OS/2 drivers. You
 * may not modify, fork, or otherwise copy this source code other than what
 * is necessary for building your own OS/2 drivers. Any request for changes
 * or fixes must be referred to the author shown above.
 *
 * This copyright and license statement may not be removed.
 */
#ifndef __DEV32LIB_H
#define __DEV32LIB_H

/* Unless otherwise specified in the individual function description,
 * All functions are intended to be used in the full 32 bit context,
 * which means ES and DS are set to a 32 bit data segment and SS is
 * set to a 32 bit stack.
 */

#define FAR16DATA __far16
#define FAR16NULL ((void FAR16DATA *)0)
#define GDTTOFAR16(sel,off) CastULONGToFar16((sel<<16)|off)

#ifndef OS2_INCLUDED
#define INCL_NOPMAPI
#include <os2.h>
#endif

#pragma pack(1)

typedef struct {
  USHORT StratEntryOffset;
  USHORT EntryCS;
  USHORT IdcEntryOffset;
  USHORT IdcCS;
  USHORT Reserved;
  USHORT ProtDS;
} DDINFO;

#define DEVNAME_SIZE 8
typedef struct {
  ULONG ulNextDD;
  USHORT usAttribs;
  USHORT pfnStrategy;
  USHORT pfnIDC;
  char abName[8];
  USHORT usDevProtCS;     //Protected mode CS
  USHORT usDevProtDS;     //Protected mode DS
  USHORT usDevRealCS;     //Real mode CS (not used)
  USHORT usDevRealDS;     //Real mode DS (not used)
  ULONG ulCaps;
  USHORT usReserved;
} DEV_HEADER;
#pragma pack()

#include <Dev32Help.h>
#include <strategy.h>
#include <Dev32rmbase.h>
#include <bsekee.h>

extern DEV_HEADER Header;

extern ULONG D32g_Flags;
/* D32g_Flags bit 0, if 1 specifies that the parameter for IDC calls is passed on the stack */

unsigned char PortInByte(unsigned short);
#pragma aux PortInByte = \
  "in   al,dx" \
  value [al] \
  parm nomemory [dx] \
  modify nomemory exact [al];

void PortOutByte(unsigned short, unsigned char);
#pragma aux PortOutByte = \
  "out  dx,al" \
  parm nomemory [dx] [al] \
  modify nomemory exact [];

unsigned short PortInWord(unsigned short);
#pragma aux PortInWord = \
  "in   ax,dx" \
  value [ax] \
  parm nomemory [dx] \
  modify nomemory exact [ax];

void PortOutWord(unsigned short, unsigned short);
#pragma aux PortOutWord = \
  "out  dx,ax" \
  parm nomemory [dx] [ax] \
  modify nomemory exact [];

unsigned long PortInDword(unsigned short);
#pragma aux PortInDword = \
  "in eax,dx" \
  parm nomemory [dx] value [eax] modify nomemory exact [eax];

void PortOutDword(unsigned short, unsigned long);
#pragma aux PortOutDword = \
  "out dx,eax" \
  parm nomemory [dx] [eax] modify nomemory exact [];

unsigned long DevGetFlagsCli(void);
#pragma aux DevGetFlagsCli = \
  "pushfd" \
  "pop eax" \
  "cli" \
  value [eax] modify nomemory exact [eax];

void DevPutFlags(unsigned long);
#pragma aux DevPutFlags = \
  "push eax" \
  "popfd" \
  parm nomemory [eax] modify nomemory exact [];

void DevCli(void);
#pragma aux DevCli = "cli";

void DevSti(void);
#pragma aux DevSti = "sti";

void DevClc(void);
#pragma aux DevClc = "clc";

void DevStc(void);
#pragma aux DevStc = "stc";

void DevInt3(void);
#pragma aux DevInt3 = "int 3";

unsigned long DevGetCS(void);
#pragma aux DevGetCS = "mov eax, cs" value [eax];

unsigned long DevGetDS(void);
#pragma aux DevGetDS = "mov eax, ds" value [eax];

unsigned long DevGetES(void);
#pragma aux DevGetES = "mov eax, es" value [eax];

unsigned long DevGetSS(void);
#pragma aux DevGetSS = "mov eax, ss" value [eax];

void DevSGDT(void *pGDT);
#pragma aux DevSGDT = \
  "sgdt [eax]" \
  parm [eax] modify exact [];

void DevSIDT(void *pIDT);
#pragma aux DevSIDT = \
  "sidt [eax]" \
  parm [eax] modify exact [];

unsigned long DevBSF(unsigned long);
#pragma aux DevBSF = \
  "bsf eax, ebx" \
  parm nomemory [ebx] value [eax] modify nomemory exact [eax];

void DevPause(void);
#pragma aux DevPause = "pause" modify nomemory;

void DevMfence(void);
#pragma aux DevMfence = "mfence" modify nomemory;

void DevSfence(void);
#pragma aux DevSfence = "sfence" modify nomemory;

void DevWbinvd(void);
#pragma aux DevWbinvd = "wbinvd" modify nomemory;

#pragma aux CastULONGToFar16 = "" parm nomemory [eax] value [eax] modify nomemory exact [];
void __far16 *CastULONGToFar16(ULONG);

#pragma aux CastFar16ToULONG = "" parm nomemory [eax] value [eax] modify nomemory exact [];
ULONG CastFar16ToULONG(void __far16 *);

/* PCI stuff ************************************************************/
#pragma pack(1)
#define MAXCAPCNT 32

typedef struct _PCI_CAP {
  UCHAR pos;
  UCHAR type;
} PCI_CAP;

typedef struct _PCI_BAR {
  ULONG bar;
  ULONG start;
  ULONG size;
  unsigned io:1;
  unsigned type:2;
  unsigned prefetchable:1;
} PCI_BAR;

struct PciDeviceInfo {
  /* Beginning of required section for Dev32lib */
  USHORT BusDevFunc;
  USHORT device;
  USHORT vendor;
  USHORT subsystem_vendor;
  USHORT subsystem_device;
  USHORT status;
  USHORT command;
  USHORT class_code_subclass;
  USHORT prog_if_rev_id;
  USHORT pcie_flags_reg; /* cached PCI-E Capabilities Register */
  UCHAR headertype;
  UCHAR irq;
  UCHAR ipin;
  UCHAR capcnt;
  UCHAR pcie_cap; /* PCI-E capability offset */
  UCHAR pcie_mpss; /* PCI-E Max Payload Size Supported */
  UCHAR pm_cap;
  UCHAR Unused;
  PCI_BAR bars[6];
  PCI_CAP caps[MAXCAPCNT];
  /* End of required section for Dev32lib */
};
typedef struct PciDeviceInfo PCI_DEVICEINFO;

typedef struct _KernalAdr
{
  ULONG D16;
  ULONG C16;
  ULONG D32;
  ULONG C32;
  ULONG D16size;
  ULONG C16size;
  ULONG D32size;
  ULONG C32size;
} KERNELADR;
#pragma pack()


#define PCI_MK_BUSDEVFUNC(b,d,f) (((b)<<8)|((d)<<3)|(f))
#define PCI_BUS_FROM_BDF(bdf) ((bdf)>>8)
#define PCI_DEV_FROM_BDF(bdf) (((bdf)>>3)&0x1f)
#define PCI_FUNC_FROM_BDF(bdf) ((bdf)&0x07)

extern int __syscall PciReadConfig(const USHORT BusDevFunc, const UCHAR reg, const UCHAR size, void *data);
/* Read a PCI config register
 * BusDevFunc: bits 15-8=Bus, bits 7-3=Device, bits 2-0=Function
 * reg: the register to read
 * size: the size of the read. 1=byte, 2=word, 4=dword
 * *data: a pointer to where to write the result
 * Returns: non-zero on failure.
 */

extern int __syscall PciWriteConfig(const USHORT BusDevFunc, const UCHAR reg, const UCHAR size, const ULONG data);
/* Write a PCI config register
 * BusDevFunc: bits 15-8=Bus, bits 7-3=Device, bits 2-0=Function
 * reg: the register to write
 * size: the size of the write. 1=byte, 2=word, 4=dword
 * data: the data to write
 * Returns: non-zero on failure.
 */

extern USHORT __syscall PciFindDevice(USHORT Vendor, USHORT Device, USHORT Index);
/* Find a PCI device by Vendor:Device ID
 * Vendor: the vendor ID to look for
 * Device: The device ID to look for
 * Index: Which device to return. zero based. 0 finds the first one, 1 finds the second one, etc.
 * Returns: 0xFFFF on failure. the BusDevFunc of the device if success.
 */

extern USHORT __syscall PciFindClass(ULONG Class, USHORT Index);
/* Find a PCI device by class ID
 * Class: the class ID to look for.
 *  MSB |                |              |  LSB
 *  00  | PCI_BASE_CLASS | PCI_SUBCLASS | PCI_PROGIF
 *  Class = MAKEULONG(MAKEUSHORT(PCI_PROGIF, PCI_SUBCLASS), MAKEUSHORT(PCI_BASE_CLASS, 0));
 * Index: Which device to return. zero based. 0 finds the first one, 1 finds the second one, etc.
 * Returns: 0xFFFF on failure. the BusDevFunc of the device if success.
 */

extern int __syscall PciGetDeviceInfo(USHORT BusDevFunc, PCI_DEVICEINFO *pDevInfo);
/* Get the PCI device's configuration information
 * BusDevFunc: bits 15-8=Bus, bits 7-3=Device, bits 2-0=Function
 * pDevInfo: Pointer to the PCI_DEVICEINFO structure where the information is returned.
 * Returns: non-zero on failure.
 */

extern int __syscall PciSetPowerStateD0(PCI_DEVICEINFO *pDevInfo);
extern int __syscall PciSetPowerStateD3hot(PCI_DEVICEINFO *pDevInfo);
extern int __syscall PciSetBusMaster(PCI_DEVICEINFO *pDevInfo);
extern UCHAR __syscall PciGetCapPos(const UCHAR type, PCI_DEVICEINFO *pDevInfo);

/* Resource Manager stuff ***********************************************/

extern int __syscall RmAddMem(HDRIVER hDriver, HRESOURCE *phResource, ULONG addr, ULONG length);
extern int __syscall RmAddIrq(HDRIVER hDriver, HRESOURCE *phResource, USHORT irq, USHORT pin, USHORT Flags);
extern int __syscall RmAddIo(HDRIVER hDriver, HRESOURCE *phResource, USHORT addr, USHORT length);
extern int __syscall RmUpdateAddIrq(HDRIVER hDriver, HADAPTER phAdapter, USHORT irq, USHORT pin, USHORT Flags);

/* The Rm1 functions are simplified for drivers that only support one adapter.
 * This avoids the necessity for the driver to manage all the handles.
 */

extern int __syscall Rm1CreateDriver(DRIVERSTRUCT *pDrs);
/* Creates a driver. Must be called first */

extern void __syscall Rm1Destroy(int iDestroyDriver);
/* Destroys everything that was created */

/* Add the specified type of resource */
extern int __syscall Rm1AddMem(ULONG addr, ULONG length);
extern int __syscall Rm1AddIrq(USHORT irq, USHORT pin, USHORT Flags);
extern int __syscall Rm1AddIo(USHORT addr, USHORT length);

extern int __syscall Rm1CreateAdapter(ADAPTERSTRUCT *pAds);
/* Creates the adapter. Must be called after adding the resources. */

extern int __syscall Rm1GetHandles(HDRIVER *phDriver, HADAPTER *phAdapter);
/* Gets the handles of the previously created adapter. Must be called after Rm1CreateAdapter. */

extern int __syscall Rm1UpdateAddIrq(USHORT irq, USHORT pin, USHORT Flags);
/* Addes an Irq resource to an existing adapter */

/* Memory functions *****************************************************/
#define PAGE_SIZE 0x1000

extern void * __syscall MapPhysToLin(ULONG ulPhysAdr, ULONG ulSize);
/* Maps a physical address to a linear address using KernVMAlloc(). This function
 * is normally for mapping the physical register space of a device to a linear
 * address so it can be accessed by a device driver. Mapping is always done in
 * whole pages so if  you map 16 bytes, you will really get 4096 bytes mapped. The
 * conversion functions in this library cannot convert pointers returned by this
 * function to Far16. Free this memory mapping with D32UnMap(). This function
 * cannot be called at interrupt time.
 */

extern int __syscall D32UnMap(void *LinAdr);
/* Free memory allocated with MapPhysToLin().
 * This function cannot be called at interrupt time
 */

extern void * __syscall D32VMAlloc(ULONG ulSize);
/* Allocate some memory using KernVMAlloc(). Tries to allocate high memory
 * first, and if that fails, allocates low memory.
 * Free this memory using D32VMFree.
 * This function cannot be called at interrupt time
 */

extern int __syscall D32VMFree(void *LinAdr);
/* Free memory allocated with D32VMAlloc().
 * This function cannot be called at interrupt time
 */

extern void * __syscall D32AllocMem(ULONG ulSize, ULONG *pPhysAdr, ULONG ulFlags);
/* Allocate memory using KernVMAlloc() and return the physical address.
 * Free this memory using D32FreeMem().
 * This function cannot be called at interrupt time
 * If ulFlags is zero, memory allocation is first attempted with VMDHA_USEHIGHMEM | VMDHA_FIXED | VMDHA_CONTIG.
 * If ulFlags is non-zero, memory allocation is first attempted with ulFlags | VMDHA_FIXED | VMDHA_CONTIG.
 * If the first attempt fails, a second attempt is made with only the VMDHA_FIXED | VMDHA_CONTIG flags.
 */

extern int __syscall D32FreeMem(void *LinAdr);
/* Free memory allocated with D32AllocMem().
 * This function cannot be called at interrupt time
 */

/* The Mem* functions are special memory allocation functions that maintain
 * a pool of memory in such a way that they can provide linear, physical
 * and far16 pointers to every buffer that is allocated. The maximum size
 * of a memory buffer that can be allocated is 64K.
 */

void * __syscall MemAlloc(ULONG ulSize);
/* Allocate some memory and return a flat linear address to it. Max 64K size
 * This function cannot be called at interrupt time.
 *
 * The amount of memory returned is the size of memory requested rounded up to
 * the nearest paragraph (16 bytes). If you request 15 bytes, you get 16 bytes.
 * If you request 48 bytes, you get 48 bytes.
 *
 * All memory returned by MemAlloc is guaranteed to be paragraph (16 byte) aligned.
 */

void * __syscall MemAllocAlign(ULONG ulSize, ULONG ulAlign);
/* Same as MemAlloc() except you can specify an alignment. Alignment specifiers
 * 0, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096 are valid. Both MemAllocAlign(n,0)
 * and MemAllocAlign(n,16) are exactly the same as MemAlloc(n) and return 16 byte
 * aligned memory.
 */

ULONG __syscall MemVerifyBounds(void *LinAdr, ULONG ulSize);
/* Finds the previously allocated memory block which contains LinAdr and then
 * verifies that the specified memory block (LinAdr+ulSize) is completely
 * contained within that previously allocated memory block. Returns zero if
 * the block of memory specified by (LinAdr+ulSize) is within the previously
 * allocated memory. Returns non-zero on failure:
 * 0=Specified block is within allocated block
 * 1=Specified block is not within allocated block
 * 2=Allocated block not found
 * 3=heap not initialized - no allocated blocks
 */

int __syscall MemFree(void *pLinAdr);
/* Free the memory that was allocated using MemAlloc()
 * This function can be called at interrupt time.
 */

ULONG __syscall MemPhysAdr(void *pLinAdr);
/* Return the physical address of memory that was allocated using MemAlloc()
 * This function can be called at interrupt time.
 */

void __far16 * __syscall MemFar16Adr(void *pLinAdr);
/* Return the far16 pointer to the memory that was allocated using MemAlloc()
 * This function can be called at interrupt time.
 */

extern ULONG D32g_HeapUseLowMem;
/* Set to 1 to cause the heap manager to return memory below 16MB.
 * Default=0 -- use high memory if possible.
 */

/* some Debug stuff *****************************************************/

/* The debug functions output to an internal memory buffer, the serial port, video, or user function.
 * The default is to output to the internal memory buffer.
 * If D32g_DbgPutc is specified, output goes there and nowhere else.
 * If D32g_ComBase is specified output goes there and nowhere else unless D32g_DbgPutc is specified.
 * If dInitVideo() then output goes to the screen and the internal memory buffer.
 */

extern ULONG D32g_DbgLevel; /* default is 1. If bit 31 is set, Bits 0-30 are a mask */
extern int D32g_DbgBufWrap; /* default is 0 (nowrap) */
extern USHORT D32g_ComBase; /* default is 0 (no COM port output) */
extern void (*D32g_DbgPutc)(ULONG Char); /* User function for debug output. */
extern void _syscall D32g_DbgBufPutc(ULONG ulCh);
extern void __syscall dprintf(int level, const char *fmt, ...);
extern void __syscall rprintf(const char *fmt, ...);
extern void __syscall dHexDump(int level, const void *p, int len, const char *fmt, ...);
extern unsigned long __syscall dCopyToUser(unsigned char *Buffer, unsigned long Count);
extern void __syscall InitComPort(ULONG BaudRate);
extern void __syscall dInitVideo(void);
extern void __syscall dEndVideo(void);

/* Trace functions ******************************************************/

extern void __syscall TraceInit(ULONG ulMajor, ULONG ulLevel);

/* Warning: This function can use a lot of stack space */
extern void __syscall TraceBuf(ULONG minCode, ULONG DataLen, void *pBuf);

extern void __syscall TraceArgs(ULONG minCode, ULONG DataLen, ...);

/* Misc functions *******************************************************/

typedef struct {
  volatile ULONG *pMsCount;
  ULONG Start;
  ULONG End;
} TIMER;

extern ULONG __syscall TimeGetSec(void);
extern ULONG __syscall TimeGetMs(void);
extern void __syscall TimeReset(void);
extern void __syscall TimeInit(void);
extern ULONG __syscall TimeDiff(void);
extern void __syscall UtMSleep(ULONG msecs);
#define msleep UtMSleep
extern void TimerInit(TIMER *pTimer, ULONG Milliseconds);

extern int TimerCheckAndBlock(TIMER *pTimer);
/* Check if the timer is done.
 *
 * This function checks the previously setup timer and returns true if
 * done, blocks for one timer interval otherwise.
 *
 * The accuracy is limited by the OS/2 timer interrupt frequency which
 * can lead to intervals up to 55ms (18.2 timer interrupts per second).
 */

extern void __syscall UtSetDriverName(char *pDevName);
/* Copies the specified string to the device driver header, enforcing the
 * rules (8 char max, blank padded)
 */

extern int __syscall UtModifyName(char *pDevName, int iChar, int iOverwrite);
/* Modifies the given string assuming it is a device driver name.
 * Finds the first '$'.
 * Checks the character immediately before the '$' to see if it is numeric.
 * If it is numeric, it is replaced with '0'+iChar. If not, '0'+iChar is
 * inserted immediately before the '$'. Non-zero status is returned on any
 * failure: iChar is not between 0 and 9, inserting a char would make the
 * name longer than 8 characters, the name is only 1 character. The buffer
 * must be at least 9 bytes long to hold the terminating null character.
 */

extern int __syscall ArgCmp(const char *CmdString, const char *ArgString);
/* Compare the given command line pointer to the given argument string and
 * return true if there is a match. CmdString characters are converted to
 * upper case before comparison so ArgString must be all upper case.
 */

extern long __syscall strtol(const char *pString, const char **pEnd, int Base);
/* Convert a string to a long value using the specified base.
 * pString is the string to be converted.
 * pEnd is a pointer to the string where conversion ended. Can be NULL.
 * Base is the base of the conversion. If 0 the function determines the base
        automatically. octal(begins with a 0) or hex(begins with 0x), or decimal.
 */

extern void _syscall StringPad(char *pDst, int iLen, char chPad);
/* Pad the string pDst with chPad to a length of iLen.
 * Terminates the string pDst[iLen]=0.
 */

extern int __syscall GetString(const CHAR *pStringIn, CHAR *pTarget, ULONG MaxLen);

/* This routine is obsolete and will eventually be removed */
extern ULONG __syscall GetHex(CHAR *pString, UCHAR *pResult, ULONG MaxLen);

static inline ULONG ReadDword(void *address) { return *(ULONG*)address; }
static inline USHORT ReadWord(void *address) { return *(USHORT*)address; }
static inline UCHAR ReadByte(void *address) { return *(UCHAR*)address; }
static inline void WriteDword(void *address, ULONG value) { *(ULONG*)address = value; }
static inline void WriteWord(void *address, USHORT value) { *(USHORT*)address = value; }
static inline void WriteByte(void *address, UCHAR value) { *(UCHAR*)address = value; }

#pragma aux D32ThunkStackTo32 modify [];
extern void D32ThunkStackTo32(void);
/* Must be called with a 16 bit stack. Results are undefined if called with a 32 bit stack. */

#pragma aux FlatToFar16 parm [eax] modify exact [eax]
extern void __far16 *FlatToFar16(void *Pointer);
/* Converts flat pointer to a Far16 pointer. This function can covert data segment pointers
 * and stack segment pointers. Attempting to convert a flat pointer that cannot be
 * converted will generate a trap 3. Use of D32ThunkStackTo32() is required instead of
 * KernThunkStackTo32() in order for FlatToFar16() to work properly for stack variables.
 */

#pragma aux Far16ToFlat parm [eax] modify exact [eax]
extern void *Far16ToFlat(void __far16 *Pointer);
/* Converts any Far16 pointer to a flat pointer */

extern ULONG __syscall DevGetDS16(void);

#pragma aux TestAndSetBit parm [eax] [ebx] value [eax] ;
extern ULONG TestAndSetBit(ULONG BitNumbar, volatile ULONG *addr);

#pragma aux ClearBit parm [eax] [ebx] modify [eax] ;
extern void ClearBit(ULONG BitNumber, volatile ULONG *addr);

extern ULONG LockXchg(volatile ULONG *addr, ULONG value);
#pragma aux LockXchg = \
  "lock xchg [ebx], eax" \
  parm [ebx] [eax] value [eax] modify [eax];

extern int __syscall ValidateFar16Adr(void __far16 *Pointer);
/* Returns non-zero if the pointer is not valid. Checks that the selector is valid
 * and that the offset is within the range defined by the selector.
 */

/* These are spin loop functions. Avoid using these if possible.
 * Use ProcBlock instead.
 */

#pragma aux udelay parm [ecx] modify [eax ecx] ;
extern void udelay(ULONG microseconds);
extern void __syscall mdelay(ULONG milliseconds);

/* The following File I/O functions are experimental and may not work, may be changed,
 * or may be removed. These functions only work at init time.
 */
extern ULONG __syscall D32FileOpen(char *pFileName, ULONG *pulHandle);
extern ULONG __syscall D32FileClose(ULONG fHandle);
extern ULONG __syscall D32FileReadAt(ULONG ulHandle, ULONG ulAt, void *pBuf, ULONG ulSize, ULONG *pulActual);
extern ULONG __syscall D32FileSize(ULONG ulHandle, ULONG *pulSize);

/* This function is only available in debug builds */
extern void __syscall MemHeapVerify(void *LinAdr);

/* some libc stuff ******************************************************/

/* ctype macros */
#define isupper(ch) ((ch) >= 'A' && (ch) <= 'Z')
#define islower(ch) ((ch) >= 'a' && (ch) <= 'z')
extern int __syscall tolower(int ch);
extern int __syscall toupper(int ch);

#ifndef __size_t
#define __size_t
typedef unsigned long size_t;
#endif

void *memchr(const void *s, short c, size_t n);
int memcmp(const void *s1, const void *s2, size_t n);
void *memcpy(void *s1, const void *s2, size_t n);
void *memset(void *s, short c, size_t n);
char *strcat(char *s1, const char *s2);
int strcmp(const char *s1, const char *s2);
char *strcpy(char *s1, const char *s2);
size_t strlcpy(char *dst, const char *src, size_t siz);
int strlen(const char *s);
#pragma intrinsic(memchr,memcmp,memcpy,memset)
#pragma intrinsic(strcat,strcmp,strlen,strcpy)

/* memmove handles overlapping memory moves */
extern void * __syscall memmove(void *s1, const void *s2, size_t n);
extern int __syscall strncmp(const char *s1, const char *s2, size_t n);
extern char * strncpy(char *s1, const char *s2, size_t __n); /* s1 will always be zero terminated */
#define strcasecmp stricmp
extern int __syscall stricmp(const char *s1, const char *s2);
extern int __syscall atoi(char *str);

#ifndef min
#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))
#endif
#define	roundup2(x, y) (((x)+((y)-1))&(~((y)-1))) /* if y is powers of two */
#define	nitems(x)	(sizeof((x)) / sizeof((x)[0]))
#define	offsetof(type, field) ((ULONG)((const volatile void *)&((type *)0)->field))

extern int abs( int __j );
extern long int labs( long int __j );
#pragma intrinsic(abs,labs)

/*
 * Very basic formatted string functions
 * Leading zero modifer is supported.
 * Length modifier is supported.
 * modifiers are: l=long, ll=long long, F=far
 * functions are: s=string, c=char, d=decimal, u=unsigned, x=hex, p=pointer
 * Formats currently understood are:
 *   %c
 *   %[0][n][l[l]]d
 *   %[0][n][l[l]]u
 *   %[0][n][l[l]]x
 *   %[n][.m][F]s
 *   %[F]p
 * Examples:
 * %s = near (flat) string
 * %Fs = far16 string
 * %d = 32 bit decimal
 * %ld = 32 bit decimal
 * %lld = 64 bit decimal
 * %x = 32 bit hex
 * %lx = 32 bit hex
 * %llx = 64 bit hex
 * %p = 32 bit flat pointer
 * %Fp = 16:16 pointer
 */

extern int __syscall sprintf(char *buf, const char *fmt, ...);
extern int __syscall snprintf(char *buf, int len, const char *fmt, ...);

extern void __syscall iprintf(const char *fmt, ...);
/* iprintf formats a string and calls DisplayMessage to output to the console. */

extern int DisplayMessage(char *pMessage);
#pragma aux DisplayMessage parm [eax] value [eax] modify nomemory [edx];
/* DisplayMessage uses the DevHelp_Save_Message call to output a message to
 * the console. This can be used by DEVICE and BASEDEV drivers.
 */

/* Timer functions */

typedef  void (__syscall *PFN_TIMER_NOTIFY)(ULONG,ULONG); /* Notify address */

extern ULONG __syscall Timer_InitTimer(ULONG ulTimerCount);

extern void __syscall Timer_DeInstallTimer(void);

extern ULONG __syscall Timer_StartTimerMS(ULONG *pHandle, ULONG ulTimerMS, PFN_TIMER_NOTIFY pfnAddrNotify, ULONG ulParm1);

extern void __syscall Timer_ResetTimer(ULONG ulTimerHandle);

extern void __syscall Timer_CancelTimer(ULONG ulTimerHandle);

/* PSD functions */
extern int __syscall PsdMsiCount(USHORT usBusDevFunc);
extern int __syscall PsdMsixCount(USHORT usBusDevFunc);
extern int __syscall PsdMsiAlloc(USHORT usBusDevFunc, ULONG *pulCount, UCHAR *pucIrq);
extern int __syscall PsdMsixAlloc(USHORT usBusDevFunc, ULONG *pulCount, UCHAR *pucIrq);
extern int __syscall PsdMsiRelease(USHORT usBusDevFunc);

#define CS_SIZE 0
#define CS_ALIGN 1
extern int __syscall CheckStructure(int iFunction, char *pName, ULONG ulSize, ULONG ulCheck);
extern int __syscall FindKernelAddresses(KERNELADR *pKrnAdr);

#endif /*__DEV32LIB_H */
