/*******************************************************************************
* FILE NAME: iddeccnv.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Definition of the class(es):                                               *
*     IDDEClientConversation - Defines a client DDE conversation.              *
*     IDDEActiveServer       - Represents a DDE server supporting a            *
*                              particular topic.                               *
*     IDDEActiveServerSet    - Set of active DDE servers.                      *
*     IDDEClientHotLinkSet   - Set of active hot links.                        *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
extern "C"
{
   #define INCL_WINDDE
   #define INCL_DOSQUEUES
   #define INCL_DOSPROCESS
   #define INCL_DOSERRORS
   #define INCL_DOSSEMAPHORES
   #include <os2.h>
   #include <string.h>
}

#include <iddeccnv.hpp>
#include <iddeinfo.hpp>
#include <iddecset.hpp>
#include <itrace.hpp>
#include <irect.hpp>
#include <iexcept.hpp>
#include <iddeevt.hpp>
#include <iobjwin.hpp>
#include <iqueue.h>
#include <icconst.h>
#include <ithread.hpp>

/*------------------------------------------------------------------------------
| An object of this class is create in the dispatchEventFromQueue function.    |
| This guarantees that when we exit that secondary thread function, (either    |
| because an exception has been thrown, or because we issued a                 |
| DosCloseQueue in the IDDEClientConversation dtor), that the DosWaitEvenSem   |
| in the IDDEClientConversation dtor will be satisfied and the dtor will       |
| safely complete.                                                             |
------------------------------------------------------------------------------*/
class IDDEFreeSemaphore {
public:
  IDDEFreeSemaphore ( unsigned long semaphore )
       : ulClSem(semaphore) { }
 ~IDDEFreeSemaphore ( ) {
    DosPostEventSem(ulClSem); }
unsigned long
  ulClSem;
};  // IDDEFreeSemaphore

class IDDEClosedConversation : public IDDEActiveServer {
public:
  IDDEClosedConversation  ( const char * appName,
                            const char * topic,
                            const IWindowHandle serverId,
                            Boolean userclose = false );
  ~IDDEClosedConversation ( );
IWindowHandle
  serverHandle ( ) const { return wndhClServer; }
Boolean
  userClosed ( ) const { return fClUsrCls; }
private:
IWindowHandle
  wndhClServer;
Boolean
  fClUsrCls;
}; // IDDEClosedConversation

class IDDEHotLinkXEvent : public IDDEClientHotLinkEvent {
public:
  IDDEHotLinkXEvent ( IEvent evt );
  ~IDDEHotLinkXEvent ( );
Boolean
  isCloseInProgress ( ) const { return fClCloseInPrgrs; }
void
  setCloseInProgress (Boolean clsInPrgrs) { fClCloseInPrgrs = clsInPrgrs; }
private:
Boolean
  fClCloseInPrgrs;
}; // IDDEHotLinkXEvent

/*------------------------------------------------------------------------------
| Transaction Queue                                                            |
|                                                                              |
| The transaction queue will maintain all transactions. When the transaction   |
| is completed and acknowledged by the server it is removed from the queue.    |
| The queue is protected from concurrent updates by the use of a semaphore     |
| (IPrivateResource).                                                          |
------------------------------------------------------------------------------*/
class IDDETransactionQueue : public IQueue<IDDEClientAcknowledgeEvent*>
{
public:
  IDDETransactionQueue ( );
  ~IDDETransactionQueue ( );
IPrivateResource
 &semaphor()
    { return priResCl; }
private:
  IDDETransactionQueue ( const IDDETransactionQueue& );
  IDDETransactionQueue& operator= ( const IDDETransactionQueue& );
IPrivateResource
  priResCl;
}; // IDDETransactionQueue

class IDDEClosedConversationSet : public ISet<IDDEClosedConversation*>
{
public:
  IDDEClosedConversationSet ( );
  ~IDDEClosedConversationSet ( );
IPrivateResource
 &semaphor()
    { return priResCl; }
private:
  IDDEClosedConversationSet ( const IDDEClosedConversationSet& );
  IDDEClosedConversationSet& operator= ( const IDDEClosedConversationSet& );
IPrivateResource
  priResCl;
}; // IDDEClosedConversationSet

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IDDECCNV_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IDdeleteElemASS                                                              |
------------------------------------------------------------------------------*/
static Boolean
  IDdeleteElemASS ( IDDEActiveServer* const& activeServer,
                    void*                    anything )
{
   delete (IDDEActiveServer*)activeServer;
   return true;
}

/*------------------------------------------------------------------------------
| IDdeleteElemCHLS                                                             |
------------------------------------------------------------------------------*/
static Boolean
  IDdeleteElemCHLS ( IDDEClientHotLinkEvent* const& hotLinkEvent,
                     void*                          item )
{
   int rc;
   if (item != 0)
   {
      IString tmpStr = IString::upperCase(hotLinkEvent->item());
      rc = strcmp(tmpStr, (char*)item);
   }
   if (item == 0 || rc == 0)
   {
      delete (IDDEHotLinkXEvent*)hotLinkEvent;
      return true;
   }
   return false;
}

/*------------------------------------------------------------------------------
| IDdeleteElemTQ                                                               |
------------------------------------------------------------------------------*/
static Boolean
  IDdeleteElemTQ ( IDDEClientAcknowledgeEvent* const& event,
                   void*                      anything )
   { delete (IDDEClientAcknowledgeEvent*)event;
     return true;  }

/*------------------------------------------------------------------------------
| IDdeleteElemCCS                                                              |
------------------------------------------------------------------------------*/
static Boolean
  IDdeleteElemCCS ( IDDEClosedConversation* const& conversation,
                    void*                            anything )
   { ITRACE_DEVELOP( conversation->application() +
        IString(" server application never responded to WM_DDE_TERMINATE "));
     delete (IDDEClosedConversation*)conversation;
     return true;  }


/*------------------------------------------------------------------------------
| IDDEClientConversation::IDDEClientConversation                               |
|                                                                              |
| Setting useEventThread will cause a thread to be used for the dispatching    |
| of handlers. A queue is also set up for dispatching events.                  |
------------------------------------------------------------------------------*/
IDDEClientConversation :: IDDEClientConversation ( Boolean useEventThread )
                            : wndhClServer(0),
                              fClBrdcstInPrgrs(false),
                              fClCaseSensitive(false),
                              fClPostMsgFail(false),
                              fClHdrActive(false),
                              strClTopic(),
                              strClApplication(),
                              pActServSetCl(0),
                              pThreadCl(0)
{
   IMODTRACE_DEVELOP("DDEClntConv::ctor");
   if (useEventThread)
   {
      PPIB ppib;
      PTIB ptib;
      unsigned long ulRc = DosCreateEventSem(0,
                                             &ulClSemaphore,
                                             0,
                                             0);
      if (ulRc)
         ITHROWSYSTEMERROR(ulRc,"DosCreateEventSem",
             IErrorInfo::outOfSystemResource, IException::recoverable);
      DosGetInfoBlocks( &ptib, &ppib );
      // ensure we have a unique queue handle by using PID
      IString tmp1((unsigned long)this);
      IString tmp2(ppib->pib_ulpid);
      IString strQueueName = IString("\\QUEUES\\") + tmp1 +
          IString("\\") + tmp2 + IString("\\.QUE");
      ITRACE_DEVELOP("Queue name is: " + strQueueName);
      ulRc = DosCreateQueue(&ulClQHandle,
                            0,
                            (char*)strQueueName);
      ITRACE_DEVELOP("DosCreateQueue rc =: " + IString(ulRc));
      if (ulRc)
         ITHROWSYSTEMERROR(ulRc,"DosCreateQueue",
             IErrorInfo::outOfSystemResource, IException::recoverable);
      IThreadMemberFn<IDDEClientConversation> *myDispatcherFn = new
          IThreadMemberFn<IDDEClientConversation>
            (*this, dispatchEventFromQueue);
      pThreadCl = new IThread(myDispatcherFn);
   }

   pwndClClient = new IObjectWindow;
   pClsdConvSetCl = new IDDEClosedConversationSet;
   pHLSetCl = new IDDEClientHotLinkSet;
   pTransQCl = new IDDETransactionQueue;
   pFormatSetCl = new IDDEFormatSet;
   wndhClClient = pwndClClient->handle();
   handleEventsFor((IWindow*)pwndClClient);
   ((IWindow*)pwndClClient)->setParent(IWindow::objectWindow());
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::IDDEClientConversation                               |
|                                                                              |
| This constructor will also begin the conversation.                           |
------------------------------------------------------------------------------*/
IDDEClientConversation :: IDDEClientConversation  ( const char* application,
                                                    const char* topic,
                                                    Boolean     useEventThread )
                            : wndhClServer(0),
                              fClBrdcstInPrgrs(false),
                              fClCaseSensitive(false),
                              fClPostMsgFail(false),
                              fClHdrActive(false),
                              strClTopic(),
                              strClApplication(),
                              pActServSetCl(0),
                              pThreadCl(0)
{
   IMODTRACE_DEVELOP("DDEClntConv::ctor");
   if (useEventThread)
   {
      PPIB ppib;
      PTIB ptib;
      unsigned long ulRc = DosCreateEventSem(0,
                                             &ulClSemaphore,
                                             0,
                                             0);
      if (ulRc)
         ITHROWSYSTEMERROR(ulRc,"DosCreateEventSem",
             IErrorInfo::outOfSystemResource, IException::recoverable);
      // ensure we have a unique queue handle by using PID
      DosGetInfoBlocks( &ptib, &ppib );
      IString tmp1((unsigned long)this);
      IString tmp2(ppib->pib_ulpid);
      IString strQueueName = IString("\\QUEUES\\") + tmp1 +
          IString("\\") + tmp2 + IString("\\.QUE");
      ITRACE_DEVELOP("Queue name is: " + strQueueName);
      ulRc = DosCreateQueue(&ulClQHandle,
                            0,
                            (char*)strQueueName);
      ITRACE_DEVELOP("DosCreateQueue rc =: " + IString(ulRc));
      if (ulRc)
         ITHROWSYSTEMERROR(ulRc,"DosCreateQueue",
             IErrorInfo::outOfSystemResource, IException::recoverable);
      IThreadMemberFn<IDDEClientConversation> *myDispatcherFn = new
          IThreadMemberFn<IDDEClientConversation>
            (*this, dispatchEventFromQueue);
      pThreadCl = new IThread(myDispatcherFn);
   }

   pwndClClient = new IObjectWindow;
   pClsdConvSetCl = new IDDEClosedConversationSet;
   pHLSetCl = new IDDEClientHotLinkSet;
   pTransQCl = new IDDETransactionQueue;
   pFormatSetCl = new IDDEFormatSet;
   wndhClClient = pwndClClient->handle();
   handleEventsFor((IWindow*)pwndClClient);
   ((IWindow*)pwndClClient)->setParent(IWindow::objectWindow());

   try {
      begin(application, topic);
   }
   catch (IException& excp)
   {
      stopHandlingEventsFor((IWindow*)pwndClClient);
      delete pwndClClient;
      delete &closedConversations();
      delete &hotLinksForUpdate();
      delete &transactions();
      if (pThreadCl) 
         delete pThreadCl;
      delete pFormatSetCl;
      IRETHROW(excp);
   }
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::~IDDEClientConversation                              |
------------------------------------------------------------------------------*/
IDDEClientConversation :: ~IDDEClientConversation()
{
   IMODTRACE_DEVELOP("DDEClntConv::dtor");
   if (inConversation())
      end();

   // iterate all sets & queues; remove, and delete the elements,
   // delete the set/q
   // if threads were used stop the thread and close the queue
   if (pThreadCl)
   {
   // wait for the secondary thread to finish
      unsigned long ulPostCount;
      unsigned long ulRc = DosCloseQueue(queueHandle());
      ITRACE_DEVELOP("DosCloseQueue rc =: " + IString(ulRc));
      ulRc = DosWaitEventSem(ulClSemaphore,
                             3000);
      if (ulRc == ERROR_TIMEOUT)
         pThreadCl->stop();
      delete pThreadCl;
   }

   closedConversations().removeAll(&IDdeleteElemCCS);
   delete &closedConversations();
   hotLinksForUpdate().removeAll(&IDdeleteElemCHLS);
   delete &hotLinksForUpdate();
   transactions().allElementsDo(&IDdeleteElemTQ);
   transactions().removeAll();
   delete &transactions();
   delete &formats();

   stopHandlingEventsFor((IWindow*)pwndClClient);
   delete pwndClClient;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::begin                                                |
|                                                                              |
| Try to begin a conversation with the server that supports the application    |
| name with the requested topic name.                                          |
|                                                                              |
| Notes:                                                                       |
|   The broadcast in progress flag (fClBrdcstInPrgrs) is used to get request   |
|   information about the server. Multiple messages may be received by         |
|   the handle methods in response to these broadcasted messages. When         |
|   broadcast in progress is set the application name and/or the topic         |
|   name can be zero length strings ( see supportingTopics and                 |
|   supportingApplications).                                                   |
------------------------------------------------------------------------------*/
Boolean IDDEClientConversation :: begin ( const char* applicationName,
                                          const char* topicName )
{
   IMODTRACE_DEVELOP("DDEClntConv::begin");
   IASSERTPARM(applicationName != 0 && topicName != 0);

   if (!fClBrdcstInPrgrs)
   {
      IASSERTPARM(applicationName[0] != '\0' && topicName[0] != '\0');
      IASSERTSTATE(!inConversation());

      strClApplication = applicationName;
      strClTopic = topicName;
   }
   CONVCONTEXT contxt = {0,0,0,0,0,0};
   contxt.cb = sizeof(CONVCONTEXT);
   Boolean bSuccess = WinDdeInitiate((HWND)clientHandle(),
                                     (PSZ)applicationName,
                                     (PSZ)topicName,
                                     &contxt);
   if (!bSuccess)
   {
      if (!inConversation() && !fClBrdcstInPrgrs)
         ITHROWLIBRARYERROR(IC_DDE_WINDDEINITIATE,IErrorInfo::accessError,
             IException::recoverable);
      ITRACE_DEVELOP("WinDdeInitiate error, no exception thrown because \
          a conversation was successfully initiated.");
   }
   if (!inConversation())
   {
      strClApplication = "";
      strClTopic = "";
   }

   // all applications have responded!
   fClBrdcstInPrgrs = false;
   return inConversation();
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::begin                                                |
|                                                                              |
| Begin a conversation with the specified server handle.                       |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: begin (
                            const IWindowHandle& serverHandle )
{
   IMODTRACE_DEVELOP("DDEClntConv::begin");
   IASSERTPARM(serverHandle.isValid());
   IASSERTSTATE(!inConversation());

   wndhClServer = serverHandle;
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::end                                                  |
|                                                                              |
| End the client's conversation with a server. All of the hot links with the   |
| server are ended, and all queued transactions are removed.                   |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: end ( )
{
   IMODTRACE_DEVELOP("DDEClntConv::end");
   if ( !(inConversation()) )
      ITHROWLIBRARYERROR1(IC_DDE_CLIENT_NOT_IN_CONVERSATION,
          IErrorInfo::invalidRequest, IException::recoverable, "end()");
   IDDEClosedConversation* pConv = new IDDEClosedConversation(application(),
                                                 topic(), serverHandle(), true);

   // clear the instance data
   IWindowHandle tmpwndh = serverHandle();
   wndhClServer = (IWindowHandle)0;
   fClCaseSensitive = false;
   strClApplication = "";
   strClTopic = "";

   // clear the hotlink set and transaction queue
   hotLinksForUpdate().semaphor().lock();  // serialize access to the set
   hotLinksForUpdate().removeAll(&IDdeleteElemCHLS);
   hotLinksForUpdate().semaphor().unlock();
   transactions().semaphor().lock();  // serialize access to the set
   transactions().allElementsDo(&IDdeleteElemTQ);
   transactions().removeAll();
   transactions().semaphor().unlock();

   // Inform the server that the client wishes to terminate the conversation.
   Boolean bSuccess = WinDdePostMsg((HWND)tmpwndh,
                                    (HWND)clientHandle(),
                                    WM_DDE_TERMINATE,
                                    0,
                                    DDEPM_RETRY);
   if (bSuccess && !fClPostMsgFail)
   {
      closedConversations().semaphor().lock();  // serialize access to the set
      closedConversations().add(pConv);
      closedConversations().semaphor().unlock();
   }
   else
   {
      ITRACE_DEVELOP(IString("WM_DDE_TERMINATE failure on attempt to initate") +
              IString("end of conversation, server application is ") +
              pConv->application() );
      delete pConv;
   }
   return *this;
}


/*------------------------------------------------------------------------------
| IDDEClientConversation::requestData                                          |
|                                                                              |
| Allows client to request data from server. This is a one time request for    |
| data, as opposed to a hot link. The data returned is posted to the callback  |
| method IDDEClientConversation::handleData through the dispatcher.            |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: requestData (
                                                         const char* item,
                                                         const char* format )
{
   IMODTRACE_DEVELOP("DDEClntConv::requestData");
   IASSERTSTATE(inConversation());
   IASSERTPARM(item != 0 && format != 0);
   IASSERTPARM(item[0] != '\0' && format[0] != '\0');

   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                 format,
                                                 0,
                                                 0,
                                                 0);

// build the event before posting the message, as OS/2 will give the shared
// memory associated with PDDESTRUCT on the Post
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_REQUEST,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pevt = new IDDEClientAcknowledgeEvent(myEvt);
   Boolean bSuccess = WinDdePostMsg((HWND)serverHandle(),
                                    (HWND)clientHandle(),
                                    WM_DDE_REQUEST,
                                    pddes,
                                    DDEPM_RETRY);
   if (bSuccess)
   {
      transactions().semaphor().lock();  // serialize access to the set
      // add the requestData transaction to the queue
      transactions().add(pevt);
      transactions().semaphor().unlock();
   }
   else
   {
      delete pevt;
      IDDEInfo__freeMemory( pddes );
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,
          IErrorInfo::accessError, IException::recoverable,
          "WinDdePostMsg (WM_DDE_REQUEST)");
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::pokeData                                             |
|                                                                              |
| Build the data structure with the data for an item in a specific format and  |
| post the message to the server to poke data.                                 |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: pokeData (
                                                      const char* item,
                                                      const void* data,
                                                      unsigned long dataLength,
                                                      const char* format )
{
   IMODTRACE_DEVELOP("DDEClntConv::pokeData");
   IASSERTSTATE(inConversation());
   // allow an empty string but not zero pointer
   IASSERTPARM(item != 0 && format != 0 && data != 0);
   IASSERTPARM(item[0] != '\0' && format[0] != '\0');

   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                  format,
                                                  0,
                                                  data,
                                                  dataLength);

   // build the event before posting the message,
   // as OS/2 will free the shared memory on us
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_POKE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pevt = new IDDEClientAcknowledgeEvent(myEvt);
   Boolean bSuccess = WinDdePostMsg((HWND)serverHandle(),
                                    (HWND)clientHandle(),
                                    WM_DDE_POKE,
                                    pddes,
                                    DDEPM_RETRY);
   if (bSuccess)
   {
      transactions().semaphor().lock();  // serialize access to the set
      // add the pokeData transaction to the queue
      transactions().add(pevt);
      transactions().semaphor().unlock();
   }
   else
   {
      delete pevt;
      IDDEInfo__freeMemory( pddes );
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,IErrorInfo::accessError,
                        IException::recoverable, "WinDdePostMsg (WM_DDE_POKE)");
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::beginHotLink                                         |
|                                                                              |
| Begin a hot link on an item. First make sure that the client does not already|
| have a hot link for the item in the specified format. Also make sure that a  |
| request for this hot link is not pending in the transactions queue.          |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: beginHotLink (
                                                        const char* item,
                                                        const char* format,
                                                        Boolean sendData,
                                                        Boolean pacing )
{
   IMODTRACE_DEVELOP("DDEClntConv::beginHotLink");
   IASSERTSTATE(inConversation());
   IASSERTPARM(item != 0 && format != 0);
   IASSERTPARM(item[0] != '\0' && format[0] != '\0');

   // ensure this is a new hotlink
   IString inItem = IString::upperCase(IString(item));
   IString inFormat = IString::upperCase(IString(format));
   IString myItem;
   IString myFormat;
   // first check the hotLink set
   IDDEClientHotLinkSet::Cursor myCurs(hotLinksForUpdate());
   IDDEClientHotLinkEvent* pEvt = 0;
   ITRACE_DEVELOP(
       "Getting ready to check for previously active hotLink match.");
   hotLinksForUpdate().semaphor().lock();  // serialize access to the set
   forCursor(myCurs)
   {
      pEvt = hotLinksForUpdate().elementAt(myCurs);
      myItem = IString::upperCase(pEvt->item());
      myFormat = IString::upperCase(pEvt->format());
      if ( ( inItem == myItem) && ( inFormat == myFormat) )
      {
         hotLinksForUpdate().semaphor().unlock();
         ITHROWLIBRARYERROR(IC_DDE_HOTLINK_ACTIVE,IErrorInfo::invalidRequest,
             IException::recoverable);
      }
   }  // end for loop
   hotLinksForUpdate().semaphor().unlock();
   // next check the transaction queue for a pending beginHotLink
   IDDETransactionQueue::Cursor myCurs1(transactions());
   IDDEClientAcknowledgeEvent* pEvt1 = 0;
   transactions().semaphor().lock();  // serialize access to the set
   forCursor(myCurs1)
   {
      pEvt1 = transactions().elementAt(myCurs1);
      myItem = IString::upperCase(pEvt1->item());
      myFormat = IString::upperCase(pEvt1->format());
      if ( ( myItem == inItem ) && ( myFormat == inFormat ) &&
           ( pEvt1->transactionType() == WM_DDE_ADVISE ) )
      {
         transactions().semaphor().unlock();
         ITHROWLIBRARYERROR(IC_DDE_HOTLINK_PENDING,IErrorInfo::invalidRequest,
             IException::recoverable);
      }
   }  // end for loop
   transactions().semaphor().unlock();

   // construct the status field
   unsigned short usStatus = 0;
   if (!sendData)
      usStatus = (unsigned short)(usStatus | DDE_FNODATA);
   if (pacing)
      usStatus = (unsigned short)(usStatus | DDE_FACKREQ);
   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                  format,
                                                  usStatus,
                                                  0,
                                                  0);

   // add the beginHotLink transaction to the Queue
   // build the event before posting the message,
   // as OS/2 will free the shared memory on us
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_ADVISE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pEvt2 = new
       IDDEClientAcknowledgeEvent(myEvt);
   Boolean bSuccess = WinDdePostMsg((HWND)serverHandle(),
                                    (HWND)clientHandle(),
                                    WM_DDE_ADVISE,
                                    pddes,
                                    DDEPM_RETRY);
   if (bSuccess)
   {
      transactions().semaphor().lock();  // serialize access to the set
      transactions().add(pEvt2);
      transactions().semaphor().unlock();
   }
   else
   {
      delete pEvt2;
      IDDEInfo__freeMemory( pddes );
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,IErrorInfo::accessError,
                      IException::recoverable, "WinDdePostMsg (WM_DDE_ADVISE)");
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::endHotLink                                           |
|                                                                              |
| The client sends a message to the server to end the specified hot link. If   |
| the client does not have the hot link with the server then an exception is   |
| thrown.  An exception is also thrown if the end hot link is pending from     |
| a previous request to end the hot link.                                      |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: endHotLink (
                                                            const char* item,
                                                            const char* format )
{
   IMODTRACE_DEVELOP("DDEClntConv::endHotLink");
   IASSERTSTATE(inConversation());
   IASSERTPARM(item != 0 && format != 0);
   IASSERTPARM(item[0] != '\0' && format[0] != '\0');

   IString inItem = IString::upperCase(IString(item));
   IString inFormat = IString::upperCase(IString(format));
   IString myItem;
   IString myFormat;
   // ensure the hotlink exists
   Boolean bSuccess = 0;
   IDDEClientHotLinkSet::Cursor myCurs(hotLinksForUpdate());
   IDDEHotLinkXEvent* pEvt = 0;
   hotLinksForUpdate().semaphor().lock();  // serialize access to the set
   for(myCurs.setToFirst();
       myCurs.isValid() && bSuccess == 0;
       myCurs.setToNext())
   {
      pEvt = (IDDEHotLinkXEvent*)hotLinksForUpdate().elementAt(myCurs);
      myItem = IString::upperCase(pEvt->item());
      myFormat = IString::upperCase(pEvt->format());
      if ( ( inItem == myItem) && ( inFormat == myFormat) )
         {
            if ( pEvt->isCloseInProgress() )
            {
               hotLinksForUpdate().semaphor().unlock();
               ITHROWLIBRARYERROR(IC_DDE_END_HOTLINK_PENDING,
                   IErrorInfo::invalidRequest,IException::recoverable);
            }
            ((IDDEHotLinkXEvent*)pEvt)->setCloseInProgress(true);
            bSuccess = true;
         }
   }  // end for loop
   hotLinksForUpdate().semaphor().unlock();
   if (!bSuccess)
      ITHROWLIBRARYERROR1(IC_DDE_NO_HOTLINK,IErrorInfo::invalidRequest,
                          IException::recoverable, "endHotLink()");

   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                  format,
                                                  0,
                                                  0,
                                                  0);

   // build the event before posting the message,
   // as OS/2 will free the shared memory on us
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_UNADVISE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pevt1 = new
                         IDDEClientAcknowledgeEvent(myEvt);
   bSuccess = WinDdePostMsg((HWND)serverHandle(),
                            (HWND)clientHandle(),
                            WM_DDE_UNADVISE,
                            pddes,
                            DDEPM_RETRY);
   if (bSuccess)
   {
      transactions().semaphor().lock();  // serialize access to the set
      // add the endHotLink transaction to the Queue
      transactions().add(pevt1);
      transactions().semaphor().unlock();
   }
   else
   {
      delete pevt1;
      IDDEInfo__freeMemory( pddes );
      ((IDDEHotLinkXEvent*)pEvt)->setCloseInProgress(false);
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,
         IErrorInfo::accessError, IException::recoverable,
         "WinDdePostMsg (WM_DDE_UNADVISE)");
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::endHotLinks                                          |
|                                                                              |
| If an item is specified then end all hot links on that item with the server. |
| If no item is specified then end all hot links with the server.              |
|                                                                              |
| Notes:                                                                       |
|   There can be several hot links on a single item. A hot link can be set up  |
|   for many different formats.                                                |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: endHotLinks (
                                                              const char* item )
{
   IMODTRACE_DEVELOP("DDEClntConv::endHotLinks");
   IASSERTSTATE(inConversation());

   if (item == 0)
     endAllHotLinks();
   else
     endAllHotLinks(item);
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::executeCommands                                      |
|                                                                              |
| The client can request the server to execute a set of commands.  The         |
| commands are sent as a character array with the length specified.            |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: executeCommands (
                                                   const void* commands,
                                                   unsigned long commandLength )
{
   IMODTRACE_DEVELOP("DDEClntConv::executeCommands");
   IASSERTSTATE(inConversation());
   IASSERTPARM(commands != 0);
   IASSERTPARM(commandLength != 0);

   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (0,
                                                  0,
                                                  0,
                                                  commands,
                                                  commandLength);

   // build the event before posting the message,
   // as OS/2 will free the shared memory on us
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_EXECUTE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pevt = new IDDEClientAcknowledgeEvent(myEvt);
   Boolean bSuccess = WinDdePostMsg((HWND)serverHandle(),
                                    (HWND)clientHandle(),
                                    WM_DDE_EXECUTE,
                                    pddes,
                                    DDEPM_RETRY);
   if (bSuccess)
   {
      transactions().semaphor().lock();  // serialize access to the set
      // add the executeCommands transaction to the Queue
      transactions().add(pevt);
      transactions().semaphor().unlock();
    }
   else
   {
      delete pevt;
      IDDEInfo__freeMemory( pddes );
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,
           IErrorInfo::accessError, IException::recoverable,
           "WinDdePostMsg (WM_DDE_EXECUTE)");
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::hotLinkInform                                        |
|                                                                              |
| This method is used to determine whether a positive or negative              |
| acknowledgement is sent to the server in response to the server message that |
| hot link data has changed.                                                   |
|                                                                              |
| Notes :                                                                      |
|   Default behavior is to return true which corresponds to a positive         |
|   acknowledgement                                                            |
------------------------------------------------------------------------------*/
Boolean IDDEClientConversation :: hotLinkInform ( IDDEClientHotLinkEvent& event)
{
   return true;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::acknowledged                                         |
|                                                                              |
| This method should be overridden by the receiver to interrogate the          |
| acknowledge event.                                                           |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: acknowledged (
                                   IDDEClientAcknowledgeEvent& event)
{ }

/*------------------------------------------------------------------------------
| IDDEClientConversation::executeAcknowledged                                  |
|                                                                              |
| This method should be overridden by the receiver to interrogate the          |
| acknowledge execute event.                                                   |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: executeAcknowledged (
                                   IDDEAcknowledgeExecuteEvent& event)
{ }

/*------------------------------------------------------------------------------
| IDDEClientConversation::pokeAcknowledged                                     |
|                                                                              |
| This method should be overridden by the receiver to interrogate the          |
| acknowledge poke event.                                                      |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: pokeAcknowledged (
                                   IDDEAcknowledgePokeEvent& event)
{ }

/*------------------------------------------------------------------------------
| IDDEClientConversation::conversationEnded                                    |
|                                                                              |
| Informs the receiver that the conversation has ended.                        |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: conversationEnded ( IDDEClientEndEvent& event )
{ }

/*------------------------------------------------------------------------------
| IDDEClientConversation::hotLinks                                             |
|                                                                              |
| Returns a set of the current hot links established with the server.          |
|                                                                              |
| Notes:                                                                       |
|   The caller has to create the hot link set that is filled by this method.   |
|   Returning a pointer to this object's hot link set would have had           |
|   detrimental side effects because the receivers cursor could be             |
|   invalidated by the addition or removal of hot links.                       |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: hotLinks (
                                              IDDEClientHotLinkSet& hotLinkSet )
{
   IDDEClientHotLinkSet::Cursor myCurs(*pHLSetCl);
   IDDEClientHotLinkEvent* pEvt = 0;
   IDDEClientHotLinkEvent* pEvt1 = 0;
   hotLinksForUpdate().semaphor().lock();  // serialize access to the set
   forCursor(myCurs)
   {
      pEvt1 = (*pHLSetCl).elementAt(myCurs);
      pEvt = new IDDEClientHotLinkEvent(*pEvt1);
      hotLinkSet.add(pEvt);
   }  // end for loop
   hotLinksForUpdate().semaphor().unlock();

   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::outstandingTransactionCount                          |
|                                                                              |
| Returns the number transactions that are still outstanding in the            |
| transaction queue.                                                           |
------------------------------------------------------------------------------*/
unsigned long IDDEClientConversation :: outstandingTransactionCount ( ) const
{
   return transactions().numberOfElements();
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::supportedTopics                                      |
|                                                                              |
| This method is used to query an application or all applications to respond   |
| with the topics that are supported. If an application is specified then      |
| only that application will respond. If no application is specified then all  |
| applications will get this message and respond to the request. Beginning a   |
| conversation with no topic name asks the application(s) to respond with all  |
| supported topics. HandleInitiateAck will detect that the broadcast message   |
| is in progress (fClBrdcstInPrgrs is set) and add the responding servers in   |
| the active server set pointed to by pActServSetCl.                           |
|                                                                              |
| If the current application is not in the set then add it as well.            |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: supportedTopics (
    IDDEActiveServerSet& activeServerSet,
    const char* applicationName )
{
   IMODTRACE_DEVELOP("DDEClntConv::supportedTopics");
   fClBrdcstInPrgrs = true;
   pActServSetCl = &activeServerSet;

   if (!applicationName)
      begin("","");
   else
      begin(applicationName, "");
   if (inConversation())  // add the current conversation if not already added
   {
      IDDEActiveServerSet::Cursor myCurs(*pActServSetCl);
      IDDEActiveServer* pServ = 0;
      Boolean bFound = false;
      forCursor(myCurs)
      {
         pServ = pActServSetCl->elementAt(myCurs);
         if ( (pServ->application() == this->application()) &&
              (pServ->topic() ==this->topic()) )
            bFound = true;
      }
      if (!bFound)
      {
         pServ = new IDDEActiveServer(application(),
                                      topic(),
                                      isCaseSensitive());
         pActServSetCl->add(pServ);
      }
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::supportingApplications                               |
|                                                                              |
| This method is used to query all applications to respond if they support the |
| specified topic. This message is sent to all applications by beginning a     |
| conversation with an empty string as the application name. HandleInitiateAck |
| will detect that the broadcast message is in progress (fClBrdcstInPrgrs is   |
| set) and add the responding servers in the active server set pointed to by   |
| pActServSetCl. If the current application is not in the set then add it      |
| as well.                                                                     |
------------------------------------------------------------------------------*/
IDDEClientConversation& IDDEClientConversation :: supportingApplications (
    IDDEActiveServerSet& activeServerSet,
    const char* topicName )
{
   IMODTRACE_DEVELOP("DDEClntConv::supportingApplications");
   IASSERTPARM(topicName != 0);
   fClBrdcstInPrgrs = true;
   pActServSetCl = &activeServerSet;

   begin("",topicName);
   if (inConversation())  // add the current conversation if not already added
   {
      IDDEActiveServerSet::Cursor myCurs(*pActServSetCl);
      IDDEActiveServer* pServ = 0;
      Boolean bFound = false;
      forCursor(myCurs)
      {
         pServ = pActServSetCl->elementAt(myCurs);
         if ( (pServ->application() == this->application()) &&
              (pServ->topic() ==this->topic()) )
            bFound = true;
      }
      if (!bFound)
      {
         pServ = new IDDEActiveServer(application(),
                                      topic(),
                                      isCaseSensitive());
         pActServSetCl->add(pServ);
      }
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::dispatchHandlerEvent                                 |
|                                                                              |
| Overloaded virtual dispatch function. Dispatches the DDE events to the       |
| virtual handle methods of this class.                                        |
|                                                                              |
| Notes:                                                                       |
|   if a WM_DDE message is received while one is being serviced then           |
|   throw an exception that DDE synchronization has been violated.             |
|   This situation can occur when useEventThread is set false (single          |
|   threaded) on the class constructor and the receiver's handle... method     |
|   puts up a dialog box.                                                      |
------------------------------------------------------------------------------*/
Boolean IDDEClientConversation :: dispatchHandlerEvent ( IEvent& evt )
{
   IMODTRACE_DEVELOP("DDEClntConv::dispatchHandlerEvent");
   if (!pThreadCl)
   {
      switch (evt.eventId())
      {
         case WM_DDE_ACK:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleAck(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                   IErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }
         case WM_DDE_DATA:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleData(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                   IErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }
         case WM_DDE_INITIATEACK:
         {
            handleInitiateAck(evt);
            evt.setResult(IEventResult(1));
            unsigned long j = 0;
            unsigned long rc = 0;
            do {
                 rc = DosFreeMem((void*)evt.parameter2().asUnsignedLong());
            } while ((rc == ERROR_INTERRUPT) && (++j <= 3));
            if (rc)
               ITHROWSYSTEMERROR(rc,"DosFreeMem",IBaseErrorInfo::accessError,
                 IException::recoverable);
            break;
         }
         case WM_DDE_TERMINATE:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleTerminate(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                   IErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }
         default:
         {
            return false;
         }

      } /* endswitch */
   }
   else      // dispatch the event on a separate thread
   {
      switch (evt.eventId())
      {
         case WM_DDE_ACK:
         case WM_DDE_DATA:
         case WM_DDE_TERMINATE:
         {
            unsigned long ulRc = 0;
            unsigned long ulSize(sizeof(IEvent));
            IEvent* pEvt = new IEvent(evt);
            ulRc = DosWriteQueue(queueHandle(),
                                 (unsigned long)0,
                                 ulSize,
                                 (void*)pEvt,
                                 (unsigned long)0);
            ITRACE_DEVELOP("DosWriteQueue rc =: " + IString(ulRc));
            if (ulRc) {  // no queue memory
               delete pEvt;
               ITHROWSYSTEMERROR(ulRc,"DosWriteQueue",
                   IErrorInfo::outOfSystemResource, IException::recoverable);
            }
            break;
         }

         case WM_DDE_INITIATEACK:
         {
            handleInitiateAck(evt);
            evt.setResult(IEventResult(1));
            unsigned long j = 0;
            unsigned long rc = 0;
            do {
                 rc = DosFreeMem((void*)evt.parameter2().asUnsignedLong());
            } while ((rc == ERROR_INTERRUPT) && (++j <= 3));
            if (rc)
               ITHROWSYSTEMERROR(rc,"DosFreeMem",IBaseErrorInfo::accessError,
                 IException::recoverable);
            break;
         }

         default:
         {
            return false;
         }

      } /* endswitch */
   }
   return true;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::handleAck                                            |
|                                                                              |
| HandleAck will take the acknowledgement returned from the server.            |
| The event is checked to see if it is in the transactions queue.              |
| Depending on the transaction type, call the appropriate acknowledge method.  |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: handleAck ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDEClntConv::handleAck");
   PDDESTRUCT pddes = (_DDESTRUCT*)(void*)evt.parameter2();
   transactions().semaphor().lock();  // serialize access to the set
   if ( transactions().numberOfElements() == 0 ) {
      transactions().semaphor().unlock();
      ITRACE_DEVELOP("WM_DDE_ACK received, no matching transactions!");
   }
   else
   {
      Boolean bFound;
      IDDEEvent evtInbound(evt);
      IString inItem = IString::upperCase(evtInbound.item());
      IString inFormat = IString::upperCase(evtInbound.format());
      IString myItem;
      IString myFormat;
      IDDEClientAcknowledgeEvent* pEvt = transactions().firstElement();
      transactions().semaphor().unlock();
      myItem = IString::upperCase(pEvt->item());
      myFormat = IString::upperCase(pEvt->format());
      if ((( myItem != inItem ) || ( myFormat != inFormat && inFormat != "")) &&
           ( pEvt->transactionType() != WM_DDE_EXECUTE ))
      //  check to see if ill behaved server is not sending positive acks!
         bFound = findTransaction(evt);
      else bFound = true;
      if (!bFound)
      {
         ITRACE_DEVELOP("WM_DDE_ACK received, no matching transaction!");
         ITRACE_DEVELOP(IString("Looking for item/format =  ") +
             evtInbound.item() + "  " + evtInbound.format());
         ITRACE_DEVELOP(IString("1st element in Q item/format =  ") +
            pEvt->item() + "  " + pEvt->format());
      }
      else
      {
         transactions().semaphor().lock();  // serialize access to the set
         // reset in case findTransaction deleted elements
         pEvt = transactions().firstElement();
         // copy the status field from the ack
         pEvt->setStatus(evtInbound.status());
         switch (pEvt->transactionType())
         {
            case WM_DDE_ADVISE:
            {
               ITRACE_DEVELOP("Ack received to a beginHotLink");
               if ( pEvt->isAckPositive() )
               {
                  // Add hot link to hot link set
                  IDDEHotLinkXEvent* pHLEvt = new
                       IDDEHotLinkXEvent(evt);
                  ITRACE_ALL(IString(
                     (unsigned long)hotLinksForUpdate().numberOfElements()) +
                     "elements in the HL set");
                  hotLinksForUpdate().semaphor().lock();
                  hotLinksForUpdate().add(pHLEvt);
                  hotLinksForUpdate().semaphor().unlock();
                  ITRACE_ALL(IString(
                     (unsigned long)hotLinksForUpdate().numberOfElements()) +
                     "elements in the HL set");
               }

               transactions().removeFirst();
               transactions().semaphor().unlock();
               this->acknowledged(*pEvt);
               break;
            }

            case WM_DDE_UNADVISE:
            {
               Boolean bRemove = false;
               ITRACE_DEVELOP("Ack received to endHotLink");
               if ( pEvt->isAckPositive() )
                  bRemove = true;
               IDDEClientHotLinkSet::Cursor myCurs(hotLinksForUpdate());
               IDDEHotLinkXEvent* pEvt1 = 0;
               // serialize access to the set
               hotLinksForUpdate().semaphor().lock();
               if (pEvt->item() == "" )
               {
                  // remove and delete all hotlink's
                  if (bRemove)
                  {
                     ITRACE_DEVELOP("Removing all hotlinks!");
                     hotLinksForUpdate().removeAll(&IDdeleteElemCHLS);
                  }
                  else
                     for( myCurs.setToFirst();
                          myCurs.isValid();
                          myCurs.setToNext())
                        ((IDDEHotLinkXEvent*)pEvt1)->setCloseInProgress(false);

               }
               else if (pEvt->format() == "")
                  {
                     // remove and delete all hotlink's for the item
                     if (bRemove)
                     {
                        ITRACE_DEVELOP(
                           "Removing all hotlinks for a given item");
                        hotLinksForUpdate().removeAll(
                            &IDdeleteElemCHLS, (char*)inItem);
                     }
                     else
                     {
                        for( myCurs.setToFirst();
                             myCurs.isValid();
                             myCurs.setToNext())
                        {
                           pEvt1 = (IDDEHotLinkXEvent*)
                              hotLinksForUpdate().elementAt(myCurs);
                           myItem = IString::upperCase(pEvt1->item());
                           if ( inItem == myItem )
                              ((IDDEHotLinkXEvent*)
                                 pEvt1)->setCloseInProgress(false);
                        }
                     }
                  }
                  else
                  {
                     // remove and delete the hotlink
                     if (bRemove)
                        ITRACE_DEVELOP("Removing a specific hotlink");
                     Boolean bSuccess = 0;
                     for(myCurs.setToFirst();myCurs.isValid() && bSuccess == 0;)
                     {
                        pEvt1 = (IDDEHotLinkXEvent*)
                           hotLinksForUpdate().elementAt(myCurs);
                        myItem = IString::upperCase(pEvt1->item());
                        myFormat = IString::upperCase(pEvt1->format());
                        if ( ( myItem == inItem ) && ( myFormat == inFormat ) )
                        {
                           bSuccess = true;
                           if (bRemove)
                           {
                              ITRACE_ALL(IString((unsigned long)
                                 hotLinksForUpdate().numberOfElements()) +
                                 "elements in the HL set");
                              hotLinksForUpdate().removeAt(myCurs);
                              ITRACE_ALL(IString((unsigned long)
                                 hotLinksForUpdate().numberOfElements()) +
                                 "elements in the HL set");
                              delete (IDDEHotLinkXEvent*)pEvt1;
                           }
                           else ((IDDEHotLinkXEvent*)pEvt1)->
                              setCloseInProgress(false);
                        }
                        else myCurs.setToNext();
                     }  // end for loop
                  }

               hotLinksForUpdate().semaphor().unlock();
               transactions().removeFirst();
               transactions().semaphor().unlock();
               this->acknowledged(*pEvt);
               break;
            }

            case WM_DDE_REQUEST:
            {
               ITRACE_DEVELOP("Ack received to requestData");
               transactions().removeFirst();
               transactions().semaphor().unlock();
               this->acknowledged(*pEvt);
               break;
            }

            case WM_DDE_EXECUTE:
            {
               ITRACE_DEVELOP("Ack received to executeCommands");
               IDDEAcknowledgeExecuteEvent* pEvt1 =
                       (IDDEAcknowledgeExecuteEvent*)transactions().firstElement();
               pEvt1->setStatus(pddes->fsStatus);
               transactions().removeFirst();
               transactions().semaphor().unlock();
               this->executeAcknowledged(*pEvt1);
               break;
            }

            case WM_DDE_POKE:
            {
               ITRACE_DEVELOP("Ack received to pokeData");
               IDDEAcknowledgePokeEvent* pEvt1 =
                     (IDDEAcknowledgePokeEvent*)transactions().firstElement();
               pEvt1->setStatus(pddes->fsStatus);
               transactions().removeFirst();
               transactions().semaphor().unlock();
               this->pokeAcknowledged(*pEvt1);
               break;
            }

            default:
            {
                transactions().removeFirst();
                transactions().semaphor().unlock();
            }

         } /* endswitch */

         delete pEvt;
      }  // end else
   }   // end else

      IDDEInfo__freeMemory(pddes);
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::handleData                                           |
|                                                                              |
| This method is called from the dispatchHandler in response to the            |
| WM_DDE_DATA message. If the data message is in response to requestData       |
| then pass the data to receiver via the data method                           |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: handleData ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDEClntConv::handleData");
   PDDESTRUCT pddes = (_DDESTRUCT*)(void*)evt.parameter2();
   IDDEEvent evtInbound(evt);
   IString inItem = IString::upperCase(evtInbound.item());
   IString inFormat = IString::upperCase(evtInbound.format());
   IString myItem;
   IString myFormat;
   unsigned short usStatus = 0;
   Boolean bPosAck = false;
   if ( pddes->fsStatus & DDE_FRESPONSE )         // response to requestData?
   {
      transactions().semaphor().lock();  // serialize access to the set
      if ( transactions().numberOfElements() != 0 )
      {
         IDDEClientAcknowledgeEvent* pEvt = transactions().firstElement();
         transactions().semaphor().unlock();
         myItem = IString::upperCase(pEvt->item());
         myFormat = IString::upperCase(pEvt->format());
         if ( (pEvt->transactionType() != WM_DDE_REQUEST) ||
                       ( myItem != inItem ) || ( myFormat != inFormat ) )
    //  check to see if ill behaved server is not sending positive acks!
            bPosAck = findTransaction(evt);
         else bPosAck = true;
         if (bPosAck)
         {
            // ensure we have the correct event
            transactions().semaphor().lock();  // serialize access to the set
            pEvt = transactions().firstElement();
            ITRACE_DEVELOP("WM_DDE_DATA from server matches request in queue");
            transactions().removeFirst();
            transactions().semaphor().unlock();
            delete pEvt;
            IDDEDataEvent dataevt(evt);
            bPosAck = this->data(dataevt);   // callback the user
            usStatus = dataevt.status();
         }
         else
         {
            ITRACE_DEVELOP("WM_DDE_DATA received, no matching transaction!");
            ITRACE_DEVELOP(IString("Looking for item/format =  ") +
               evtInbound.item() + "  " + evtInbound.format());
            ITRACE_DEVELOP(IString("1st element in Q item/format =  ") +
               pEvt->item() + "  " + pEvt->format());
         }
      }
      else {
         transactions().semaphor().unlock();
         ITRACE_DEVELOP("WM_DDE_DATA received, no outstanding transactions!");
      }
   }
   else             // data or inform from hotlink
   {
      ITRACE_ALL("Trying to find hotLinkTrans, \
         looking for item and format of" );
      ITRACE_ALL(evtInbound.item() + "  " + evtInbound.format());
      IDDEClientHotLinkSet::Cursor myCurs(hotLinksForUpdate());
      const IDDEHotLinkXEvent* pEvt1 = 0;
      hotLinksForUpdate().semaphor().lock();  // serialize access to the set
      for(myCurs.setToFirst(); myCurs.isValid() && bPosAck == false;)
      {
         pEvt1 = (IDDEHotLinkXEvent*)hotLinksForUpdate().elementAt(myCurs);
         myItem = IString::upperCase(pEvt1->item());
         myFormat = IString::upperCase(pEvt1->format());
         if ( ( myItem == inItem ) && ( myFormat == inFormat ) )
            bPosAck = true;
         else myCurs.setToNext();
      }  // end for loop
      hotLinksForUpdate().semaphor().unlock();
      if (!bPosAck)
         bPosAck = findTransaction(evt, true);
      if (bPosAck)
      {
         if ( pddes->fsStatus & DDE_FNODATA ) // hotlink inform
         {
            ITRACE_DEVELOP("Hot Link Inform matches active link");
            IDDEClientHotLinkEvent hlevt(evt);
            bPosAck = this->hotLinkInform(hlevt);
            usStatus = hlevt.status();
         }
         else
         {
            ITRACE_DEVELOP("Hot Link Data matches active link");
            IDDEDataEvent hlevt(evt);
            bPosAck = this->data(hlevt);
            usStatus = hlevt.status();
         }
      }
      else
      {
         ITRACE_DEVELOP("WM_DDE_DATA received, no matching hotlink!");
         ITRACE_DEVELOP(IString("Looking for item/format =  ") +
            evtInbound.item() + "  " + evtInbound.format());
      }
   }  // end data or inform from hotLink

  // Send an ack if one has been requested
   if (pddes->fsStatus & DDE_FACKREQ )
   {
      if (bPosAck)
      {
         usStatus |=  DDE_FACK;
         // ensure none of the negative flags are on
         usStatus &= ~(DDE_FBUSY | DDE_NOTPROCESSED);
      }
      PDDESTRUCT pddest = (PDDESTRUCT)buildDDEStruct (
                             (char*)DDES_PSZITEMNAME(pddes),
                             0,
                             usStatus,
                             0,
                             0);
      pddest->usFormat = pddes->usFormat;
      Boolean fError = WinDdePostMsg((HWND)serverHandle(),
                                     (HWND)clientHandle(),
                                     WM_DDE_ACK,
                                     pddest,
                                     DDEPM_RETRY);
      if (!fError)
      {
         // don't throw exception as server will also crash
         fClPostMsgFail = true;
         IEvent myEvt(clientHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)clientHandle()),
                      IEventParameter2((void*)0));
         IDDEClientEndEvent endEvt(myEvt, IDDEEndEvent::error, application(),
            topic());
         Boolean bEnded = true;
         try {
            this->end();
         }
         catch (IInvalidRequest excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded(endEvt);
         this->end();
         this->conversationEnded(endEvt);
         fClPostMsgFail = false;
      }
   }

   IDDEInfo__freeMemory(pddes);
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::handleInitiateAck                                    |
|                                                                              |
| The initiateAck is sent by the server in response to the initiate message.   |
| If fClBrdcstInPrgrs is set then add the active server to the server set      |
| pointed to by pActServSetCl. SupportingTopics and SupportingApplications     |
| use this to add servers to the server list that support the broadcast        |
| message.                                                                     |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: handleInitiateAck ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDEClntConv::handleInitiateAck");
   IDDEBeginEvent initevt(evt);
   Boolean bTerminate = false;
   if (fClBrdcstInPrgrs)
   {
      IDDEActiveServer* pServ = new IDDEActiveServer(initevt.application(),
                                                     initevt.topic(),
                                                     initevt.isCaseSensitive());
      pActServSetCl->add(pServ);
      bTerminate = true;
   }
   else
   {
      if (!inConversation())
      {
         IString myapp = IString::upperCase(application());
         IString mytop = IString::upperCase(topic());
         IString inapp = IString::upperCase(initevt.application());
         IString intop = IString::upperCase(initevt.topic());
         if ( (myapp == inapp) && (mytop == intop) )
         {
            wndhClServer = (IWindowHandle)(evt.parameter1());
            if (initevt.isCaseSensitive())
               fClCaseSensitive = true;
         }
         else
         {
            ITRACE_DEVELOP(
               "Application and Topic don't match, conversation discarded");
            bTerminate = true;
         }
      }
      else
      {
         ITRACE_DEVELOP(
            "More than one server responded, conversation discarded");
         bTerminate = true;
      }
   }
   // terminate the conversation if necessary
   if (bTerminate)
   {
      Boolean bSuccess = WinDdePostMsg((HWND)evt.parameter1(),
                                       (HWND)clientHandle(),
                                       WM_DDE_TERMINATE,
                                       0,
                                       DDEPM_RETRY);
      if (!bSuccess)
         ITRACE_DEVELOP(
            IString("WM_DDE_TERMINATE failure on attempt to initate end of ") +
            IString("conversation, server application is ") +
            initevt.application() );
      else
      {
         IDDEClosedConversation* pConv = new
            IDDEClosedConversation(initevt.application(),
                                   initevt.topic(),
                                   (HWND)evt.parameter1());
         closedConversations().semaphor().lock();  // serialize access to the set
         closedConversations().add(pConv);
         closedConversations().semaphor().unlock();
      }
   }
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::handleTerminate                                      |
|                                                                              |
| When the terminate message is received from the server the client must       |
| respond with a terminate message if initiated from the server. If the        |
| terminate was initiated by the client then the terminate message from        |
| server is in response to the client and the client should not send another   |
| terminate message.                                                           |
|                                                                              |
| If the client initiated the closing of the conversation then the             |
| conversation will be in the closed conversation set. This is used to         |
| determine that a WM_DDE_TERMINATE message should not be sent back to the     |
| server. The receiver is notified of the closing of the conversation through  |
| the method conversationEnded only if the closing is not the result of a      |
| broadcast message.                                                           |
|                                                                              |
| If the conversation is not in the closed conversation set then the closing   |
| was initiated by the server and the client must respond with the             |
| WM_DDE_TERMINATE message.                                                    |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: handleTerminate ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDEClntConv::handleTerminate");
// iterate the closed list first
   IDDEClosedConversationSet::Cursor myCurs(closedConversations());
   IDDEClosedConversation* pConv = 0;
   Boolean bSuccess = false;
   IEvent myEvt(clientHandle(),        // create dummy event for callbacks
                (unsigned long)WM_DDE_TERMINATE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)0));
   closedConversations().semaphor().lock();  // serialize access to the set
   for(myCurs.setToFirst(); myCurs.isValid() && bSuccess == false;)
   {
      pConv = closedConversations().elementAt(myCurs);
      if ( pConv->serverHandle() == evt.parameter1() )
      {
         bSuccess = true;
         closedConversations().removeAt(myCurs);
         closedConversations().semaphor().unlock();
// check flag here and callback if necessary (not from broadcast)
         if ( pConv->userClosed() )
         {
            IDDEClientEndEvent endEvt(myEvt, IDDEEndEvent::client,
                                      pConv->application(), pConv->topic());
            this->conversationEnded(endEvt);
         }
         delete pConv;
      }
      else myCurs.setToNext();
   }  // end for loop

   if (!bSuccess)
   {
      closedConversations().semaphor().unlock();
      if (evt.parameter1() == serverHandle())
      {
         IString tmpapp = application();
         IString tmptop = topic();
         fClCaseSensitive = false;
         strClApplication = "";
         strClTopic = "";
         Boolean fError = WinDdePostMsg((HWND)serverHandle(),
                                        (HWND)clientHandle(),
                                        WM_DDE_TERMINATE,
                                        0,
                                        DDEPM_RETRY);
         if (!fError)
            ITRACE_DEVELOP(
               "Failure on attempt to respond to WM_DDE_TERMINATE.");
         wndhClServer = (IWindowHandle)0;
      // clear the hotlink set and transaction queue
         hotLinksForUpdate().semaphor().lock();  // serialize access to the set
         hotLinksForUpdate().removeAll(&IDdeleteElemCHLS);
         hotLinksForUpdate().semaphor().unlock();
         transactions().semaphor().lock();  // serialize access to the set
         transactions().allElementsDo(&IDdeleteElemTQ);
         transactions().removeAll();
         transactions().semaphor().unlock();

         IDDEClientEndEvent endEvt1(myEvt, IDDEEndEvent::server, tmpapp,
            tmptop);
         this->conversationEnded(endEvt1);
      }
      else
        ITRACE_DEVELOP("Unknown server sent WM_DDE_TERMINATE");
   }
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::endAllHotLinks                                       |
|                                                                              |
| End the hot links for the specified item in all formats. If no hot links     |
| for the requested item exist then an exception is thrown. An exception is    |
| also thrown if a request to end all hot links on this item has already been  |
| requested and is pending.                                                    |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: endAllHotLinks ( const char* item )
{
   IMODTRACE_DEVELOP("DDEClntConv::endAllHotLinks");
   IASSERTPARM(strlen(item));
   IString inItem = IString::upperCase(IString(item));
   IString myItem;

// ensure at least one hotlink exists for this item
   ITRACE_DEVELOP("Getting ready to check for active hotLink match.");
   if (hotLinksForUpdate().numberOfElements() == 0)
      ITHROWLIBRARYERROR(IC_DDE_NO_HOTLINKS,IErrorInfo::invalidRequest,
         IException::recoverable);
   Boolean bSuccess = 0;
   Boolean bFound = 0;
   IDDEClientHotLinkSet::Cursor myCurs(hotLinksForUpdate());
   IDDEHotLinkXEvent* pEvt = 0;
   hotLinksForUpdate().semaphor().lock();  // serialize access to the set
   forCursor(myCurs)
   {
      pEvt = (IDDEHotLinkXEvent*)hotLinksForUpdate().elementAt(myCurs);
      myItem = IString::upperCase(pEvt->item());
      if ( inItem == myItem )
      {
         bFound = true;
         if ( !(pEvt->isCloseInProgress()) )
         {
            bSuccess = true;
            ((IDDEHotLinkXEvent*)pEvt)->setCloseInProgress(true);
         }
      }
   }
   hotLinksForUpdate().semaphor().unlock();
   if (!bFound)
      ITHROWLIBRARYERROR(IC_DDE_NO_HOTLINKS_ITEM,IErrorInfo::invalidRequest,
         IException::recoverable);
   if (!bSuccess)
      ITHROWLIBRARYERROR(IC_DDE_END_HOTLINKS_ITEM_PENDING,
         IErrorInfo::invalidRequest,IException::recoverable);

   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                  0,
                                                  0,
                                                  0,
                                                  0);
// build the event before posting the message,
// as OS/2 will free the shared memory on us
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_UNADVISE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pevt = new IDDEClientAcknowledgeEvent(myEvt);

   bSuccess = WinDdePostMsg((HWND)serverHandle(),
                            (HWND)clientHandle(),
                            WM_DDE_UNADVISE,
                            pddes,
                            DDEPM_RETRY);
   if (bSuccess)
   {
      // add the endHotLink transaction to the Queue
      transactions().semaphor().lock();  // serialize access to the set
      transactions().add(pevt);
      transactions().semaphor().unlock();
   }
   else
   {
      delete pevt;
      IDDEInfo__freeMemory( pddes );
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,
         IErrorInfo::accessError, IException::recoverable,
         "WinDdePostMsg (WM_DDE_UNADVISE)");
   }
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::endAllHotLinks                                       |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: endAllHotLinks ( )
{
   IMODTRACE_DEVELOP("DDEClntConv::endAllHotLinks");
// ensure at least one hotlink exists
   ITRACE_DEVELOP("Getting ready to check for active hotLink match.");
   if (hotLinksForUpdate().numberOfElements() == 0)
      ITHROWLIBRARYERROR(IC_DDE_NO_HOTLINKS,IErrorInfo::invalidRequest,
         IException::recoverable);

   Boolean bFound = 0;
   IDDEClientHotLinkSet::Cursor myCurs(hotLinksForUpdate());
   IDDEHotLinkXEvent* pEvt = 0;
   hotLinksForUpdate().semaphor().lock();  // serialize access to the set
   forCursor(myCurs)
   {
      pEvt = (IDDEHotLinkXEvent*)hotLinksForUpdate().elementAt(myCurs);
      if ( !(pEvt->isCloseInProgress()) )
      {
         bFound = true;
         ((IDDEHotLinkXEvent*)pEvt)->setCloseInProgress(true);
      }
   }
   hotLinksForUpdate().semaphor().unlock();
   if (!bFound)
      ITHROWLIBRARYERROR(IC_DDE_END_HOTLINKS_PENDING,
         IErrorInfo::invalidRequest,IException::recoverable);

   PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (0,
                                                  0,
                                                  0,
                                                  0,
                                                  0);

// add the endHotLink transaction to the Queue
// build the event before posting the message,
// as OS/2 will free the shared memory on us
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_UNADVISE,
                IEventParameter1((unsigned long)clientHandle()),
                IEventParameter2((void*)pddes));
   IDDEClientAcknowledgeEvent* pevt = new
     IDDEClientAcknowledgeEvent(myEvt);
   Boolean bSuccess = WinDdePostMsg((HWND)serverHandle(),
                                    (HWND)clientHandle(),
                                    WM_DDE_UNADVISE,
                                    pddes,
                                    DDEPM_RETRY);
   if (bSuccess)
   {
      transactions().semaphor().lock();  // serialize access to the set
      transactions().add(pevt);
      transactions().semaphor().unlock();
   }
   else
   {
      delete pevt;
      IDDEInfo__freeMemory( pddes );
      ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,IErrorInfo::accessError,
                    IException::recoverable, "WinDdePostMsg (WM_DDE_UNADVISE)");
   }
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::findTransaction                                      |
------------------------------------------------------------------------------*/
Boolean IDDEClientConversation :: findTransaction ( const IEvent& evt,
                                                    Boolean       removeMatch )
{
   IMODTRACE_DEVELOP("DDEClntConv::findTransaction");
   PDDESTRUCT pddes = (_DDESTRUCT*)(void*)evt.parameter2();
   IEvent myEvt(clientHandle(),
                (unsigned long)WM_DDE_UNADVISE,
                IEventParameter1((unsigned long)serverHandle()),
                IEventParameter2((void*)pddes));
   IDDEEvent evtInbound(myEvt);
   IString inItem = IString::upperCase(evtInbound.item());
   IString inFormat = IString::upperCase(evtInbound.format());
   IString myItem;
   IString myFormat;
   Boolean bOK = true;
   Boolean bFound = false;
   unsigned long ulCount = 0;
   if (removeMatch)
      ulCount = 1;
   IDDETransactionQueue::Cursor myCurs(transactions());
   IDDEClientAcknowledgeEvent* pEvt = 0;
   transactions().semaphor().lock();  // serialize access to the set
   for(myCurs.setToFirst();myCurs.isValid() && (bFound == false)
             && (bOK == true);)
   {
      pEvt = transactions().elementAt(myCurs);
      myItem = IString::upperCase(pEvt->item());
      myFormat = IString::upperCase(pEvt->format());
      if ( ( myItem == inItem ) && ( myFormat == inFormat ) )
         bFound = true;
      else
      {
         ulCount += 1;
         myCurs.setToNext();
         if ( pEvt->transactionType() == WM_DDE_REQUEST )
            // quit walking the queue
            bOK = false;
      }
   }  // end for loop
   transactions().semaphor().unlock();
   if (bFound)
   {
      for (unsigned long ulNum = 0;(ulNum < ulCount) && (bOK == true);ulNum++)
      {
         transactions().semaphor().lock();  // serialize access to the set
         pEvt = transactions().firstElement();
         transactions().semaphor().unlock();
// create a ddestruct, and event to send
         PDDESTRUCT pddest = (PDDESTRUCT)buildDDEStruct (pEvt->item(),
                                                         pEvt->format(),
                                                         DDE_FACK,
                                                         0,
                                                         0);
         IEventParameter2 evtprm((void*)pddest);
         IEvent* pmyEvt = new IEvent(evt.handle(),
                                     (unsigned long)WM_DDE_ACK,
                                     evt.parameter1(),
                                     evtprm);
         handleAck(*pmyEvt);
         delete pmyEvt;
      }
      return true;
   }
   else return false;
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::dispatchEventFromQueue                               |
|                                                                              |
| When threads are used events are queued up. This method dispatches these     |
| events.                                                                      |
------------------------------------------------------------------------------*/
void IDDEClientConversation :: dispatchEventFromQueue ( )
{
   IMODTRACE_DEVELOP("DDEClntConv::dsptchEvtFromQ");
   HAB hab = WinInitialize(0);
   REQUESTDATA reqData;
   unsigned long ulSize;
   unsigned long ulRc;
   void* pv;
   IEvent* pEvt;
   BYTE byt;
   IDDEFreeSemaphore safeDtor(ulClSemaphore);
   while (true) {
      ulRc = DosReadQueue(queueHandle(),
                          &reqData,
                          &ulSize,
                          &pv,
                          (unsigned long)0,
                          (unsigned long)0,
                          &byt,
                          (HEV)0);
      if ( ulRc == ERROR_QUE_INVALID_HANDLE )
         break;
      pEvt = (IEvent*)(pv);
      switch (pEvt->eventId())
      {
         case WM_DDE_ACK:
         {
            handleAck(*pEvt);
            break;
         }
         case WM_DDE_DATA:
         {
            handleData(*pEvt);
            break;
         }
         case WM_DDE_TERMINATE:
         {
            handleTerminate(*pEvt);
            break;
         }

      } /* endswitch */
      delete pEvt;
   } /* endwhile */
}

/*------------------------------------------------------------------------------
| IDDEClientConversation::buildDDEStruct                                       |
------------------------------------------------------------------------------*/
void* IDDEClientConversation :: buildDDEStruct ( const char*    itemName,
                                                 const char*    dataFormat,
                                                 unsigned short status,
                                                 const void*    xferData,
                                                 unsigned long  dataLength)
{
   Boolean bAtomAdded = false;
   void* pStruct = IDDEInfo__buildDDEStruct( itemName,
                                             dataFormat,
                                             status,
                                             xferData,
                                             dataLength,
                                             &bAtomAdded );
   if (bAtomAdded)
   {
      formats().add(((PDDESTRUCT)pStruct)->usFormat);
      ITRACE_DEVELOP("Added atom to format list.");
   }
   return pStruct;
}

/*------------------------------------------------------------------------------
| IDDEActiveServer::IDDEActiveServer                                           |
------------------------------------------------------------------------------*/
IDDEActiveServer :: IDDEActiveServer ( const char * appName,
                                       const char * topic,
                                       Boolean caseSens )
                      : strClApplication(appName),
                        strClTopic(topic),
                        fClCaseSensitive(caseSens)
{ }

/*------------------------------------------------------------------------------
| IDDEActiveServer::~IDDEActiveServer                                          |
------------------------------------------------------------------------------*/
IDDEActiveServer :: ~IDDEActiveServer ( )
{ }

/*------------------------------------------------------------------------------
| IDDEActiveServerSet::~IDDEActiveServerSet                                    |
------------------------------------------------------------------------------*/
IDDEActiveServerSet :: ~IDDEActiveServerSet ( )
{
// remove and delete all old members of the active server set
   removeAll(&IDdeleteElemASS);
}

/*------------------------------------------------------------------------------
| IDDEClientHotLinkSet::IDDEClientHotLinkSet                                   |
------------------------------------------------------------------------------*/
IDDEClientHotLinkSet :: IDDEClientHotLinkSet ( )
          : ISet<IDDEClientHotLinkEvent*>(3)
{ }

/*------------------------------------------------------------------------------
| IDDEClientHotLinkSet::~IDDEClientHotLinkSet                                  |
------------------------------------------------------------------------------*/
IDDEClientHotLinkSet :: ~IDDEClientHotLinkSet ( )
{
// remove and delete all old members of the client hot link set
   removeAll(&IDdeleteElemCHLS);
}

/*------------------------------------------------------------------------------
| IDDEClientHotLinkSet::semaphor                                               |
------------------------------------------------------------------------------*/
IPrivateResource& IDDEClientHotLinkSet :: semaphor ( )
{
   return priResCl;
}

/*------------------------------------------------------------------------------
| IDDEHotLinkXEvent::IDDEHotLinkXEvent                                         |
------------------------------------------------------------------------------*/
IDDEHotLinkXEvent :: IDDEHotLinkXEvent ( IEvent evt )
        : IDDEClientHotLinkEvent(evt),
          fClCloseInPrgrs(0)
{ }

/*------------------------------------------------------------------------------
| IDDEHotLinkXEvent::~IDDEHotLinkXEvent                                        |
------------------------------------------------------------------------------*/
IDDEHotLinkXEvent :: ~IDDEHotLinkXEvent ( )
{ }

/*------------------------------------------------------------------------------
| IDDEClosedConversation::IDDEClosedConversation                               |
------------------------------------------------------------------------------*/
IDDEClosedConversation :: IDDEClosedConversation  (const char * appName,
                                                   const char * topic,
                                                   const IWindowHandle serverId,
                                                   Boolean userclose )
        : IDDEActiveServer(appName, topic, false), wndhClServer(serverId),
                            fClUsrCls(userclose)
{ }

/*------------------------------------------------------------------------------
| IDDEClosedConversation::~IDDEClosedConversation                              |
------------------------------------------------------------------------------*/
IDDEClosedConversation :: ~IDDEClosedConversation ( )
{ }

/*------------------------------------------------------------------------------
| IDDEClosedConversationSet::IDDEClosedConversationSet                         |
------------------------------------------------------------------------------*/
IDDEClosedConversationSet :: IDDEClosedConversationSet ( )
              : ISet<IDDEClosedConversation*>(10)
{ }

/*------------------------------------------------------------------------------
| IDDEClosedConversationSet::~IDDEClosedConversationSet                        |
------------------------------------------------------------------------------*/
IDDEClosedConversationSet :: ~IDDEClosedConversationSet ( )
{ }

/*------------------------------------------------------------------------------
| IDDETransactionQueue::IDDETransactionQueue                                   |
------------------------------------------------------------------------------*/
IDDETransactionQueue :: IDDETransactionQueue ( )
     : IQueue<IDDEClientAcknowledgeEvent*>(5)
{ }

/*------------------------------------------------------------------------------
| IDDETransactionQueue::~IDDETransactionQueue                                  |
------------------------------------------------------------------------------*/
IDDETransactionQueue :: ~IDDETransactionQueue ( )
{ }

IDDEClientHotLinkSet& IDDEClientConversation :: hotLinksForUpdate ( ) const
/*------------------------------------------------------------------------------
| IDDEClientConversation::hotLinksForUpdate                                    |
------------------------------------------------------------------------------*/
{
   return *pHLSetCl;
}

unsigned long IDDEClientConversation :: hotLinkCount ( ) const
/*------------------------------------------------------------------------------
| IDDEClientConversation::hotLinkCount                                         |
------------------------------------------------------------------------------*/
{
   return hotLinksForUpdate().numberOfElements();
}

/****************************************************************/
/*   IDDEActiveServerSet Member Functions                       */
/****************************************************************/

IDDEActiveServerSet :: IDDEActiveServerSet ( )
              : ISet<IDDEActiveServer*>(10)
/*------------------------------------------------------------------------------
| IDDEActiveServerSet::IDDEActiveServerSet                                     |
------------------------------------------------------------------------------*/
{ }
