/*******************************************************************************
* FILE NAME: iflyhhdr.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iflyhhdr.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C" {
  #define INCL_WIN
  #define INCL_WINMESSAGEMGR
  #define INCL_WINWINDOWMGR
  #define INCL_WINPOINTERS
  #define INCL_WININPUT
  #include <iwindefs.h>
}

#include <iflyhhdr.hpp>
#include <iflytext.hpp>
#include <iflypriv.hpp>
#include <iapp.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <ictlevt.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IFLYHHDR_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IHandleStringElement::IHandleStringElement                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IHandleStringElement::IHandleStringElement( unsigned long      handle,
                                            const IString&     flyText,
                                            const IString&     longText )
: handleKey(handle), flyStr(flyText), longStr(longText),
  fStrings(true)
{
    //-----------------------------------------------------------------------
    // Defect: 28390
    // Initialize our data storage
    //-----------------------------------------------------------------------
    // save our resource id values                          // 28390
    flyResId      = 0;                                      // 28390
    longResId     = 0;                                      // 28390
    // null out or library pointers                         // 28390
    flyResLibptr  = 0;                                      // 28390
    longResLibptr = 0;                                      // 28390
}

/*------------------------------------------------------------------------------
| IHandleStringElement::IHandleStringElement                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IHandleStringElement::IHandleStringElement( unsigned long      handle,
                                            const IResourceId& flyText,
                                            const IResourceId& longText )
: handleKey(handle), flyStr(), longStr(),
  fStrings(false)
{
    //-----------------------------------------------------------------------
    // Defect 28390 start
    // - save our own copy of the IResourceLibrary that is supplied
    //   in the passed in IResourceId's
    //-----------------------------------------------------------------------
    // save our resource id values
    flyResId      = flyText.id();
    longResId     = longText.id();

    // save a copy of the resource library into our own storage area
    // First check to see if the resource is from a dll or the exe
    IResourceLibrary temp;

    //  Here we check that to see if the resourceLibrary handle is from
    //  the exe or a dll.
    if( temp.handle() != flyText.resourceLibrary().handle() &&
        flyText.resourceLibrary().handle() != 0 )
    {
        // since the handles are not the same and not 0 (ie. exe), then the
        // flyText.resourceLibrary() is not a IResourceLibrary but a
        // IDynamicLinkLibrary, so create a copy of the
        // IDynamicLinkLIbrary
        IDynamicLinkLibrary *tmpResLibptr =
            new IDynamicLinkLibrary (flyText.resourceLibrary().handle());

        // save the pointer to the copy of the resource library
        flyResLibptr = tmpResLibptr;
    }
    else
    {
        IResourceLibrary *tmpResLibptr =
            new IResourceLibrary (flyText.resourceLibrary());
        // save the pointer to the copy of the resource library
        flyResLibptr = tmpResLibptr;
    }


    //
    //  Make sure the long help is ok also
    //
    //  Here we check that to see if the resourceLibrary handle is from
    //  the exe or a dll.
    if( temp.handle() != longText.resourceLibrary().handle() &&
        longText.resourceLibrary().handle() != 0 )
    {
        // since the handles are not the same and not 0 (ie. exe), then the
        // flyText.resourceLibrary() is not a IResourceLibrary but a
        // IDynamicLinkLibrary, so create a copy of the
        // IDynamicLinkLIbrary
        IDynamicLinkLibrary *tmpResLibptr =
            new IDynamicLinkLibrary (longText.resourceLibrary().handle());

        // save the pointer to the copy of the resource library
        longResLibptr = tmpResLibptr;
    }
    else
    {
        IResourceLibrary *tmpResLibptr =
            new IResourceLibrary (longText.resourceLibrary());
        // save the pointer to the copy of the resource library
        longResLibptr = tmpResLibptr;
    }
    //-----------------------------------------------------------------------
    // Defect 28390 end
    //-----------------------------------------------------------------------
}

/*------------------------------------------------------------------------------
| IHandleStringElement::~IHandleStringElement                                  |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IHandleStringElement::~IHandleStringElement( )
{
    if( flyResLibptr )                  // 28390
        delete flyResLibptr;            // 28390
    if( longResLibptr )                 // 28390
        delete longResLibptr;           // 28390
}

/*------------------------------------------------------------------------------
| IHandleStringSet::IHandleStringSet                                           |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IHandleStringSet::IHandleStringSet()
: IKeySet < IHandleStringElement*, unsigned long > ()
{}

/*------------------------------------------------------------------------------
| IHandleStringSet::~IHandleStringSet                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IHandleStringSet::~IHandleStringSet()
{}

/*------------------------------------------------------------------------------
| IFlyOverData::IFlyOverData                                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverData::IFlyOverData( IFlyText* flyText, ITextControl* longText,
                            unsigned long delay, unsigned long initialDelay,
                            IFlyOverHelpHandler* flyHelp )
: fFlytxt(flyText), fLongtxt(longText),
  fLongOffset(0), fFlyOffset(0),
  fTimer(),
  fPrevPos(),
  fDefault(),
  fSetPrevToNull(false),
  fHandle(0),
  fPrevHandle(0),
  fControlHandle(0),
  fControlWindow(0),
  fFly(flyHelp),
  fResLib(0),
  fHandleStringSet(0),
  fTimerDelay(delay),            // Save the original timer value.
  fInitialTimerDelay(initialDelay), // 8032
  bFlytxtCreated(false)             // 8032
{
  //----------------------------------------------------------
  // DEFECT 8032 : create our own IFlyText if not provided
  //               and only if there is a long text item.       // 27153
  //               fLongtxt is used as the IWindow* owner,      // 27153
  //               thus it cannot be null.                      // 27153
  //----------------------------------------------------------
  if (!fFlytxt && fLongtxt ) {                                  // 27153
     bFlytxtCreated = true;
     fFlytxt = new IFlyText(DEFAULT_FLYTEXT_ID, fLongtxt );
  }
  //----------------------------------------------------------
}

/*------------------------------------------------------------------------------
| IFlyOverData::~IFlyOverData                                                  |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverData::~IFlyOverData()
{
  if (fHandleStringSet)
  {
    IHandleStringSet::Cursor cursor(*fHandleStringSet);
    forCursor (cursor)
    {
      delete fHandleStringSet->elementAt( cursor );
    }
    fHandleStringSet->removeAll();
  }
  delete fHandleStringSet;
  delete fResLib;
  fTimer.stop();

  //----------------------------------------------------------
  // DEFECT 8032 : created our own IFlyText if not provided
  //----------------------------------------------------------
  if ( bFlytxtCreated )
     delete fFlytxt;
  //----------------------------------------------------------
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::IFlyOverHelpHandler                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler::IFlyOverHelpHandler( IFlyText* flyText,
                                          unsigned long initialDelay,
                                          unsigned long delay )
: fFlyOverData( new IFlyOverData(flyText, 0, delay, initialDelay, this))
{
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::IFlyOverHelpHandler                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler::IFlyOverHelpHandler( ITextControl* longText,
                                          unsigned long initialDelay,
                                          unsigned long delay )
: fFlyOverData( new IFlyOverData(0, longText, delay, initialDelay, this))
{
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::IFlyOverHelpHandler                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler::IFlyOverHelpHandler( IFlyText* flyText,
                                          ITextControl* longText,
                                          unsigned long initialDelay,
                                          unsigned long delay )
: fFlyOverData( new IFlyOverData(flyText, longText, delay, initialDelay, this))
{
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::~IFlyOverHelpHandler                                    |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler::~IFlyOverHelpHandler()
{
  delete fFlyOverData;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::flyTextControl                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyText* IFlyOverHelpHandler::flyTextControl( ) const
{
  return fFlyOverData->fFlytxt;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::longTextControl                                         |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
ITextControl* IFlyOverHelpHandler::longTextControl( ) const
{
  return fFlyOverData->fLongtxt;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setFlyTextControl                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setFlyTextControl(IFlyText* flyText)
{
  //-------------------------------------------------------------------------
  // defect 27153
  // if user calls setFlyTextControl, then we must be sure to delete our own
  // instance if we allocated one and clean up our flags.
  //-------------------------------------------------------------------------
  if ( flyText )                                                    // 27153
  {  // Fly text specified.                                         // 27153
    if ( fFlyOverData->bFlytxtCreated  &&  fFlyOverData->fFlytxt )  // 27153
    {  // Get rid of the default one now.                           // 27153
       delete fFlyOverData->fFlytxt;                                // 27153
       fFlyOverData->bFlytxtCreated = false;                        // 27153
    }                                                               // 27153
    fFlyOverData->fFlytxt = flyText;
  }                                                                 // 27153
  else if ( ! fFlyOverData->bFlytxtCreated )                        // 27153
  {  // No fly text specified, but the caller supplied one earlier. // 27153
     // Replace the caller's previous fly text with a default one.  // 27153
       if( this->longTextControl() )                                // 27153
       {
         fFlyOverData->fFlytxt =                                        // 27153
            new IFlyText( DEFAULT_FLYTEXT_ID, this->longTextControl() );// 27153
         fFlyOverData->bFlytxtCreated = true;                           // 27153
       }
  }                                                                 // 27153
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setLongTextControl                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setLongTextControl(
                                                        ITextControl* longText )
{
  fFlyOverData->fLongtxt = longText;
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::resourceLibrary                                         |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IResourceLibrary& IFlyOverHelpHandler::resourceLibrary( ) const
{
  // If we've got our own, return it, else use the current user default.
  if ( fFlyOverData->fResLib )
    return *fFlyOverData->fResLib;
  else
    return IApplication::current().userResourceLibrary();
}

#ifndef IC_MOTIF
/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setResourceLibrary                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setResourceLibrary(
                                                   const IModuleHandle &resMod )
{
  // Construct a DLL from the input handle.

  delete fFlyOverData->fResLib;
  fFlyOverData->fResLib = 0;
  if ( resMod )
    fFlyOverData->fResLib = new IDynamicLinkLibrary( resMod );
  else
    fFlyOverData->fResLib = new IResourceLibrary();

  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setResourceLibrary                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setResourceLibrary(
                                                        const char *resDLLName )
{
  // Construct a DLL from the input dll name.

  delete fFlyOverData->fResLib;
  fFlyOverData->fResLib = 0;
  if ( resDLLName )
    fFlyOverData->fResLib = new IDynamicLinkLibrary( resDLLName );
  else
    fFlyOverData->fResLib = new IResourceLibrary();

  return *this;
}
#endif

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setFlyStringTableOffset                                 |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setFlyTextStringTableOffset(
                                                                long newOffset )
{
  fFlyOverData->fFlyOffset = newOffset;
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setLongStringTableOffset                                |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setLongStringTableOffset(
                                                                long newOffset )
{
  fFlyOverData->fLongOffset = newOffset;
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::flyTextStringTableOffset                                |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
long IFlyOverHelpHandler::flyTextStringTableOffset ( ) const
{
  return fFlyOverData->fFlyOffset;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::longStringTableOffset                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
long IFlyOverHelpHandler::longStringTableOffset  ( ) const
{
  return fFlyOverData->fLongOffset;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::defaultText                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IString IFlyOverHelpHandler::defaultText ( ) const
{
  return fFlyOverData->fDefault;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setDefaultText                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setDefaultText( unsigned long id )
{
  fFlyOverData->fDefault = this->resourceLibrary().loadString( id );
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setDefaultText                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setDefaultText( const IString &id )
{
  fFlyOverData->fDefault = id;
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::flyHelpText                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IString IFlyOverHelpHandler::flyHelpText(const IWindowHandle& handle ) const
{
  if ( fFlyOverData->fHandleStringSet &&
       fFlyOverData->fHandleStringSet->containsElementWithKey( handle ))
  {
    IHandleStringElement*
      elementWithHandle(fFlyOverData->fHandleStringSet->elementWithKey( handle ));
    return elementWithHandle->flyStr;
  }
  else
  {
    return IString();
  }
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::longHelpText                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IString IFlyOverHelpHandler::longHelpText(const IWindowHandle& handle ) const
{
  if ( fFlyOverData->fHandleStringSet &&
       fFlyOverData->fHandleStringSet->containsElementWithKey( handle ))
  {
    IHandleStringElement*
      elementWithHandle(fFlyOverData->fHandleStringSet->elementWithKey( handle ));
    return elementWithHandle->longStr;
  }
  else
  {
    return IString();
  }
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setHelpText                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setHelpText
                                               ( const IWindowHandle& handle,
                                                 const IString&       flyText,
                                                 const IString&       longText )
{
  if ( !fFlyOverData->fHandleStringSet )
    fFlyOverData->fHandleStringSet = new IHandleStringSet();

  removeHelpText( handle );

  fFlyOverData->fHandleStringSet->add(
                           new IHandleStringElement(handle,flyText,longText));
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setHelpText                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setHelpText
                                               ( const IWindowHandle& handle,
                                                 const IResourceId&   flyText,
                                                 const IResourceId&   longText )
{
  if ( !fFlyOverData->fHandleStringSet )
    fFlyOverData->fHandleStringSet = new IHandleStringSet();

  removeHelpText( handle );

  fFlyOverData->fHandleStringSet->add(
                           new IHandleStringElement(handle,flyText,longText));
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::removeHelpText                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::removeHelpText
                                                 ( const IWindowHandle& handle )
{
  if ( fFlyOverData->fHandleStringSet &&
       fFlyOverData->fHandleStringSet->containsElementWithKey( handle ))
  {
    IHandleStringElement *oldElement((IHandleStringElement*)
                      fFlyOverData->fHandleStringSet->elementWithKey( handle ));
    fFlyOverData->fHandleStringSet->removeElementWithKey( handle );
    delete oldElement;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setDelayTime                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setDelayTime( unsigned long milliseconds )
{
  fFlyOverData->fTimer.setInterval( milliseconds/2 );
  // Save the new timer delay.
  fFlyOverData->fTimerDelay = milliseconds;
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::delayTime                                               |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IFlyOverHelpHandler::delayTime( ) const
{
  // Return the last set time delay.
  // return fFlyOverData->fTimer.interval();
  return fFlyOverData->fTimerDelay;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setInitialDelayTime                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::setInitialDelayTime( unsigned long milliseconds )
{
  // Save the new timer delay.
  fFlyOverData->fInitialTimerDelay = milliseconds;
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::initialDelayTime                                        |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IFlyOverHelpHandler::initialDelayTime( ) const
{
  // Return the last set time delay.
  // return fFlyOverData->fTimer.interval();
  return fFlyOverData->fInitialTimerDelay;
}


/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::startHandlingEventsFor                                  |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::handleEventsFor( IWindow* window )
{
  IASSERTPARM(window != 0);
  IFlyOverHelpHandler::Inherited::handleEventsFor( window );
  if (!fFlyOverData->fTimer.isStarted())
  {
    fFlyOverData->fTimer.setInterval( fFlyOverData->fInitialTimerDelay/2 );
    fFlyOverData->fTimer.start(
      new ITimerMemberFn<IFlyOverData>(*fFlyOverData, IFlyOverData::timerpop));

    fFlyOverData->fPrevPos = IWindow::pointerPosition();
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::stopHandlingEventsFor                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverHelpHandler& IFlyOverHelpHandler::stopHandlingEventsFor( IWindow* window )
{
  IASSERTPARM(window != 0);
  IFlyOverHelpHandler::Inherited::stopHandlingEventsFor( window );
  fFlyOverData->fTimer.stop();
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::dispatchHandlerEvent                                    |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
Boolean IFlyOverHelpHandler::dispatchHandlerEvent( IEvent& event )
{
  if (event.eventId() == WM_CONTROLPOINTER || event.eventId() == WM_MOUSEMOVE)
  {
    if (!fFlyOverData->fTimer.isStarted())  // REMOVE
       int x = 1;
    else
       int y = 1;

    if (!fFlyOverData->fTimer.isStarted() &&
        WinIsChild( event.handle(), WinQueryActiveWindow( IWindow::desktopWindow()->handle() )))
    {
      // Kick the timer off.
      // Note: the interval is divided by two becuase it takes to timer pops
      //       in order to show fly over help for a window.
      fFlyOverData->fTimer.setInterval( fFlyOverData->fInitialTimerDelay/2 );
      fFlyOverData->fTimer.start(
        new ITimerMemberFn<IFlyOverData>(*fFlyOverData, IFlyOverData::timerpop));
      fFlyOverData->fPrevPos = IWindow::pointerPosition();
    }
    if (event.eventId() == WM_CONTROLPOINTER)
    {
      IControlEvent controlEvent(event);
      // If the mouse is of the fly-over text control then do NOT change
      // the fHandle data member
      if ( fFlyOverData->fFlytxt &&
         ( fFlyOverData->fFlytxt->handle() == controlEvent.controlHandle() ))
      {
         // Just do nothing for this case (in other words, leave the
         // fHandle data member at it's previous value)
      }
      else
      {
        fFlyOverData->fHandle = controlEvent.controlHandle();
        if (!fFlyOverData->fHandle)
          fFlyOverData->fHandle = event.handle();
      }
    }
    else
      fFlyOverData->fHandle = event.handle();
  }
  // If the window the handler is attached to is moving,
  // stop the timer and hide the flytext control.
  else if (event.eventId() == WM_ADJUSTWINDOWPOS)
  {
    PSWP pswp((PSWP)event.parameter1().asUnsignedLong());
    if (pswp->fl & SWP_MOVE)
    {
      if ( fFlyOverData->fFlytxt )
      {
         fFlyOverData->fFlytxt->hide();
      }
      fFlyOverData->fPrevHandle = 0;
    }
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFlyOverData::timerpop                                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IFlyOverData::timerpop( unsigned long timerId )
{
  // Get the mouse location.

  POINTL pt;
  WinQueryPointerPos( HWND_DESKTOP, &pt );

  // If either of the controls is no longer valid, return.

  if (fFlytxt && !fFlytxt->isValid())
    return;
  if (fLongtxt && !fLongtxt->isValid())
    return;

  // Get the handle of the control that is under the mouse.
  // Note:  controls that are disabled will not be returned since
  //        they do not respond to hit-test.
  fControlHandle = WinWindowFromPoint( IWindow::desktopWindow()->handle(), &pt, true );

  // If the control that is under the mouse is a descendant of the
  // handle that was in the control pointer or mousemove event or
  // the mouse is over the fly text control (this prevents the fly text
  // control from being removed once displayed for a window) and
  // the control is a child of the active window then proceed to show
  // fly over help.

  Boolean overFlyTextWindow(fFlytxt && (fFlytxt->handle() == fControlHandle));

  if (( WinIsChild( fControlHandle, fHandle ) &&
        WinIsChild( fControlHandle, WinQueryActiveWindow( IWindow::desktopWindow()->handle() ))) ||
        overFlyTextWindow )
  {
    if (overFlyTextWindow)
      return;
    if (pt.x == fPrevPos.x() && pt.y == fPrevPos.y())  // If the mouse has been
                                                       // at the same location.
    {
      fControlWindow = IWindow::windowWithHandle( fControlHandle );
      IWindowHandle testHandle = fControlHandle;

      // If the controlWindow is null, then we are dealing with
      // an aggregate control that is not wrappered.

      if (!fControlWindow)
      {
        // Query the handle's parent and then try to get the
        // IWindow pointer that corresponds to the parent handle.

        IWindowHandle parentHandle(WinQueryWindow(fControlHandle, QW_PARENT ));
        fControlWindow = IWindow::windowWithHandle( parentHandle );
        testHandle = parentHandle;
      }

      if (fControlWindow && (fControlWindow != fFlytxt))
      {
        // Look through the handle string set to see if we have
        // a hit based on the window handle.

        if ( fHandleStringSet &&
             fHandleStringSet->containsElementWithKey( testHandle ))
        {
          // We go a hit in the handle set.
          IHandleStringElement*
                   mouseOverElement(fHandleStringSet->elementWithKey(testHandle));

          if ( mouseOverElement->fStrings )
          {
            if ( fFlytxt && !bFlytxtCreated )            // DEFECT 8032
              setFlyText(mouseOverElement->flyStr);

            if (fLongtxt) {                                // DEFECT 8032
               if( mouseOverElement->longStr.size() > 0 )
                 setLongText( mouseOverElement->longStr );
               else if ( mouseOverElement->flyStr.size() > 0 )
                 setLongText( mouseOverElement->flyStr );
               else
                 setLongText( fFly->defaultText() );
            }
          }
          else
          {
            // DEFECT 8032
            unsigned long flyid(mouseOverElement->flyResId);
            IString flyInfo = IString();
            // Defect: 28390
            flyInfo = mouseOverElement->flyResLibptr->tryToLoadString(flyid);

            unsigned long longid(mouseOverElement->longResId);
            IString longInfo = IString();
            // Defect: 28390
            longInfo = mouseOverElement->longResLibptr->tryToLoadString(longid);

            if ( fFlytxt && !bFlytxtCreated )            // DEFECT 8032
            {
               if (flyInfo.length() > 0)
                 setFlyText(flyInfo/*, textHdr-szText*/ );
               else          //if id = 0 or an invalid resource id
                 setFlyText(fFly->defaultText()/*, textHdr->szText*/ );
            }

            if ( fLongtxt )
            {
               if (longInfo.length() > 0)                // DEFECT 8032
                 setLongText( longInfo );
               else if ( flyInfo.length() > 0)
                 setLongText( flyInfo );
               else
                 setLongText( fFly->defaultText() );
            }
          }
        }
        else
        {
          unsigned long controlId(fControlWindow->id());

          IString flyInfo = IString();
          flyInfo = fFly->resourceLibrary().tryToLoadString(controlId + fFlyOffset);
          if ( fFlytxt && !bFlytxtCreated )
          {
            if (flyInfo.length() > 0)
              setFlyText(flyInfo);
            else
              setFlyText(fFly->defaultText());
          }

          if ( fLongtxt )
          {
            IString longInfo = IString();
            longInfo = fFly->resourceLibrary().tryToLoadString(controlId + fLongOffset);
            if (longInfo.length() > 0)                // DEFECT 8032
              setLongText( longInfo );
            else if ( flyInfo.length() > 0)
              setLongText( flyInfo );
            else
              setLongText( fFly->defaultText() );
          }
        }
      }

      // Set the previous handle to the current handle for the next timerpop.
      fPrevHandle = testHandle;  // CWC??

      // set the interval to 1/3 second so we can tell "quickly" if we've
      // moved off of the window.
      fTimer.setInterval(333);
    }
    else // The mouse has moved.
    {
      // If we are no longer over the same control.
      if (fPrevHandle != fControlHandle)
      {
        // Reset previous handle to null and reset the interval.
        fPrevHandle = 0;
        fTimer.setInterval(fTimerDelay/2);
        if (fFlytxt)
          fFlytxt->hide();
        if (fLongtxt)
          setLongText( fDefault );
      }
    }
    // Update the previous point to the current point.
    fPrevPos.setX( pt.x ).setY( pt.y );
  }
  else
  {
    // Mouse is no longer over a window that is a child of the
    // window the flyOverHelpHandler what attached to.

    if (fFlytxt)
    {
      fFlytxt->hide();
    }
    if (fLongtxt)
    {
      setLongText( fDefault );
    }
    fPrevHandle = 0;
    fTimer.stop();
  }
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setFlyText                                              |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverData& IFlyOverData::setFlyText( const IString& helpText )
{
  // only if flytext was set
  if( !fFlytxt )                                    // 27153
    return *this;                                   // 27153

  if(helpText.size()!=0)
  {
    if(fFlytxt->text()!=helpText)
    {
      fFlytxt->hide();
      fFlytxt->setText(IString::change(helpText, (char*)"\xA", (char*)""));
    }
    // If the previous handle is null, this means the mouse has moved
    // onto a new window so show the fly text control.
    if ( fPrevHandle == 0 )
    {
      positionTextWindow();
    }
  }
  else
  {
    if(fFlytxt->isVisible())
      fFlytxt->hide();
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverHelpHandler::setLongText                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFlyOverData& IFlyOverData::setLongText( const IString& helpText )
{
  // Make sure we have long text to change
    if( !fLongtxt )               // 27153
      return *this;               // 27153

  Boolean preventLongTextUpdate(fLongtxt->sendEvent(IC_UM_QRY_PREVENTUPDATE));
  if (!preventLongTextUpdate)
  {
    if (helpText.size()!=0)
      fLongtxt->setText( helpText );
    else
      if ( fDefault.length() != 0 )
        fLongtxt->setText( fDefault );
      else
        // At a minimum, set the long text to a single blank to avoid
        // the frame handler from hiding the long text control if the
        // long text control happens to be a frame extension.
        fLongtxt->setText( " " );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IFlyOverData::positionTextWindow                                             |
|                                                                              |
| Reposition and show the flyText window if the mouse has moved over a         |
| new window since the last timer pop.                                         |
|                                                                              |
| Note:  The window handle of the window the mouse is over and the IWindow*    |
|        is passed into this function.  The window handle will not correspond  |
|        to the IWindow object in the case of aggregate controls (spin button, |
|        combo box, mle, etc).  The IWindow pointer is needed to query the     |
|        layoutAdjustment for such controls as combo box.                      |
------------------------------------------------------------------------------*/
IFlyOverData& IFlyOverData::positionTextWindow( )
{
  if (fPrevHandle != fControlHandle)
  {
    fFlytxt->hide();
    SWP swp;

    // Get the location and size of the window.

    IQUERYWINDOWPOS(fControlHandle, &swp);

    // Map the location of the window the mouse is over to desktop
    // coordinates.
    IPoint pt(0,0);
    pt = IWindow::mapPoint(pt, fControlHandle,
                           IWindow::desktopWindow()->handle() );

    IRectangle controlRect( pt, ISize(swp.cx, swp.cy) );

    // Get the control's visible rectangle.
    IRectangle visibleRect(fControlWindow->visibleRectangle());

    // If the rectangle is empty then this might be a graphic push
    // button (so we would need the parents visible rect instead)
    if ( visibleRect.size() == ISize(0,0) )
    {
       IWindow* pParent = fControlWindow->parent();
       if ( pParent && ( fControlWindow->id() == pParent->id() ))
       {
          visibleRect = pParent->visibleRectangle();
          fControlWindow = pParent;
       }
    }

    // if the visible rectangle does not equal the rectangle
    // from query the control (as in the case of a drop down
    // combo box, then convert the origin point of the
    // visible rectangle to desktop coordinates and position
    // the fly text relative to it.
    if (controlRect.size() == visibleRect.size())
      fFlytxt->setRelativeWindowRect( controlRect );
    else
    {
      pt = IWindow::mapPoint( visibleRect.minXMinY(),
                              fControlWindow->parent()->handle(),
                              IWindow::desktopWindow()->handle());
      fFlytxt->setRelativeWindowRect( IRectangle( pt, visibleRect.size() ));
    }
    fFlytxt->postEvent(IC_UM_FLY_PAINT);
  }
  return *this;
}

