/*******************************************************************************
* FILE NAME: ireslock.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IResource                                                                *
*     IPrivateResource                                                         *
*     ISharedResource                                                          *
*     IResourceLock                                                            *
*                                                                              *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#ifdef IC_REF_COUNT_SEMS
  #pragma options("/Oi+")   // Ensure optimization is on
#endif
extern "C" {
  #define INCL_DOSSEMAPHORES
  #define INCL_DOSERRORS
  #define INCL_WINMESSAGEMGR
  #include <iwindefs.h>
}

#include "ireslock.hpp"
#include <istring.hpp>
#include <iexcept.hpp>
#include <itrace.hpp>
#include <iplatfrm.hpp>
#include <isafeid.hpp>
#include <stdio.h>
#include <icconst.h>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IRESLOCK_CPP_
  #include <ipagetun.h>
#endif


/***************************************************************/
/* FUNCTION:  IPrivateResource :: handle                       */
/*                                                             */
/* PURPOSE:   returns handle to the resource                   */
/*            Cannot be inlined due to being virtual           */
/***************************************************************/
ISemaphoreHandle& IPrivateResource :: handle()
{
  return resourceHandleCl;
}


/***************************************************************/
/* FUNCTION:  ISharedResource :: handle                        */
/*                                                             */
/* PURPOSE:   returns handle to the resource                   */
/*            Cannot be inlined due to being virtual           */
/***************************************************************/
ISemaphoreHandle& ISharedResource :: handle()
{
  return resourceHandleCl;
}


/***************************************************************/
/* FUNCTION:  IResource :: IResource                           */
/*                                                             */
/* PURPOSE:   Constructor for base resource object             */
/***************************************************************/
IResource ::  IResource()
{
  lClReferenceCount = 0;
  reslockGate = 0;
}


/***************************************************************/
/* FUNCTION:  IResource :: ~IResource                          */
/*                                                             */
/* PURPOSE:   Destructor for base resource object              */
/***************************************************************/
IResource :: ~IResource()
{
  IMODTRACE_ALL("IResource::~IResource");

  if (reslockGate)
    delete reslockGate;

  reslockGate = 0;
  ITRACE_ALL(IString("Use Count is: ")+IString(lClReferenceCount));
  lClReferenceCount = 0;
}


/***************************************************************/
/* FUNCTION:  IPrivateResource :: IPrivateResource             */
/*                                                             */
/* PURPOSE:   Constructor the key for a private resource.      */
/*                                                             */
/* NOTE:  A possible future enhancement is to change the       */
/*        NT implementation of this constructor to use a       */
/*        critical section object which is optimized, instead  */
/*        of a mutex object.                                   */
/***************************************************************/
IPrivateResource :: IPrivateResource()
                        : resourceHandleCl(0)
{
  IMODTRACE_ALL("IPrivateResource::IPrivateResource");

#ifdef IC_PM
  unsigned long ulRC = 0;
  ulRC = DosCreateMutexSem( 0, (PHMTX)&resourceHandleCl, 0, false );
  if (ulRC != 0)
  {
    ITHROWSYSTEMERROR( ulRC,
                       "DosCreateMutexSem",
                       IBaseErrorInfo::accessError,
                       IException::recoverable );
  }
#endif
#ifdef IC_WIN
  if (IPlatform::isWin32S())
  {
    ITRACE_DEVELOP("Win32s: IPrivateResource() does not create semaphore");
    return;
  }
  resourceHandleCl = CreateMutex( 0, false, 0 );
  if (resourceHandleCl == 0)
  {
    ITHROWGUIERROR2( "CreateMutex",
                     IBaseErrorInfo::accessError,
                     IException::recoverable );
  }
#endif
}


/***************************************************************/
/* FUNCTION:  IPrivateResource :: ~IPrivateResource            */
/*                                                             */
/* PURPOSE:   Destructor.                                      */
/***************************************************************/
IPrivateResource :: ~IPrivateResource()
{
  IMODTRACE_ALL("IPrivateResource::~IPrivateResource");

  // Copied from IResource::~IResource
  if (reslockGate)
    delete reslockGate;

  reslockGate=0;
  if (resourceHandleCl != 0)
  {
#ifdef IC_PM
    unsigned long ulRC = 0;
    ulRC = DosCloseMutexSem( (HMTX)resourceHandleCl );
    if (ulRC != 0)
    {
      ITHROWSYSTEMERROR( ulRC,
                         "DosCloseMutexSem",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
    }
#endif
#ifdef IC_WIN
    if (IPlatform::isWin32S())
    {
      ITRACE_DEVELOP("Win32s: ~IPrivateResource() does not close semaphore");
    }
    else
    {
      if ( !CloseHandle( resourceHandleCl ) )
      {
        ITHROWGUIERROR2( "CloseHandle",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
      }
    }
#endif
  }

  ITRACE_ALL(IString("Handle is: ")+IString(handle().asUnsigned()).d2x());
  resourceHandleCl = 0;
}


/***************************************************************/
/* FUNCTION:  ISharedResource :: ISharedResource               */
/*                                                             */
/* PURPOSE:   Construct the key for a named semaphore.         */
/***************************************************************/
ISharedResource :: ISharedResource(const char* pszKeyName)
                     : keyNameCl(""), resourceHandleCl(0)
{
  IMODTRACE_ALL("ISharedResource::ISharedResource");

  unsigned long ulRC = 0;
#ifdef IC_PM
  keyNameCl += "\\SEM32\\";
  keyNameCl += pszKeyName;
  ulRC = DosCreateMutexSem( (char*)keyNameCl,
                            (PHMTX)&resourceHandleCl, 0, false );
  if ( ulRC == ERROR_DUPLICATE_NAME )
  {
    ulRC = DosOpenMutexSem( (char*)keyNameCl,
                            (PHMTX)&resourceHandleCl );
    if (ulRC != 0)
    {
      ITHROWSYSTEMERROR( ulRC,
                         "DosOpenMutexSem",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
    }
  }
  else if (ulRC != 0)
  {
    ITHROWSYSTEMERROR( ulRC,
                       "DosCreateMutexSem",
                       IBaseErrorInfo::accessError,
                       IException::recoverable );
  }
#endif
#ifdef IC_WIN
  if (IPlatform::isWin32S())
  {
    ITRACE_DEVELOP("Win32s: ISharedResource() does not create semaphore");
    return;
  }
  // Backslash is NOT allowed in semaphore path names
  keyNameCl += pszKeyName;

  // If we are running on NT, need to insure we allow sharing to
  // services (i.e. cannot use default security attributes)
  if (IPlatform::isWinNT())
  {
    SECURITY_DESCRIPTOR sd;
    SECURITY_ATTRIBUTES sa;

    InitializeSecurityDescriptor(&sd, SECURITY_DESCRIPTOR_REVISION);
    SetSecurityDescriptorDacl(&sd, TRUE, (PACL) NULL, FALSE);
    sa.nLength = sizeof(sa);
    sa.lpSecurityDescriptor = &sd;
    sa.bInheritHandle = TRUE;

    resourceHandleCl = CreateMutex( &sa, false, (char*)keyNameCl );
  }
  else
    resourceHandleCl = CreateMutex( 0, false, (char*)keyNameCl );

  if (resourceHandleCl == 0)
  {
    ulRC = GetLastError();
    ITRACE_ALL( IString("ulRC=") + IString(ulRC) );
    if (ulRC == ERROR_ALREADY_EXISTS)
    {
      resourceHandleCl = OpenMutex( MUTEX_ALL_ACCESS,
                                    true, (char*)keyNameCl );
      if (resourceHandleCl == 0)
      {
        ITHROWGUIERROR2( "OpenMutex",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
      }
    }
    else if (ulRC != 0)
    {
      ITHROWSYSTEMERROR( ulRC,
                         "CreateMutex",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
    }
  }
#endif

  ITRACE_ALL(IString("Handle is:")+IString(handle().asUnsigned()).d2x());
}


/***************************************************************/
/* FUNCTION:  ISharedResource :: ~ISharedResource              */
/*                                                             */
/* PURPOSE:   Destructor.                                      */
/***************************************************************/
ISharedResource :: ~ISharedResource()
{
  IMODTRACE_ALL("ISharedResource::~ISharedResource");

  // Copied from IResource::~IResource
  if (reslockGate)
    delete reslockGate;

  reslockGate=0;
  if (resourceHandleCl != 0)
  {
#ifdef IC_PM
    unsigned long ulRC = 0;
    ulRC = DosCloseMutexSem( (HMTX)resourceHandleCl );
    if (ulRC != 0)
    {
      ITHROWSYSTEMERROR( ulRC,
                         "DosCloseMutexSem",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
    }
#endif
#ifdef IC_WIN
    if (IPlatform::isWin32S())
    {
      ITRACE_DEVELOP("Win32s: ~ISharedResource() does not close semaphore");
    }
    else
    {
      if ( !CloseHandle( resourceHandleCl ) )
      {
        ITHROWGUIERROR2( "CloseHandle",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
      }
    }
#endif

    ITRACE_ALL( IString("Handle is:") +
                IString(resourceHandleCl.asUnsigned()).d2x() );
  }

  ITRACE_ALL(IString("Handle is: ")+IString(handle().asUnsigned()).d2x());
  resourceHandleCl = 0;
}


/***************************************************************/
/* FUNCTION: IResource :: lock                                 */
/*                                                             */
/* PURPOSE: provide ability to explicitly acquire a resouce    */
/*          lock intead of requiring x = new IResourceLock(yy) */
/***************************************************************/
IResource& IResource :: lock(long lTimeOut)
{
  IMODTRACE_ALL("IResource::lock");

  if (reslockGate == 0)
    reslockGate = new IResourceLock( *this, lTimeOut );
  else
    reslockGate->setLock( lTimeOut );

  ITRACE_ALL(IString("Use Count is: ")+IString(lClReferenceCount));
  return *this;
}


/***************************************************************/
/* FUNCTION: IResource :: unlock                               */
/*                                                             */
/* PURPOSE: provide ability to explicitly release a resouce    */
/*          lock intead of requiring delete x                  */
/***************************************************************/
IResource& IResource :: unlock()
{
  IMODTRACE_ALL("IResource::unlock");

  if (reslockGate != 0)
    reslockGate->clearLock();

  ITRACE_ALL(IString("Use Count is: ")+IString(lClReferenceCount));
  return *this;
}


/***************************************************************/
/* FUNCTION: IResourceLock :: setLock                          */
/*                                                             */
/* PURPOSE: sets (requests) the semaphore                      */
/***************************************************************/
IResourceLock& IResourceLock::setLock(long lTimeOut)
{
  IMODTRACE_ALL("IResourceLock::setLock");

  unsigned long ulRC = 0;
  {
#ifdef IC_PM
    if (presCl->handle())
    {
      ITRACE_ALL( IString("Acquiring Handle: ")+
                  IString(presCl->handle().asUnsigned()).d2x() );
      ulRC = WinRequestMutexSem( (HMTX)presCl->handle(), lTimeOut );
    }
    if (ulRC == ERROR_TIMEOUT)
    {
      ITHROWLIBRARYERROR( IC_SEMAPHORES_EXHAUSTED,
                          IBaseErrorInfo::outOfSystemResource,
                          IException::recoverable );
    }
    else if (ulRC != 0)
    {
      ITHROWSYSTEMERROR( ulRC,
                         "WinRequestMutexSem",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
    }
#endif
#ifdef IC_WIN
    if (presCl->handle())
    {
      ITRACE_ALL( IString("Acquiring Handle: ")+
                  IString(presCl->handle().asUnsigned()).d2x() );
      if (IPlatform::isWin32S())
      {
        ITRACE_DEVELOP("Win32s: setLock() does not wait on a semaphore");
        return( *this );
      }
      else
      {
        ulRC = WaitForSingleObject( presCl->handle(), lTimeOut );
      }
    }
    if ( !IPlatform::isWin32S() )
    {
      if (ulRC == WAIT_FAILED)
      {
        ITHROWGUIERROR2( "WaitForSingleObject",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
      }
      else if (ulRC == WAIT_TIMEOUT)
      {
        ITHROWLIBRARYERROR( IC_SEMAPHORES_EXHAUSTED,
                            IBaseErrorInfo::outOfSystemResource,
                            IException::recoverable );
      }
      else if (ulRC != WAIT_OBJECT_0)
      {
        ITHROWSYSTEMERROR( ulRC,
                           "WaitForSingleObject",
                           IBaseErrorInfo::accessError,
                           IException::recoverable );
      }
    }
#endif
  }
  // Only bump ref count if semaphore request was successful
  ISafeIncDec::increment(presCl->lClReferenceCount);
  ITRACE_ALL( IString("Use Count is: ")+
              IString(presCl->lClReferenceCount) );

  return( *this );
}


/***************************************************************/
/* FUNCTION:  IResourceLock :: IResourceLock                   */
/*                                                             */
/* PURPOSE:   Construct an instance of a resource lock.        */
/***************************************************************/
IResourceLock :: IResourceLock(IResource& res, long lTimeOut)
{
  IMODTRACE_ALL("IResourceLock::IResourceLock");

  this->presCl = &res;
  setLock( lTimeOut );
}


/***************************************************************/
/* FUNCTION:  IResourceLock :: ~IResourceLock                  */
/*                                                             */
/* PURPOSE:   Free the lock on the resource.                   */
/***************************************************************/
IResourceLock :: ~IResourceLock()
{
  IMODTRACE_ALL("IResourceLock::~IResourceLock");

  if ( this->presCl && this->presCl->lClReferenceCount )
    clearLock();
}


/***************************************************************/
/* FUNCTION: IResourceLock :: clearLock                        */
/*                                                             */
/* PURPOSE: clears the associated semaphore                    */
/***************************************************************/
IResourceLock& IResourceLock::clearLock()
{
  IMODTRACE_ALL("IResourceLock::clearLock");
  {
#ifdef IC_PM
    if (presCl->handle())
    {
      ITRACE_ALL( IString("Releasing Handle:")+
                  IString(presCl->handle().asUnsigned()).d2x() );
      unsigned long ulRC = 0;
      ulRC = DosReleaseMutexSem( (HMTX)presCl->handle() );
      if (ulRC != 0)
      {
        ITHROWSYSTEMERROR( ulRC,
                           "DosReleaseMutexSem",
                           IBaseErrorInfo::accessError,
                           IException::recoverable );
      }
    }
#endif
#ifdef IC_WIN
    if (presCl->handle())
    {
      ITRACE_ALL( IString("Releasing Handle:")+
                  IString(presCl->handle().asUnsigned()).d2x() );
      if (IPlatform::isWin32S())
      {
        ITRACE_DEVELOP("Win32s: clearLock() does not release semaphore");
      }
      else
      {
        if ( !ReleaseMutex( presCl->handle() ) )
        {
          ITHROWGUIERROR2( "ReleaseMutex",
                           IBaseErrorInfo::accessError,
                           IException::recoverable );
        }
      }
    }
#endif
  }
  // Only drop ref count if semaphore release was successful
  ISafeIncDec::decrement(presCl->lClReferenceCount);
  ITRACE_ALL( IString("Use Count is: ")+
              IString(presCl->lClReferenceCount) );

  return( *this );
}

